#
## <SHAREFILE=algebra/invar/invar.mpl >
## <DESCRIBE>
##        SEE ALSO: algebra/invar/invar.tex (30K), algebra/invar/invar.dvi (44K)
##
##                The invar package is a package of routines mainly for
##                computing the invariant ring of permutation groups or
##                finite linear groups over Q or an algebraic number field.
##                AUTHOR: Gregor Kemper, kemper@kalliope.iwr.uni-heidelberg.de
## </DESCRIBE>

# PROCEDURES FOR CALCULATING THE RING OF INVARIANTS

# Written by Gregor Kemper, 
# email kemper@kalliope.iwr.uni-heidelberg.de
# Version 2 (June 1993)

# This script defines the invar-package containing a bunch of procedures
# and some permutation groups


macro(alglinsolve=invar['alglinsolve'],aver_orb=invar['aver_orb'],
      showbasis=invar['showbasis'],check_nullspace=invar['check_nullspace'],
      classes=invar['classes'],derive=invar['derive'],
      elements=invar['elements'],elesym=invar['elesym'],
      generators=invar['generators'],inv=invar['inv'],
      invarspace=invar['invarspace'],invring=invar['invring'],
      lingroup=invar['lingroup'],mklingroup=invar['mklingroup'],
      monoms=invar['monoms'],monoms_base=invar['monoms_base'],
      mul=invar['mul'],onesoff=invar['onesoff'],out=invar['out'],
      poincare=invar['poincare'],poinrep=invar['poinrep'],
      showprims=invar['showprims'],primaries=invar['primaries'],
      proj=invar['proj'],reci=invar['reci'],reduce=invar['reduce'],
      showrels=invar['showrels'],relations=invar['relations'],
      renormal=invar['renormal'],repfrac=invar['repfrac'],
      represent=invar['represent'],showseconds=invar['showseconds'],
      secondaries=invar['secondaries'],throw=invar['throw'],
      valueof=invar['valueof']);


# INVRING
# The master-routine of the package which does everything and calls all
# the other routines.
# Ex.: known:=invring(permgroup({[[1,2],[3,4]]}));
# or   known:=invring(lingroup({[[a,0],[0,-a]],[[0,1],[1,0]]},a**2+1),inter);
# Include 'norels' in flags if you don't want the relations.
# Include 'inter' in flags if you want to have the possibility to interfere if
# difficulties arise when searching good primary generators!
# Set global variable protocol_level 0 or -1 if you want less or no protocol!

invring:=proc(G,flags,Known)
    local n,i,x,y,t,known,Inter,Norels,T;
    global _known;
    
    if not ( type(G,function) and
    ( op(0,G)=permgroup or
      op(0,G)=` lingroup` )) then
        ERROR(`Wrong argument type!`)
    fi;
    if op(0,G)=permgroup then readlib(group); permgroup(op(G)) fi;
    if nargs>=3 then Known:=table(); known:=Known
    else 
        if assigned(_known) then 
            if type(_known,table) then print(`Warning: _known is overwritten!`)
            else ERROR(`_known should be unassigned or a table!`)
            fi
        fi;
        _known:=table(); known:=_known
    fi;
    if op(0,G)=permgroup then known[group][permgroup]:=[op(G)]; n:=op(1,G)
    else
        known[group][geners]:=op(1,G);
        n:=nops(op(1,G)[1]);
        if nops(G)>=2 then known[group][minpol]:=op(2,G) fi
    fi;
    Inter:=0;
    if nargs>=2 then
        if has(flags,'inter') then Inter:='inter' fi;
        if has(flags,'norels') then Norels:='norels' fi
    fi;
    
    out(); out(`CALCULATING THE INVARIANT RING OF`);
    out(`   G`=G); out(); out();
    
    T:=time();
    
    for i from 1 to n do
        if assigned(x.i) then ERROR(``.(evaln(x.i)).
                    ` must be an unassigned name!`) fi;
        if assigned(y.i) then ERROR(``.(evaln(y.i)).
                    ` must be an unassigned name!`) fi;
        if assigned(s.i) then ERROR(``.(evaln(s.i)).
                    ` must be an unassigned name!`) fi
    od;
    known[xvars]:=[x.(1..n)];
    if assigned(s) then ERROR(`s must be an unassigned name!`) fi;
    known[svar]:=s;
    onesoff(s,known);
    primaries(",Inter,known);
    secondaries(",t,known);
    if Norels<>'norels' then
        relations(known)
        else out(`Calculation of relations not desired.`); out()
    fi;
    out(`Time used:`,time()-T,'seconds'); out();
    RETURN()
end:


# LINGROUP
# check if a linear group is well defined

lingroup:=proc(gens,minpol)
    local n,ele,row,alg;
    
    if nargs>=2 then
        indets(minpol);
        if nops(")<>1 then ERROR(`Bad second argument!`) fi;
        alg:="[1];
        if not type(minpol,polynom(rational,alg)) then
                                ERROR(`Bad second argument!`) fi
    fi;
    if ( not type(gens,set(listlist)) or nops(gens)=0 )then
                                    ERROR(`Bad first argument!`) fi;
    n:=nops(gens[1]);
    for ele in gens do
        if nops(ele)<>n then ERROR(`Bad first argument!`) fi;
        for row in ele do
            if nops(row)<>n then ERROR(`Bad first argument!`) fi
        od;
        if not type(ele,list(list(polynom(rational,alg)))) then
                                        ERROR(`Bad first argument!`) fi
    od;
    '` lingroup`(args)'
end:
    

# OUT
# Output of a piece of protocol

out:=proc()
    local lev,narg,pl;
    
    if nargs>0 and type(args[1],integer) then
        lev:=args[1]; narg:=[args[2..nargs]]
    else lev:=1; narg:=[args]
    fi;
    if assigned(protocol_level) then pl:=protocol_level else pl:=1 fi;
    if pl<lev then RETURN() fi;
    if nops(narg)>0 and narg[1]=print then print(op(narg[2..nops(narg)]))
    else lprint(narg[])
    fi;
    RETURN()
end:


# MKLINGROUP
# If G is given as a permutation group, it is transformed into a matrix-group,
# matrices being (here) lists of lists.

mklingroup:=proc(G)
local n,g,conv;
    
    conv:=proc(p,n)
        local i,pl;
        pl:=convert(p,permlist,n);
        array(sparse,1..n,1..n,[seq((pl[i],i)=1, i=1..n)]);
        convert(",listlist)
    end;
    
    if nargs>=1 then g:=G else g:=_known[group] fi;
    if not type(g,table) then ERROR(`Wrong argument-type!`) fi;
    if not assigned(g[geners]) then
        if type(g[permgroup],function) then g[permgroup]:=[op(g[permgroup])]
        elif type(g[permgroup],list) then
        else ERROR(`G[permgroup] is not what it should be!`)
        fi;
        n:=g[permgroup][1]; readlib(group);
        g[geners]:=(map(conv,g[permgroup][2],n));
        if assigned(g[eles]) then g[eles]:=map(conv,g[eles],n) fi
    elif assigned(g[degree]) and assigned(g[identity]) then RETURN()
    else n:=nops(g[geners][1])
    fi;
    g[degree]:=n; g[identity]:=convert(array(identity,1..n,1..n),listlist);
    if assigned(g[minpol]) then g[alg]:=indets(g[minpol])[] fi;
    RETURN()
end:


# REDUCE
# reduce a polynomial in an algebraic number

reduce:=proc(expr,G)
    if nargs>=2 then G else _known[group] fi;
    rem(expr,"[minpol],"[alg])
end:


# MUL
# a times b; they might contain an algebaic number!

mul:=proc(a,b,minpol)
    local i,j,k,n,res;
    option remember;
    
    n:=nops(a);
    res:=[seq([seq(expand(sum('a[i][k]*b[k][j]', k=1..n)), j=1..n)], i=1..n)];
    if nargs>=3 then
        indets(minpol)[];
        if has(res,") then res:=[seq(map(rem,res[i],minpol,"), i=1..n)] fi
    fi;
    RETURN(res)
end:


# ELEMENTS
# assigns to G[eles] a set of all elements of G.

elements:=proc(G)
    local x,g,mipo;
    
    if nargs>=1 then g:=G else g:=_known[group] fi;
    if assigned(g[eles]) then RETURN() fi;
    out(`Calculating all elements of the group G ...`);
    if assigned(g[permgroup]) then                  # can take a shortcut!
        if type(g[permgroup],function) then
                g[permgroup]:=[op(g[permgroup])] fi;
        readlib(group);
        g[eles]:=group[cosets](permgroup(g[permgroup][]),
                        permgroup(op(1,g[permgroup]),{}))
    fi;
    mklingroup(g);
    if assigned(g[minpol]) then mipo:=g[minpol] else mipo:=NULL fi;
    if not assigned(g[eles]) then
        g[eles]:={g[identity]}; g[order]:=0;
    
        while g[order]<nops(g[eles]) do
            g[order]:=nops(g[eles]);
            for x in g[geners] do
                g[eles]:=g[eles] union map(mul,g[eles],x,mipo)
            od
        od
    else g[order]:=nops(g[eles])
    fi;
    out(`It has `.(g[order]).` elements.`); out();
    out(2,`They are`); out(2,print,map(array,g[eles])[]);
    RETURN()
end:


# RECI
# 1/x, where x is a polynomial in an algebraic number

reci:=proc(x,minpol)
    local alg,ret;
    option remember;
    
    if nargs<2 then RETURN(1/x) fi;
    alg:=indets(minpol)[];
    if not has(x,alg) then RETURN(1/x) fi;
    if gcdex(x,minpol,alg,'ret')<>1 then ERROR(`Zero-division`) fi;
    ret
end:


# INV
# The inverse of the matrix x

inv:=proc(x,G)
    local g,res,A;
    
    if nargs>=2 then g:=G else g:=_known[group] fi;
    if assigned(g[inverses][x]) then g[inverses][x]
    else
        A:=array(x);
        res:=linalg[inverse](A);
        if has(eval(res),g[alg]) then
            linalg[det](A);
            "*reci(",g[minpol]);
            res:=map((x,d,g)->reduce(normal(d*x),g),res,",g)
        fi;
        g[inverses][x]:=convert(res,listlist)
    fi
end:    


# CLASSES
# calculates the conjugacy-classes of G and writes them into G[ccls],
# returns an element of each class.

classes:=proc(G)
    local still,res,ladder,class,g,mipo;
    
    if nargs>=1 then g:=G else g:=_known[group] fi;
    if not assigned(g[ccls]) then
        elements(g);
        out(`Calculating the conjugacy classes of G ...`);
        if assigned(g[minpol]) then mipo:=[g[minpol]] else mipo:=[] fi;
        still:=g[eles]; res:=NULL;
        ladder:=map((x,g)->[inv(x,g),x],g[eles],g);
        while still<>{} do
            class:=map((a,x,m)->mul(a[1],mul(x,a[2],m[]),m[]),
                                        ladder,still[1],mipo);
            res:=res,class;
            still:=still minus class
        od;
        g[ccls]:=sort([res],(a,b)->evalb(nops(a)<nops(b)));
        g[classlengths]:=map(nops,g[ccls]);
        out(`G has conjugacy classes of lengths`,g[classlengths]); out();
        out(2,`They are`);
        for class in g[ccls] do out(2,print,map(array,class)[]) od
    fi;
    map(x->x[1],g[ccls])
end:


# THROW
# applies a matrix to an expression in the variables 'vars'.
# CAVEAT: In literature, throwing a linear map s to a polynomial functoin f is 
# often defined as: sf(x):=f(s^(-1)(x)). Here, we take the transpose s^t and 
# define sf(x):=f(s^t(x)). These are isomorphic ways to have a linear group 
# operate on the space of polynomial functions.

throw:=proc(expr,mat,o_and_n_vars,Known)
    local ex,i,j,ovars,nvars,known,mipo,alg;
    
    if nargs<4 or type(Known,table) then
        if nargs>=4 then known:=Known else known:=_known fi;
        if nargs>=3 and type(o_and_n_vars,list) then
            ovars:=o_and_n_vars[1]; nvars:=o_and_n_vars[2]
        else ovars:=known[vars]; nvars:=ovars
        fi;
        if assigned(known[group][minpol]) then known[group][minpol] else 0 fi;
        RETURN(throw(expr,mat,[ovars,nvars],"))
    fi;
    ex:=expand(expr);
    mipo:=Known; if "<>0 then alg:=indets(")[] fi;
    if not has([ex,mat],alg) then
        ovars:=o_and_n_vars[1]; nvars:=o_and_n_vars[2];
        expand(subs([seq(ovars[i]=sum('mat[j][i]*nvars[j]', j=1..nops(nvars)), 
                                                        i=1..nops(ovars))],ex))
    elif type(ex,`+`) then map(throw,ex,args[2..4])
    elif type(ex,`*`) then
        throw(convert([op(2..nops(ex),ex)],`*`),args[2..4]);
        expand(throw(op(1,ex),args[2..4])*");
        if has(",alg) then expand(rem(",mipo,alg)) fi;
        "
    elif type(ex,`^`) then
        throw(op(1,ex)^(op(2,ex)-1),args[2..4]);
        expand(throw(op(1,ex),args[2..4])*");
        if has(",alg) then expand(rem(",mipo,alg)) fi;
        "
    else
        ovars:=o_and_n_vars[1]; nvars:=o_and_n_vars[2];
        subs([seq(ovars[i]=sum('mat[j][i]*nvars[j]', j=1..nops(nvars)), 
                                                    i=1..nops(ovars))],ex)
    fi;
    
    throw(args):="
end:


# AVER_ORB
# ('char'^(-1))-weighted sum of s out of known[group] applied to 'expr'.
# This is a pseudo-invriant of weight 'char'!

aver_orb:=proc(expr,char,Known)
    local chi,i,j,known;
    
    if nargs>=3 then known:=Known else known:=_known fi;
    if nargs>=2 then chi:=char else chi:=1 fi;
    if type(chi,list) then
        classes(known[group]);
        sum('chi[i]*sum('throw(expr,
            inv(known[group][ccls][i][j],known[group]),0,known)',
            j=1..known[group][classlengths][i])',
            i=1..nops(known[group][ccls]))
    else
        elements(known[group]);
        sum('throw(expr,inv(known[group][eles][i],known[group]),0,known)',
                                                    i=1..known[group][order])
    fi;
    expand(");
    if "<>0 then "/icontent(") fi;
    "
end:


# INVARSPACE
# A basis for the subspace of all pseudo-invariants of weight 'char'
# (default 1) and degree 'deg'. If the group comes from a permgroup, the
# projection of the deg'th elementary symmetric polynomial (if nonzero) is
# taken as first basis polynomial.
# 'char' can be either a set or list of equations relating the generators of
# the group to the character-values, or a list of character-values
# corresponding to the conjugacy classes.

invarspace:=proc(deg,char,Known)
    local i,n,alfa,ans,unknowns,pars,chi,res,I,mons,eq,known;
    
    if nargs>=3 then known:=Known else known:=_known fi;
    mklingroup(known[group]);
    
    if nargs>=2 and char<>1 then
        if type(char,{list(`=`),set(`=`)}) and 
        {map(lhs,char)[]}=known[group][geners] then chi:=char
        elif type(char,list(polynom(rational,known[group][alg]))) then
            proc(mat,known)
                local k;
                member(true,map((c,mat)->member(mat,c),
                            known[group][ccls],mat),k);
                k
            end;
            chi:=map((mat,char,pro,known)->(mat=char[pro(mat,known)]),
                                known[group][geners],char,",known)
        else ERROR(`Third argument is not what it should be!`)
        fi
    else chi:=map(x->(x=1),known[group][geners])
    fi;
    if assigned(known[group][invspace][deg,chi]) then
        RETURN(known[group][invspace][deg,chi])
    fi;
    
    if nargs>=2 and char<>1 then `pseudo-` else `` fi;
    out(`Calculating the space of `.".`invariants of degree `.deg.`.`);
    
    mons:=monoms(deg,map(x->[x,1],known[vars]));
    if known[trans][permgroup]=true then
        # smuggle in an elementary symmetric polynomial!
        proj(elesym(deg,known[xvars]),known);
        if "<>0 then
            coeffs(",known[vars],'res');
            mons:=[["",deg],op({mons[]} minus {[op(1,[res]),deg]})]
        fi
    fi;
    ans:=sum('alfa[i]*mons[i][1]', i=1..nops(mons));
    unknowns:={seq(alfa[i], i=1..nops(mons))};
    # Condition for ans being an invariant
    eq:=map((mat,expr,chi,known)->coeffs(expand(throw(expr,mat,0,known)-
        subs(chi,mat)*expr),known[vars]),
        known[group][geners],ans,chi,known);
    out(2,`Solving a system of `.(nops(")).
            ` linear equations in `.(nops(unknowns)).` unknowns ...`);
    if has(eq,known[group][alg]) then
        alglinsolve(eq,unknowns,known[group][minpol])
    else
        readlib(`solve/linear`)(eq,unknowns)
    fi;
    
    ans:=expand(subs(",ans));
    pars:=indets(ans) intersect unknowns;
    if member(alfa[1],pars) then pars:=[alfa[1],op(pars minus {alfa[1]})]
                            # true if an elementary symmetric was smuggled in!
    else pars:=[pars[]]
    fi;
    n:=nops(pars);
    
    [seq(subs(pars[i]=1,map(x->(x=0),pars),ans), i=1..n)];
    res:=map(f->f/icontent(f),");
    for i from 1 to n do
        if assigned(I.deg._.i) then
            ERROR(``.(evaln(I.deg._.i)).` must be an unassigned name!`)
        fi
    od;
    if n=0 then `is none!`
    elif n=1 then `is one linearly independent invariant ("I`.deg.`_1").`
    else `are `.n.` linearly independent invariants ("I`.deg.
                                    `_1" through "I`.deg._.n.`").`
    fi;
    out(`There `.");
    if n>0 then
        if n>1 then out(2,`They are`) else out(2,`It is`) fi;
        for i from 1 to n do out(2,`   I`.deg._.i=res[i]) od
    fi;
    out();
    known[group][invspace][deg,chi]:=res
end:


# MONOMS
# all products of elements of 'polys_with_degrees' of degree 'deg'

monoms:=proc(deg,polys_with_degrees)
    local d,f,res,factor,reduced,pwd;
    option remember;
    
    if nargs>=2 then pwd:=polys_with_degrees
    else pwd:=map(x->[x,1],_known[vars])
    fi;
    if nops(pwd)=0 then
        if deg=0 then RETURN([[1,0]])
                 else RETURN([])
        fi
    fi;

    f:=pwd[1];
    reduced:=pwd[2..nops(pwd)];
    if f[2]=0 then RETURN(monoms(deg,reduced)) fi;    
                                                    # avoiding infinite loop!
    res:=NULL; d:=0; factor:=1;

    while d<=deg do
        monoms(deg-d,reduced);                      # recursion!!
        res:=res,map((f,fac,deg)->[fac*f[1],deg],",factor,deg)[];
        d:=d+f[2]; factor:=factor*f[1]
    od;
    [res]
end:


# MONOMS_BASE
# a Q-basis for the subspace of polynomials of degree 'deg' in the 
# free module with basis 'base' over the polynomial ring generated 
# by 'indep'

monoms_base:=proc(deg,indep,base)
    local f,res;
    option remember;

    res:=NULL;
    for f in base do
        deg-f[2];       
        if ">=0 then 
            monoms(",indep);
        res:=res,map((f,fac,deg)->[f[1]*fac,deg],",f[1],deg)[]
        fi
    od;
    [res]
end:


# ELESYM
# the 'k'-th elementary symmetric function in variables 'var'

elesym:=proc(k,Var)
    local n,i,res,var;
    
    if nargs>=2 then var:=Var else var:=_known[xvars] fi;
    if k=0 then RETURN(1) fi;
    res:=0; n:=nops(var);
    for i from 1 to n-k+1 do
        res:=res+var[i]*elesym(k-1,var[i+1..n])
    od;
    expand(res)
end:


# ONESOFF
# splits off unit-characters from known[group]

onesoff:=proc(s_name,Known)
    local i,k,n,d,pr,copr,S,T,bc,sname,s,G,select_base, known;
    
    select_base:=proc(A,d,known)
        local S,i,j,k,x,n,cand;
        n:=known[group][degree]; i:=0;
        for j while i<d do
            cand:=linalg[submatrix](A,1..n,[j]);
            icontent(sum('cand[k,1]*x[k]', k=1..n));
            if "<>0 then map((v,d)->v/d,cand,") else cand fi;
            if assigned(S) then linalg[concat](S,") else " fi;
            if linalg[kernel](subs(known[group][alg]=
                                RootOf(known[group][minpol]),"))={} then
                S:=";
                i:=i+1
            fi
        od;
        op(S)
    end;
    
    if nargs>=2 then known:=Known else known:=_known fi;
    if nargs>=1 then sname:=s_name else sname:=s fi;
    elements(known[group]);
    known[trans][permgroup]:=assigned(known[group][permgroup]);
    out(0,`Splitting off unit-characters ...`);
    
    n:=known[group][degree];
    pr:=evalm(convert(map(array,known[group][eles]),`+`));
    # |G| times a G-invariant projection onto the space fixed by G
    
    d:=linalg[trace](pr)/known[group][order];       # dimension of fixed space
    known[prim]:=[seq([``.sname.i,1,'outer'], i=1..d)];
    known[values]:=NULL;
    
    if d=0 then
        known[vars]:=known[xvars];
        known[trans][newbasis]:=known[group][identity];
        known[trans][proj]:=";
        out(`There is none.`); out();
        RETURN([sname,0])
    fi;
    
    copr:=evalm(known[group][order]-pr); # |G| times complementary projection
    
    # Now choose independent columns to form transition matrix
    d:=n-d;
    S:=linalg[concat](select_base(copr,d,known),select_base(pr,n-d,known));
    T:=array(inv(convert(S,listlist),known[group]));
    
    out(); out(`"Outer" primary generators:`);
    for i from 1 to n-d do
        sum('S[k,d+i]*known[xvars][k]', k=1..n);
        known[values]:=known[values],``.sname.i=";
        out(`   `.sname.i="")
    od;
    out();
    
    S:=linalg[submatrix](S,1..n,1..d);
    T:=linalg[submatrix](T,1..d,1..n);
    known[trans][newbasis]:=convert(S,listlist);
    known[trans][proj]:=convert(T,listlist);
    
    # make new linear group
    bc:=proc(m,G,s,t)
        evalm(t&*array(m)&*s);
        if has(",G[alg]) then map(reduce,",G) fi;
        convert(",listlist)
    end;
    
    if assigned(known[group][minpol]) then
        G[minpol]:=known[group][minpol]; G[alg]:=known[group][alg]
    fi;
    G[geners]:=map(bc,known[group][geners],G,S,T);
    G[eles]:=map(bc,known[group][eles],G,S,T);
    G[order]:=known[group][order];
    mklingroup(G);
    known[group]:=G;
    
    known[vars]:=[y.(1..d)];
    out(`New variables:`);
    for i to d do
        known[vars][i];
        out(`   `."=renormal(",known))
    od;
    out(2); out(2,`Action of G on new variables generated by`);
    out(2,print,map(array,known[group][geners])[]);
    out(`Done with splitting off unit-characters!`); out();
    RETURN([sname,n-d])
end:


# PROJ
# projects 'expr' to the non-1-subspace and expresses it in the y-variables.

proj:=proc(expr,Known)
    local known;
    
    if nargs>=2 then known:=Known else known:=_known fi;
    throw(expr,known[trans][proj],[known[xvars],known[vars]],known)
end:
    

# RENORMAL
# expresses 'expr' in the x-variables.

renormal:=proc(expr,Known)
    local known;
    
    if nargs>=2 then known:=Known else known:=_known fi;
    throw(expr,known[trans][newbasis],[known[vars],known[xvars]],known)
end:


# POINCARE
# The "weighted poincare-series" of G with character 'char' (default 1);
# 'char' given as a list corresponding to G[ccls].

poincare:=proc(char,svar,G)
    local chi,S,i,g,mipo;
    
    if nargs>=3 then g:=G else g:=_known[group] fi;
    if nargs>=1 then chi:=char else chi:=1 fi;
    if type(chi,list) then classes(g)
    else elements(g)
    fi;
    if nargs>=2 and type(svar,name) then S:=svar else S:='s' fi;
    if assigned(g[poinseries][chi]) then
        g[poinseries][chi];
        RETURN(subs(indets(")[1]=S,"))
    fi;
    if assigned(g[minpol]) then mipo:=[g[minpol]] else mipo:=[] fi;
    out(`Calculating the poincare-series ...`);
    if assigned(g[ccls]) then
        if chi=1 then chi:=[1 $ nops(g[ccls])] fi;
        [seq(chi[i]*g[classlengths][i]*
            reci(linalg[det](evalm(&*()-S*array(g[ccls][i][1]))),mipo[]), 
                                                    i=1..nops(g[ccls]))];
        normal(convert(",`+`)/g[order])
    else
        map((mat,mipo,S)->reci(linalg[det](evalm(&*()-S*array(mat))),mipo[]),
                                                    [g[eles][]],mipo,S);
        normal(convert(",`+`)/g[order])
    fi;
    if has(",g[alg]) then
        {coeffs(reduce(expand(numer(")-i*denom(")),g),g[alg])};
        normal(subs(solve(",i),i))
    fi;
    i:=";
    out(`It is`); out(print,'P'(S)=i); out();
    g[poinseries][chi]:=i
end:


# POINREP
# The best representation of the poincare-series relevant to invariant theory 
# with exponents > last_exp[1][1],...,last_exp[1][G[degree]] (default [0...0]).

poinrep:=proc(last_exp,Known)
    local n,i,nvec,num,next,pnum,pden,known;
    
    next:=proc(vec)
        local i,j,n,r,d,q;
        n:=nops(vec);
        if vec[1]=0 then RETURN([1$n]) fi;
        d:=sum('vec[j]', j=1..n);
        for i from n-1 by -1 to 1 do
            if i=1 then 1 else vec[i-1] fi;
            if vec[i]>" then
                q:=iquo(d-sum('vec[j]', j=1..i)+1,n-i,'r');
                RETURN([op(vec[1..i-1]),vec[i]-1,q $ n-i-r,q+1 $r])
            fi
        od;
        q:=iquo(d+1,n,'r');
        [q $ n-r,q+1 $r]
    end;
    
    if nargs>=2 then known:=Known else known:=_known fi;
    poincare(1,known[svar],known[group]);
    pnum:=numer("); pden:=denom("");
    n:=known[group][degree];
    if nargs>=1 and type(last_exp,list) then
            nvec:=last_exp[1] else nvec:=[0 $n] fi;
    do
        nvec:=next(nvec);
        if divide(pnum*product('1-known[svar]^nvec[i]', i=1..n),pden,'num')
        and not member(true,map(x->evalb(x<0),[coeffs(num,known[svar])]))
        then
            out(`Try the representation`);
            out(print,'P'(known[svar])=
                    num/product('1-known[svar]^nvec[i]', i=1..n));
            out();
            RETURN([nvec,[seq(i $ coeff(num,known[svar],i), 
                                    i=0..degree(num,known[svar]))]])
        fi
    od
end:


# PRIMARIES
# Finds primary generators. If Inter='inter', 'primaries' might ask the user
# what to do.

primaries:=proc(s_name,Inter,Known)
    local n,i,j,k,poin,max,cand,pattern,s,mat,base,trials,idea,I,i0,known;
    
    if nargs>=3 then known:=Known else known:=_known fi;
    if nargs>=1 and type(s_name,list) then s:=s_name[1]; i0:=s_name[2]
    else i0:=0
    fi;
    out(0,`Finding primary generators ...`); out();
    #readlib(select);
    poin:=0; n:=known[group][degree];
    
    do              # try to make the different representations of P(s) real
        poin:=poinrep(poin,known);
        max:=poin[1][n];
        pattern:=[seq(nops(select((x,y)->x=y,poin[1],i)), i=1..max)];
        mat:=NULL; base:=NULL;
        for i to max do
            if pattern[i]>0 then
                invarspace(i,1,known);
                mat:=mat,[pattern[i],nops(")];
                base:=base,[seq(I.i._.j=""[j], j=1..nops(""))];
            fi
        od;
        mat:=[mat]; base:=[base];
        
        for trials do
            if assigned(idea) then idea
            else
                mat:=`primaries/nextmat`(mat);
                if "=`no more!` then
                    out(`No more linearly independent choices!`);
                    mat:=0; break
                fi;
                [seq(seq(sum('mat[i][j,k]*lhs(base[i][k])',k=1..nops(base[i])), 
                                j=1..linalg[rowdim](mat[i])), i=1..nops(mat))]
            fi;
            cand:=subs(base[],");
            out(`Trying invariants`,seq(``.s.(i+i0)=""[i], i=1..n));
            idea:='idea';
        
            # Checking if cand is a good list of primaries ...
            if known[trans][permgroup]=true and
                    poin[1]=[seq(i, i=2..nops(known[xvars]))] then
                out(`They come from the elementary-symmetric functions:`);
                out(`No further checking necessary!!`);
                break
            # Have to check for zeroes<>(0..0)
            elif check_nullspace(cand,known) then
                out(`OH HAPPY DAY: They qualify!! `.
                                `(No common zeroes other than [0,..,0])`);
                break
            else
                out(
                `Not so good: There are common zeroes other than (0,..,0))`);
                if nargs>=2 and Inter='inter' then
                    lprint();
                    lprint(`'invar' is troubled and seeks counsel: Shall it`);
                    lprint(`- proceed as before and try another combination `.
                            `(type ";" and RETURN)`);
                    lprint(`- give up and pass to the next representation of `.
                                `the poincare-series`);
                    lprint(`     (type "hopeless;" and RETURN)`);
                    lprint(`- try a combination of Ii_j's of your desire `.
                            `(type it in with ";" and RETURN)`);
                    [readstat(`So ...> `)];
                    if "=[] then
                    elif "=['hopeless'] then mat:=0; break
                    else idea:="
                    fi
                elif trials>=10 then mat:=0; break
                fi
            fi
        od;
            
        if mat=0 then
            out(`WEEP AND MOURN: `.
                `Pass to next representation of the poincare-series ...`)
        else break
        fi
    od;
    if assigned(known[prim]) then known[prim] else [] fi;
    known[prim]:=["[],seq([``.s.(i+i0),poin[1][i]], i=1..n)];
    known[used]:={cand[]};
    if not assigned(known[values]) then known[values]:=NULL fi;
    for i from 1 to n do
        known[values]:=known[values],``.s.(i+i0)=cand[i];
    od;
    out(); out(0,`The primaries are (by their symbolic names):`);
    for i in known[prim] do
        if nops(i)=2 then out(0,`   `.(i[1]).` of degree `.(i[2]))
        else out(0,`   `.(i[1]).` of degree `.(i[2]).`  (outer primary)`)
        fi
    od;
    out(0); out(`Done with finding primary generators`); out();
    RETURN(poin[2])
end:


# CHECK_NULLSPACE
# checks whether the members of 'ideal' have no common zero other than (0..0)

check_nullspace:=proc(ideal,Known)
    local n,i,temp,known,var;
    
    if nargs>=2 then known:=Known else known:=_known fi;
    if assigned(known[vars]) then var:=known[vars]
    else var:=[indets(ideal)[]]
    fi;
    n:=nops(var);
    if has(ideal,known[group][alg]) then temp:=[ideal[],known[group][minpol]]
    else temp:=ideal
    fi;
    for i to n do
        subs(var[i]=1,temp);
        if {"[]}={0} then RETURN(false) fi;
        readlib(`grobner/gsolve`)(")=[];
        if " then temp:=subs(var[i]=0,temp)     # no zero with var[i]<>0!
        else RETURN(false)
        fi
    od;
    true
end:


# PRIMARIES/NEXTMAT
# Service-routine for 'primaries':
# Succession of lists of matrices for building combinations.

`primaries/nextmat`:=proc(last)
    local i,j,k,vec,newmat,ans,a,upsidedown,isregular,vectomat;
    global nextmat_remember;
    
    upsidedown:=proc(A)
        local i;
        linalg[rowdim](A);
        linalg[submatrix](A,[seq("-i, i=0.."-1)],1..linalg[coldim](A))
    end;
    
    isregular:=
        A->evalb(linalg[coldim](A)-nops(linalg[kernel](A))=linalg[rowdim](A));
    
    vectomat:=proc(pat,vec)
    global nextmat_remember;
        if pat[1]=pat[2] then
            array(identity,1..pat[1],1..pat[1]);
            RETURN(array(convert(",listlist)))
        fi;
        nextmat_remember[point];
        nextmat_remember[point]:="+pat[1]*pat[2];
        linalg[matrix](pat[1],pat[2],vec["".."-1])
    end;
        

    if type(last,listlist) then                     # Start new sequence
        readlib(`solve/linear`);
        nextmat_remember[form]:=[seq(last[nops(last)-i], i=0..nops(last)-1)];
        [seq(array([seq([seq(a[i,j,k], k=1.."[i][1])], j=1.."[i][1])]), 
                                                            i=1..nops("))];
        nextmat_remember[multipier]:=
            subs('form'=",mat->zip((x,y)->evalm(x&*y),form,mat));
        proc(par)
            local i;
            [seq([0$(par[1]-i),1,0$(par[2]-par[1]+i-1)], i=1..par[1])]
        end;
        map(",nextmat_remember[form]); map(array,");
        nextmat_remember[yet]:={nextmat_remember[multipier](")};
        RETURN(map(upsidedown,[seq(""[nops("")-i], i=0..nops("")-1)]))
    fi;
    
    if {map(x->evalb(x[1]=x[2]),nextmat_remember[form])[]}={true} then 
        RETURN(`no more!`)
    fi;
    
    proc(mat)
        if linalg[rowdim](mat)=linalg[coldim](mat) then RETURN() fi;
        convert(mat,listlist);
        op(map(op,"))
    end;
    vec:=map(",map(upsidedown,[seq(last[nops(last)-i], i=0..nops(last)-1)]));
        
    do                                                  # Find next good one!
        vec:=`primaries/nextvec`(vec);
        nextmat_remember[point]:=1;
        newmat:=map(vectomat,nextmat_remember[form],vec);
        
        if member(false,map(isregular,newmat)) then next fi;
        
        for ans in nextmat_remember[yet] do
            map(op,{seq(entries(linalg[matadd](newmat[i],ans[i])), 
                                            i=1..nops(newmat))});       
            [`solve/linear`(",indets("))];
            if "<>[] then newmat:=0; break fi
        od;
        if newmat<>0 then 
            nextmat_remember[yet]:=nextmat_remember[yet] union
                            {nextmat_remember[multipier](newmat)};
            RETURN(map(upsidedown,[seq(newmat[nops(newmat)-i], 
                                            i=0..nops(newmat)-1)]))
        fi
    od
end:


# PRIMARIES/NEXTVEC
# A service-routine to 'primaries/nextmat':
# yields a nice succession of all vectors over the integrals if you do
# `primaries/nextvec`([0,0,0,0]): `primaries/nextvec`("): 
# `primaries/nextvec`(") ... or alike.

`primaries/nextvec`:=proc(vec)
    local n,i,new;
    
    n:=nops(vec);
    new:=array(vec);
    for i from 1 to n do
        if new[i]>0 then new[i]:=-new[i]; RETURN(convert(new,list))
        else new[i]:=-new[i]
        fi
    od;
    for i from 1 to n-1 do
        if new[i]>0 then
            new[1]:=new[i]-1;
            if i>1 then new[i]:=0 fi;
            new[i+1]:=new[i+1]+1;
            RETURN(convert(new,list))
        fi
    od;
    new[1]:=new[n]+1;
    new[n]:=0;
    RETURN(convert(new,list))
end:


# REPRESENT
# represents 'expr' in terms of the Cohen-Macaulay-basis contained in 'known'.
# Core-routine of the package!

represent:=proc(expr,Known)
    local alfa,n,i,deg,terms,ansatz,leftside,rightside,equ,sol,unknowns,known;
    
    if nargs>=2 then known:=Known else known:=_known fi;
    deg:=expr[2];

    if expr[1]=0 then RETURN(0)
    elif has(expr[1],known[vars]) then expand(subs(known[vars][1]=1,expr[1]))
                                        # expr is already evaluated
    else `represent/evalprod`(expr,known)
    fi;
    if indets(") minus {known[vars][],known[group][alg]} <> {} then
            ERROR(`Expression contains outer primaries or x-variables!`) fi;
    rightside:=";
    
    if (type(known[leftsides][deg],list) and 
            known[leftsides][deg][1]=nops(known[base])) then                                                        
                                                  # left side known!
        unknowns:=known[leftsides][deg][2];
        ansatz:=known[leftsides][deg][3];
        leftside:=known[leftsides][deg][4];
        n:=nops(unknowns)
    else
        select(p->nops(p)=2,known[prim]);       # leave out outer primaries     
        terms:=monoms_base(deg,",known[base]);
        n:=nops(terms);
        if n=0 then RETURN('FAIL') fi;
        unknowns:=[seq(alfa[i], i=1..n)];
        ansatz:=convert([seq(unknowns[i]*terms[i][1], i=1..n)],`+`);
        leftside:=expand(convert([seq(unknowns[i]*
                        `represent/evalprod`(terms[i],known), i=1..n)],`+`))
    fi;

    equ:={coeffs(leftside-rightside,known[vars][2..known[group][degree]])};
    out(`Solving a system of `.(nops(equ)).` linear equations in `.
                                                n.` unknowns ...`);
    if assigned(known[group][minpol]) then
        sol:=[alglinsolve(equ,{unknowns[]},known[group][minpol])]
    else sol:=[alglinsolve(equ,{unknowns[]})]
    fi;

    if nops(sol)=0 then RETURN('FAIL') fi;
    if nops(sol)>1 then RETURN(`MULTIPLE SOLUTIONS!`) fi;
    for equ in op(sol) do              # still a bunch of solutions!
        if has(rhs(equ),unknowns) then 
            RETURN(`MULTIPLE SOLUTIONS!`)
        fi 
    od;
    known[leftsides][deg]:=
                [nops(known[base]),unknowns,ansatz,leftside];
                         # remember, since this will turn up again!
    subs(op(sol),ansatz);
end:


# ALGLINSOLVE
# solves a system of linear equations, possibly over a number field given by 
# 'minpol'. Presumably quicker than solve/linear/algnum.

alglinsolve:=proc(equations,unknowns,Minpol)
local i,k,t,eqn,eqns,var,vars,pivot,sol,sols,a,ans,x,minpol;
    
    readlib(`solve/linear`);
    if nargs>=3 then minpol:=Minpol
    else if assigned(_known[group][minpol]) then
                        minpol:=_known[group][minpol] fi
    fi;
    a:=indets(minpol)[];
    if not ( assigned(minpol) and has(equations,a) ) then 
        RETURN(`solve/linear`(equations,unknowns))
    fi;
    ans:=sum('x[i]*a**i', i=0..degree(minpol,a)-1);
    eqns := equations minus {0};
    for k while eqns <> {} do
        if 2 < printlevel then
            lprint(`solve: linear: # equations is:`,nops(eqns))
        fi;
        eqn := eqns[1];
        for i from 2 to nops(eqns) do
            if length(eqns[i]) < length(eqn) then eqn := eqns[i] fi
        od;
        vars := indets(eqn) intersect unknowns;
        if vars = {} then RETURN() else var := vars[1] fi;
        eqns := eqns minus {eqn};
        pivot := coeff(eqn,var,1);
        for i from 2 to nops(vars) do
            t := coeff(eqn,vars[i],1);
            if length(t) < length(pivot) then pivot := t; var := vars[i] fi
        od;
        if 3 < printlevel then
            lprint(`solve: linear: best unknown/equation`,var,eqn)
        fi;
        if type(pivot,integer) then pivot:=1/pivot
        else
            {coeffs(rem(ans*pivot-1,minpol,a),a)};
            `solve/linear`(",{seq(x[i-1], i=1..degree(minpol,a))});
            pivot:=subs(",ans)
        fi;
        eqn := -expand(rem(pivot*subs(var = 0,eqn),minpol,a));
        sol[k] := var,eqn;
        proc(x,m,a)
            expand(rem(x,m,a));
            if "<>0 then "/icontent(") fi
        end;
        eqns:=map(",subs(var = eqn,eqns),minpol,a) minus {0}
    od;
    
    sols:={};
    for i from k-1 by -1 to 1 do
        if 2 < printlevel then
            lprint(`solve: linear: backsubstitution at:`,i)
        fi;
        var := sol[i][1];
        eqn := sol[i][2];
        eqn := expand(rem(subs(sols,eqn),minpol,a));
        sols := sols union {var = eqn}
    od;
    map(x->x=x,unknowns minus map(lhs,sols)) union sols
end:


# REPRESENT/EVALPROD
# evaluates a symbolic product.

`represent/evalprod`:=proc(expr,Known)
    local known;
    
    if nargs>=2 then known:=Known else known:=_known fi;
    subs(known[vars][1]=1,[known[values]]);
    if assigned(known[group][minpol]) then
        expand(reduce(subs(",expr[1]),known[group]))
    else
        expand(subs(",expr[1]),known[group])
    fi
end:


# REPFRAC
# Represent a homogeneous rational invariant in terms of a Cohen-Macaulay basis
# stored in 'known'. 'expr' is just the invariant, without degree.

repfrac:=proc(expr,Known)
    local i,numden,what,deg,ans,alfa,unknowns,known;
    
    if nargs>=2 then known:=Known else known:=_known fi;
    normal(expr);
    if indets(") minus {known[vars][],known[group][alg]} <> {} then
            ERROR(`Expression contains outer primaries or x-variables!`) fi;
    numden:=[numer("),denom(")];
    {known[vars][]};
    if degree(""[1],")<degree(""[2],") then what:=1 else what:=2 fi;
    
    # find a factor to extend numer or denom to an invariant
    for deg from 0 do
        monoms(deg,map(x->[x,1],known[vars]));
        ans:=sum('alfa[i]*"[i][1]', i=1..nops("));
        unknowns:={seq(alfa[i], i=1..nops(""))};
        expand(ans*numden[what]);
        map((mat,expr,known)->coeffs(expand(throw(expr,mat,0,known)-expr),
                            known[vars]),known[group][geners],",known);
        if has(",known[group][alg]) then
            alglinsolve(",unknowns,known[group][minpol])
        else
            readlib(`solve/linear`)(",unknowns)
        fi;
        ans:=expand(subs(",ans));
        indets(ans) intersect unknowns;
        if "<>{} then
            subs("[1]=1,map(x->x=0,"),ans);
            ans:="/icontent(");
            break
        fi
    od;
    out(`fraction extended by a factor of degree `.deg);
    out(2,`It is`,ans);
    
    proc(x,ext,known)
        expand(ext*x);
        if has(",known[group][alg]) then reduce(",known[group]) fi;
        represent([",degree(",{known[vars][]})],known)
    end;
    map(",numden,ans,known);
    if has(",'FAIL') then 'FAIL'
    elif has(",`MULTIPLE SOLUTIONS!`) then `MULTIPLE SOLUTIONS!`
    else normal("[1]/"[2])
    fi
end:


# SECONDARIES
# finds secondary generators.

secondaries:=proc(degrees,t_name,Known)
    local t,i,deg,candset,cand,shortest,temp,ready,yet,known;

    shortest:=proc(cands)
        local n,res,i;
        res:=cands[1]; n:=length(res);
        for i from 2 to nops(cands) do
            if length(cands[i])<n then res:=cands[i]; n:=length(res) fi
        od;
        res
    end;
    
    if nargs>=3 then known:=Known else known:=_known fi;
    if nargs>=2 and type(t_name,name) then t:=t_name fi;

    out(0,`Finding secondary generators ...`); out();
    out(cat(`We'll get a basis of invariants of degrees `,
                degrees[1],seq(`,`.(degrees[i]), i=2..nops(degrees)),`.`));

    out();
    
    known[base]:=[]; yet:=0;                # index of last new invariant
    known[second]:=[]; known[rel]:=[];

    # M A I N   L O O P

    for deg in degrees do
        if deg=0 then 
            out(`It starts with 1 (ever so amazing!)`); out();
            known[base]:=[op(known[base]),[1,0]]
        else                                    # degree not zero
            ready:=false; 
            do                      # try combinations of old invariants first
                candset:={monoms(deg,known[second])[]} minus known[used];
                if "={} then break fi;
                cand:=shortest(candset);
                known[used]:={known[used][],cand};
                out(`Trying `,cand[1],` of degree `.deg);
                derive(cand,known);     # maybe it can be derived!
                if "<>'FAIL' then                           # relation found!
                    #known[equ][cand[1]]:=";
                    known[equ][cand[1]]:=`known[equ] disabled!`;
                    if not ready then out(`Not good:`) fi;
                    out(cand[1]=collect("",
                        map(x->x[1],known[second]),distributed),`(derived)`);
                    out();
                    next
                fi;

                represent(cand,known);                  # derive failed!
                if "=`MULTIPLE SOLUTIONS!` then
                            ERROR(`Invariants not independent!`)
                elif "<>'FAIL' then                     # relation found!
                    #known[equ][cand[1]]:=";
                    known[equ][cand[1]]:=`known[equ] disabled!`;
                    if not ready then out(`Not good:`) fi;
                    out(cand[1]=collect("",
                        map(x->x[1],known[second]),distributed));
                    out();
                    cand[1]-"";
                    known[rel]:=[known[rel][],["/icontent("),deg]]
                else                # cand is good (=independent) invariant!
                    out(`Next invariant: `,cand[1]); out();
                    known[base]:=[op(known[base]),cand];
                    ready:=true;
                    break
                fi
            od;             # end of `while nops(candset)>0`

            while not ready do
                        # didn't get through with combinations of old,
                        # have to search new invariant

                candset:={invarspace(deg,1,known)[]} minus known[used];
                if candset={} then 
                    ERROR(`Not enough independent invariants!`)
                fi;
                cand:=shortest(candset);
                known[used]:={known[used][],cand};
                member(cand,invarspace(deg,1,known),'temp');
                out(`Trying invariant I`.deg._.temp.` of degree `.deg);
                represent([cand,deg],known);
                if "=`MULTIPLE SOLUTIONS!` then
                    ERROR(`Invariants not independent!`)
                elif "='FAIL' then
                           # cand is good (=independent) invariant!
                    yet:=yet+1;
                    if assigned(``.t.yet) then
                        ERROR(``.(evaln(``.t.yet)).
                            ` must be an unassigned name!`)
                    fi;
                    temp:=``.t.yet;
                    known[second]:=[known[second][],[temp,deg]];
                    known[used]:={known[used][],[temp,deg]};
                    known[values]:=known[values],temp=cand;
                    out(`We take it and call it "`.temp.`"!`); out();
                    known[base]:=[op(known[base]),[temp,deg]];
                    ready:=true
                else                                # cand was dependent
                    out(`Not good:`); out('Candidate'="); out()
                fi
            od                        # end of `didn't get through`
        fi
    od;                                          # end of `for deg`

    out(0,`The secondary generators are:`);
    for i in known[second] do out(0,`   `.(i[1]).` of degree `.(i[2])) od;
    out(0);
    out(`The invariant ring is  a free module over Q[{primaries}] with basis`);
    for i in known[base] do out(` `,i[1],` of degree `.(i[2])) od;
    out(); out(`Ready with finding secondary generators!`); out();
    RETURN(known[base])
end:

# RELATIONS
# calculates the syzygies.

relations:=proc(Known)
    local deg,wanted,missing,temp,shortest,i,known;

    shortest:=proc(cands)
        local n,res,i;
        res:=cands[1]; n:=length(res);
        for i from 2 to nops(cands) do
            if length(cands[i])<n then res:=cands[i]; n:=length(res) fi
        od;
        res
    end;
    
    if nargs>=1 then known:=Known else known:=_known fi;
    out(0,`Finding relations between secondary generators ...`);
    #for temp in known[base] do known[equ][temp[1]]:=temp[1] od;
    for temp in known[base] do known[equ][temp[1]]:=`known[equ] disabled!` od;
    
    # MAIN LOOP
    known[second];
    if "=[] then 0 else "[nops(")][2]*known[base][nops(known[base])][2] fi;
    for deg to " do
        wanted:=NULL;
        for temp in known[second] do
            wanted:=wanted,
                map(proc(f,s,d) if f[2]+s[2]=d then [s[1]*f[1],d] fi end,
                                            known[base],temp,deg)[]
        od;
        wanted:=
            select((f,s)->not member([f[1]],s),{wanted},{indices(known[equ])});
        if "<>{} then out(); out(`Degree `.deg.`:`) fi;
        while wanted<>{} do                     # treating members of wanted
            missing:=shortest(wanted);
            wanted:=wanted minus {missing};
            out(`calculating `,missing[1]);
            temp:=derive(missing,known);
            if temp<>'FAIL' then
                #known[equ][missing[1]]:=temp;
                known[equ][missing[1]]:=`known[equ] disabled!`;
                out(missing[1]=collect(temp,
                    map(x->x[1],known[second]),distributed),`(derived)`);
            else             # bad luck: have to calculate it!
                temp:=represent(missing,known);
                if temp=`MULTIPLE SOLUTIONS!` then 
                                    ERROR(`Invariants not independent!`)
                elif temp='FAIL' then ERROR(`Not enough invariants!`)
                else
                    #known[equ][missing[1]]:=temp;
                    known[equ][missing[1]]:=`known[equ] disabled!`;
                    out(missing[1]=collect(temp,
                        map(x->x[1],known[second]),distributed));
                    missing[1]-temp;
                    known[rel]:=[known[rel][],["/icontent("),deg]]
                fi
            fi                              # end of 'bad luck'
            
        od                # end of 'treating members of wanted'
    od;                                           # end of MAIN LOOP
    
    out(); out(`The relations are:`);
    for i in known[rel] do out(` `,i[1],`of degree `.(i[2])) od;
    out(); out(`Ready with finding relations!`); out();
    RETURN(known[rel])
end:


# DERIVE
# Tries to derive a relation from the ones already known.

derive:=proc(wanted,Known)
    local terms,n,ansatz,alfa,mons,equ,temp,s,t,b,i,known;
    
    if nargs>=2 then known:=Known else known:=_known fi;
    out(2,`Try to derive`,wanted[1],`from the known relations ...`);
    s:=map(x->x[1],known[prim]);
    t:=map(x->x[1],known[second]);
    b:=map(x->x[1],known[base]);
    terms:=monoms_base(wanted[2],[known[prim][],known[second][]],known[rel]);
    n:=nops(terms);
    ansatz:=expand(wanted[1]-
                    convert([seq(alfa[i]*terms[i][1], i=1..n)],`+`));
    terms:=[coeffs(ansatz,[s[],t[]],'mons')]; mons:=[mons];
    equ:={};
    for i from 1 to nops(terms) do
        temp:=coeffs(mons[i],s);
        if not member(temp,b) then equ:=equ union {terms[i]} fi
    od;
    out(2,`   Solving a system of `.(nops(equ)).` linear equations in `.
                                                n.` unknowns ...`);
    if assigned(known[group][minpol]) then [known[group][minpol]] else [] fi;
    equ:=alglinsolve(equ,{seq(alfa[i], i=1..n)},"[]);
    if equ=NULL then out(2,`... failed!`); RETURN('FAIL') fi;
    out(2,`... succeeded!`);
    RETURN(expand(subs(equ,ansatz)))
end:


# Retrieval-functions PRIM, SECOND, BASE, REL, GENERATORS

showprims:=proc(Known)
    if nargs>=1 then Known else _known fi;
    "[prim]
end:

showseconds:=proc(Known)
    if nargs>=1 then Known else _known fi;
    "[second]
end:

showbasis:=proc(Known)
    if nargs>=1 then Known else _known fi;
    "[base]
end:

showrels:=proc(Known)
    if nargs>=1 then Known else _known fi;
    "[rel]
end:

generators:=proc(G)
    if nargs>=1 then G else _known[group] fi;
    "[geners]
end:


# VALUEOF
# the value of some expression in the s1,s2... and t1,t2...

valueof:=proc(expr,Known)
    if nargs>=2 then Known else _known fi;
    subs("[values],expr)
end:


macro(alglinsolve=alglinsolve,aver_orb=aver_orb,
      check_nullspace=check_nullspace,classes=classes,derive=derive,
      elements=elements,elesym=elesym,generators=generators,inv=inv,
      invarspace=invarspace,invring=invring,lingroup=lingroup,
      mklingroup=mklingroup,monoms=monoms,monoms_base=monoms_base,mul=mul,
      onesoff=onesoff,out=out,poincare=poincare,poinrep=poinrep,
      primaries=primaries,proj=proj,reci=reci,reduce=reduce,
      relations=relations,renormal=renormal,repfrac=repfrac,
      represent=represent,secondaries=secondaries,showbasis=showbasis,
      showprims=showprims,showrels=showrels,showseconds=showseconds,
      throw=throw,valueof=valueof):


# DEFINITION OF A COUPLE OF PERMUTATION GROUPS

Z2:=permgroup(2,{[[1,2]]}):

Z3:=permgroup(3,{[[1,2,3]]}):
S3:=permgroup(3,{[[1,2,3]],[[1,2]]}):

V4:=permgroup(4,{[[1,2],[3,4]],[[1,3],[2,4]]}):
Z4:=permgroup(4,{[[1,2,3,4]]}):
D4:=permgroup(4,{[[1,2,3,4]],[[1,3]]}):
A4:=permgroup(4,{[[1,2],[3,4]],[[1,3],[2,4]],[[1,2,3]]}):
S4:=permgroup(4,{[[1,2]],[[1,3]],[[1,4]]}):

Z5:=permgroup(5,{[[1,2,3,4,5]]}):
D5:=permgroup(5,{[[1,2,3,4,5]],[[2,5],[3,4]]}):
F20:=permgroup(5,{[[1,2,3,4,5]],[[2,3,5,4]]}):
A5:=permgroup(5,{[[1,2,3,4,5]],[[1,2,3]]}):
S5:=permgroup(5,{[[1,2]],[[2,3]],[[3,4]],[[4,5]]}):

#    save invar,
#         `primaries/nextmat`,`primaries/nextvec`,`represent/evalprod`,
#         Z2,Z3,S3,V4,Z4,D4,A4,S4,Z5,D5,F20,A5,S5,
#         `invar.m`:
#quit
