#
## <SHAREFILE=engineer/polycon/polycon.mpl >
## <DESCRIBE>
##                SEE ALSO: engineer/polycon.tex  (52K)
##                This package includes functions for analysis of polynomial and
##                rational control systems, i.e. control systems of the type
##
##                        dx/dt=f(x(t),u(t)),  y=h(x,u)
##                or
##                       x(t+1)=f(x(t),u(t)),  y=h(x,u)
##
##                where h and all components of f are rational functions in
##                x = (x1,...,xn) and u.
##                AUTHOR: Krister Forsman, krister@isy.liu.se
## </DESCRIBE>


# Functions of the polycon package.

# Extended k:th Lie derivative of h w.r.t. f.
# Takes inputs into account, up to 10:th derivative of inputs.
# Syntax: lieder(h,f,k,vars,invar)  [invar may be a list]
# K. Forsman 1992-04-14, 1992-08-06

lieder:=proc()
   local dinvar,f,h,i,ind,invar,n,nu,res,var,vars;

   if nargs<2 then ERROR(`not enough input arguments.`) fi;

   h:=args[1]; f:=args[2];
   if not type(h,scalar) then
     ERROR(`first argument should be a scalar.`) fi;
   if not type(f,vector) then
     ERROR(`second argument should be a vector.`) fi;
   if nargs=2 then ind:=1; else ind:=args[3]; fi;
   if not type(ind,integer) then
      ERROR(`third argument should be an integer.`) fi;
   n:=linalg[vectdim](f);
   if nargs<4 then vars:=[x.(1..n)]; else vars:=args[4] fi;
   if not type(vars,list) then
      ERROR(`fourth argument should be a list of variables.`)
   fi;
   if nops(vars)<>n then
     ERROR(`wrong number of variables given.`) fi;
   if nargs<5 then invar:=[u]; elif type(args[5],name) then
      invar:=[args[5]] else invar:=args[5] fi;
   if not type(invar,list(name)) then
     ERROR(`fifth argument should be a name or a list of names of input variables.`,
           `If default: u must be unassigned.`) fi;
   nu:=nops(invar):

   for var in invar do
      f:=subs(var=cat(var,0),op(f)):
      h:=subs(var=cat(var,0),h)
   od:

   dinvar:=[]:
   for var in invar do
      dinvar:=[op(dinvar),[seq(cat(var,i),i=0..10)]]
   od:
   vars:=[op(vars),op(map(op,[seq(dinvar[i][1..10],i=1..nu)]))]:

   f:=linalg[vector]([op(convert(f,list)),
	op(map(op,[seq(dinvar[i][2..11],i=1..nu)]))]);
   res:=h:
   for i to ind do res:=auxlieder(res,f,vars) od
end:


# Auxiliary Lie-derivative function.
# Lie derivative of h w.r.t. f without safety jacket. Variables: vars
# Syntax: auxlieder(h,f,vars)
# K. Forsman 1992-02-04

auxlieder:=proc(h,f,vars)
   linalg[dotprod](f,map((a,b) -> diff(b,a),vars,h))
end:


# List of successive Lie-derivatives. Requires lieder.
# Changed and renamed (from Lieder2) 1992-02-24 to replace former Lieder1.
# K. Forsman 1992-02-24, 1992-02-25

liederlist:=proc(h,f,n,stvars,invar)
   local i,L,tmp:
   L:=[h]: tmp:=h:
   for i to n do
      tmp:=lieder(tmp,f,1,stvars,invar):
      L:=[op(L),tmp]; od
end:


######################################################################

######################################################################


# Extended k:th Lie homomorphism of h w.r.t. f.
# Takes inputs into account, up to 10:th time-shift of input.
# Syntax: liehom(h,f,k,vars,invar)   [invar may be a list]
# K. Forsman 1992-04-14, 1992-08-06

liehom:=proc()
   local dinvar,f,h,i,ind,invar,n,nu,res,var,vars;

   if nargs<2 then ERROR(`not enough input arguments.`) fi;

   h:=args[1]; f:=args[2];
   if not type(h,scalar) then
     ERROR(`first argument should be a scalar.`) fi;
   if not type(f,vector) then
     ERROR(`second argument should be a vector.`) fi;
   f:=convert(f,list):
   if nargs=2 then ind:=1; else ind:=args[3]; fi;
   if not type(ind,integer) then
      ERROR(`third argument should be an integer.`) fi;
   n:=nops(f):
   if nargs<4 then vars:=[x.(1..n)]; else vars:=args[4] fi;
   if not type(vars,list) then
      ERROR(`fourth argument should be a list of variables.`)
   fi;
   if nops(vars)<>n then
     ERROR(`wrong number of state variables given.`) fi;
   if nargs<5 then invar:=[u]; elif type(args[5],name) then
      invar:=[args[5]] else invar:=args[5] fi;
   if not type(invar,list(name)) then
     ERROR(`fifth argument should be a name or a list of names of input variables.`,
           `If default: u must be unassigned.`) fi;
   nu:=nops(invar):

   for var in invar do
      f:=subs(var=cat(var,0),f):
      h:=subs(var=cat(var,0),h)
   od:

   dinvar:=[]:
   for var in invar do
      dinvar:=[op(dinvar),[seq(cat(var,i),i=0..10)]]
   od:
   vars:=[op(vars),op(map(op,[seq(dinvar[i][1..10],i=1..nu)]))]:

   f:=[op(f),op(map(op,[seq(dinvar[i][2..11],i=1..nu)]))]: res:=h:
   for i to ind do
      res:=subs(zip((X,Y)->X=Y,vars,f),res)
   od
end:

# List of successive Lie-homomorphisms. Requires liehom.
# K. Forsman 1992-02-04, 1992-02-24

liehomlist:=proc(h,f,n,stvars,invar)
   local L,i,tmp:
   L:=[h]: tmp:=h:
   for i to n do
      tmp:=liehom(tmp,f,1,stvars,invar):
      L:=[op(L),tmp]:
   od
end:


######################################################################
#
######################################################################

# Derivative of p w.r.t. t: diff(u0,t$k)=uk, diff(y0,t$k)=yk
# Uses auxlieder. Syntax: uyder(p,k,invar,outvar)
# K. Forsman 1992-02-12, 1992-02-25

uyder:=proc()
   local i,invar,dinvar,k,outvar,outvars,p:

   p:=args[1]:
   if not type(p,scalar) then
     ERROR(`first argument should be a scalar.`) fi;
   if nargs=1 then k:=1 else k:=args[2] fi:
   if nargs<3 then invar:=u else invar:=args[3] fi;
   if not type(invar,name) then
     ERROR(`third argument should be an input variable name.`,
           `If default: u must be unassigned.`) fi;
   if nargs<4 then outvar:=y else outvar:=args[4] fi;
   if not type(outvar,name) then
     ERROR(`fourth argument should be an output variable name.`,
           `If default: y must be unassigned.`) fi;

   dinvar:=[seq(cat(invar,i),i=0..10)];
   outvars:=[seq(cat(outvar,i),i=0..10)];

   p:=subs({invar=cat(invar,0),outvar=cat(outvar,0)},p):
   for i to k do
      p:=auxlieder(p,linalg[vector]([op(dinvar[2..11]),op(outvars[2..11])]),
         [op(dinvar[1..10]),op(outvars[1..10])]);
   od
end:


# Difference of p: delta^k(u0)=uk, delta^k(y0)=yk
# Uses liehom.
# Syntax: uyhom(p,k,invar,outvar)
# K. Forsman 1992-02-24, 1992-02-25

uyhom:=proc()
   local i,invar,dinvar,k,outvar,outvars,p:

   p:=args[1]:
   if not type(p,scalar) then
     ERROR(`first argument should be a scalar.`) fi;
   if nargs=1 then k:=1 else k:=args[2] fi:
   if nargs<3 then invar:=u else invar:=args[3] fi;
   if not type(invar,name) then
     ERROR(`third argument should be an input variable name.`,
           `If default: u must be unassigned.`) fi;
   if nargs<4 then outvar:=y else outvar:=args[4] fi;
   if not type(outvar,name) then
     ERROR(`fourth argument should be an output variable name.`,
           `If default: y must be unassigned.`) fi;

   dinvar:=[seq(cat(invar,i),i=0..10)];
   outvars:=[seq(cat(outvar,i),i=0..10)];

   p:=subs({invar=cat(invar,0),outvar=cat(outvar,0)},p):
   liehom(p,linalg[vector]([op(dinvar[2..11]),op(outvars[2..11])]),k,
            [op(dinvar[1..10]),op(outvars[1..10])])
end:

# Input-output relation for continuous time rational systems
# in state-space form. Uses liederlist.
# Syntax: ss2ioc(f,h,stvars,invars,outvar)   {invars can be a list}
# K. Forsman 1992-06-16, 1992-08-07

ss2ioc:=proc()
   local f,h,i,invars,L,n,outvar,p,stvars,var:

   if nargs<2 then ERROR(`not enough input arguments.`) fi;

   f:=args[1]: h:=args[2]:
   if not type(f,vector) then
     ERROR(`first argument should be a vector.`) fi;
   if not type(h,scalar) then
     ERROR(`second argument should be a scalar.`) fi;
   n:=linalg[vectdim](f):
   if nargs<3 then
       stvars:=[x.(1..n)] else stvars:=args[3] fi;
   if not type(stvars,list(name)) then
     ERROR(`third argument should be a list of names of state variables`,
           `If default: x1,...,xn must be unassigned.`) fi;
   if nops(stvars)<>n then
     ERROR(`wrong number of state variables given.`) fi;

   if nargs<4 then invars:=[u]; elif type(args[4],name) then
      invars:=[args[4]] else invars:=args[4] fi;
   if not type(invars,list(name)) then
     ERROR(cat(`fourth argument should be a name or a list of names of`,
            `input variables.`),`If default: u must be unassigned.`) fi;

   for var in invars do
      for i from 0 to n do
         if not type(eval(cat(var,i)),name) then
           ERROR(cat(cat(var,i),` must be unassigned.`)) fi;
         od                 #This is a little tough.
   od:

   if nargs<5 then outvar:=y else outvar:=args[5] fi;
   if not type(outvar,name) then
     ERROR(`fifth argument should be an output variable name.`,
           `If default: y must be unassigned.`) fi;
   for i from 0 to n do
      if not type(eval(cat(outvar,i)),name) then
        ERROR(cat(cat(outvar,i),` must be unassigned.`)) fi;
      od:

   for var in invars do h:=subs(var=cat(var,0),h) od:

   L:=liederlist(h,f,n,stvars,invars):
   L:=zip((X,Y)->(X-Y),[seq(cat(outvar,i),i=0..n)],[op(L)]);
   L:=map(numer@expand,L):
   p:=grobner[finduni](cat(outvar,n),L,{op(stvars),cat(outvar,n)});
   if p=NULL then
         print(`WARNING: System realization is not minimal.`);
         print(`(Disregard potential warning message that follows.)`);
         for i from n by -1 to 1 while p=NULL do
         p:=grobner[finduni](cat(outvar,i-1),L[1..i],
                   {op(stvars),cat(outvar,i-1)}); od:
   fi;
   sort(p,[seq(cat(outvar,n-i),i=0..n)],'plex')
end:


######################################################################

######################################################################

# Input-output relation for discrete time rational systems
# in state-space form. Uses liehomlist.
# Syntax: ss2iod(f,h,stvars,invars,outvar)
# K. Forsman 1992-04-14, 1992-08-07

ss2iod:=proc()
   local f,h,i,invars,L,n,outvar,p,stvars,var:

   if nargs<2 then ERROR(`not enough input arguments.`) fi;

   f:=args[1]: h:=args[2]:
   if not type(f,vector) then
     ERROR(`first argument should be a vector.`) fi;
   if not type(h,scalar) then
     ERROR(`second argument should be a scalar.`) fi;
   n:=linalg[vectdim](f):
   if nargs<3 then
      stvars:=[x.(1..n)] else stvars:=args[3] fi;
   if not type(stvars,list(name)) then
     ERROR(`third argument should be a list of names of state variables`,
           `If default: x1,...,xn must be unassigned.`) fi;
   if nops(stvars)<>n then
     ERROR(`wrong number of state variables given.`) fi;

   if nargs<4 then invars:=[u]; elif type(args[4],name) then
      invars:=[args[4]] else invars:=args[4] fi;
   if not type(invars,list(name)) then
     ERROR(`fifth argument should be a name or a list of names of input variables.`,
           `If default: u must be unassigned.`) fi;
   for var in invars do
      for i from 0 to n do
         if not type(eval(cat(var,i)),name) then
           ERROR(cat(cat(var,i),` must be unassigned.`)) fi;
         od                 #This is a little tough.
   od:

   if nargs<5 then outvar:=y else outvar:=args[5] fi;
   if not type(outvar,name) then
     ERROR(`fifth argument should be an output variable name.`,
           `If default: y must be unassigned.`) fi;
   for i from 0 to n do
      if not type(eval(cat(outvar,i)),name) then
        ERROR(cat(cat(outvar,i),` must be unassigned.`)) fi;
      od:

   for var in invars do h:=subs(var=cat(var,0),h) od:

   L:=liehomlist(h,f,n,stvars,invars):
   L:=zip((X,Y)->(X-Y),[seq(cat(outvar,i),i=0..n)],[op(L)]);
   L:=map(numer@expand,L):

   p:=grobner[finduni](cat(outvar,n),L,{op(stvars),cat(outvar,n)});
   if p=NULL then
         print(`WARNING: System realization is not minimal.`);
         print(`(Disregard potential warning message that follows.)`);
         for i from n by -1 to 1 while p=NULL do
         p:=grobner[finduni](cat(outvar,i-1),L[1..i],
                   {op(stvars),cat(outvar,i-1)}); od:
   fi;
   sort(p,[seq(cat(outvar,n-i),i=0..n)],'plex')
end:


######################################################################
######################################################################

######################################################################
######################################################################

# Expert version of ss2io: no safety jackets, returns the entire GB.
# K. Forsman 1992-04-14, 1992-06-16

xss2ioc:=proc()
   local f,h,i,invars,L,n,outvar,stvars,tmplist,var:

   f:=args[1]: h:=args[2]:
   n:=linalg[vectdim](f):
   if nargs<3 then
       stvars:=[x.(1..n)] else stvars:=args[3] fi;
   if nargs<4 then invars:=[u]; elif type(args[4],name) then
      invars:=[args[4]] else invars:=args[4] fi;
   if nargs<5 then outvar:=y else outvar:=args[5] fi;

   for var in invars do h:=subs(var=cat(var,0),h) od:

   L:=liederlist(h,f,n,stvars,invars):
   L:=zip((X,Y)->(X-Y),[seq(cat(outvar,i),i=0..n)],[op(L)]);
   L:=map(numer@expand,L):
   tmplist:=[op(stvars),seq(cat(outvar,n-i),i=0..n)];
   grobner[gbasis](L,tmplist,'plex')
end:


xss2iod:=proc()
   local f,h,i,invars,L,n,outvar,stvars,tmplist,var:

   f:=args[1]: h:=args[2]:
   n:=linalg[vectdim](f):
   if nargs<3 then
      stvars:=[x.(1..n)] else stvars:=args[3] fi;
   if nargs<4 then invars:=[u]; elif type(args[4],name) then
      invars:=[args[4]] else invars:=args[4] fi;
   if nargs<5 then outvar:=y else outvar:=args[5] fi;

   for var in invars do h:=subs(var=cat(var,0),h) od:

   L:=liehomlist(h,f,n,stvars,invars):
   L:=zip((X,Y)->(X-Y),[seq(cat(outvar,i),i=0..n)],[op(L)]);
   L:=map(numer@expand,L):
   tmplist:=[op(stvars),seq(cat(outvar,n-i),i=0..n)];
   grobner[gbasis](L,tmplist,'plex')
end:

# I/O form to state space form for given states. Continuous time.
# Returns a list consisting of the rhs vector field and the output map.
# SYNTAX: io2ssc(p,S,stvars,invar,outvar,ssmod) where p is the I/O-relation
# and S is a list containing the suggested states:
# S=[s1,...,sn] si\in k<invar,outvar> and stvars is a list of
# suggested state variables.
# The parameter ssmod determines if the state space model should allow
#   1) derivatives of the input
#   2) implicit state equations
# If ssmod='kalman' then neither 1 nor 2 are allowed
#          'der' then 1 but not 2 is allowed
#          'impl' then 2 but not 1 is allowed
#          'derimpl' then both 1 and 2 are allowed
# Default is ssmod='kalman'
# Uses uyder (and thus lieder).
# K. Forsman 1992-04-14, 1992-08-06

io2ssc:=proc()
   local DS,dstvars,explxflag,explyflag,f,failind,G,h,i,invar,invars,j,L,n,
         OKvars,outvar,outvars,p,res,S,ssm,ssmod,stvars,xflag,yflag:

   if nargs<2 then ERROR(`not enough input arguments.`) fi;
   p:=args[1]: S:=args[2]: n:=nops(S):

   if not type(p,scalar) then
     ERROR(`first argument should define an I/O equation.`) fi;
   if type(p,equation) then p:=numer(lhs(p)-rhs(p)) fi;
   if not type(S,list) then
     ERROR(`second argument should be a list defining the states.`) fi;
   if nargs<3 then stvars:=[x.(1..n)] else stvars:=args[3] fi;
   if not type(stvars,list(name)) then
     ERROR(`third argument should be a list of names of state variables`,
           `If default: x1,...,xn must be unassigned.`) fi;
   if nops(stvars)<>n then
     ERROR(`wrong number of state variables given.`) fi;

   if nargs<4 then invar:=u else invar:=args[4] fi;
   if not type(invar,name) then
     ERROR(`fourth argument should be an input variable name.`,
           `If default: u must be unassigned.`) fi;
   for i from 0 to n do
      if not type(eval(cat(invar,i)),name) then
        ERROR(cat(cat(invar,i),` must be unassigned.`)) fi;
      od:                #This is a little tough.

   if nargs<5 then outvar:=y else outvar:=args[5] fi;
   if not type(outvar,name) then
     ERROR(`fifth argument should be an output variable name.`,
           `If default: y must be unassigned.`) fi;
   for i from 0 to n do
      if not type(eval(cat(outvar,i)),name) then
        ERROR(cat(cat(outvar,i),` must be unassigned.`)) fi;
      od:

   if nargs<6 then ssmod:='kalman' else ssmod:=args[6] fi;

# Simple internal representation of the parameter ssmod.
# ssm[1] <->  derivatives of input allowed
# ssm[2] <-> implicit equations allowed
   if ssmod='derimpl' then ssm:=[true,true]
      elif ssmod='der' then ssm:=[true,false]
      elif ssmod='impl' then ssm:=[false,true]
      else ssm:=[false,false] fi;

   p:=subs({invar=cat(invar,0),outvar=cat(outvar,0)},p):
   S:=subs({invar=cat(invar,0),outvar=cat(outvar,0)},S):

   DS:=map(uyder,S,1,invar,outvar):
   L:=zip((X,Y)->(X-Y),stvars,S):
   dstvars:=[seq(cat('d',stvars[j]),j=1..n)];
   for i to n do if not type(cat('d',(stvars[i])),name) then
      ERROR(cat('d',stvars[i],` should not be assigned.`)) fi od;

   L:=[p,op(L),op(zip((X,Y)->(X-Y),dstvars,DS))]:
   L:=map(numer@expand,L):
   f:=linalg[vector](n):
   invars:=[seq(cat(invar,j),j=0..n)];
   outvars:=[seq(cat(outvar,j),j=0..n)];
   xflag:=true: explxflag:=true:

# Search for allowable expressions for the time derivatives of the states
# and for the output:
   if ssm[1] then
      OKvars:={op(invars),op(stvars)}:
      for i to n while (xflag and explxflag) do
         G:=grobner[gbasis](L,[dstvars[1..i-1],op(dstvars[i+1..n]),
              op(outvars),dstvars[i]],'plex'):
         f[i]:=G[nops(G)];
         xflag:=evalb(indets(f[i]) minus OKvars = {dstvars[i]});
         if not xflag then failind:=i fi:
         if xflag and (not ssm[2])
            then explxflag:=evalb(degree(f[i],dstvars[i])=1) fi;
      od;
      G:=grobner[gbasis](L,[op(dstvars),seq(outvars[n-j+1],j=0..n)],'plex'):
      h:=G[nops(G)]:
      yflag:=evalb(indets(h) minus OKvars = {outvars[1]});
      if yflag and (not ssm[2]) then
         explyflag:=evalb(degree(h,outvars[1])=1) fi;

   else
      OKvars:={invars[1],op(stvars)}:
      for i to n while xflag do
         G:=grobner[gbasis](L,[op(dstvars[1..i-1]),op(dstvars[i+1..n]),
              op(outvars),op(invars[2..n+1]),dstvars[i]],'plex'):
         f[i]:=G[nops(G)];
         xflag:=evalb(indets(f[i]) minus OKvars = {dstvars[i]});
         if not xflag then failind:=i fi:
         if xflag and (not ssm[2]) and explxflag
            then explxflag:=evalb(degree(f[i],dstvars[i])=1) fi;
      od;
      G:=grobner[gbasis](L,[op(dstvars),op(invars[2..n+1]),
           seq(outvars[n-j+1],j=0..n)],'plex'):
      h:=G[nops(G)]:
      yflag:=evalb(indets(h) minus OKvars = {outvars[1]});
      if yflag and (not ssm[2]) then
         explyflag:=evalb(degree(h,outvars[1])=1) fi;
   fi;

# If fail, return depressing message:
   if not xflag then
      print(cat(`No expression exists for the time derivative of `,
                 stvars[failind],`.`))
   fi;
   if not yflag then
      print(cat(`No expression exists for `,outvar,`.`))
   fi;

# If success in at least one respect, return the results obtained,
# considering the specification given by ssmod.

   res:=[]:

   if xflag then
      if ssm[2] then res:=[op(f)]
      elif explxflag then
           f:=convert(f,list): f:=zip(solve,f,dstvars):
           res:=[linalg[vector](f)]:
      else
         print(cat(`No explicit expression exists for `,dstvars));
      fi;
   fi;

   if yflag then
      if ssm[2] then res:=[op(res),h]
      elif explyflag then
           h:=solve(h,outvars[1]):
           res:=[op(res),h]:
      else
         print(cat(`No explicit expression exists for `,outvar));
      fi;
   fi;

   res
end:


######################################################################

######################################################################

# I/O-form to state space form for given states. Discrete time.
# Returns a list consisting of the rhs "vector field" and
# the output map.
# SYNTAX: io2ssd(p,S,stvars,invar,outvar,ssmod) where p is the I/O-relation
# and S is a list# containing the suggested states:
# L=[s1,...,sn] si\in k<u,y> and  stvars is a list of
# suggested state variables.
# The parameter ssmod determines if the state space model should allow
#   1) non-causality in the input
#   2) implicit state equations
# If ssmod=`kalman` then neither 1 nor 2 are allowed
#          `nc` then 1 but not 2 is allowed
#          `impl` then 2 but not 1 is allowed
#          `ncimpl` then both 1 and 2 are allowed
# Default is ssmod=`kalman`
# Requires uyhom (and thus liehom).
# K. Forsman 1992-04-14, 1992-08-06

io2ssd:=proc()
   local DS,dstvars,explxflag,explyflag,f,failind,G,h,i,invar,invars,j,L,n,
         OKvars,outvar,outvars,p,res,S,ssm,ssmod,stvars,xflag,yflag:

   if nargs<2 then ERROR(`not enough input arguments.`) fi;
   p:=args[1]: S:=args[2]: n:=nops(S):

   if not type(p,scalar) then
     ERROR(`first argument should define an I/O equation.`) fi;
   if type(p,equation) then p:=numer(lhs(p)-rhs(p)) fi;
   if not type(S,list) then
     ERROR(`second argument should be a list defining the states.`) fi;
   if nargs<3 then stvars:=[x.(1..n)] else stvars:=args[3] fi;
   if not type(stvars,list(name)) then
     ERROR(`third argument should be a list of names of state variables`,
           `If default: x1,...,xn must be unassigned.`) fi;
   if nops(stvars)<>n then
     ERROR(`wrong number of state variables given.`) fi;

   if nargs<4 then invar:=u else invar:=args[4] fi;
   if not type(invar,name) then
     ERROR(`fourth argument should be an input variable name.`,
           `If default: u must be unassigned.`) fi;
   for i from 0 to n do
      if not type(eval(cat(invar,i)),name) then
        ERROR(cat(cat(invar,i),` must be unassigned.`)) fi;
      od:                #This is a little tough.

   if nargs<5 then outvar:=y else outvar:=args[5] fi;
   if not type(outvar,name) then
     ERROR(`fifth argument should be an output variable name.`,
           `If default: y must be unassigned.`) fi;
   for i from 0 to n do
      if not type(eval(cat(outvar,i)),name) then
        ERROR(cat(cat(outvar,i),` must be unassigned.`)) fi;
      od:

   if nargs<6 then ssmod:='kalman' else ssmod:=args[6] fi;

# Simple internal representation of the parameter ssmod.
# ssm[1] <->  input allowed to occur non-causally
# ssm[2] <-> implicit equations allowed
   if ssmod='ncimpl' then ssm:=[true,true]
      elif ssmod='nc' then ssm:=[true,false]
      elif ssmod='impl' then ssm:=[false,true]
      else ssm:=[false,false] fi;

   p:=subs({invar=cat(invar,0),outvar=cat(outvar,0)},p):
   S:=subs({invar=cat(invar,0),outvar=cat(outvar,0)},S):

   DS:=map(uyhom,S,1,invar,outvar):
   L:=zip((X,Y)->(X-Y),stvars,S):
   dstvars:=[seq(cat('d',stvars[j]),j=1..n)];
   for i to n do if not type(cat('d',(stvars[i])),name) then
      ERROR(cat('d',stvars[i],` should not be assigned.`)) fi od;

   L:=[p,op(L),op(zip((X,Y)->(X-Y),dstvars,DS))]:
   L:=map(numer@expand,L):
   f:=linalg[vector](n):
   invars:=[seq(cat(invar,j),j=0..n)];
   outvars:=[seq(cat(outvar,j),j=0..n)];
   xflag:=true: explxflag:=true:

# Search for allowable expressions for the time lags of the states
# and for the output.

   if ssm[1] then
      OKvars:={op(invars),op(stvars)}:
      for i to n while (xflag and explxflag) do
         G:=grobner[gbasis](L,[dstvars[1..i-1],dstvars[i+1..n],
              op(outvars),dstvars[i]],'plex'):
         f[i]:=G[nops(G)];
         xflag:=evalb(indets(f[i]) minus OKvars = {dstvars[i]});
         if not xflag then failind:=i fi:
         if xflag and (not ssm[2])
            then explxflag:=evalb(degree(f[i],dstvars[i])=1) fi;
      od;
      G:=grobner[gbasis](L,[op(dstvars),seq(outvars[n-j+1],j=0..n)],'plex'):
      h:=G[nops(G)]:
      yflag:=evalb(indets(h) minus OKvars = {outvars[1]});
      if yflag and (not ssm[2]) then
         explyflag:=evalb(degree(h,outvars[1])=1) fi;

   else
      OKvars:={invars[1],op(stvars)}:
      for i to n while xflag do
         G:=grobner[gbasis](L,[op(dstvars[1..i-1]),op(dstvars[i+1..n]),
              op(outvars),op(invars[2..n+1]),dstvars[i]],'plex'):
         f[i]:=G[nops(G)];
         xflag:=evalb(indets(f[i]) minus OKvars = {dstvars[i]});
         if not xflag then failind:=i fi:
         if xflag and (not ssm[2]) and explxflag
            then explxflag:=evalb(degree(f[i],dstvars[i])=1) fi;
      od;
      G:=grobner[gbasis](L,[op(dstvars),op(invars[2..n+1]),
           seq(outvars[n-j+1],j=0..n)],'plex'):
      h:=G[nops(G)]:
      yflag:=evalb(indets(h) minus OKvars = {outvars[1]});
      if yflag and (not ssm[2]) then
         explyflag:=evalb(degree(h,outvars[1])=1) fi;
   fi;

# If fail, return depressing message:
   if not xflag then
      print(cat(`No expression found for the time lag of `,
                 stvars[failind],`.`))
   fi;
   if not yflag then
      print(cat(`No expression found for `,outvar,`.`))
   fi;

# If success in at least one respect, return the results obtained,
# considering the specification given by ssmod.

   res:=[]:

   if xflag then
      if ssm[2] then res:=[op(f)]
      elif explxflag then
           f:=convert(f,list): f:=zip(solve,f,dstvars):
           res:=[linalg[vector](f)]:
      else
         print(cat(`No explicit expression found for `,dstvars));
      fi;
   fi;

   if yflag then
      if ssm[2] then res:=[op(res),h]
      elif explyflag then
           h:=solve(h,outvars[1]):
           res:=[op(res),h]:
      else
         print(cat(`No explicit expression found for `,outvar));
      fi;
   fi;

   res
end:

# Find state space transformation for two I/O-equivalent continuous time
# systems. Syntax: sstrac(f,a,g,b,ost,nst,oinp,ninp,form)
# The original system is in the variables ost (default: x) and the new one
# is in the variables nst (default: z).
# The parameter form determines the form of the output:
#   if form='expl' (default) then the output is a vector
#   and if form='impl' then the output is a list of polynomials, the i:th
#   one defining zi as an algebraic function of the x.
# The transformation obtained may involve the input. Requires liederlist.
# K. Forsman 1992-06-16, 1992-08-06

sstrac:=proc()
   local a,b,explflag,f,failind,form,g,i,L,Lf,Lg,n,ninp,nst,oinp,ost,T:

   if nargs<4 then ERROR(`not enough input arguments.`) fi;

   f:=args[1]: a:=args[2]: g:=args[3]: b:=args[4]:
   if not type(f,vector) then
     ERROR(`first argument should be a vector.`) fi;
   if not type(a,scalar) then
     ERROR(`second argument should be a scalar.`) fi;
   if not type(g,vector) then
     ERROR(`third argument should be a vector.`) fi;
   if not type(b,scalar) then
     ERROR(`fourth argument should be a scalar.`) fi;

   n:=linalg[vectdim](f):
   if n<>linalg[vectdim](g) then
      ERROR(`incompatible system dimensions.`) fi;

   if nargs<5 then ost:=[x.(1..n)] else ost:=args[5] fi;
   if not type(ost,list(name)) then
     ERROR(`fifth argument should be a list of names of state variables`,
           `If default: x1,...,xn must be unassigned.`) fi;
   if nops(ost)<>n then
     ERROR(`wrong number of old state variables given.`) fi;


   if nargs<6 then nst:=[z.(1..n)] else nst:=args[6] fi;
   if not type(nst,list(name)) then
     ERROR(`sixth argument should be a list of names of state variables`,
           `If default: z1,...,zn must be unassigned.`) fi;
   if nops(nst)<>n then
     ERROR(`wrong number of new state variables given.`) fi;

   if nargs<7 then oinp:=u else oinp:=args[7] fi;
   if not type(oinp,name) then
      ERROR(`seventh argument should be an input variable name`,
            `If default: u must be unassigned.`) fi;
   if nargs<8 then ninp:=u else ninp:=args[8] fi;
   if not type(ninp,name) then
      ERROR(`eighth argument should be an input variable name`,
            `If default: u must be unassigned.`) fi;
   if nargs<9 then form:='expl' else form:=args[9] fi;

   Lf:=liederlist(a,f,n,ost,oinp):
   Lg:=liederlist(b,g,n,nst,ninp):
   L:=zip((X,Y)->(X-Y),Lf,Lg):
   L:=map(numer@expand,L): # Support rational functions, even though proof
                           # does not cover this case...

   T:=linalg[vector](n);
   explflag:=true:

   for i to n while explflag do
      T[i]:=grobner[finduni](nst[i],L,{op(nst)}):
      explflag:=evalb(form='impl' or degree(T[i],nst[i])=1);
      if not explflag then failind:=i fi;
   od:

   if form='impl' then convert(T,list);
   else
     if not explflag then
        print(cat(`No explicit expression for `,nst[failind],` exists.`));
     else
        T:=zip(solve,convert(T,list),nst);
     fi;
   fi
end:

######################################################################

######################################################################


# Find state space transformation for two I/O-equivalent discrete time
# systems.  Syntax: sstrad(f,a,g,b,ost,nst,oinp,ninp,form)
# The original system is in the variables ost (default: x) and the new one
# is in the variables nst (default: z).
# The parameter form determines the form of the output:
#   if form='expl' (default) then the output is a vector
#   and if form='impl' then the output is a list of polynomials, the i:th
#   one defining zi as an algebraic function of the x.
# The transformation obtained may involve the input. Requires liehomlist.
# K. Forsman 1992-06-16, 1992-08-06

sstrad:=proc()
   local a,b,explflag,f,failind,form,g,i,L,Lf,Lg,n,ninp,nst,oinp,ost,T:

   if nargs<4 then ERROR(`not enough input arguments.`) fi;

   f:=args[1]: a:=args[2]: g:=args[3]: b:=args[4]:
   if not type(f,vector) then
     ERROR(`first argument should be a vector.`) fi;
   if not type(a,scalar) then
     ERROR(`second argument should be a scalar.`) fi;
   if not type(g,vector) then
     ERROR(`third argument should be a vector.`) fi;
   if not type(b,scalar) then
     ERROR(`fourth argument should be a scalar.`) fi;

   n:=linalg[vectdim](f):
   if n<>linalg[vectdim](g) then
      ERROR(`incompatible system dimensions`) fi;

   if nargs<5 then ost:=[x.(1..n)] else ost:=args[5] fi;
   if not type(ost,list(name)) then
     ERROR(`fifth argument should be a list of names of state variables`,
           `If default: x1,...,xn must be unassigned.`) fi;
   if nops(ost)<>n then
     ERROR(`wrong number of old state variables given.`) fi;

   if nargs<6 then nst:=[z.(1..n)] else nst:=args[6] fi;
   if not type(nst,list(name)) then
     ERROR(`sixth argument should be a list of names of state variables`,
           `If default: z1,...,zn must be unassigned.`) fi;
   if nops(nst)<>n then
     ERROR(`wrong number of new state variables given.`) fi;

   if nargs<7 then oinp:=u else oinp:=args[7] fi;
   if not type(oinp,name) then
      ERROR(`seventh argument should be an input variable name`,
            `If default: u must be unassigned.`) fi;
   if nargs<8 then ninp:=u else ninp:=args[8] fi;
   if not type(ninp,name) then
      ERROR(`eighth argument should be an input variable name`,
            `If default: u must be unassigned.`) fi;
   if nargs<9 then form:='expl' else form:=args[9] fi;

   Lf:=liehomlist(a,f,n,ost,oinp):
   Lg:=liehomlist(b,g,n,nst,ninp):
   L:=zip((X,Y)->(X-Y),Lf,Lg):
   L:=map(numer@expand,L): # Support rational functions, even though proof
                           # does not cover this case...

   T:=linalg[vector](n);
   explflag:=true:

   for i to n while explflag do
      T[i]:=grobner[finduni](nst[i],L,{op(nst)}):
      explflag:=evalb(form='impl' or degree(T[i],nst[i])=1);
      if not explflag then failind:=i fi;
   od:

   if form='impl' then convert(T,list);
   else
     if not explflag then
        print(cat(`No explicit expression for `,nst[failind],` exists.`));
     else
        T:=zip(solve,convert(T,list),nst);
     fi;
   fi
end:

# Go from a given rational parametrization of a hypersurface
# to a state space description of a continuous time system.
# Syntax: par2ssc(H,vars)  where H is a list of polynomials in
# the variables vars.
# K. Forsman 1992-06-16, 1992-08-10

par2ssc:=proc()
   local A,b,G,H,i,n,vars:

   H:=args[1]: n:=nops(H)-1:
   if nargs=1 then vars:=[x.(1..n)] else vars:=args[2] fi;
   if not type(vars,list(name)) then
     ERROR(`second argument should be a list of names of state variables`,
           `If default: x1,...,xn must be unassigned.`) fi;
   if nops(vars)<>n then
      ERROR(`H and vars have incompatible dimensions.`) fi;

   G:=linalg[grad](H[1],vars):
   A:=linalg[matrix]([convert(G,list)]):
   for i from 2 to n do
      G:=linalg[grad](H[i],vars):
      A:=linalg[stack](A,G):
   od;
   if det(A)=0 then
      ERROR(`H[1], ..., H[n-1] are algebraically dependent.`) fi;
   b:=linalg[vector](H[2..(n+1)]);
   linalg[linsolve](A,b)
end:

# "Observer relation" for continuous time rational SISO system
# in state-space form. Requires liederlist. Uses finduni.
# Syntax: obsvc(f,h,obsvar,stvars,invars,outvar,form)
# obsvar must be a member of stvars. invars may be a list.
# K. Forsman 1992-06-16, 1992-08-07

obsvc:=proc()
   local dinvars,f,form,G,h,i,ind,invars,IOvars,n,L,obsflag,obsvar,
         OKvars,outvar,res,stvars,var,varflag:

   if nargs<3 then ERROR(`not enough input arguments.`) fi;
   f:=args[1]: h:=args[2]: obsvar:=args[3]:

   if not type(f,vector) then
     ERROR(`first argument should be a vector.`) fi;
   if not type(h,scalar) then
     ERROR(`second argument should be a scalar.`) fi;
   if not type(obsvar,name) then
     ERROR(`third argument should be a variable name.`) fi;

   n:=linalg[vectdim](f):
   if nargs<4 then stvars:=[x.(1..n)] else stvars:=args[4] fi;
   if not type(stvars,list(name)) then
     ERROR(`fourth argument should be a list of names of state variables`,
           `If default: x1,...,xn must be unassigned.`) fi;
   if nops(stvars)<>n then
     ERROR(`wrong number of state variables given.`) fi;

   varflag:=member(obsvar,stvars,ind):
   if not varflag then
     ERROR(cat(obsvar,` is not an element of `,stvars,`.`)) fi;

   if nargs<5 then invars:=[u] elif type(args[5],name) then
      invars:=[args[5]] else invars:=args[5] fi;
   if not type(invars,list(name)) then
     ERROR(cat(`fifth argument should be a name or a list of names of`,
            `input variables.`),`If default: u must be unassigned.`) fi;
   for var in invars do
      for i from 0 to n do
         if not type(eval(cat(var,i)),name) then
           ERROR(cat(cat(var,i),` must be unassigned.`)) fi;
         od:                #This is a little tough.
   od:

   if nargs<6 then outvar:=y else outvar:=args[6] fi;
   if not type(outvar,name) then
     ERROR(`sixth argument should be an output variable name.`,
           `If default: y must be unassigned.`) fi;
   for i from 0 to n do
      if not type(eval(cat(outvar,i)),name) then
        ERROR(cat(cat(outvar,i),` must be unassigned.`)) fi;
      od:

   if nargs<7 then form:='expl' else form:=args[7] fi;

   for var in invars do
      h:=subs(var=cat(var,0),h):
      f:=subs(var=cat(var,0),op(f)):
   od:

   dinvars:=[]:
   for i from 0 to n do
      for var in invars do
         dinvars:=[cat(var,i),op(dinvars)]
      od:
   od:
   IOvars:=[]:
   for i from 0 to n do
      for var in [op(invars),outvar] do
         IOvars:=[cat(var,i),op(IOvars)]
      od:
   od:

   OKvars:=convert(IOvars,set): #To handle parameters in the state eqs.
   for i to n do
      OKvars:=(indets(f[i]) minus convert(stvars,set)) union OKvars
   od:

   L:=liederlist(h,f,n,stvars,invars):
   L:=zip((X,Y)->(X-Y),[seq(cat(outvar,i),i=0..n)],[op(L)]);
   L:=map(numer@expand,L):
   G:=grobner[gbasis](L,[op(stvars[1..ind-1]),
         op(stvars[ind+1..n]),obsvar,op(IOvars)],'plex');

#  Check the GB for relations involving obsvar and IOvars only:

   res:=[]; obsflag:=false:
   for i to nops(G) do
      if (indets(G[i]) minus OKvars = {obsvar}) then
         obsflag:=true:
         if form='impl' then res:=[op(res),collect(G[i],obsvar)];
         elif degree(G[i],obsvar)=1 then res:=[op(res),solve(G[i],obsvar)];
      fi;
      fi;
   od;
   if not obsflag then
      print(cat(obsvar,` is not algebraically observable.`))
   elif form='expl' then
      if res=[] then
         print(cat(`No explicit expression for `,obsvar,` exists.`))
      else res; fi;
   else res; fi
end:

######################################################################

######################################################################

# "Observer relation" for discrete time rational SISO system
# in state-space form. Requires liehomlist. Uses finduni.
# Syntax: obsvd(f,h,obsvar,stvars,invars,outvar,form)
# obsvar must be a member of stvars.
# K. Forsman 1992-06-16, 1992-08-07

obsvd:=proc()
   local dinvars,f,form,G,h,i,ind,invars,IOvars,n,L,obsflag,obsvar,
         OKvars,outvar,res,stvars,var,varflag:

   if nargs<3 then ERROR(`not enough input arguments.`) fi;
   f:=args[1]: h:=args[2]: obsvar:=args[3]:

   if not type(f,vector) then
     ERROR(`first argument should be a vector.`) fi;
   if not type(h,scalar) then
     ERROR(`second argument should be a scalar.`) fi;
   if not type(obsvar,name) then
     ERROR(`third argument should be a variable name.`) fi;

   n:=linalg[vectdim](f):
   if nargs<4 then stvars:=[x.(1..n)] else stvars:=args[4] fi;
   if not type(stvars,list(name)) then
     ERROR(`fourth argument should be a list of names of state variables`,
           `If default: x1,...,xn must be unassigned.`) fi;
   if nops(stvars)<>n then
     ERROR(`wrong number of state variables given.`) fi;

   varflag:=member(obsvar,stvars,ind):
   if not varflag then
     ERROR(cat(obsvar,` is not an element of `,stvars,`.`)) fi;

   if nargs<5 then invars:=[u] elif type(args[5],name) then
      invars:=[args[5]] else invars:=args[5] fi;
   if not type(invars,list(name)) then
     ERROR(cat(`fifth argument should be a name or a list of names of`,
            `input variables.`),`If default: u must be unassigned.`) fi;
   for var in invars do
      for i from 0 to n do
         if not type(eval(cat(var,i)),name) then
           ERROR(cat(cat(var,i),` must be unassigned.`)) fi;
         od:                #This is a little tough.
   od:

   if nargs<6 then outvar:=y else outvar:=args[6] fi;
   if not type(outvar,name) then
     ERROR(`sixth argument should be an output variable name.`,
           `If default: y must be unassigned.`) fi;
   for i from 0 to n do
      if not type(eval(cat(outvar,i)),name) then
        ERROR(cat(cat(outvar,i),` must be unassigned.`)) fi;
      od:

   if nargs<7 then form:='expl' else form:=args[7] fi;

   for var in invars do
      h:=subs(var=cat(var,0),h):
      f:=subs(var=cat(var,0),op(f)):
   od:

   dinvars:=[]:
   for i from 0 to n do
      for var in invars do
         dinvars:=[cat(var,i),op(dinvars)]
      od:
   od:

   IOvars:=[]:
   for i from 0 to n do
      for var in [op(invars),outvar] do
         IOvars:=[cat(var,i),op(IOvars)]
      od:
   od:

   OKvars:=convert(IOvars,set): #To handle parameters in the state eqs.
   for i to n do
      OKvars:=(indets(f[i]) minus convert(stvars,set)) union OKvars
   od:

   L:=liehomlist(h,f,n,stvars,invars):
   L:=zip((X,Y)->(X-Y),[seq(cat(outvar,i),i=0..n)],[op(L)]);
   L:=map(numer@expand,L):
   G:=grobner[gbasis](L,[op(stvars[1..ind-1]),
         op(stvars[ind+1..n]),obsvar,op(IOvars)],'plex');

#  Check the GB for relations involving obsvar and IOvars only:

   res:=[]; obsflag:=false:
   for i to nops(G) do
      if (indets(G[i]) minus OKvars = {obsvar}) then
         obsflag:=true:
         if form='impl' then res:=[op(res),collect(G[i],obsvar)];
         elif degree(G[i],obsvar)=1 then res:=[op(res),solve(G[i],obsvar)];
      fi;
      fi;
   od;
   if not obsflag then
      print(cat(obsvar,` is not algebraically observable.`))
   elif form='expl' then
      if res=[] then
         print(cat(`No explicit expression for `,obsvar,` found.`))
      else res; fi;
   else res; fi
end:

# Find the transformed system T(f) where f is a vector and T a list
# expressing the new states in terms of the old ones.
# Continuous time.
# Syntax: newsysc(f,T,ost,nst,invar,form)
# The parameter form determines the form of the output:
#   if form='expl' (default) then the output is a vector (rhs of the new system)
#   and if form='impl' then the output is a list (polynomials defining dz).
#   Thus form='impl' means that implicit state equations are allowed.
# Requires lieder.
# K. Forsman 1992-02-24, 1992-03-02

newsysc:=proc()
   local dz,dzdef,explflag,f,failind,form,g,i,invar,j,L,n,nst,ost,T,zdef:

   if nargs<2 then ERROR(`not enough input arguments.`) fi;

   f:=args[1]: T:=args[2]:
   if not type(f,vector) then
     ERROR(`first argument should be a vector.`) fi;
   if not type(T,list) then
     ERROR(`second argument should be a list.`) fi;

   n:=linalg[vectdim](f);

   if nargs<3 then ost:=[x.(1..n)] else ost:=args[3] fi;
   if not type(ost,list) then
      ERROR(`third argument should be a list of state variables.`) fi;
   if nargs<4 then nst:=[z.(1..n)] else nst:=args[4] fi;
   if not type(nst,list) then
      ERROR(`fourth argument should be a list of state variables.`) fi;
   if nargs<5 then invar:=u else invar:=args[5] fi;
   if not type(invar,name) then
      ERROR(`fifth argument should be an input variable name.`,
            `If default: u must be unassigned.`) fi;
   if nargs<6 then form:='expl' else form:=args[6] fi;

   f:=subs(invar=cat(invar,0),op(f)):
   T:=subs(invar=cat(invar,0),T):

   zdef:=zip((X,Y)->(X-Y),nst,T):
   dz:=[seq(cat('d',nst[j]),j=1..n)];
   for i to n do if not type(cat('d',(nst[i])),name) then
      ERROR(cat('d',nst[i],` must not be assigned.`)) fi od;

   dzdef:=zip((X,Y)->(X-Y),dz,map(lieder,T,f,1,ost,invar)):
   L:=[op(dzdef),op(zdef)]:
   L:=map(numer@expand,L):
   g:=linalg[vector](n);
   explflag:=true:

   for i to n while explflag do
      g[i]:=grobner[finduni](dz[i],L,{op(ost),op(dz)}):
      explflag:=evalb(form='impl' or degree(g[i],dz[i])=1);
      if not explflag then failind:=i fi;
   od:

   if form='impl' then
      g:=convert(g,list);
   elif not explflag then
        print(cat(`No explicit expression for `,dz[failind],` exists.`));
   else
        g:=zip(solve,g,linalg[vector](dz));
   fi
end:

######################################################################

######################################################################


# Find the transformed system T(f) where f is a vector and T a list
# expressing the new states in terms of the old ones.
# Discrete time.
# Syntax: newsysd(f,T,ost,nst,invar,form)
# The parameter form determines the form of the output:
#   if form='expl' (default) then the output is a vector (rhs of the new system)
#   and if form='impl' then the output is a list (polynomials defining dz).
#   Thus form='impl' means that implicit state equations are allowed.
# Requires liehom.
# K. Forsman 1992-02-25, 1992-03-02

newsysd:=proc()
   local dz,dzdef,explflag,f,failind,form,g,i,invar,j,L,n,nst,ost,T,zdef:

   if nargs<2 then ERROR(`not enough input arguments.`) fi;

   f:=args[1]: T:=args[2]:
   if not type(f,vector) then
     ERROR(`first argument should be a vector.`) fi;
   if not type(T,list) then
     ERROR(`second argument should be a list.`) fi;

   n:=linalg[vectdim](f);

   if nargs<3 then ost:=[x.(1..n)] else ost:=args[3] fi;
   if not type(ost,list) then
      ERROR(`third argument should be a list of state variables.`) fi;
   if nargs<4 then nst:=[z.(1..n)] else nst:=args[4] fi;
   if not type(nst,list) then
      ERROR(`fourth argument should be a list of state variables.`) fi;
   if nargs<5 then invar:=u else invar:=args[5] fi;
   if not type(invar,name) then
      ERROR(`fifth argument should be an input variable name.`,
            `If default: u must be unassigned.`) fi;
   if nargs<6 then form:='expl' else form:=args[6] fi;

   f:=subs(invar=cat(invar,0),op(f)):
   T:=subs(invar=cat(invar,0),T):

   zdef:=zip((X,Y)->(X-Y),nst,T):
   dz:=[seq(cat('d',nst[j]),j=1..n)];
   for i to n do if not type(cat('d',(nst[i])),name) then
      ERROR(cat('d',nst[i],` must not be assigned.`)) fi od;

   dzdef:=zip((X,Y)->(X-Y),dz,map(liehom,T,f,1,ost,invar)):
   L:=[op(dzdef),op(zdef)]:
   L:=map(numer@expand,L):
   g:=linalg[vector](n);
   explflag:=true:

   for i to n while explflag do
      g[i]:=grobner[finduni](dz[i],L,{op(ost),op(dz)}):
      explflag:=evalb(form='impl' or degree(g[i],dz[i])=1);
      if not explflag then failind:=i fi;
   od:

   if form='impl' then
      convert(g,list);
   elif not explflag then
        print(cat(`No explicit expression for `,dz[failind],` exists.`));
   else
        g:=zip(solve,g,linalg[vector](dz));
   fi
end:

# Function for computing critical d for local Lyapunov functions.
# lambda eliminated before finduni or gbasis is applied.
# Syntax: loclyap(f,V,vars,lvar,met)
# K. Forsman 1992-03-03, 1992-08-06

loclyap:=proc()
   local f,g,gdeg,i,ind,L,lvar,md,met,mindeg,minind,n,Q,
         V,vars,Vx,Vxdeg:

   if nargs<2 then ERROR(`not enough input arguments.`) fi;

   f:=args[1]: V:=args[2]:
   if not type(f,vector) then
     ERROR(`first argument should be a vector.`) fi;
   if not type(V,scalar) then
     ERROR(`second argument should be a scalar.`) fi;
   n:=linalg[vectdim](f):
   if nargs<3 then vars:=[x.(1..n)] else vars:=args[3] fi;
   if not type(vars,list(name)) then
     ERROR(`third argument should be a list of names of state variables`,
           `If default: x1,...,xn must be unassigned.`) fi;
   if nops(vars)<>n then
     ERROR(`wrong number of variables given in parameter vars.`) fi;
   if nargs<4 then lvar:=d else lvar:=args[4] fi;
   if not type(lvar,name) then
     ERROR(`fourth argument should be a name.`,
           `If default: d must be unassigned.`) fi;
   if nargs<5 then met:='pol' else met:=args[5] fi;

   Vx:=linalg[grad](V,linalg[vector](vars)):
   Q:=linalg[dotprod](f,Vx):
   g:=evalm(linalg[grad](Q,linalg[vector](vars))):
   gdeg:=map(degree,convert(g,list),{op(vars)});
   Vxdeg:=map(degree,convert(Vx,list),{op(vars)});
   md:=zip(max,gdeg,Vxdeg);
   mindeg:=min(op(md));
   member(mindeg,md,minind);
   ind:=[$(1..(minind-1)),$((minind+1)..n)];
   L:=[];

   for i in ind do
	L:=[op(L),g[i]*Vx[minind]-g[minind]*Vx[i]];
   od;
   L:=map(expand,[V-lvar,Q,op(L)]);

   if met='tot' then
         grobner[gbasis](L,[op(vars),lvar],'plex')
      else
         factor(grobner[finduni](lvar,L,{op(vars),lvar}));
   fi
end:

# K. Forsman 1992-06-15

############################################################


polycon := `The polynomial and rational control systems package`:
#save `polycon.m`;
#quit
