#
## <SHAREFILE=numerics/Newton/Newton.mpl >
## <DESCRIBE>
##      An implementation of a Newton iteration for solving non-linear
##      systems of equations and a Maple worksheet with some examples.
##                AUTHOR: Ross Taylor, taylor@sun.soe.clarkson.edu
## </DESCRIBE>
## <UPDATE=R4 >

Newton := proc (Eqns1:: {equation,list,vector},
                 Vars:: {equation,list(name=numeric),vector(name=numeric)})

local neqns,        # number of equations
      nvars,        # number of variables
      Fvec,         # vector of functions
      Jmat,         # array containing jacobian element functions
      i, j, k,      # various counters
      dfij,         # used to flag zero in jacobian matrix
      delx,         # change in x vector
      del1,         # adjusted change in x vector
      nrm,          # norm of function vector
      ff,           # function vector
      JJ,           # jacobian matrix
      itercount,    # records current iteration number
      a,s,t,        # used in processing optional arguments
      eps,          # tolerance
      infolevel,    # used to control printed output
      maxit,        # maximum number of iterations
      nextstep,     # name of proc used to control steps
      x,            # vector of variable names
      xx,           # vector of variable values
      xstart,       # vector of initial values for entries in x
      Eqns,         # copy of Eqns1 in argument list
      Varlist,      # copy of Vars in argument list
      Varseq;       # Sequence of variables

# Determine number of equations
if type (Eqns1, equation) and type (Vars, equation) then
  Eqns := [Eqns1];
  Varlist := [Vars];
else
  Eqns := Eqns1;
  Varlist := Vars;
fi;

neqns := vectdim(Eqns);
nvars := vectdim(Varlist);
x := vector(nvars);
xstart := vector(nvars);

for i from 1 to nvars do
     x[i] := lhs(Varlist[i]);
     xstart[i] := rhs(Varlist[i]);
od;

if neqns <> nvars then
   ERROR(`Numbers of equations and variables are not the same`);
fi;

# look over optional arguments and set parameters
eps := 10^(-Digits+2);
infolevel := vector([0,0,0,0,0,0]);
maxit := 25;
nextstep := NULL;
for a in [args[3..nargs]] do
        if not type(a,equation) then
          ERROR(`Incorrect optional argument`,a)
        fi;
        s := lhs(a); t := rhs(a);
        if s = 'tolerance'  then
          eps := t;
          if eps < 10^(-Digits+2) then
            lprint(`WARNING: tolerance may be too small`)
          fi;
        elif s = 'output' then
            if type(t,list) or type(t,set) then
                if member (`norm`,t) then infolevel[1] := 1 fi;
                if member(`variables`,t) then infolevel[2] := 1 fi;
                if member(`functions`,t) then infolevel[3] := 1 fi;
                if member(`jacobian`,t) then infolevel[4] := 1 fi;
                if member(`sparsity`,t) then infolevel[5] := 1 fi;
                if member(`symbolics`,t) then infolevel[6] := 1 fi;
            fi;
        elif s = 'iterations' then
            maxit := t;
        elif s = 'steps' then
            if type (t,procedure) then
               nextstep := t;
            else
               ERROR (`Optional argument should be a procedure name`);
            fi;
        fi;
od;
# Define function vector and jacobian matrix of appropriate size
Fvec:=vector(neqns):
Jmat := array(sparse,1..neqns,1..neqns):

# Create variable sequence
Varseq := seq(x[k],k=1..neqns);

# Create function vector
for i from 1 to neqns do
  if type(Eqns[i],`=`) then
    Fvec[i]:=unapply(lhs(Eqns[i])-rhs(Eqns[i]),Varseq);
  else
   Fvec[i]:=unapply(Eqns[i],Varseq);
  fi;
od;

if infolevel[6] > 0 then print(Fvec(Varseq)); fi;

# Now for the Jacobian matrix
for i from 1 to neqns do
  for j from 1 to neqns do
    if has(Fvec[i](Varseq),x[j]) then
      dfij := diff(Fvec[i](Varseq),x[j]);
      Jmat[i,j] := unapply(dfij,Varseq);
    fi;
  od;
od;

if infolevel[6] > 0 then print(Jmat(Varseq)); fi;
if infolevel[5] > 0 then
  print(plots[sparsematrixplot](Jmat(Varseq),title=`Sparsity pattern of jacobian (upside down)`));
fi;

# Define a few vectors and matrices for working space
xx := vector(neqns):
xx := xstart;
delx := vector(neqns);
ff := vector(neqns);
del1 := vector(neqns);
JJ := array(sparse,1..neqns,1..neqns);

# Evaluate the function vector and jacobian matrix and the norm of the function vector

ff:=Fvec(seq(xx[k],k=1..neqns));
JJ:=Jmat(seq(xx[k],k=1..neqns));
nrm := evalf(norm(ff,1));

# Perform iterations
itercount := 0;
while nrm > eps do

   # Print out various things according to inflolevel
   if infolevel[1] > 0 then print(`Iteration `. itercount, `   Norm = `, nrm); fi;
   if infolevel[2] > 0 then print(seq(x[k]=xx[k],k=1..nvars)); fi;
   if infolevel[3] > 0 then print(seq(`f`[k]=ff[k],k=1..nvars)); fi;
   if infolevel[4] > 0 then print(JJ); fi;

   # Solve linear system to get vector of corrections
   delx:=linsolve(JJ,ff);

   # Modify steps if required by optional arguments
   if nextstep <> NULL then
      del1:=nextstep(delx,xx,x);
   else
      del1 := delx;
   fi;

   # Compute new set of variable values and the next function vector etc.
   xx:=evalf(evalm(xx-del1));
   ff:=Fvec(seq(xx[k],k=1..neqns));
   JJ:=Jmat(seq(xx[k],k=1..neqns));
   nrm := evalf(norm(ff,1));

   # Increment iteration count and check that we have not done too many iterations
   itercount := itercount+1;
   if itercount > (maxit + 1) then ERROR(`Maximum number of iterations exceeded`);  fi;

od:

# If we have made it this far we probably have a converged solution
if neqns=1 then 
  RETURN(x[1]=xx[1]);
else 
  RETURN([seq(x[k]=xx[k],k=1..nvars)]);
fi;
end:

#
# Here we give some procedures for controlling the Newton step.
# The first limits the correction to some maximum step size that
# the user determines.  The use of these routines will become
# clear later.

maxchange := proc(delx, dxmax)
# Procedure to limit the maximum step size
# delx - original step
# dxmax - maximum step size (must be positive number)
if delx > dxmax then
  dxmax elif delx < -dxmax then-dxmax
else
  delx
fi;
end:

# Now a routine that uses maxchange.
teststep := proc(delx)
# Procedure for limiting Newton steps (called from Newton)
# delx - vector of original steps
#           proc returns a vector of steps modified accoring to the rules in this proc
# maxstep is a global vriable set outside the proc so that user need not pass a value by call to Newton
local i, nvars, newstep;
if not type(maxstep,numeric)  then ERROR(`Please set global variable named maxstep before using teststep`); fi;
nvars := linalg[vectdim](delx);
newstep := linalg[vector](nvars);
for i from 1 to nvars do newstep[i] := maxchange(delx[i], maxstep); od;
evalm(newstep);
end:

# Here is a proc for simply damping (or accelerating) the Newton step.
# Nothing fancy like line searching here.  Just multiplication of the
# Newton step vector with a user determined (global) number.
dampstep := proc(delx)
# Procedure for limiting Newton steps (called from Newton)
# delx - vector of original steps
# dampfactor is a global vriable set outside the proc so that user
# need not pass a value by call to Newton.
local i, newstep, nvars;
if not type(dampfactor,numeric) then ERROR(`Please set global variable named dampfactor before using dampstep`); fi;
nvars := linalg[vectdim](delx);
newstep := linalg[vector](nvars);
for i from 1 to nvars do newstep[i] := dampfactor * delx[i]; od;
evalm(newstep);
end:

trustregion := proc (x, delx, xmin, xmax) 
local temp; 
temp := x-delx; 
if temp < xmin then 
  RETURN (1/2*x-1/2*xmin);
elif xmax < temp then 
  RETURN (1/2*x-1/2*xmax); 
else 
  RETURN (delx);
fi; 
end:

lowerbound := proc (x, delx, xmin) 
local temp; 
temp := x-delx; 
if temp < xmin then 
  RETURN (1/2*x-1/2*xmin);
else 
  RETURN (delx);
fi; 
end:

upperbound := proc (x, delx, xmax) 
local temp; 
temp := x-delx; 
if xmax < temp then 
  RETURN (1/2*x+1/2*xmin);
else 
  RETURN (delx);
fi; 
end:
