#
## <SHAREFILE=algebra/galois/galois.mpl >
## <DESCRIBE>
##   (update - extensions to degree 8)
##                Computes the Galois group of a polynomial of degree <= 8
##                Extends the functionality of the library routine galois
##                from Q[x] to Q[t1,t2,...,tm][x] i.e. polynomials with
##                polynomial coefficients.  Note, the Galois group computed
##                does not specialize correctly for particular values of
##                the parameters t1, t2, ..., tm.  The code is presently being
##                extended to higher degree.
##                AUTHORS: R. Sommeling, T. Mattman,
##                AUTHORS: J McKay, mckay@vax2.concordia.ca
## </DESCRIBE>
## <UPDATE=R4update >

# Copyright 1987 R. Sommeling; all rights reserved
#                                                           April 25, 1987
# Extensions to degree 8 case,
# Copyright 1992 T.Mattman; all rights reserved
#                                                           February 18, 1992

# Input:  A polynomial f in x, the indeterminate x and a non-negative
#         integer r.
# Output: A primitive polynomial of degree binomial(n,r) in x whose roots are
#         the products of r distinct roots of f.

rsetpol := proc(f,x,r)
local a,b,g,h,hj,i,j,k,n,rb,s,s2;
options `Copyright 1987 by Ron Sommeling`;
    if not type(x,'name') then
        ERROR('`second argument should be a variable`')
    fi;
    g := normal(f);
    if not type(g,'polynom'('rational',x)) then
        ERROR('`first argument should be a polynomial`')
    fi;
    if not type(r,'integer') or (r < 0) then
        ERROR('`third argument should be a natural number`')
    fi;
    g := expand(g);
    g := expand(g/lcoeff(g,x));
    n := degree(g,x);
    if r = 0 then x-1
    elif r = 1 then g
    elif r = n then x-(-1)^n*coeff(g,x,0)
    elif n < r then 1
    else
        b := binomial(n,r);
        rb := r*b;
        h := array(1 .. r);
        a := array(0 .. n,['coeff(g,x,n-i)' $ ('i' = 0 .. n)]);
        s := a_to_s(op(a),n,rb);
        s2 := array(1 .. b);
        for i to b do
            for j to r do
                hj := s[i*j];
                for k to j-1 do  hj := normal(hj+h[k]*s[i*(j-k)]) od;
                h[j] := -hj/j
            od;
            s2[i] := (-1)^r*h[r]
        od;
        a := s_to_a(op(s2),b);
        i := 'i';
        g := sum(a[b-i]*x^i,i = 0 .. b);
        g/content(g,x)
    fi
end:

# Input:  A polynomial f in x and the indeterminate x.
# Output: A primitive polynomial of degree n(n-1) in x whose roots are
#         xi + 2 xj, where xi and xj are distinct roots of f.

twoseqpol := proc(f,x)
local a,d,g,i,j,n,s,s2,si;
options `Copyright 1987 by Ron Sommeling`;
    if not type(x,'name') then
        ERROR('`second argument should be a variable`')
    fi;
    g := normal(f);
    if not type(g,'polynom'('rational',x)) then
        ERROR('`first argument should be a polynomial`')
    fi;
    g := expand(g);
    g := expand(g/lcoeff(g,x));
    n := degree(g,x);
    d := n*(n-1);
    s2 := array(1 .. d);
    a := array(0 .. n,['coeff(g,x,n-i)' $ ('i' = 0 .. n)]);
    s := a_to_s(op(a),n,d);
    for i to d do
        si := 0;
        for j from 0 to i do
            si := normal(si+2^j*binomial(i,j)*(s[j]*s[i-j]-s[i]))
        od;
        s2[i] := si
    od;
    a := s_to_a(op(s2),d);
    i := 'i';
    g := sum(a[d-i]*x^i,i = 0 .. d);
    g/content(g,x)
end:

# Input:  An array a(0..n) of the coefficients of a monic polynomial f of
#         degree n (so a[0]=1), an integer n, and an integer m.
# Result: An array(0..m) with the powersums x1^i + ... + xn^i stored
#         in the i'th element for i from 0 to m, where the xi s are
#         the roots of f.

a_to_s := proc(a,n,m)
local s,i,j,n2,si;
options `Copyright 1987 by Ron Sommeling`;
    s := array(0 .. m);
    s[0] := n;
    for i to m do
        if n < i then si := -a[n]*s[i-n]; n2 := n-1
        else si := -i*a[i]; n2 := i-1
        fi;
        for j to n2 do  si := normal(si-a[j]*s[i-j]) od;
        s[i] := si
    od;
    RETURN(op(s))
end:

# Input:  An array s(1..m-1) of the powersums of a monic polynomial f of
#         degree m, an integer m .
# Result: An array 'a[0..m-1]' such that a[i] == coeff(f,x,m-i)

s_to_a := proc(s,m)
local a,ai,i,j;
options `Copyright 1987 by Ron Sommeling`;
    a := array(0 .. m,[1]);
    for i to m do
        ai := s[i];
        for j to i-1 do  ai := normal(ai+a[j]*s[i-j]) od;
        a[i] := -ai/i
    od;
    RETURN(op(a))
end:

# Copyright 1987 R. Sommeling; all rights reserved
#
# Input:  An univariate polynomial f in x with integer coefficients,
#         the indeterminate x, a prime p such that p doesn't divide the
#         leading coefficient of f and f is squarefree mod p.
# Output: A sequence of non-increasing integers which are the degrees of
#         the factors of f mod p. To compute these degrees it is not
#         necessary to compute the actual factors of f mod p.
#                                                           April 25, 1987

cyclepattern := proc(f,x,p)
local a,i,n;
options `Copyright 1987 by Ron Sommeling`;
    a := modp1(ConvertIn(f,x),p);
    a := modp1(DistDeg(a),p);
    n := nops(a);
    a := ['a[n-i]' $ (i = 0 .. n-1)];
    a := map(proc(L,p)
             local i,j,k;
                 i := modp1(Degree(L[1]),p);
                 j := L[2];
                 k := iquo(i,j);
                 if j = 0 then NULL else j $ k fi
             end                                 ,a,p);
    op(a)
end:

# Copyright 1987 R. Sommeling; all rights reserved
#
# Extensions to degree 8 case
# Copyright 1992 T. Mattman; all rights reserved
#
# The array "groups" contains all transitive groups of degree up to 8.
#
# groups[n, 0] = set of all transitive groups of degree n which are
#                contained in the alternating group of degree n.
# groups[n, 1] = set of all transitive groups of degree n which are not
#                contained in the alternating group of degree n.
#
# A group is represented by a list of 3 or more elements:
# - The first element is a string which is the name of the group as
#   given in [1].
# - The second element is the order of the group.
# - The third element is a set of generators of the group (up to conjugacy).
#   A generator is a string which is the disjoint cycle representation of
#   the generator.
# - The fourth element (for groups of degree 4 up to 8) is a list of zeros
#   and ones indicating whether a cycle type occurs in the group (see [1]).
#   0 = cycle type doesn't occur; 1 = cycle type occurs.
#   The cycle type are ordered lexicographically starting with 2 1 1 ... 1
#   (a 2-cycle) and ending with n (an n-cycle).
# - For degree 4 up to 7,
#   - The fifth up to the one but last element are sorted lists of the
#     orbit-length partition of 2-sets, 3-sets, etc. (see [2]).
#   - The last element (for groups of degree 4 up to 7) is a sorted list of
#     the orbit-length partition of 2-sequences (see [2]).
# - For degree 8,
#    - The initial elements are as described for degree 4 through 7.
#    - An additional element follows which indicates the Galois group
#      of a factor of a resolvent polynomial. If the group is even, this
#      is the terminal element.
#    - If the group is not even, additional elements are added, one for each
#      of the r-set or 2-sequences polynomials. These extra elements indicate
#      whether or not the factors of the polynomial are irreducible over
#      the field extended by the square root of the discriminant.
#
# Example1: One of the elements in the set "groups[6, 1]" is:
#
#          [`S4/Z4`, 24, {`(1 2)(3 4)(5 6)`, `(1 2 3)(4 5 6)`},
#            [0, 1, 1, 0, 0, 1, 1, 0, 0, 0],
#              [3, 12], [8, 12], [6, 24]]
#
# This means that this is a transitive group of degree 6 which is not
# contained in the A6 (the alternating group of order 6).
# The name of the group is "S4/Z4", its order is 24 and it is generated
# by (1 2)(3 4)(5 6) and (1 2 3)(4 5 6) (up to conjugacy).
# Only the following cycle types appear in the group:
#
#                       1 1 1 1 1 1
#                       2 2 1 1
#                       2 2 2
#                       3 3
#                       4 1 1
#
# The orbit-length partition of 2-sets is 3, 12.
# The orbit-length partition of 3-sets is 8, 12.
# The orbit-length partition of 2-sequences is 6, 24.
#
# Example2: One of the elements in the set "groups[8, 1]" is:
#
#  [`T40`, 192, {`(1 6)(2 5)(3 7)(4 8)`, `(3 5 7)(4 6 8)`,
#                   `(1 3)(2 4)`},
#     [0, 1, 1, 1, 0, 0, 0, 1, 0, 1, 0, 1,
#         0, 1, 0, 0, 0, 0, 1, 0, 1],
#     [4, 24], [24, 32], [6, 16, 48], [8, 48],
#     [5, 8, `+T34`, {`+T34`, `+T41`}],
#     [{-4}, {-24}], [{-24}, {-32}], [{-6}, {16}, {-48}],
#     [{-8}, {-48}]],
#
# The first 8 elements are interpretted as above. The 9th element indicates
# that the degree 8 factor of the 5th polynomial has group T34. The other
# groups which are otherwise indistinguishable from T40 will have one of
# T34 or T41 as Galois group for this factor. The polynomials are numbered
# as they occur in the groups array listing. Hence for degree 8, the 1st
# polynomial is the 2-set, the 2nd is the 3-set, the 3rd is the 4-set and
# the 4th is the 2-sequence. The 5th polynomial referred to here is a special
# polynomial of degree 35 with roots of the form (x1+x2+x3+x4-x5-x6-x7-x8)^2.
#    The remaining elements indicate irreducibility over sqrt(D). A "-"
# means irreducible while no sign means reducible. Thus the factors of the
# 4-set polynomial factor as follows over the extended field:
#    degree 6 factor - irreducible
#    degree 16 factor - reducible
#    degree 48 factor - irreducible
#
# Note that two dummy entries are included for degree 12. They represent the
# Galois groups corresponding to degree 12 factors of the 2-set polynomials
# of the degree 8 groups +T42 and +T45.
#
# References:
# [1] G. Butler & J. McKay, The transitive permutation groups of degree up
#     to eleven, Comm. Alg. 11 (1983), pp. 863-911.
# [2] L.H. Soicher & J. McKay, Computing Galois groups over the rationals,
#     Number Theory 20 (1985), pp. 273-281.
# [3] J. McKay & E. Regener, Actions of permutation groups on r-sets,
#     Comm. Alg. 13 (1985), pp.619-630.
#                                                           April 25, 1987
#                                                           February 18, 1992

groups := array(1..12, 0..1, [

[ {[`+Id`, 1, {}]}, {} ],
[ {}, {[`S2`, 2, {`(1 2)`}]} ],
[ {[`+A3`, 3, {`(1 2 3)`}]},
  {[`S3`, 6, {`(1 2 3)`, `(1 2)`}]} ],

[ {[`+V4`, 4, {`(1 2)(3 4)`, `(1 3)(2 4)`},
     [0, 1, 0, 0],
       [2, 2, 2], [4, 4, 4]],
   [`+A4`, 12, {`(1 2 3)`, `(1 2)(3 4)`},
     [0, 1, 1, 0],
       [6], [12]]},

  {[`Z4`, 4, {`(1 2 3 4)`},
     [0, 1, 0, 1],
       [2, 4], [4, 4, 4]],
   [`D4`, 8, {`(1 2 3 4)`, `(1 3)`},
     [1, 1, 0, 1],
       [2, 4], [4, 8]],
   [`S4`, 24, {`(1 2 3 4)`, `(1 2)`},
     [1, 1, 1, 1],
       [6], [12]]} ],

[ {[`+Z5`, 5, {`(1 2 3 4 5)`},
     [0, 0, 0, 0, 0, 1],
       [5, 5], [5, 5, 5, 5]],
   [`+D5`, 10, {`(1 2 3 4 5)`, `(2 5)(3 4)`},
     [0, 1, 0, 0, 0, 1],
       [5, 5], [10, 10]],
   [`+A5`, 60, {`(1 2 3 4 5)`, `(1 2 3)`},
     [0, 1, 1, 0, 0, 1],
       [10], [20]]},

  {[`F20`, 20, {`(1 2 3 4 5)`, `(2 3 5 4)`},
     [0, 1, 0, 0, 1, 1],
       [10], [20]],
   [`S5`, 120, {`(1 2 3 4 5)`, `(1 2)`},
     [1, 1, 1, 1, 1, 1],
       [10], [20]]} ],

[ {[`+A4`, 12, {`(1 3 5)(2 4 6)`, `(1 2)(5 6)`},
     [0, 1, 0, 0, 0, 1, 0, 0, 0, 0],
       [3, 12], [4, 4, 6, 6], [6, 12, 12]],
   [`+S4/V4`, 24, {`(1 3 5)(2 4 6)`, `(1 6)(2 5)`},
     [0, 1, 0, 0, 0, 1, 0, 1, 0, 0],
       [3, 12], [4, 4, 12], [6, 24]],
   [`+3^2.4`, 36, {`(1 2 3)`, `(1 5 2 4)(3 6)`},
     [0, 1, 0, 1, 0, 1, 0, 1, 0, 0],
       [6, 9], [2, 18], [12, 18]],
   [`+PSL2(5)`, 60, {`(1 2 3 4 5)`, `(1 6)(2 5)`},
     [0, 1, 0, 0, 0, 1, 0, 0, 1, 0],
       [15], [10, 10], [30]],
   [`+A6`, 360, {`(1 2 3 4 5)`, `(4 5 6)`},
     [0, 1, 0, 1, 0, 1, 0, 1, 1, 0],
       [15], [20], [30]]},

  {[`Z6`, 6, {`(1 2 3 4 5 6)`},
     [0, 0, 1, 0, 0, 1, 0, 0, 0, 1],
       [3, 6, 6], [2, 6, 6, 6], [6, 6, 6, 6, 6]],
   [`S3`, 6, {`(1 5)(2 4)(3 6)`, `(1 6)(2 5)(3 4)`},
     [0, 0, 1, 0, 0, 1, 0, 0, 0, 0],
       [3, 3, 3, 6], [2, 6, 6, 6], [6, 6, 6, 6, 6]],
   [`D6`, 12, {`(1 2 3 4 5 6)`, `(2 6)(3 5)`},
     [0, 1, 1, 0, 0, 1, 0, 0, 0, 1],
       [3, 6, 6], [2, 6, 12], [6, 12, 12]],
   [`3.S3`, 18, {`(1 2 3)`, `(1 4)(2 5)(3 6)`},
     [0, 0, 1, 1, 0, 1, 0, 0, 0, 1],
       [6, 9], [2, 18], [6, 6, 18]],
   [`2.A4`, 24, {`(1 3 5)(2 4 6)`, `(1 2)`},
     [1, 1, 1, 0, 0, 1, 0, 0, 0, 1],
       [3, 12], [6, 6, 8], [6, 12, 12]],
   [`S4/Z4`, 24, {`(1 2)(3 4)(5 6)`, `(1 2 3)(4 5 6)`},
     [0, 1, 1, 0, 0, 1, 1, 0, 0, 0],
       [3, 12], [8, 12], [6, 24]],
   [`3^2.2^2`, 36, {`(1 2 3 4 5 6)`, `(1 3)(2 4)`},
     [0, 1, 1, 1, 0, 1, 0, 0, 0, 1],
       [6, 9], [2, 18], [12, 18]],
   [`2.S4`, 48, {`(1 2 3 4)`, `(1 5)(3 6)`},
     [1, 1, 1, 0, 0, 1, 1, 1, 0, 1],
       [3, 12], [8, 12], [6, 24]],
   [`3^2.D4`, 72, {`(1 2 3 4 5 6)`, `(1 3)`},
     [1, 1, 1, 1, 1, 1, 0, 1, 0, 1],
       [6, 9], [2, 18], [12, 18]],
   [`PGL2(5)`, 120, {`(1 2 3 4 5)`, `(1 6)(2 3)(4 5)`},
     [0, 1, 1, 0, 0, 1, 1, 0, 1, 1],
       [15], [20], [30]],
   [`S6`, 720, {`(1 2 3 4 5 6)`, `(1 2)`},
     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
       [15], [20], [30]]} ],

[ {[`+Z7`, 7, {`(1 2 3 4 5 6 7)`},
     [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1],
       [7, 7, 7], [7, 7, 7, 7, 7], [7, 7, 7, 7, 7, 7]],
   [`+F21`, 21, {`(1 2 3 4 5 6 7)`, `(2 3 5)`},
     [0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1],
       [21], [7, 7, 21], [21, 21]],
   [`+PSL3(2)`, 168, {`(1 2 3 4 5 6 7)`, `(2 3)(4 7)`},
     [0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 1],
       [21], [7, 28], [42]],
   [`+A7`, 2520, {`(1 2 3 4 5 6 7)`, `(1 2 3)`},
     [0, 1, 0, 1, 0, 1, 1, 0, 1, 0, 1, 0, 0, 1],
       [21], [35], [42]]},

  {[`D7`, 14, {`(1 2 3 4 5 6 7)`, `(2 7)(3 6)(4 5)`},
     [0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1],
       [7, 7, 7], [7, 7, 7, 14], [14, 14, 14]],
   [`F42`, 42, {`(1 2 3 4 5 6 7)`, `(2 4 3 7 5 6)`},
     [0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 1],
       [21], [14, 21], [42]],
   [`S7`, 5040, {`(1 2 3 4 5 6 7)`, `(1 2)`},
     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
       [21], [35], [42]]} ],

[ {[`+T2:2.4`, 8, {`(1 3 5 7)(2 4 6 8)`, `(1 6)(2 5)(3 8)(4 7)`},
     [0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 0],
     [4, 4, 4, 8, 8], [8, 8, 8, 8, 8, 8, 8],
     [2, 2, 2, 4, 4, 8, 8, 8, 8, 8, 8, 8], [8, 8, 8, 8, 8, 8, 8]],
   [`+T3:2^3`, 8, {`(1 5)(2 6)(3 7)(4 8)`, `(1 7)(2 8)(3 5)(4 6)`,
                `(1 6)(2 5)(3 8)(4 7)`},
     [0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 0, 0, 0, 0, 0, 0, 0, 0],
     [4, 4, 4, 4, 4, 4, 4], [8, 8, 8, 8, 8, 8, 8],
     [2, 2, 2, 2, 2, 2, 2, 8, 8, 8, 8, 8, 8, 8],
     [8, 8, 8, 8, 8, 8, 8]],
   [`+T4:D4`, 8, {`(1 3 5 7)(2 4 6 8)`, `(1 8)(2 7)(3 6)(4 5)`},
     [0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 0],
     [4, 4, 4, 4, 4, 8], [8, 8, 8, 8, 8, 8, 8],
     [2, 2, 2, 4, 4, 4, 4, 8, 8, 8, 8, 8, 8],
     [8, 8, 8, 8, 8, 8, 8]],
   [`+T5:Q8`, 8, {`(1 6 2 5)(4 8 3 7)`, `(1 7 2 8)(3 5 4 6)`},
     [0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 0],
     [4, 8, 8, 8], [8, 8, 8, 8, 8, 8, 8],
     [2, 2, 2, 8, 8, 8, 8, 8, 8, 8, 8],
     [8, 8, 8, 8, 8, 8, 8]],
   [`+T9`, 16, {`(1 3 5 7)(2 4 6 8)`, `(1 7)(2 8)(3 5)(4 6)`,
                 `(1 6)(2 5)(3 8)(4 7)`},
     [0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 0],
     [4, 4, 4, 8, 8], [8, 8, 8, 16, 16],
     [2, 2, 2, 4, 4, 8, 8, 8, 16, 16],
     [8, 8, 8, 16, 16]],
   [`+T10`, 16, {`(1 3 5 7)(2 4 6 8)`, `(1 6)(2 5)(3 7)(4 8)`},
     [0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 0],
     [4, 4, 4, 16], [8, 8, 8, 16, 16],
     [2, 4, 4, 4, 8, 8, 8, 16, 16],
     [8, 8, 8, 16, 16]],
   [`+T11`, 16, {`(1 6 2 5)(4 8 3 7)`, `(1 5)(2 6)(3 7)(4 8)`,
                  `(1 3)(2 4)(5 8)(6 7)`},
     [0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 0],
     [4, 8, 8, 8], [8, 8, 8, 16, 16],
     [2, 2, 2, 8, 8, 8, 8, 16, 16],
     [8, 8, 8, 16, 16]],
   [`+T12:SL(2,3)`, 24, {`(1 7 2 8)(3 5 4 6)`, `(3 5 7)(4 6 8)`},
     [0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 1, 0, 0],
     [4, 24], [8, 24, 24], [6, 8, 8, 24, 24], [8, 24, 24]],
   [`+T13:2.A4`, 24, {`(1 6)(2 5)(3 8)(4 7)`, `(3 5 7)(4 6 8)`},
     [0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0,
         0, 0, 0, 0, 0, 0, 1, 0, 0],
     [4, 12, 12], [8, 24, 24], [2, 6, 6, 8, 24, 24], [8, 24, 24],
     [1, 4, `+`]],
   [`+T14:S4`, 24, {`(3 5 7)(4 6 8)`, `(1 4)(2 3)(5 6)(7 8)`},
     [0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 0],
     [4, 12, 12], [8, 24, 24], [2, 6, 6, 8, 12, 12, 24],
     [8, 24, 24]],
   [`+T18`, 32, {`(1 3 5 7)(2 4 6 8)`, `(1 7)(2 8)(3 5)(4 6)`,
                  `(1 6)(2 5)(3 7)(4 8)`},
     [0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 0],
     [4, 4, 4, 16], [8, 16, 16, 16], [2, 4, 4, 4, 8, 8, 8, 32],
     [8, 8, 8, 32]],
   [`+T19`, 32, {`(1 3 5 7)(2 4 6 8)`, `(1 7)(2 8)(3 6)(4 5)`},
     [0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 0],
     [4, 8, 16], [8, 16, 32], [2, 4, 8, 8, 16, 32],
     [8, 16, 32], [1, 8, `T21`, {`T21`, `T31`}]],
   [`+T20`, 32, {`(1 3 5 7)(2 4 6 8)`, `(1 2)(7 8)`},
     [0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 0],
     [4, 8, 16], [8, 8, 8, 32], [2, 4, 8, 8, 16, 16, 16],
     [8, 16, 16, 16]],
   [`+T22`, 32, {`(1 6 2 5)(4 8 3 7)`, `(1 5)(2 6)(3 7)(4 8)`,
                  `(1 6)(2 5)(3 7)(4 8)`, `(1 7)(2 8)(3 5)(4 6)`},
     [0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 0],
     [4, 8, 8, 8], [8, 8, 8, 32], [2, 2, 2, 8, 8, 16, 16, 16],
     [8, 16, 16, 16]],
   [`+T24:2.S4`, 48, {`(1 6)(2 5)(3 8)(4 7)`, `(3 5 7)(4 6 8)`,
                  `(1 3)(2 4)`},
     [0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 1, 0, 0],
     [4, 12, 12], [8, 24, 24], [2, 6, 6, 8, 24, 24], [8, 24, 24],
     [1, 4, `-`]],
   [`+T25:2^3.7`, 56, {`(1 2 3 4 5 6 7)`, `(1 8)(2 4)(3 7)(5 6)`},
     [0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 0, 0, 0, 0, 0, 0, 1, 0],
     [28], [56], [14, 56], [56],
     [5, 7, `+Z7`, {`+Z7`, `+F21`, `+PSL3(2)`}]],
   [`+T29`, 64, {`(1 3 5 7)(2 4 6 8)`, `(1 7)(2 8)(3 5)(4 6)`,
                  `(1 7)(2 8)(3 6)(4 5)`},
     [0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 0],
     [4, 8, 16], [8, 16, 32], [2, 4, 8, 8, 16, 32], [8, 16, 32],
     [1, 8, `T31`, {`T21`, `T31`}]],
   [`+T32`, 96, {`(1 7)(2 8)(3 5)(4 6)`, `(1 6)(2 5)(3 7)(4 8)`,
                  `(3 5 7)(4 6 8)`},
     [0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 1, 0, 0],
     [4, 24], [24, 32], [6, 8, 8, 48], [8, 48], [1, 4, `+`]],
   [`+T33`, 96, {`(1 7 3 5)(2 8 4 6)`, `(2 4 3)(6 8 7)`},
     [0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 1, 0, 0],
     [12, 16], [8, 48], [2, 12, 24, 32], [24, 32],
     [5, 6, `+A4`, {`+A4`, `+S4/V4`}]],
   [`+T34`, 96, {`(1 4)(2 3)`, `(2 4 3)(6 8 7)`,
                  `(1 8)(2 5)(3 6)(4 7)`},
     [0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 0],
     [12, 16], [8, 48], [2, 12, 12, 12, 32], [24, 32]],
   [`+T36:2^3.(7.3)`, 168, {`(1 2 3 4 5 6 7)`, `(1 8)(2 4)(3 7)(5 6)`,
                  `(2 3 5)(4 7 6)`},
     [0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0,
         0, 0, 0, 0, 0, 0, 1, 1, 0],
     [28], [56], [14, 56], [56],
     [5, 7, `+F21`, {`+Z7`, `+F21`, `+PSL3(2)`}]],
   [`+T37:L(2,7)`, 168, {`(1 2 3 4 5 6 7)`, `(2 3 5)(4 7 6)`,
                  `(1 8)(2 7)(3 4)(5 6)`},
     [0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 0, 1, 0],
     [28], [56], [14, 14, 42], [56]],
   [`+T39`, 192, {`(1 6)(2 5)(3 7)(4 8)`, `(3 5 7)(4 6 8)`,
                   `(1 3)(2 4)`},
     [0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0,
         0, 1, 0, 0, 0, 0, 1, 0, 0],
     [4, 24], [24, 32], [6, 8, 8, 48], [8, 48], [1, 4, `-`]],
   [`+T41`, 192, {`(1 7 3 5)(2 8 4 6)`, `(2 4 3)(6 8 7)`,
                   `(1 8)(2 5)(3 6)(4 7)`},
     [0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0,
         0, 1, 0, 0, 0, 0, 1, 0, 0],
     [12, 16], [8, 48], [2, 12, 24, 32], [24, 32],
     [5, 6, `+S4/V4`, {`+A4`, `+S4/V4`}]],
   [`+T42`, 288, {`(1 3)(2 4)`, `(6 8 7)`,
                   `(1 5)(2 6)(3 7)(4 8)`},
     [0, 1, 0, 1, 1, 0, 1, 1, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 1, 0, 0],
     [12, 16], [8, 48], [2, 32, 36], [24, 32],
     [1, 12, `+12T1`, {`+12T1`, `+12T2`}]],
   [`+T45`, 576, {`(1 3)(2 4)`, `(6 8 7)`,
                   `(1 5)(2 6)(3 7)(4 8)`, `(1 8)(2 5)(3 6)(4 7)`},
     [0, 1, 0, 1, 1, 0, 1, 1, 0, 0, 1, 0,
         0, 1, 0, 0, 0, 0, 1, 0, 0],
     [12, 16], [8, 48], [2, 32, 36], [24, 32],
     [1, 12, `+12T2`, {`+12T1`, `+12T2`}]],
   [`+T48:2^3.L(3,2)`, 1344, {`(1 2 3 4 5 6 7)`, `(2 3)(4 7)`,
                    `(1 8)(2 4)(3 7)(5 6)`},
     [0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0,
         0, 1, 0, 0, 0, 0, 1, 1, 0],
     [28], [56], [14, 56], [56],
     [5, 7, `+PSL3(2)`, {`+Z7`, `+F21`, `+PSL3(2)`}]],
   [`+T49:A8`, 20160, {`(1 2 3 4 5 6 7)`, `(6 8 7)`},
     [0, 1, 0, 1, 1, 0, 1, 1, 0, 0, 1, 0,
         0, 1, 1, 0, 1, 0, 1, 1, 0],
     [28], [56], [70], [56]]},

  {[`T1:8`, 8, {`(1 4 6 8 2 3 5 7)`},
     [0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 1],
     [4, 8, 8, 8], [8, 8, 8, 8, 8, 8, 8],
     [2, 4, 8, 8, 8, 8, 8, 8, 8, 8],
     [8, 8, 8, 8, 8, 8, 8], [], [{4}, {8, 8, 8}],
     [{8, 8, 8, 8, 8, 8, 8}],
     [{2}, {4}, {8, 8, 8, 8, 8, 8, 8, 8}],
     [{8, 8, 8, 8, 8, 8, 8}]],
   [`T6`, 16, {`(1 4 6 8 2 3 5 7)`, `(1 7)(2 8)(3 6)(4 5)`},
     [0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 1],
     [4, 8, 8, 8], [8, 8, 8, 16, 16],
     [2, 4, 8, 8, 8, 8, 16, 16], [8, 16, 16, 16], [],
     [{-4}, {-8, 8, 8}], [{-8, -8, -8}, {16, 16}],
     [{-2}, {4}, {-8, -8, 8, 8}, {16, 16}],
     [{-8}, {16, 16, 16}]],
   [`T7`, 16, {`(1 4 6 8 2 3 5 7)`, `(3 4)(7 8)`},
     [0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 1],
     [4, 8, 16], [8, 8, 8, 16, 16],
     [2, 4, 8, 8, 16, 16, 16], [8, 8, 8, 16, 16], [],
     [{4}, {8}, {16}], [{8, 8, 8}, {16, 16}],
     [{2}, {4}, {8, 8}, {16, 16, 16}], [{8, 8, 8}, {16, 16}]],
   [`T8`, 16, {`(1 4 6 8 2 3 5 7)`, `(1 6)(2 5)(3 4)`},
     [0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 1],
     [4, 8, 16], [8, 8, 8, 16, 16],
     [2, 4, 8, 8, 16, 16, 16], [8, 16, 16, 16], [],
     [{-4}, {-8}, {16}], [{-8, -8, -8}, {16, 16}],
     [{-2}, {4}, {-8, -8}, {16, 16, 16}], [{-8}, {16, 16, 16}]],
   [`T15`, 32, {`(1 4 6 8 2 3 5 7)`, `(1 6)(2 5)(3 4)`,
                  `(3 4)(7 8)`},
     [0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 1],
     [4, 8, 16], [8, 16, 16, 16],
     [2, 4, 8, 8, 16, 32], [8, 16, 32], [],
     [{-4}, {-8}, {16}], [{-8}, {-16, -16, 16}],
     [{-2}, {4}, {-8, -8}, {16}, {32}], [{-8}, {16}, {32}]],
   [`T16`, 32, {`(1 4 6 8 2 3 5 7)`, `(1 5)(2 6)(3 7)(4 8)`},
     [0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 0, 0, 1],
     [4, 8, 16], [8, 8, 8, 32], [2, 4, 16, 16, 16, 16],
     [8, 16, 16, 16], [], [{4}, {8}, {16}], [{8, 8, 8}, {32}],
     [{2}, {4}, {16, 16, 16, 16}], [{8}, {16, 16, 16}]],
   [`T17`, 32, {`(1 4 6 8 2 3 5 7)`, `(1 7)(2 8)(3 5)(4 6)`},
     [0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1,
         0, 1, 0, 0, 0, 0, 0, 0, 1],
     [4, 8, 16], [8, 16, 32], [2, 4, 16, 16, 32],
     [8, 8, 8, 32], [], [{-4}, {-8}, {16}], [{-8}, {16}, {32}],
     [{-2}, {4}, {16, 16}, {32}], [{-8, -8, -8}, {32}]],
   [`T21`, 32, {`(1 6 2 5)(3 7)(4 8)`, `(1 7)(2 8)(3 5)(4 6)`},
     [0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1,
         0, 1, 0, 0, 0, 0, 0, 0, 0],
     [4, 8, 8, 8], [8, 8, 8, 32], [2, 2, 2, 16, 16, 16, 16],
     [8, 16, 16, 16], [], [{4}, {-8, -8, 8}], [{8, 8, 8}, {32}],
     [{-2, -2, 2}, {16, 16, 16, 16}], [{8}, {16, 16, 16}]],
   [`T23`, 48, {`(3 5 7)(4 6 8)`, `(1 3)(2 4)(7 8)`},
     [0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 0, 1, 0, 1],
     [4, 24], [8, 24, 24], [6, 16, 24, 24], [8, 48], [],
     [{-4}, {-24}], [{-8}, {-24, -24}], [{-6}, {16}, {-24, -24}],
     [{-8}, {48}]],
   [`T26`, 64, {`(1 4 6 8 2 3 5 7)`, `(1 6)(2 5)(3 4)`,
                  `(1 5)(2 6)(3 7)(4 8)`},
     [0, 1, 1, 1, 0, 0, 0, 0, 0, 1, 0, 1,
         0, 1, 0, 0, 0, 0, 0, 0, 1],
     [4, 8, 16], [8, 16, 32], [2, 4, 16, 16, 32],
     [8, 16, 32], [], [{-4}, {-8}, {16}], [{-8}, {16}, {-32}],
     [{-2}, {4}, {-16, 16}, {32}], [{-8}, {-16}, {-32}]],
   [`T27`, 64, {`(1 4 6 8 2 3 5 7)`, `(1 2)`},
     [1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1,
         0, 1, 0, 0, 0, 0, 0, 0, 1],
     [4, 8, 16], [8, 8, 8, 32], [2, 4, 16, 16, 16, 16],
     [8, 16, 16, 16], [], [{-4}, {-8}, {-16}], [{-8, -8, -8}, {-32}],
     [{-2}, {-4}, {-16, -16, -16, 16}], [{-8}, {-16, -16, -16}]],
   [`T28`, 64, {`(1 4 6 8 2 3 5 7)`, `(1 5)(2 6)`},
     [0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1,
         0, 1, 0, 0, 0, 0, 0, 0, 1],
     [4, 8, 16], [8, 16, 32], [2, 4, 16, 16, 32],
     [8, 16, 32], [], [{4}, {8}, {-16}], [{8}, {16}, {32}],
     [{2}, {-4}, {16, 16}, {32}], [{8}, {16}, {32}]],
   [`T30`, 64, {`(1 3 5 7)(2 4 6 8)`, `(1 2)(7 8)`,
                  `(1 5)(2 6)(3 4)`},
     [0, 1, 1, 1, 0, 0, 0, 0, 0, 1, 0, 1,
         0, 1, 0, 0, 0, 0, 0, 0, 0],
     [4, 8, 16], [8, 16, 32], [2, 4, 16, 16, 32],
     [8, 16, 32], [], [{-4}, {-8}, {-16}], [{-8}, {16}, {-32}],
     [{-2}, {-4}, {-16, 16}, {-32}], [{-8}, {-16}, {32}]],
   [`T31`, 64, {`(1 6 2 5)(3 7)(4 8)`, `(1 7)(2 8)(3 5)(4 6)`,
                  `(1 2)`},
     [1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1,
         0, 1, 0, 0, 0, 0, 0, 0, 0],
     [4, 8, 8, 8], [8, 8, 8, 32], [2, 2, 2, 16, 16, 16, 16],
     [8, 16, 16, 16], [], [{-4}, {-8, -8, -8}], [{-8, -8, -8}, {-32}],
     [{-2, -2, -2}, {-16, -16, -16, 16}], [{-8}, {-16, -16, -16}]],
   [`T35`, 128, {`(1 4 6 8 2 3 5 7)`, `(1 6)(2 5)(3 4)`,
                   `(1 2)`},
     [1, 1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1,
         0, 1, 0, 0, 0, 0, 0, 0, 1],
     [4, 8, 16], [8, 16, 32], [2, 4, 16, 16, 32],
     [8, 16, 32], [], [{-4}, {-8}, {-16}], [{-8}, {-16}, {-32}],
     [{-2}, {-4}, {-16, 16}, {-32}], [{-8}, {-16}, {-32}]],
   [`T38`, 192, {`(3 4)`, `(1 7)(2 8)(3 5)(4 6)`,
                   `(3 5 7)(4 6 8)`},
     [1, 1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 1,
         0, 1, 0, 0, 0, 1, 1, 0, 0],
     [4, 24], [24, 32], [6, 16, 48], [8, 48], [1, 4, `+`],
     [{-4}, {-24}], [{-24}, {-32}], [{-6}, {16}, {-48}], [{-8}, {-48}]],
   [`T40`, 192, {`(1 6)(2 5)(3 7)(4 8)`, `(3 5 7)(4 6 8)`,
                   `(1 3)(2 4)`},
     [0, 1, 1, 1, 0, 0, 0, 1, 0, 1, 0, 1,
         0, 1, 0, 0, 0, 0, 1, 0, 1],
     [4, 24], [24, 32], [6, 16, 48], [8, 48],
     [5, 8, `+T34`, {`+T34`, `+T41`}],
     [{-4}, {-24}], [{-24}, {-32}], [{-6}, {16}, {-48}],
     [{-8}, {-48}]],
   [`T43:PGL(2,7)`, 336, {`(1 2 3 4 5 6 7)`, `(2 4 3 7 5 6)`,
                   `(1 8)(2 7)(3 4)(5 6)`},
     [0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 0,
         0, 1, 0, 0, 0, 1, 0, 1, 1],
     [28], [56], [28, 42], [56], [5, 8, `+T41`, 3],
     [{-28}], [{-56}], [{28}, {-42}], [{-56}]],
   [`T44`, 384, {`(1 2)`, `(1 3 5 7)(2 4 6 8)`,
                   `(1 3)(2 4)`},
     [1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 1, 1,
         0, 1, 0, 0, 0, 1, 1, 0, 1],
     [4, 24], [24, 32], [6, 16, 48], [8, 48],
     [5, 8, `+T41`, {`+T34`, `+T41`}], [{-4}, {-24}],
     [{-24}, {-32}], [{-6}, {16}, {-48}], [{-8}, {-48}]],
   [`T46`, 576, {`(1 3)(2 4)`, `(6 8 7)`,
                   `(1 6 2 5)(3 7)(4 8)`},
     [0, 1, 0, 1, 1, 0, 1, 1, 0, 0, 1, 1,
         0, 1, 0, 0, 0, 0, 0, 0, 1],
     [12, 16], [8, 48], [2, 32, 36], [24, 32], [],
     [{12}, {-16}], [{8}, {48}], [{2}, {32}, {-36}], [{24}, {32}]],
   [`T47`, 1152, {`(1 4)`, `(1 2)`,
                    `(1 5)(2 6)(3 7)(4 8)`},
     [1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1,
         1, 1, 0, 0, 0, 0, 1, 0, 1],
     [12, 16], [8, 48], [2, 32, 36], [24, 32], [],
     [{-12}, {-16}], [{-8}, {-48}], [{-2}, {-32}, {-36}],
     [{-24}, {-32}]],
   [`T50:S8`, 40320, {`(1 2 3 4 5 6 7 8)`, `(1 2)`},
     [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
         1, 1, 1, 1, 1, 1, 1, 1, 1],
     [28], [56], [70], [56], [], [{-28}], [{-56}], [{-70}],
     [{-56}]]} ], [], [], [],

[ {[`+12T1`, 0, {}, [], [6, 24, 36], [], [], [], [], [],
   [1, 6, `3.S3`, {`3.S3`, `3^2.2^2`}]],
   [`+12T2`, 0, {}, [], [6, 24, 36], [], [], [], [], [],
   [1, 6, `3^2.2^2`, {`3.S3`, `3^2.2^2`}]]},
   {}] ]):

# Copyright 1987 R. Sommeling; all rights reserved
#                                                           April 25, 1987
# Extensions to degree 8 case,
# Copyright 1992 T. Mattman; all rights reserved
#                                                           February 18, 1992

# Input:  An irreducible univariate polynomial f over the rationals of degree
#         up to 8.
# Output: A sequence of three elements. The first is the name of the Galois
#         group of f, the second is its order and the third is a set of
#         generators of the group (up to conjugacy).
#
# Main reference:
#
#    L.H. Soicher & J. McKay, Computing Galois groups over the rationals,
#    Number Theory 20 (1985), pp. 273-281.

unprotect(galois);
galois := proc(f)
local D,e,g,grp,i,lc,minelm,n,n2,p,part,possible,rm,shape,sol,stable,x,
sqdiscrim;
options remember,`Copyright 1987 by Ron Sommeling`;
    g := expand(normal(f));
    x := indets(g);

# Check whether the first argument has the right form.
    if x = {} then RETURN(`Id`,1,{}) fi;
    if
    (1 < nops(x)) or not type(x[1],name) or not type(g,polynom(rational))
     then
        ERROR(`not a polynomial of the right form`)
    fi;
    x := x[1];
    n := degree(g,x);
    n2 := iquo(n,2);
    if 8 < n then ERROR(`the degree of the polynomial should be < 9`) fi;
    g := g/icontent(g);
    if type(factor(g),{`*`,`^`}) then
        ERROR(`the polynomial is reducible`)
    fi;
    if 1 < printlevel then lprint(`Computing the Galois group of `,g) fi;

# Check whether the discriminant is a square. The discriminant is a square
# iff the Galois group contains only even permutations.
    lc := lcoeff(g);
    D := discrim(g,x);
    sqdiscrim := issqr(D);
    if 1 < printlevel then
        if not sqdiscrim then lprint('D' = D,`(nonsquare)`)
        elif D = 1 then lprint('D' = D)
        else lprint(`D = `(isqrt(D))^2)
        fi
    fi;
    if sqdiscrim then possible := groups[n,0]
    else possible := groups[n,1]
    fi;
    if nops(possible) = 1 then
        if 0 < printlevel then
            print(`The Galois group of `(f));
            print(` is `)
        fi;
        RETURN(op(possible[1][1 .. 3]))
    fi;
    if 1 < printlevel then lprint(`Possible groups:`,short(possible)) fi;

# Find the shapes of some elements in the Galois group by computing the
# degrees of the factors of f mod p for some small primes p.
# See van der Waerden, Algebra I (1971).
# We compute these shapes until one of the following holds:
# 1. There is only one possible Galois group left. (Is always An or Sn)
# 2. The set of possible Galois groups hasn't changed by computing the last
#    n+1 shapes AND the set of possible Galois groups contains an element
#    whose set of possible shapes is minimal w.r.t. set inclusion.
#    This group is assigned to the variable "minelm".
    part := initpart(n);
    p := 1;
    stable := 0;
    minelm := findminelm(possible);
    while stable < n+1 do
        p := nextprime(p);
        if (lc mod p = 0) or (D mod p = 0) then next fi;
        shape := cyclepattern(g,x,p);
        if 1 < printlevel then lprint('p' = p,`gives shape`,shape) fi;
        e := encode(part,shape);
        rm := {};
        if 0 < e then
            for i to nops(possible) do
                grp := possible[i];
                if grp[4][e] = 0 then rm := rm union {grp} fi
            od
        fi;
        if rm <> {} then
            possible := possible minus rm;
            if 1 < printlevel then
                lprint(`Removing`,short(rm));
                lprint(`Possible groups left:`,short(possible))
            fi;
            if nops(possible) = 1 then
                if 0 < printlevel then
                    print(`The Galois group of `(f));
                    print(` is `)
                fi;
                RETURN(op(possible[1][1 .. 3]))
            fi;
            minelm := findminelm(possible);
            stable := 0
        elif minelm <> false then stable := stable+1
        fi
    od;
    if 1 < printlevel then lprint(`The Galois group is probably`,minelm[1])
    fi;

# Call respol which will determine the group using resolvent polynomials.
    sol := respol(g,x,D,minelm,possible,n2,n,sqdiscrim);
    if 0 < printlevel then
        print(`The Galois group of `(f));
        print(` is `)
    fi;
    sol
end:


# Input:  An irreducible univariate polynomial g over the rationals, its
#         variable x, its discriminant D, its degree n, n2 the floor of
#         n/2, sqdiscrim indicating whether or not D is a square, gposs
#         a set of groups which includes the Galois group of g, and gmin,
#         a minimal element of gposs with respect to the set of cycle
#         shapes, if such exists, false otherwise.
# Output: A sequence of three elements. The first is the name of the Galois
#         group of g, the second is its order and the third is a set of
#         generators of the group (up to conjugacy).

respol := proc(g,x,D,gmin,gposs,n2,n,sqdiscrim)
local decidable,equal,grp,h,i,minelm,possible,orbitpart,orbittype,R,rm,rsets;
options `Copyright 1987 by Ron Sommeling`;

    minelm := gmin;
    possible := gposs;

# If the degree exceeds 7, call hideg to calculate the group.
    if n >= 8 then
        RETURN(hideg(g,x,D,minelm,possible,n2,n,sqdiscrim));
    fi;

# Find out what is the best thing to do now: looking at 2-sequences or at
# r-sets for some r>1. We distinguish 3 cases:
# 1. The group "minelm" is still in the list of possible groups AND there
#    is a (r-set or 2-sequence) resolvent such that by factoring this
#    resolvent we can decide whether the Galois group is "minelm" or not.
#    In this case we take the resolvent of the lowest degree with this
#    property.
# 2. Otherwise, if there is a resolvent such that by factoring this resolvent
#    we can always reduce the set of possible groups, then take the resolvent
#    of the lowest degree with this property.
# 3. Otherwise, we are in one of the 4 special cases. See the description of
#    the functions "special5" and "special6".
    do
        decidable := false;
        for orbittype to n2 do
            equal := 0;
            if minelm = false then orbitpart := possible[1][orbittype+4]
            else orbitpart := minelm[orbittype+4]
            fi;
            for i to nops(possible) do
                if possible[i][orbittype+4] = orbitpart then equal := equal+1
                fi
            od;
            if (equal = 1) and (minelm <> false) then
                decidable := orbittype; break
            fi;
            if (decidable = false) and (equal < nops(possible)) then
                decidable := orbittype
            fi
        od;
        if decidable <> false then rsets := evalb(decidable < n2)
        else
            if degree(g,x) = 5 then
                RETURN(special5(g,x,D,possible))
            else
                RETURN(special6(g,x,sqfreepart(D),possible))
            fi;
        fi;
        if 1 < printlevel then
            if rsets then
                lprint(cat(
                `Using the orbit-length partition of `,decidable+1,`-sets.`
                ))
            else lprint(`Using the orbit-length partition of 2-sequences.`)
            fi
        fi;

# Construct the (r-set or 2-sequence) polynomial. If it is not squarefree
# then apply suitable Tschirnhaus transformations to f until the resolvent
# is squarefree. Then factor the resolvent and use the degrees of the factors
# to reduce the set of possible groups.
        h := g;
        do
            if 1 < printlevel then
                lprint(`Calculating a resolvent polynomial...`)
            fi;
            if rsets then R := rsetpol(h,x,decidable+1)
            else R := twoseqpol(h,x)
            fi;
            if issqfree(R,x) then break
            elif rsets then h := expand(subs(x = x+1,h))
            else h := numer(subs(x = 1/(x+1),h))
            fi
        od;
        if 1 < printlevel then
            lprint(`Factoring the resolvent polynomial...`)
        fi;
        orbitpart := degrees(factor(R),x);
        if 1 < printlevel then
            lprint(`Orbit-length partition is`,orbitpart)
        fi;
        orbitpart := [orbitpart];
        rm := {};
        for i to nops(possible) do
            grp := possible[i];
            if grp[decidable+4] <> orbitpart then rm := rm union {grp} fi
        od;
        possible := possible minus rm;
        if 1 < printlevel then
            lprint(`Removing`,short(rm));
            lprint(`Possible groups left:`,short(possible))
        fi;
        if nops(possible) = 1 then
            RETURN(op(possible[1][1 .. 3]))
        fi;
        if not member(minelm,possible) then minelm := false fi
    od
end:

short := proc(grps)
options `Copyright 1987 by Ron Sommeling`;
    map(proc(grp) grp[1] end,grps)
end:

issqfree := proc(f,x)
options `Copyright 1987 by Ron Sommeling`;
    degree(gcd(f,diff(f,x)),x) = 0
end:

# The weight array indicates the relative difficulty of generating
# the resolvent polynomials (2-set,3-set,..., and 2-sequence) for
# the degrees 8 through 12. Note that a weight above n2 (floor of
# n/2) ensures that the corresponding polynomial will be used only
# if no other polynomial can reduce the set of possible groups. The
# order of the polynomials in weight corresponds to that in the groups
# array.

weight := array(8..12, [[0,1,5,2], [0,2,3,1], [0,2,3,4,1],
                        [0,2,3,4,1], [0,2,3,4,5,1]]):

# Copyright 1992 T. Mattman; all rights reserved
#                                                           February 18, 1992

# Input:  An irreducible univariate polynomial f over the rationals, its
#         variable x, its discriminant D, its degree n, n2 the floor of
#         n/2, sqdiscrim indicating whether or not D is a square, gposs
#         a set of groups which includes the Galois group of g, and gmin
#         a minimal element of gposs with respect to the set of cycle
#         shapes, if such exists, false otherwise.
# Output: A sequence of three elements. The first is the name of the Galois
#         group of f, the second is its order and the third is a set of
#         generators of the group (up to conjugacy).
#
# The following code follows closely that written by Ron Sommeling for
# the procedure galois.

hideg := proc(f,x,D,gmin,gposs,n2,n,sqdiscrim)
local decidable,decwt,equal,grp,i,minelm,possible,orbitpart,
      orbittype,R,rm,rsets;
options `Copyright 1992 by T. Mattman`;
    minelm := gmin;
    possible := gposs;

# Initialize table of resolvent polynomials.
    for i to n2 do
        R[i] := false
    od;
    if n=8 then
        R[5] := false
    fi;

# Find out what is the best thing to do now: looking at 2-sequences or at
# r-sets for some r>1. We distinguish 3 cases:
# 1. The group "minelm" is still in the list of possible groups AND there
#    is a (r-set or 2-sequence) resolvent of weight less than n2 such that
#    by factoring this resolvent we can decide whether the Galois group is
#    "minelm" or not. In this case we take the resolvent of the lowest
#    weight with this property.
# 2. Otherwise, if there is a resolvent such that by factoring this resolvent
#    we can always reduce the set of possible groups, then take the resolvent
#    of the lowest weight with this property.
# 3. Otherwise, if the polynomial's discriminant is not square, call rtdpol
#    to test irreducibility over the field extended by the square root of the
#    discriminant. Finally call galfac to use Galois groups of factors of the
#    resolvent polynomials to distinguish the group.
    do
        decidable := false;
        decwt := n;
        for orbittype to n2 do
            equal := 0;
            if minelm = false then orbitpart := possible[1][orbittype+4]
            else orbitpart := minelm[orbittype+4]
            fi;
            for grp in possible do
                if grp[orbittype+4] = orbitpart then
                    equal := equal+1
                fi
            od;
            if ((equal = 1) and (minelm <> false))
                and (weight[n][orbittype] < n2) then
                decidable := orbittype;
                break
            fi;
            if (weight[n][orbittype] < decwt) and (equal < nops(possible)) then
                decidable := orbittype;
                decwt := weight[n][orbittype]
            fi;
        od;
        if decidable = false then
            if not sqdiscrim then
                rtdpol(f,x,D,n,n2,possible,R,'possible','R');
                if nops(possible) = 1 then
                    RETURN(op(possible[1][1..3]))
                fi
            fi;
            RETURN(galfac(f,x,possible,R,n,n2))
        fi;
        rsets := evalb(decidable < n2);
        if 1 < printlevel then
            if rsets then
                lprint(cat(
                `Using the orbit-length partition of `,decidable+1,`-sets.`
                ))
            else lprint(`Using the orbit-length partition of 2-sequences.`)
            fi
        fi;

# Call resbuild to build and factor the required polynomial. Use the
# degrees of the factors to reduce the set of possible groups.
        resbuild(f,x,decidable,R,n,n2,'R');
        orbitpart := degrees(R[decidable],x);
        if 1 < printlevel then
            lprint(`Orbit-length partition is`,orbitpart)
        fi;
        orbitpart := [orbitpart];
        rm := {};
        for grp in possible  do
            if grp[decidable+4] <> orbitpart then rm := rm union {grp} fi
        od;
        possible := possible minus rm;
        if 1 < printlevel then
            lprint(`Removing`,short(rm));
            lprint(`Possible groups left:`,short(possible))
        fi;
        if nops(possible) = 1 then
            RETURN(op(possible[1][1 .. 3]))
        fi;
        if not member(minelm,possible) then minelm := false fi
    od
end:

# Input:  An irreducible univariate polynomial f over the rationals, its
#         variable x, its discriminant D, its degree n, n2 the floor of
#         n/2, gposs a set of groups which includes the Galois group of g,
#         gR a table of factored resolvent polynomials for f, and possout
#         and Rout, the names under which the updated set of possible groups
#         and table of resolvent polynomials respectively are to be passed
#         back.
# Output: NULL.
#
# The following code is based on that written by Ron Sommeling for
# the procedure galois.

rtdpol := proc(f,x,D,n,n2,gposs,gR,possout,Rout)
local decidable,deg,equal,grp,orbittype,possible,R,rm,rsets,rtdfac,testdeg;
options `Copyright 1992 by T. Mattman`;
    possible := gposs;
    R := gR;

# If there is a set of factors of like degree of a resolvent polynomial
# such that by factoring over sqrt(D), we can reduce the number of possible
# groups, we choose the first such that occurs; the order of search is
# to go through the list of resolvent polynomials as they occur in the
# array groups and then for each polynomial to go through the factors
# in order of increasing degree.
# If no such factorization will reduce the set of groups, we return NULL.
    do
        decidable := false;
        for orbittype to n2 while (decidable = false) do
            rtdfac := possible[1][orbittype+5+n2];
            for deg to nops(rtdfac) while (decidable = false) do
                equal := 0;
                for grp in possible do
                    if grp[orbittype+5+n2][deg] = rtdfac[deg] then
                        equal := equal+1
                    fi
                od;
                if (decidable = false) and (equal < nops(possible)) then
                    decidable := orbittype;
                    testdeg := deg
                fi
            od
        od;
        if decidable = false then
            possout := possible;
            Rout := op(R);
            RETURN()
        fi;
        rsets := evalb(decidable < n2);
        if 1 < printlevel then
            lprint(`Factoring polynomials over sqrt(D).`);
            if rsets then
                lprint(cat(`Using degree `,abs(rtdfac[testdeg][1]),
                        ` factors of `,decidable+1,`-sets polynomial.`))
            else
                lprint(cat(`Using degree `,abs(rtdfac[testdeg][1]),
                           ` factors of 2-sequences polynomial.`))
            fi
        fi;

# Call resbuild to build and factor the required resolvent polynomial.
# Call rtdcons to factor our set of factors and use this information
# to reduce the set of possible groups.
        resbuild(f,x,decidable,R,n,n2,'R');
        rtdfac := rtdcons(R[decidable],abs(rtdfac[testdeg][1]),D,x);
        if 1 < printlevel then
            lprint(`The polynomials factor as `,rtdfac);
            lprint(`'-' indicates irreducble; '+' indicates reducible`)
        fi;
        rm := {};
        for grp in possible do
            if grp[decidable+5+n2][testdeg] <> rtdfac then
                rm := rm union {grp}
            fi
        od;
        possible := possible minus rm;
        if 1 < printlevel then
            lprint(`Removing`,short(rm));
            lprint(`Possible groups left:`,short(possible))
        fi;
        if nops(possible) = 1 then
            possout := possible;
            Rout := op(R);
            RETURN()
        fi
    od
end:

# Input:  A factored polynomial R in the variable x. Deg indicates the degree
#         of factors of R to be tested. They are tested for irreducibility
#         over sqrt(d).
# Output: A set consisting of deg or -deg for each factor of degree deg of R.
#         "-" indicates that the corresponding polynomial is irreducible over
#         sqrt(d); otherwise it factors.
#
# The method of constructing the "gd" polynomial is taken from the routine
# special6 written by Ron Sommeling.

rtdcons := proc(R,deg,d,x)
local fac,gd,i,j,y;
options `Copyright 1992 by T. Mattman`;

# To determine whether or not a given polynomial, g, with roots ai,
# is irreducible over sqrt(d),  we construct the polynomial, gd, with roots
# ai + sqrt(d), ai - sqrt(d). Then g is irreducible over sqrt(d) iff gd
# is irreducible.
    fac := {};
    for i to nops(R) do
        if degree(op(i,R),x) = deg then
            for j do
                gd := resultant(subs(x=x-y,op(i,R)),y^2-j^2*d,y);
                gd := gd/icontent(gd);
                if issqfree(gd,x) then break fi
            od;
            if type(factor(gd),{`*`,`^`}) then
                fac := fac union {deg}
            else
                fac := fac union {-deg}
            fi
        fi
    od;
    fac
end:

# Input:  An irreducible univariate polynomial f over the rationals, its
#         variable x, its degree n, n2 the floor of n/2, gposs
#         a set of groups which includes the Galois group of g, and gR
#         a table of resolvent polynomials for f.
# Output: A sequence of three elements. The first is the name of the Galois
#         group of f, the second is its order and the third is a set of
#         generators of the group (up to conjugacy).
#

galfac := proc(f,x,gposs,gR,n,n2)
local D,fac,facpos,fsig,gal,grp,i,j,minelm,par,possible,R,rm,rsets;
options `Copyright 1992 by T. Mattman`;
    possible := gposs;
    R := gR;
    rm := {};

# We first check parity of factors of the resolvent polynomials.
# In the groups array, the "name" of the Galois group of such a factor
# may appear as `+` or `-`. This indicates the parity of that group.
# We test the discriminant of the corresponding polynomial derived from
# f to see if it is square or not. Groups with inappropriate parity
# for this factor are removed from the set of possible groups. If we
# find a group with the correct parity, we return it immediately.
    for i to nops(possible) while ((nops(possible) - nops(rm)) > 1) do
        fsig := possible[i][5+n2];
        if member(fsig[3], {`+`,`-`}) then
            if 1 < printlevel then
                lprint(cat(`Checking parity of degree `,fsig[2],
                        ` factor of`));
                rsets := evalb(fsig[1] < n2);
                if rsets then
                    lprint(cat(fsig[1]+1,`-sets polynomial`))
                else
                    lprint(`2-sequences polynomial`)
                fi
            fi;
            resbuild(f,x,fsig[1],R,n,n2,'R');
            fac := false;
            for j to nops(R[fsig[1]]) while fac = false do
                if degree(op(j,R[fsig[1]]),x) = fsig[2] then
                    fac := op(j,R[fsig[1]])
                fi
            od;
            par := issqr(discrim(fac,x));
            if 1 < printlevel then
                if par then
                    lprint(`The parity is even.`)
                else
                    lprint(`The parity is odd.`)
                fi
            fi;
            for grp in possible do
                if (grp[5+n2][1] = fsig[1]) and
                   (grp[5+n2][2] = fsig[2]) then
                    if (par and (grp[5+n2][3] = `-`)) or
                       (not par and (grp[5+n2][3] = `+`)) then
                        rm := rm union {grp}
                    else
                        RETURN(op(grp[1..3]))
                    fi
                fi
            od
        fi
    od;
    possible := possible minus rm;

# To distinguish any remaining groups, we call respol to find the Galois
# group of a factor of a resolvent polynomial.
    if nops(possible) > 1 then
        fsig := possible[1][5+n2];
        if 1 < printlevel then
            lprint(cat(`Checking Galois group of degree `,fsig[2],
                        ` factor of`));
            if fsig[1] <= n2 then
                rsets := evalb(fsig[1] < n2);
                if rsets then
                    lprint(cat(fsig[1]+1,`-sets polynomial`))
                else
                    lprint(`2-sequences polynomial`)
                fi
            else
                lprint(`a special resolvent polynomial.`)
            fi
        fi;
        resbuild(f,x,fsig[1],R,n,n2,'R');
        fac := false;
        for i to nops(R[fsig[1]]) while fac = false do
            if degree(op(i,R[fsig[1]]),x) = fsig[2] then
                fac := op(i,R[fsig[1]])
            fi
        od;
        D := discrim(fac,x);
        facpos := posscons(fsig[2],fsig[4]);
        minelm := findminelm(facpos);
        gal := respol(fac,x,D,minelm,facpos,
                      iquo(fsig[2],2),fsig[2],issqr(D));
        if 1 < printlevel then
            lprint(`The Galois group of the factor is `,gal[1])
        fi;
        rm := {};
        for grp in possible do
            if (grp[5+n2][1] = fsig[1]) and
               (grp[5+n2][2] = fsig[2]) then
                if gal[1] <> grp[5+n2][3] then
                    rm := rm union {grp}
                fi
            fi
        od;
        possible := possible minus rm
    fi;
    op(possible[1][1..3])
end:

# Input:  An irreducible univariate polynomial f over the rationals, its
#         variable x, the type of resolvent polynomial to build, gR a table
#         of resolvent polynomials already constructed, n the degree of f,
#         n2 the floor of n/2 and Rout the name under which the updated
#         resolvent table should be returned.
# Output: The table of factored resolvent polynomials.
#
# Much of this code is taken from the procedure galois written by
# Ron Sommeling.

resbuild := proc(f,x,rttype,gR,n,n2,Rout)
local g,oldg,R,r,rd,rdf,rsets,sumsets,y;
options `Copyright 1992 by T. Mattman`;
    R := gR;
    if R[rttype] = false then
        g := f;
        if (n <> 8) or not member(rttype, {3,5}) then

# Construct the (r-set or 2-sequence) polynomial. If it is not squarefree
# then apply suitable Tschirnhaus transformations to f until the resolvent
# is squarefree.
            rsets := evalb(rttype < n2);
            do
                if 1 < printlevel then
                    lprint(`Calculating a resolvent polynomial...`)
                fi;
                if rsets then r := rsetpol(g,x,rttype+1)
                else r := twoseqpol(g,x)
                fi;
                if issqfree(r,x) then break
                elif rsets then g := expand(subs(x = x+1,g))
                else g := numer(subs(x = 1/(x+1),g))
                fi
            od;
            if 1 < printlevel then
                lprint(`Factoring the resolvent polynomial...`)
            fi;
            R[rttype] := factor(r)
        else

# To factor the degree 8 4-set polynomial, we make use of the degree 35
# polynomial rd with roots of the form (x1+x2+x3+x4-x5-x6-x7-x8)^2. rd
# may be constructed from r, having roots (x1+x2+x3+x3-x5-x6-x7-x8) using
# the fact that rd(x^2) = r(x). Note that if the sum of the roots is zero
# (ie. if g has coefficient zero for its x^7 term) then r is also the
# polynomial with roots of the form (x1+x2+x3+x4), ie. the 4-set
# resolvent polynomial. Because of their similarity, factoring rd gives a
# partial factorization of r.
            if rttype = 5 then
                sumsets := 3
            else
                sumsets := rttype
            fi;
            do
                if 1 < printlevel then
                    lprint(`Calculating a resolvent polynomial...`)
                fi;
                oldg := g;

# Apply Tschirnhaus transformation to ensure the coeff(g,x,7) = 0.
                g := expand(subs(x=x-coeff(g,x,7)/(8*lcoeff(g,x)),g));
                g := g/icontent(g);
                r := rsumpol(g,x,sumsets+1);
                if issqfree(r,x) then break
                else
                    g := numer(subs(x = 1/(x+1),oldg))
                fi
            od;
            if 1 < printlevel then
                lprint(`Factoring the resolvent polynomial...`)
            fi;
            rd := subs(x=y^(1/2),r);
            rdf := factor(rd);
            R[5] := subs(y=x,rdf);
            R[sumsets] := factor(subs(y=x^2,rdf))
        fi
    fi;
    Rout := op(R);
end:

# Input:  An integer n and a set grps of group names. n is the degree of
#         the Galois groups listed in grps.
# Output: A set of groups correspondng to grps but including all
#         information for the groups as stored in the groups array.
#

posscons :=  proc(n, grps)
local chklst,evengp,gp, possible;
options `Copyright 1992 by T. Mattman`;
    evengp := evalb(substring(grps[1], 1..1) = `+`);
    if evengp then
        chklst := groups[n,0]
    else
        chklst := groups[n,1]
    fi;
    possible := {};
    for gp in chklst do
        if member(gp[1],grps) then
            possible := possible union {gp}
        fi
    od;
    possible
end:

# Input:  A polynomial f in x, the indeterminate x and a non-negative
#         integer r.
# Output: A primitive polynomial of degree binomial(n,r) in x whose roots are
#         the sums of r distinct roots of f.
#
# The method used to construct the polynomials is taken from
# D.Casperson, D.Ford & J.McKay, Symmetric functions, m-sets, and
# Galois groups, preprint.

rsumpol := proc(f,x,r)
local a,arg,b,g,H,i,j,k,l,n,s;
options `Copyright 1992 by T. Mattman`;
    if not type(x,name) then
        ERROR(`second argument should be a variable`)
    fi;
    g := normal(f);
    if not type(g,polynom(rational,x)) then
        ERROR(`first argument should be a polynomial`)
    fi;
    if not type(r,integer) or (r < 0) then
        ERROR(`third argument should be a natural number`)
    fi;
    g := expand(g);
    g := expand(g/lcoeff(g,x));
    n := degree(g,x);
    if r = 0 then x-1
    elif r = 1 then g
    elif r = n then x+coeff(g,x,n-1)
    elif n < r then 1
    else
        b := binomial(n, r);
        a := array(0 .. n,['coeff(g,x,n-i)' $ ('i' = 0 .. n)]);
        s := a_to_s(op(a),n,b);
        H[0,0] := 1;
        H[1,0] := s[0];
        for i from 1 to b do
            H[0,i] := 0;
            H[1,i] := s[i]/i!
        od;
        for i from 2 to r do
            for j from 0 to b do
                H[i,j] := 0;
                for l from 1 to i do
                    arg := (-1)^(l+1)/i;
                    for k from 0 to j do
                        H[i,j] := H[i,j] + arg*H[i-l,j-k]*H[1,k];
                        arg := arg*l
                    od
                od
            od
        od;
        for i from 1 to b do
            s[i] := H[r,i]*i!
        od;
        a := s_to_a(op(s),b);
        i := 'i';
        g := sum(a[b-i]*x^i, i=0..b);
        g/content(g,x)
    fi
end:

# Input:  An integer n.
# Output: The integer m such that n = k^2 m, where all prime factors of k are
#         smaller than 1999 and all prime factors of m smaller than 1999
#         appear with multiplicity 1.
# We don't compute the complete squarefree part, because that could take a
# very long time if n is big. Note that 510510 is the product of the primes
# up to 17 while the big number used in this procedure is the product of the
# primes from 19 to 1999.

sqfreepart := proc(n)
local m1,m2,m;
options `Copyright 1987 by Ron Sommeling`;
    m := n;
    m2 := 2;
    while (1 < m2) and (1 < m) do
        m1 := m/igcd(m,510510); m2 := igcd(m1,510510); m := m/m2^2
    od;
    m2 := 2;
    while (1 < m2) and (1 < m) do
        m1 := m/igcd(m,
5668344243265471819408349879264900561344506260438297113121754471676061\
2560791973390670744989333350841745309402061968741709452975735415298674\
9560019905142897690259093457210799603632782717922065955901830529319296\
1167931511458206100875119670995542639325967180196434319724253387877603\
6716818371965880567092149001036019530948612161197976056057227524736840\
1870162331444571403713689995433289772675349111069849906442564926271706\
4250647347752986985165434242366290330943902166943407887478097884195588\
7151142775136584305953026725917860957221689991171352057774946363103279\
4081865124575338024762333449029062271815188481933090660812302255777593\
0961309507305546661753623302876891787345038544171957581961149093965350\
7058614796835748370471316444340873208399321825992890475512470915258564\
3599469820961428065196995970020581366019108556944629640933475067617
        );
        m2 := igcd(m1,
5668344243265471819408349879264900561344506260438297113121754471676061\
2560791973390670744989333350841745309402061968741709452975735415298674\
9560019905142897690259093457210799603632782717922065955901830529319296\
1167931511458206100875119670995542639325967180196434319724253387877603\
6716818371965880567092149001036019530948612161197976056057227524736840\
1870162331444571403713689995433289772675349111069849906442564926271706\
4250647347752986985165434242366290330943902166943407887478097884195588\
7151142775136584305953026725917860957221689991171352057774946363103279\
4081865124575338024762333449029062271815188481933090660812302255777593\
0961309507305546661753623302876891787345038544171957581961149093965350\
7058614796835748370471316444340873208399321825992890475512470915258564\
3599469820961428065196995970020581366019108556944629640933475067617
        );
        m := m/m2^2
    od;
    m
end:

# Input:  An integer n.
# Output: An array part(0..n,0..n) needed for the encoding of shapes up to
#         degree n.
#         part[i,j] = number of shapes of degree i with all elements <= j.

initpart := proc(n)
local i,j,part;
options `Copyright 1987 by Ron Sommeling`;
    part := array(0 .. n,0 .. n);
    part[0,0] := 1;
    for i to n do
        part[i,0] := 0;
        for j to i do  part[i,j] := part[i,j-1]+part[i-j,min(i-j,j)] od
    od;
    op(part)
end:

# Input:  The partition array as described above and a sequence of integers
#         representing a shape.
# Output: The number of shapes of the same degree which are smaller than the
#         given one w.r.t. the lexicographical ordering.

encode := proc(part)
local e,i,n;
options `Copyright 1987 by Ron Sommeling`;
    n := sum(args[i],i = 2 .. nargs);
    e := 0;
    for i from 2 to nargs do  e := e+part[n,args[i]-1]; n := n-args[i] od;
    e
end:

# Input:  A set of groups.
# Output: If there is a group in the set whose set of possible shapes is
#         minimal w.r.t. set inclusion, then this element, otherwise false.

findminelm := proc(grps)
local i,j,m,n,s,t;
options `Copyright 1987 by Ron Sommeling`;
    n := nops(grps);
    s := grps[1][4];
    m := nops(s);
    for i from 2 to n do
        t := grps[i][4]; s := [(s[j]*t[j]) $ (j = 1 .. m)]
    od;
    for i to n do  if grps[i][4] = s then RETURN(grps[i]) fi od;
    false
end:

# Input:  A polynomial in x in factored form and the indeterminate x.
# Output: The sequence of the degrees of the factors in non-decreasing order.
#
# Modified to deal with repeated factors and possible initial factor
# indicating sign:  Copyright 1992 T. Mattman

degrees := proc(factored,x)
local degs, sgn;
options `Copyright 1987 by Ron Sommeling`,
        `Modifications Copyright 1992 by T. Mattman`;
    if not type(factored,{`*`,`^`}) then
        degree(factored,x)
    elif type(factored,`^`) then
        degree(op(1,factored),x) $ op(2,factored)
    else
        if type(op(1,factored),negative) then
            sgn := -1
        else
            sgn := 1
        fi;
        degs := map(degrees,convert(sgn*factored,list),x);
        degs := sort(degs,numeric);
        op(degs)
    fi
end:

# Case distinction between F20 and S5. We factor a resolvent of degree 6
# derived from a resolvent of S5/D5 as suggested by J. Buhler.

special5 := proc(g,x,D,possible)
local a2,a4,a6,c,cx4,d,Dn,e,f,grp,h,h2,k,lc,R;
options `Copyright 1987 by Ron Sommeling`;
    h := g;
    do
        if 2 < printlevel then lprint('`Calculating a S5/F20 resolvent...`')
        fi;
        lc := lcoeff(h,x);
        cx4 := coeff(h,x,4);
        Dn := sqfreepart(D);
        k := lc^6*isqrt(iquo(D,Dn));
        if irem(cx4,5) = 0 then
            h2 := expand(lc^4*subs(x = (x-iquo(cx4,5))/lc,h))
        else
            h2 := expand(3125*lc^4*subs(x = 1/5*(x-cx4)/lc,h));
            k := 9765625*k
        fi;
        c := coeff(h2,x,3);
        d := 2*coeff(h2,x,2);
        e := 4*coeff(h2,x,1);
        f := 4*coeff(h2,x,0);
        a2 := -5*e-3*c^2;
        a4 := -50*d*f+15*e^2-2*c^2*e+4*c*d^2+3*c^4;
        a6 := 250*c*f^2-50*d*e*f-10*c^2*d*f+5*e^3-11*c^2*e^2+14*c*d^2*e+7*c^4*e
            -4*d^4-4*c^3*d^2-c^6;
        R := x^6+a2*Dn*x^4+a4*Dn^2*x^2-32*k*Dn^3*x+a6*Dn^3;
        if issqfree(R,x) then break else h := numer(subs(x = 1/(x+1),h)) fi
    od;
    if 1 < printlevel then lprint('`Factoring this S5/F20 resolvent...`') fi;
    if type(factor(R),'`*`') then
        grp := 'F20';
        if 1 < printlevel then
            lprint('`Reducible, so the Galois group is`',grp)
        fi
    else
        grp := 'S5';
        if 1 < printlevel then
            lprint('`Irreducible, so the Galois group is`',grp)
        fi
    fi;
    if possible[1][1] = grp then op(possible[1][1 .. 3])
    else op(possible[2][1 .. 3])
    fi
end:

# Case distinction between S4/Z4 and 2.S4, between 3^2.2^2 and 3^2.D4
# and between PGL2(5) and S6.
# In all cases first the 3-set resolvent is computed and factored, then one
# of its factors g is taken (of degree resp. 12, 2, 20) and a polynomial is
# constructed whose roots are xi + sqrt(D) and xi - sqrt(D), where xi is a
# root of g and D is the discriminant of f. Reducibility of this polynomial
# determines the Galois group.

special6 := proc(f,x,D,possible)
local degs,factored,g,i,gd,grp,grps,h,R,y;
options `Copyright 1987 by Ron Sommeling`;
    grps := short(possible);
    if x = y then y := 'x' fi;
    h := f;
    do
        if 2 < printlevel then
            lprint('`Calculating a resolvent polynomial...`')
        fi;
        R := rsetpol(h,x,3);
        if issqfree(h,x) then break else h := expand(subs(x = x+1,h)) fi
    od;
    if grps = {'`S4/Z4`','`2.S4`'} then
        if 2 < printlevel then
            lprint('`Factoring the resolvent polynomial...`')
        fi;
        factored := factor(R);
        if degree(op(1,factored),x) = 12 then g := op(1,factored)
        else g := op(2,factored)
        fi;
        grp := '`S4/Z4`'
    elif grps = {'`3^2.2^2`','`3^2.D4`'} then
        if 2 < printlevel then
            lprint('`Factoring the resolvent polynomial...`')
        fi;
        factored := factor(R);
        if degree(op(1,factored),x) = 2 then g := op(1,factored)
        else g := op(2,factored)
        fi;
        grp := '`3^2.2^2`'
    elif grps = {'`PGL2(5)`','S6'} then g := R; grp := '`PGL2(5)`'
    fi;
    for i do
        if 2 < printlevel then
            lprint('`Constructing a special resolvent polynomial...`')
        fi;
        gd := resultant(subs(x = x-y,g),y^2-i^2*D,y);
        gd := gd/icontent(gd);
        if issqfree(gd,x) then break fi
    od;
    if 2 < printlevel then
        lprint('`Factoring the special resolvent polynomial...`')
    fi;
    degs := degrees(factor(gd),x);
    if 1 < nops([degs]) then
        if 2 < printlevel then
            lprint('`Reducible - orbit lengths`',degs);
            lprint('`so the Galois group is`',grp)
        fi;
        if possible[1][1] = grp then op(possible[1][1 .. 3])
        else op(possible[2][1 .. 3])
        fi
    else
        if 1 < printlevel then
            lprint(`Irreducible, so the Galois group is`,op(1,grps minus {grp})
                )
        fi;
        if possible[1][1] = grp then op(possible[2][1 .. 3])
        else op(possible[1][1 .. 3])
        fi
    fi
end:

readlib(issqr):

