#
## <SHAREFILE=algebra/MatPade/MatPade.mpl >
## <DESCRIBE>
##                SEE ALSO: algebra/MatPade/MatPade.mws
##
##                A set of functions to compute matrix-type Pade approximants.
##                Examples of such approximants include Hermite Pade,
##                Simultaneous Pade, right and left matrix Pade approximants.
##                AUTHOR: G. Labahn, glabahn@daisy.uwaterloo.ca
## </DESCRIBE>

MatPade :=
`See ?SigmaBases, ?HermitePade, ?PHPA, ?SimultaneousPade, `
. `?RightMatrixPade, ?LeftMatrixPade, ?Pade`:

SigmaBases := proc(F,z,n,s,Sigma)

# Input:
# ---- F:  an m x 1  matrix of formal power series
# ---- z:  variable of the vector of power series 
# ---- n:  a vector of integers 
# ---- s:  an integer
#
# Output:
# An m-by-m matrix polynomial P. The rows of P provide a basis for 
# L^Sigma
#
# Method:
# Uses the algorithm of Beckermann and Labahn (1991) 
# (to appear in SIAM J. of Matrix Analysis (1993)
#

  local lambda,sigma,l,m,FF,d,i,c,P,pi,d_pi,j;

# Initialization

  sigma := 0; 
  if type(F,matrix) then
     m := linalg[rowdim](F);
     for l from 1 to m do
        FF[l] := convert(series(F[l,1],z,Sigma+1),polynom);
     od;
  elif type(F,list) then
     m := nops(F);
     FF := [seq(convert(series(F[l],z,Sigma+1),polynom),l=1..m)];
  elif type(F,vector) then
     m := linalg[vecdim](F);
     FF := [seq(convert(series(F[l],z,Sigma+1),polynom),l=1..m)];
  fi;

  d:= linalg[vector](m);
  for i from 1 to m do
    d[i] := n[i] + 1;
  od;

  P := linalg[matrix](m,m,proc(i,j) if i<>j then 0 else 1 fi end );

# Recursive Step

  while sigma < Sigma do
     lambda := NULL; 

     for l from 1 to m do
        c[l] :=expand(convert([seq(subs(z = z^s,P[l,j])*FF[j],j=1..m)] , `+`));
        c[l] := coeff(expand(c[l]/z^sigma),z,0);
        if  c[l] <> 0 then 
          lambda := lambda,l; 
        fi;
     od;
     # Make lambda into a set
     lambda := {lambda};

# Case Lamda sub sigma = {}

     if lambda = {} then 
        sigma := sigma + 1;

# Case Lamda sub sigma <> {}

     else
        pi := lambda[1];
        d_pi := d[pi];
        for l in lambda  do
          if d[l] > d_pi then d_pi := d[l]; pi := l; fi; 
        od;

        for l in lambda do
          if l <> pi then
              P := linalg[addrow](P,pi,l,-normal(c[l]/c[pi])):
          fi;
        od;
        P := linalg[mulrow](P,pi,z);
        P:= map(normal,P);
        d[pi] := d[pi] - 1; 
        sigma := sigma + 1;
     fi;

# Output
  
   od;
   eval(P);
end:

PHPA := proc( f , z, n ,s, sigma)
  local F, P, x, m, i, sols,j,alpha, dct, poly;

  F := map(series,f,z,sigma+1);
  m := nops(F);
  F := linalg[matrix](m,1,F);
  if type(z,`=`) then
    x := op(1,z);
    F := map(proc(m,i,j) subs(i=j,m) end,eval(F),x,x+op(2,z));
    F := map(normal,eval(F));
  else 
    x := z;
  fi;
  
  if nargs = 6 then sols:= false 
  else sols := true; fi;

  P := SigmaBases(F,x,n,s,sigma);
  for i from 1 to m do
     dct[i] := min(seq(n[j]+1-degree(P[i,j],x),j=1..m));
     if sols then
       if dct[i] > 0 then poly[i] := 1 else poly[i] := 0 fi;
     else
       poly[i]:= convert( [ seq( _c(i,j)*x^j , j=0..dct[i]-1)],`+`);
     fi;
  od;
  alpha := linalg[matrix](1,m,[seq(poly[i],i=1..m)]);
  P := map(expand,evalm( alpha &* P ));
  P := map(sort,eval(P),x);
  P := map(collect,eval(P),x);
  if type(z,`=`) then 
    subs( op(1,z) = op(1,z) - op(2,z),[seq(P[1,i],i=1..m)]);
  else
    [seq(P[1,i],i=1..m)];
  fi;

end:

HermitePade := proc( f , z, n )
  local Sigma;
  Sigma := convert(n,`+`) + nops(f) - 1;
  PHPA(f, z, n, 1, Sigma, args[4..nargs]);
end:

SimultaneousPade := proc( f , z, n)
  local F, Sigma, P, j,N, alpha, dct, i, mu, sols, x, poly;

  Sigma := convert(n,`+`);
  F := map(series,f,z,Sigma+1);
  mu := nops(F);

  if type(z,`=`) then
    x := op(1,z);
    F := map(proc(x,i,j) subs(i=j,x) end, F,x,x+op(2,z)); 
  else x := z;
  fi;

  if nargs = 4 then sols:= false 
  else sols := true; fi;

  F :=  linalg[matrix](mu+1,1,
                      [- convert([seq(x^(j-1)*subs(x=x^mu,F[j]),j=1..mu)],`+`),
                         seq(x^(j-1),j=1..mu)]);
  N := [seq( Sigma - n[i],i=1..mu+1)]; 
  P := SigmaBases(F,x,N,mu,mu*(Sigma+1));
  for i from 1 to mu+1 do
     dct[i] := min(seq(N[j] + 1 -degree(P[i,j],x),j=1..mu+1));
     if sols then
       if dct[i] > 0 then poly[i]:= 1 else poly[i]:= 0 fi;
     else
       poly[i]:= convert( [ seq( _c(i,j)*x^j , j=0..dct[i]-1)],`+`);
     fi;
  od;
  alpha := linalg[matrix](1,mu+1,[seq(poly[i],i=1..mu+1)]);
  P := map(expand,evalm( alpha &* P ));
  P := map(sort,eval(P),x);
  P := map(collect,eval(P),x);
  if type(z,`=`) then 
    subs( op(1,z) = op(1,z) - op(2,z),[seq(P[1,i],i=1..mu+1)]);
  else
    [seq(P[1,i],i=1..mu+1)];
  fi;

end:

RightMatrixPade := proc( f , z, n )
  local F, Sigma, P, r, sols, j, k, N, Q, a, coeffslist,
      dct, den, i, num, p, q, seqlist, x, poly;

  Sigma := convert(n,`+`);
  F := map(series,evalm(f),z,Sigma+1);
  p := linalg[rowdim](F);
  q := linalg[coldim](F);

  if nargs >= 4 and type(args[4],integer) then 
       r := args[4];
       if nargs = 5 then sols := false; 
       else sols := true; fi;
  elif nargs = 4 then sols := false; r := q;
  else r := q; sols := true; 
  fi;

  if type(z,`=`) then
    x := op(1,z);
    F := map(proc(r,i,j) subs(i=j,r) end,eval(F),x,x+op(2,z)); 
  else x := z; 
  fi;

  F := evalm(linalg[matrix](1,p,[seq(x^j,j=0..p-1)]) &*
             linalg[augment]( 
                  linalg[matrix](p,p,proc(i,j) if i=j then 1 else 0 fi end),
                  evalm(- subs(x = x^p,eval(F) ) )));
  F := linalg[transpose](map(series,eval(F),x,p*(Sigma+1)));
  F := map(convert,eval(F),polynom);

  N := [seq(n[1],i=1..p),seq(n[2],i=1..q)]; 
  P := SigmaBases(F,x,N,p,p*(n[1] + n[2] +1));
  
  seqlist:= NULL;
  for i from 1 to p + q  do
     dct[i] := min(seq(N[j] + 1 - degree(P[i,j],x),j=1..p+q));
     if sols then
       if dct[i] > 0 then 
          poly[i] := linalg[matrix](1,r,[seq( convert(
                [seq(_c(i,j)*x^j,j=0..dct[i]-1)],`+`),k=1..r)]); 
          seqlist:= seqlist,[seq(_c(i,j)=0,j=0..dct[i]-1)]; 
       else poly[i]:= linalg[matrix](1,r,0) 
       fi;
     else
       poly[i]:= linalg[matrix](1,r,[seq( convert(  
		[seq(_c(i,j,k)*x^j,j=0..dct[i]-1)],`+`),k=1..r)]);;
     fi;
  od;

  if sols then
     seqlist := [seqlist];
     if nops(seqlist) >= r then
        coeffslist := [seq(op(1,seqlist[i]),i=1..r)];
        seqlist := {op(map(op,seqlist))};
     else
        coeffslist := map(op,seqlist);
        seqlist := {op(coeffslist)};
     fi;
     for i from 1 to p + q do
         for j from 1 to r do
             a := op(1,op(j,coeffslist));
             poly[i][1,j] := sort(subs(subs((a=0)=(a=1),seqlist),
                     poly[i][1,j]),x);
         od;
     od;
  fi;

  Q := linalg[submatrix](P,1..p+q,1..p);
  P := linalg[submatrix](P,1..p+q,p+1..p+q);
  
  num := evalm( convert([ seq(linalg[matrix](p,1,
                   eval(linalg[row](Q,i)))&*poly[i],i=1..p+q) ],`+`)); 
  den := evalm( convert([ seq(linalg[matrix](p,1,
                   eval(linalg[row](P,i)))&*poly[i],i=1..p+q) ],`+`)); 
  num := map(expand,num);
  den := map(expand,den);
  num := map(collect,eval(num),x);
  den := map(collect,eval(den),x);
  if type(z,`=`) then 
    subs( op(1,z) = op(1,z) - op(2,z), [ eval(num), eval(den) ]);
  else
    [ eval(num), eval(den) ]; 
  fi;
end:

LeftMatrixPade := proc( f , z, n )
  local ans;

  ans := RightMatrixPade(linalg[transpose](f), args[2..nargs]);
  [ eval(linalg[transpose](ans[1])), eval(linalg[transpose](ans[2]))];

end:

Pade := proc( f , z , n)
   local F, ans, sols;

  if nargs = 4 then sols:= false 
  else sols := true; fi;

  if sols then
     F := series(f,z,n[1] + n[2] + 1);
     ans := convert(F,ratpoly,n[1],n[2]);
     [numer(ans),denom(ans)];
  else
     ans := HermitePade([-1,f],z,n,'all');
  fi;

end:

#save `MatPade.m`;
#quit
