#
## <SHAREFILE=geometry/algcurv/algcurve.mpl >
## <DESCRIBE>
## SEE ALSO: geometry/algcurve.tex  (149K)
##               A package of routines for computing with algebraic curves
##               in 2 dimensions.  Includes facilities for computing singular
##               points, tangents, inflection points, curve intersection points,
##               as well as this routine for plotting curves.
##               AUTHOR: Stefan Schwendi, schwendi@iam.unibe.ch
## </DESCRIBE>



##
##    Title   :  algcurve,  the algebraic curves package  ( file 1/5)
##
##    Created :  Jul 24  1992
##
##    Author  :  Stefan Schwendimann
##               Institut fuer Informatik und 
##               angewandte Mathematik
##		 Laenggassstrasse 51 
##		 CH--3012 Bern
##
##		 <schwendi@iam.unibe.ch>
##
##    Documentation :   algcurve.tex (algcurve.ps)
##


macro ( 
  scalarmul   =  linalg['scalarmul'],
  crossprod   =  linalg['crossprod'],
  
  affvars     =  algcurve['affvars'],
  homvars     =  algcurve['homvars'],

  iszerolist  =  `algcurve/iszerolist`,
  getindets   =  `algcurve/getindets`,  
  hompoly     =  `algcurve/hompoly`,
  hompoint    =  `algcurve/hompoint`,
  affpoly     =  `algcurve/affpoly`,
  affpoint    =  `algcurve/affpoint`,
  poly2coord  =  `algcurve/poly2coord`,
  coord2poly  =  `algcurve/coord2poly`,
  standard    =  `algcurve/standard`,
  homequal    =  `algcurve/homequal`,
  grad        =  `algcurve/grad`
  
):


# ___  ISZEROLIST  _______________________________________________________

iszerolist := proc(v)
local i;
  for i from 1 to nops(v) do
    if simplify(v[i]) <> 0 then RETURN(false) fi
  od;
  RETURN(true)
end:  




# ___  GETINDETS  ________________________________________________________

getindets := proc (fromargs)
local inds;
  inds:= indets(fromargs,name);
  if inds intersect {op(affvars)} <> {} then
    if inds intersect {op(homvars)} <> {} then
      ERROR(`homogeneous and affine variables in the same expression`)
    fi;
    affvars
  elif inds intersect {op(homvars)} <> {} then
    homvars
  else
    ERROR(`neither homogeneous nor affine variables in expression`)  
  fi
end:


# ___  TPOLY  ________________________________________________________

`type/tpoly` := proc(p)
local vars;
  if type([args],[anything,list(name)]) then
    vars:= args[2]
  elif type([args],[anything]) then
    vars:= indets(p,name)
  else  
    ERROR(`wrong type or number of arguments`)
  fi;  
  type(p, { polynom(algebraic,vars), list(polynom(algebraic,vars)) } )
end:  


# ___  LISTLIST  ________________________________________________________

`type/listlist` := proc(A)
local z,n,b;
options `Copyright 1990 by the University of Waterloo`;
  if nargs <> 1 then ERROR(`wrong number of arguments`) fi;
  if type(A,list) then
    if A = [] then RETURN(true) fi;
    n := nops(op(1,A));
    b := true;
    for z in A while b do  b := type(z,list) and (n = nops(z)) od;
    RETURN(b)
  else false
  fi
end:

# ___  DOTPROD  ________________________________________________________

`algcurve/dotprod`:= proc(u,v)
local i;
  convert( [ seq(u[i]*v[i],i=1..nops(u)) ], `+`)
end:  




# ___  PROJPOLY  ________________________________________________________

`type/projpoly` := proc(F)
local lF,vars,deg,i,i0;
  if type([args],[anything,list(name),posint]) then
    vars:= args[2];
    deg:= args[3]
  elif type([args],[anything,list(name)]) then
    vars:= args[2];
    deg:= 0  
  elif type([args],[anything,posint]) then
    vars:= homvars;
    deg:= args[2]
  elif type([args],[anything]) then
    vars:= homvars;
    deg:= 0
  else  
    ERROR (`wrong number or type of arguments`)
  fi;
  vars:= convert(vars,set); 
  if not type(F,polynom(algebraic,vars)) then
    RETURN (false)
  fi;
  lF:= expand(F);  
  if type(lF,`+`) then lF:= [op(lF)] else lF:= [lF] fi;
  if deg=0 then 
    deg:= degree(lF[1],vars); 
    i0:= 2 
  else
    i0:= 1  
  fi;
  if deg = 0 then RETURN(false) fi;
  for i from i0 to nops(lF) do
    if degree(lF[i],vars) <> deg then
      RETURN (false)
    fi
  od;
  RETURN (true)
end:


# ___  PROJCOORD  ______________________________________________________

`type/projcoord` := proc (p)
local dim;
  if type ([args],[anything,posint]) then
    dim:= args[2]
  elif type ([args],[anything]) then
    dim:= 2
  else
    ERROR(`wrong number or type of arguments`)
  fi;  
  if nops(p)-1 <> dim then RETURN(false) fi;
  type(p,list) and (not iszerolist(p))
end:




# ___  HOMPOLY  ________________________________________________________

hompoly := proc (f)
local hompoly0,avars,hvars,nr,sublist,j;

    hompoly0 := proc(f,avars,homvar,sublist)
    local lf,deg;
      lf:= expand(f);
      deg:= degree(lf,avars);
      if deg = 0 then deg:= 1 fi;
      expand ( homvar^deg * subs(sublist,lf) )
    end:

  if type([args],[anything,list(name),list(name),nonnegint]) then
    avars:= args[2];
    hvars:= args[3];
    nr:= args[4]
  elif type([args],[anything]) then
    avars:= affvars;
    hvars:= homvars;
    nr:= 0
  elif type([args],[anything,nonnegint]) then
    avars:= affvars;
    hvars:= homvars;
    nr:= args[2]
  else
    ERROR(`wrong number or type of arguments`)
  fi;
  if not type(f,tpoly(avars)) then  ERROR(`illegal first argument`)  fi;
  if nr > nops(avars)      then  ERROR(`invalid projection number`) fi; 
  if nops(hvars) <> nops(avars)+1 then
    ERROR(`wrong number of homogeneous variables`)
  fi;
  sublist := [seq(avars[j]=hvars[j]/hvars[nr+1],j=1..nr),
    seq(avars[j]=hvars[j+1]/hvars[nr+1],j=nr+1..nops(avars))];
  avars:= convert(avars,set);  
  if type(f,list) then
    map(hompoly0,f,avars,hvars[nr+1],sublist)
  else
    hompoly0(f,avars,hvars[nr+1],sublist)    
  fi  
end:     





# ___  HOMPOINT  ______________________________________________________


hompoint := proc(p)
local nr,dim, hompoint0;

    hompoint0 := proc(p,nr,dim)
      [op(p[1..nr]),1,op(p[nr+1..dim])]
    end:
    
  if type ([args],[list]) then
    nr:= 0
  elif type([args],[list,nonnegint]) then
    nr:= args[2]
  else
    ERROR(`wrong number or type of arguments`)  
  fi;
  if type(p,listlist) then
    dim:= nops(p[1]);
    if nr > dim then ERROR(`invalid projection number`) fi;  
    map(hompoint0,p,nr,dim)
  else
    dim:= nops(p);
    if nr > dim then ERROR(`invalid projection number`) fi; 
    hompoint0(p,nr,dim)
  fi
end:
    
    
      
# ___  AFFPOLY  ______________________________________________________

affpoly := proc (F)
local avars,hvars,nr,sublist,j;
  if type([args],[anything,list(name),list(name),nonnegint]) then
    hvars:= args[2];
    avars:= args[3];
    nr:= args[4]
  elif type([args],[anything,nonnegint]) then
    avars:= affvars;
    hvars:= homvars;
    nr:= args[2]    
  elif type([args],[anything]) then
    avars:= affvars;
    hvars:= homvars;
    nr:= 0
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  if not type(F,tpoly(hvars)) then ERROR(`illegal first argument`)  fi;
  if nr > nops(avars) then ERROR(`invalid projection number`) fi; 
  if nops(hvars) <> nops(avars)+1 then
    ERROR(`wrong number of affine variables`)
  fi;
  sublist := [seq(hvars[j]=avars[j],j=1..nr),hvars[nr+1]=1,
    seq(hvars[j]=avars[j-1],j=nr+2..nops(hvars))];
  subs(sublist,F)  
end:
  



# ___  AFFPOINT  ______________________________________________________

affpoint := proc(p)
local nr,dim,affpoint0;

    affpoint0 := proc(p,nr,dim) local i,t;
      if p[nr+1] = 0 then RETURN(NULL) fi;
      #scalarmul([p[1..nr],p[nr+2..dim]],1/p[nr+1])
      t := p[nr+1];
      [seq(p[i]/t,i=1..nr),seq(p[i]/t,i=nr+2..dim)]
    end:
  
  if type([args],[list,nonnegint]) then
    nr:= args[2]
  elif type ([args],[list]) then
    nr:= 0
  else
    ERROR(`wrong number or type of arguments`)  
  fi;
  if type(p,listlist) then
    dim:= nops(p[1]);
    if nr > dim-1 then ERROR(`invalid projection number`) fi;  
    map(affpoint0,p,nr,dim)
  else
    dim:= nops(p);
    if nr > dim-1 then ERROR(`invalid projection number`) fi; 
    affpoint0(p,nr,dim)
  fi
end:
 


# ___  POLY2COORD  ______________________________________________________

poly2coord := proc(F)
local vars,varset,lF,v,n,i,j,m;
  if type([args],[anything,list(name)])  then
    vars:= args[2]
  elif type([args],[anything]) then
    vars:= homvars
  else
    ERROR(` wrong number or type of arguments`) 
  fi;
  if not type(F,projpoly(vars)) then 
    ERROR(`first argument is not a homogeneous polynomial in `,eval(vars))
  fi;  
  lF:= collect(F,vars,'distributed');
  if nops(vars) <> 3 then
    ERROR(`number of variables  <> 3`)
  fi;  
  varset:= convert(vars,set);
  n:= degree(lF,varset);
  v:= array([seq(0,i=1..(n+1)*(n+2)/2)]);
  if type(lF,`+`) then lF:= [op(lF)] else lF:= [lF] fi;
  for m from 1 to nops(lF) do
    i:= degree(lF[m],vars[1]);
    j:= degree(lF[m],vars[2]);
    v[(n-i)*(n-i+1)/2 + 1 + n-i-j] := coeffs(lF[m],varset)
  od;  
  convert(v,list)
end:
   	    



# ___  COORD2POLY  ______________________________________________________

coord2poly := proc(C)
local vars,n,i,j,vind,F;
  if type([args],[list,list(name)])  then
    vars:= args[2]
  elif type([args],[list]) then
    vars:= homvars
  else
    ERROR(`wrong number or type of arguments`) 
  fi;
  if nops(vars) <> 3 then
    ERROR(`number of variables  <> 3`)
  fi;  
  n:= -3/2 + sqrt(1+8*nops(C))/2;
  if not type(n,posint) then 
    ERROR(`illegal dimension`)
  fi;  
  vind:= 0;
  F:= 0;
  for i from n by -1 to 0 do
    for j from n-i by -1 to 0 do
      vind:= vind+1;
      F:= F + C[vind]*vars[1]^i*vars[2]^j*vars[3]^(n-i-j)
    od
  od;
  eval(F)    
end:
    
macro (
  poly2coord = algcurve['poly2coord'],
  coord2poly = algcurve['coord2poly']
):



# ___  STANDARD  ______________________________________________________

standard := proc (p)
local i,j,k,tmp, ind,faclist,fac,intfac,comfac, denomj,denomlist,
      expo,exptable, lp,lpdim,ispoly,vars,dummy, polystandard;
      
    polystandard := proc(f,vars)
      sort(primpart(f,vars),vars)
    end:    
      
  if type([args],[list]) then
    ispoly:= false
  elif type([args],[anything,list(name)]) then  
    vars:= args[2];
    ispoly:= true    
  elif type([args],[anything]) then
    vars:= getindets(p);
    ispoly:= true  
  else  
    ERROR(` wrong number or type of arguments`) 
  fi;
  if ispoly then 
    if nops(vars) <> 3 or not type(p,projpoly(vars)) then
      RETURN(polystandard(p,vars))
    fi;
    lp:= poly2coord(p,vars) 
  else  
    lp:= p
  fi;  
  lpdim:= nops(lp);
  lp:= array(lp);
  for i from 1 to lpdim while lp[i] = 0 do od;
  if i > lpdim then ERROR(`argument is zero-tuple`) fi;
  for j from i+1 to lpdim do lp[j]:= normal (lp[j]/lp[i]*dummy) od;
  lp[i]:= dummy;
  denomlist:= []; exptable:= table();
  intfac:= 1;
  for j from i+1 to lpdim do
    tmp:= op(1,[op(lp[j])]);
    if type(tmp,fraction) then
      tmp:= denom(tmp);  k:= tmp/gcd(intfac,tmp); 
      intfac:= intfac*k;
      denomj:= denom(lp[j])/tmp
    else
      denomj:= denom(lp[j])
    fi;
    if denomj <> 1 then 
      if type(denomj,`*`) then faclist:=[op(denomj)] else faclist:=[denomj] fi;
      for k from 1 to nops(faclist) do
        if type(faclist[k],anything^integer) then
          expo:= op(2,faclist[k]);  fac:= op(1,faclist[k])
        else 
          expo:= 1; fac:= faclist[k]
        fi;	
        if member(fac,denomlist,'ind') then
          if exptable[ind] < expo then exptable[ind]:= expo fi
        else
          denomlist:= [op(denomlist),fac];
	  exptable[nops(denomlist)]:= expo
        fi	
      od
    fi
  od;  
  comfac:= convert([seq(denomlist[k]^exptable[k],k=1..nops(denomlist))],`*`);
  for k from i to lpdim do lp[k]:= intfac*lp[k]*comfac/dummy od;
  lp:= convert(lp,list);
  if ispoly then
    coord2poly(lp,vars)
  else
    eval(lp)
  fi    
end:

 
 


# ___  HOMEQUAL  ______________________________________________________

homequal := proc (p1,p2)
local cp,i,k;
  if not type([args],[list,list]) then
    ERROR(`wrong number or type of arguments`)  
  fi;
  if nops(p1) <> nops(p2) then
    ERROR(`dimension mismatch`)
  fi;
  if nops(p1) = 3 then
    iszerolist(convert(crossprod(p1,p2),list));
  else
    cp := { seq(p1[i] = k*p2[i],i=1..nops(p1)) };
    evalb(solve(cp,k) <> NULL)
  fi
end:
  
  
# ___  GRAD  ______________________________________________________
    
grad := proc (F)
local i,vars;
  if type([args],[anything,list(name)]) then
    vars:= args[2]
  elif type([args],[anything]) then
    vars:= getindets(F)
  else
    ERROR(`wrong number or type of arguments`)
  fi;
  if not type(F,polynom(algebraic,vars)) then
    ERROR(`illegal first argument`)
  fi;
  [seq(diff(F,vars[i]),i=1..nops(vars))]
end:           


##
##    Title   :  algcurve,  the algebraic curves package  ( file 2/5)
##
##    Created :  Jul 24  1992
##
##    Author  :  Stefan Schwendimann
##               Institut fuer Informatik und 
##               angewandte Mathematik
##		 Laenggassstrasse 51 
##		 CH--3012 Bern
##
##		 <schwendi@iam.unibe.ch>
##
##    Documentation :   algcurve.tex (algcurve.ps)
##


macro ( 
  crossprod   =  linalg['crossprod'],
  genmatrix   =  linalg['genmatrix'],
  coldim      =  linalg['coldim'],
  transpose   =  linalg['transpose'],
  inverse     =  linalg['inverse'],
  multiply    =  linalg['multiply'],

  affvars     =  algcurve['affvars'],
  homvars     =  algcurve['homvars'],

  iszerolist  =  `algcurve/iszerolist`,
  getindets   =  `algcurve/getindets`,
  dotprod     =  `algcurve/dotprod`,  
  hompoly     =  algcurve['hompoly'],
  hompoint    =  algcurve['hompoint'],
  affpoly     =  algcurve['affpoly'],
  affpoint    =  algcurve['affpoint'],
  standard    =  algcurve['standard'],
  
  homtrans    =  `algcurve/homtrans`,
  afftrans    =  `algcurve/afftrans`,
  image2mat   =  `algcurve/image2mat`,
  mat2image   =  `algcurve/mat2image`,
  appoly      =  `algcurve/appoly`,
  appoly0     =  `algcurve/appoly0`,
  appolyE     =  `algcurve/appolyE`,
  appmat      =  `algcurve/appmat`,
  appimage    =  `algcurve/appimage`,
  NonCurvePt  =  `algcurve/NonCurvePt`,
  NonVertexTrans = `algcurve/NonVertexTrans`,
  PtTrans     =  `algcurve/PtTrans`,
  multcomp    =  `algcurve/multcomp`,
  commoncomp  =  `algcurve/commoncomp`,
  homxyfac    =  `algcurve/homxyfac`

):




# ___  HOMTRANS  ________________________________________________________

homtrans := proc (T)
local avars,hvars,nr;
  if type([args],[anything,list(name),list(name),nonnegint]) then
    avars:= args[2];
    hvars:= args[3];
    nr:= args[4]
  elif type([args],[anything]) then
    avars:= affvars;
    hvars:= homvars;
    nr:= 0
  elif type([args],[anything,nonnegint]) then
    avars:= affvars;
    hvars:= homvars;
    nr:= args[2]
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  if not type(T,list) or not type(T,tpoly(avars)) then
    ERROR(`illegal first argument`)
  fi;  	 
  if nr > nops(avars) then ERROR(`invalid projection number`) fi; 
  if nops(hvars) <> nops(avars)+1 then
    ERROR(`wrong number of homogeneous variables`)
  fi;
  hompoly(hompoint(T,nr),avars,hvars,nr)
end:


# ___  AFFTRANS  ________________________________________________________
  
afftrans := proc (T)
local avars,hvars,nr;
  if type([args],[anything,list(name),list(name),nonnegint]) then
    avars:= args[3];
    hvars:= args[2];
    nr:= args[4]
  elif type([args],[anything]) then
    avars:= affvars;
    hvars:= homvars;
    nr:= 0
  elif type([args],[anything,nonnegint]) then
    avars:= affvars;
    hvars:= homvars;
    nr:= args[2]
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  if not type(T,list(projpoly(hvars))) then
    ERROR(`illegal first argument`)
  fi; 
  if nr > nops(avars) then ERROR(`invalid projection number`) fi; 
  if nops(hvars) <> nops(avars)+1 then
    ERROR(`wrong number of affine variables`)
  fi;
  affpoint(affpoly(T,hvars,avars,nr),nr)
end:




# ___  IMAGE2MAT  ________________________________________________________

image2mat := proc (homimage)
local hvars;
  if type([args],[anything,list(name)]) then
    hvars:= args[2]
  elif type([args],[anything]) then
    hvars:= homvars
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  if not type (homimage, list(projpoly(hvars,1))) then
    ERROR(`first argument is not a list of linear homogeneous polynomials`)
  fi; 
  genmatrix(homimage,hvars)
end:          


# ___  MAT2IMAGE  ________________________________________________________

mat2image := proc (hommat)
local hvars;
  if type([args],[matrix,list(name)]) then
    hvars:= args[2]
  elif type([args],[matrix]) then
    hvars:= homvars
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  if nops(hvars) <> coldim(hommat) then
    ERROR(`dimension mismatch`)
  fi;
  convert(multiply(hommat,hvars),list)
end:




# ___  APPOLY  ____________________________________________________________

appoly0 := proc(F,pt,vars)
local i;
  subs ( [seq(vars[i] = pt[i], i=1..nops(pt))] , F )
end:  

appolyE := proc(F,pt,vars)
  simplify(appoly0(F,pt,vars))
end:  


appoly := proc (F,pt)
local vars,i;
  if type([args],[anything,list,list(name)]) then
    vars:= args[3]
  elif type([args],[anything,list]) then
    vars:= getindets(F)
  else
    ERROR(`wrong type or number of arguments`)
  fi;  
  if not type(F,tpoly(vars)) then
    ERROR(`illegal first argument`)
  fi;  
  if type(pt,listlist) then 
    if nops(pt[1]) <> nops(vars) then ERROR(`dimension mismatch`) fi;
    RETURN( [seq( appoly0(F,pt[i],vars), i=1..nops(pt) )] )
  else 
    if nops(pt) <> nops(vars) then ERROR(`dimension mismatch`) fi;
    RETURN(appoly0(F,pt,vars))
  fi    
end:


# ___  APPIMAGE  __________________________________________________________

appimage := proc (im ,F)
local vars;
  if type([args],[list,anything,list(name)]) then
    vars:= args[3]
  elif type([args],[list,anything]) then
    vars:= getindets([im,F])
  else
    ERROR(`wrong type or number of arguments`)
  fi;  
  if not type(im,list(tpoly(vars))) then
    ERROR(`illegal first argument`)
  fi;
  if not type(F,tpoly(vars)) then
    ERROR(`illegal second argument`)
  fi;  
  appoly0(F,im,vars)
end:



  
# ___  APPMAT  ____________________________________________________________
  
appmat := proc (mat,F)
local vars,polycase,imlist,i;
  if type([args],[matrix,anything,list(name)]) then
    vars:= args[3];
    polycase:= true
  elif type([args],[matrix,list]) then
    polycase:= false  
  elif type([args],[matrix,anything]) then
    vars:= homvars;
    polycase:= true
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  if polycase then
    if not type(F,projpoly(vars)) then
      ERROR(`second argument is not a homogeneous polynomial in`,vars)
    fi;  
    imlist:= convert(multiply(mat,vars),list);  
    appoly0 (F,imlist,vars)
  elif type(F,listlist) then
    [seq( convert(multiply(mat,F[i]),list), i=1..nops(F))]
  else
    convert(multiply(mat,F),list)  
  fi  
end:



# ___  NONCURVEPT  _________________________________________________________ 


NonCurvePt := proc (F)
local f,homcase, avars,hvars,nr, v1,v2,exlist, dir,left,right,up,down;
  if type([args],[anything,nonnegint,listlist,list(name)]) then
    nr:= args[2];
    exlist:= args[3];
    hvars:= args[4];
    homcase:= true
  elif type([args],[anything,nonnegint]) then
    nr:= args[2];
    exlist:= [];
    hvars:= homvars;
    homcase:= true  
  elif type([args],[anything,list(name)]) then
    nr:= 0;
    exlist:= [];
    avars:= args[2];
    homcase:= evalb(nops(avars)=3);  
    if homcase then hvars:= avars fi
  elif type([args],[anything]) then
    nr:= 0;
    exlist:= [];
    avars:= getindets(F);
    homcase:= evalb(nops(avars)=3);
    if homcase then hvars:= avars fi   
  else 
    ERROR(`wrong type or number of arguments`)
  fi; 
  if homcase then
    if nops(hvars) <> 3 then ERROR(`number of homogeneous variables <> 3`) fi;  
    if not type(F,projpoly(hvars)) then 
      ERROR(`first argument is not a homogeneous polynomial in`,hvars)
    fi;  
    if exlist <> [] then exlist:= affpoint(exlist,nr) fi;
    avars:= affvars;
    f:= affpoly(F,hvars,avars,nr)   
  else
    if nops(avars) <> 2 then ERROR(`number of affine variables <> 2`) fi;  
    if not type(F,polynom(algebraic,avars)) then
      ERROR(`first argument ist not a polynomial in`,avars)
    fi;
    f:= F
  fi;
  dir:= right;
  v1:=0; v2:= 0;
  while  appoly0(f,[v1,v2],avars) = 0   or  member([v1,v2],exlist) do
    if dir = right then
      v1:= v1+1;
      if v1 > -v2 then dir:= up fi
    elif dir = up then
      v2:= v2+1;
      if v2 = v1 then dir:= left fi
    elif dir = left then
      v1:= v1-1;
      if v1 = -v2 then dir:= down fi
    else # dir = down
      v2:= v2-1;
      if v2 = v1 then dir:= right fi
    fi;
  od;
  if homcase then
    hompoint([v1,v2],nr)
  else
    [v1,v2]
  fi
end:  # NonCurvePt

macro (NonCurvePt = algcurve['NonCurvePt']):




# ___  NONVERTEXTRANS  ______________________________________________________ 

NonVertexTrans := proc (F)
local vars, prelist, P, i, prenr, a;
  if type([args],[anything,listlist,list(name)]) then
    vars:= args[3];
    prelist:= args[2]
  elif type([args],[anything]) then
    vars:= homvars;
    prelist:= [];
  else
    ERROR(`wrong type or number of arguments`)
  fi; 
  if nops(vars) <> 3  or nops(prelist) > 3 then
    ERROR(`wrong number of variables or too long list`);
  fi;
  if not type(F,projpoly(vars)) then 
    ERROR(`arg is not a homogeneous polynomial in`,vars) 
  fi;
  prenr:= nops(prelist);
  for i from 1 to prenr do P[i]:= prelist[i] od;
  if prenr > 1 and iszerolist(convert(crossprod(P[1],P[2]),list)) then
    ERROR(`dependent tupels in list`);
  fi;  
  if prenr = 0 then
    P[1]:= NonCurvePt(F,0,[],vars)
  fi;
  if prenr <= 1 then
    a:= P[1][2];
    if a <> 0 then 
      P[2]:= NonCurvePt(F,1,[ [1/a,1,P[1][3]/a] ],vars)
    else
      P[2]:= NonCurvePt(F,1,[],vars)
    fi  
  fi;
  if prenr <= 2 then
    P[3]:= NonCurvePt(F*dotprod(crossprod(P[1],P[2]),vars),2,[],vars)
  fi;
  transpose(linalg['matrix'](3,3,[P[1],P[2],P[3]]))
end:


# ___  PTTRANS  ______________________________________________________ 

PtTrans := proc (L1,L2)
local A,B,BB;
  if type([args],[listlist,listlist]) then
    A:= L1;
    B:= L2
  elif type([args],[listlist]) then
    B:= L1
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  BB:= inverse(transpose(linalg['matrix'](B)));
  if nargs = 1 then RETURN(eval(BB)) fi;
  if nops(A) <> nops(B) then ERROR(`different length of lists`) fi;
  A:= transpose(linalg['matrix'](A));
  multiply(A,BB)
end:  



# ___  MULTCOMP  ______________________________________________________ 

multcomp := proc(F)
local vars, i;
  if type([args],[anything,list(name)]) then
    vars:= args[2]
  elif type([args],[anything]) then
    vars:= getindets(F)
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  if not type(F,polynom(algebraic,vars)) then
    ERROR(`illegal first argument`)
  fi;  
  for i from 1 to nops(vars) do
    if has(F,vars[i]) and simplify(discrim(F,vars[i])) = 0 then RETURN(true) fi
  od;
  RETURN(false) 
end:



# ___  COMMONCOMP  ______________________________________________________ 

commoncomp := proc(F,G)
local vars, i;
  if type([args],[anything,anything,list(name)]) then
    vars:= args[3]
  elif type([args],[anything,anything]) then
    vars:= getindets([F,G])
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  if not type([F,G],list(polynom(algebraic,vars))) then
    ERROR(`illegal first or second argument`)
  fi; 
  for i from 1 to nops(vars) do
    if has(F,vars[i]) and has(G,vars[i]) then
      if simplify(resultant(F,G,vars[i])) = 0 then RETURN(true) fi
    fi  
  od;
  RETURN(false) 
end:




# ___  HOMXYFAC  ______________________________________________________ 

homxyfac := proc (F)
local vars,G,S,i,k,r,sols,Grts,multtab;
  if type([args],[anything,list(name)]) then
    vars:= args[2]
  elif type([args],[anything]) then
    vars:= affvars
  else
    ERROR(`wrong type or number of arguments`)
  fi;    
  if nops(vars) <> 2  or  not type(F,projpoly(vars)) then
    ERROR(`number of variables <> 2  or polynomial is not homogeneous`)
  fi;
  S:= NULL;
  G:= F;
  for i from 1 to 2 do
    for k from 0 do
      r:= rem(G,vars[i],vars[i],'G');
      if r <> 0 then  G:= G*vars[i]+r; break fi
    od;
    if k > 0 then S:= S,vars[i]^k fi
  od;  
  sols:= [solve(subs(vars[1]=1,G),vars[2])]; Grts:= [];
  for i from 1 to nops(sols) do
    if not member(sols[i],Grts,'k') then
      Grts:= [op(Grts),sols[i]];
      multtab[nops(Grts)]:= 1
    else
      multtab[k]:= multtab[k]+1
    fi
  od;      
  [S,seq( (vars[1]*numer(Grts[i])-denom(Grts[i])*vars[2])^multtab[i],
          i=1..nops(Grts))]
end: 


  
 

##
##    Title   :  algcurve,  the algebraic curves package  ( file 3/5)
##
##    Created :  Jul 24  1992
##
##    Author  :  Stefan Schwendimann
##               Institut fuer Informatik und 
##               angewandte Mathematik
##		 Laenggassstrasse 51 
##		 CH--3012 Bern
##
##		 <schwendi@iam.unibe.ch>
##
##    Documentation :   algcurve.tex (algcurve.ps)
##


macro ( 
  nullspace   =  linalg['nullspace'],

  affvars     =  algcurve['affvars'],
  homvars     =  algcurve['homvars'],

# ---  alg1  ----
  iszerolist  =  `algcurve/iszerolist`,
  getindets   =  `algcurve/getindets`,
  dotprod     =  `algcurve/dotprod`,
  hompoly     =  algcurve['hompoly'],
  hompoint    =  algcurve['hompoint'],
  affpoly     =  algcurve['affpoly'],
  affpoint    =  algcurve['affpoint'],
  coord2poly  =  algcurve['coord2poly'],
  standard    =  algcurve['standard'],
  grad        =  algcurve['grad'],

# ---  alg2  ----
  appoly0     =  `algcurve/appoly0`,
  appolyE     =  `algcurve/appolyE`,
  appoly      =  algcurve['appoly'],
  multcomp    =  algcurve['multcomp'],
  homxyfac    =  `algcurve/homxyfac`,

# ---  alg3  ----
  singularities =  `algcurve/singularities`,
  isregular     =  `algcurve/isregular`,
  issingular    =  `algcurve/issingular`,
  isordinary    =  `algcurve/isordinary`,
  curvetaylor   =  `algcurve/curvetaylor`,
  multiplicity  =  `algcurve/multiplicity`,
  singtangent   =  `algcurve/singtangent`,
  tangent       =  `algcurve/tangent`,
  unityvec      =  `algcurve/unityvec`,
  filter        =  `algcurve/filter`,
  processmult   =  `algcurve/processmult`,
  linearsystem  =  `algcurve/linearsystem`,
  randsyspoly   =  `algcurve/randsyspoly`,
  iPoints       =  `algcurve/iPoints`,
  cPoints       =  `algcurve/cPoints`,
  getPoints     =  `algcurve/getPoints`,
  shorter       =  `algcurve/shorter`,
  RegCurvePts   =  `algcurve/RegCurvePts`
):


# ___  SINGULARITIES  ________________________________________________________

singularities := proc (F)
local vars,sols,t,oncurve,homcase,i;
  if type([args],[anything,list(name)]) then
    vars:= args[2]
  elif type([args],[anything]) then
    vars:= getindets(F)
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  if not type(F,polynom(algebraic,vars)) then
    ERROR(`illegal first argument`)
  fi;
  sols:= [solve(convert(grad(F,vars),set),convert(vars,set))];
  sols:= `algcurve/patchsolve`(sols):
  sols:= [seq(subs(sols[i],eval(vars)),i=1..nops(sols))];
  homcase:= evalb( nops(vars) = 3  and type(F,projpoly(vars) ));
  if homcase then
    oncurve:= unapply('evalb'('not iszerolist(t) and  appolyE'(F,t,vars)=0),t);
    RETURN (map(standard,select(oncurve,sols)))
  else
    oncurve:= unapply('evalb'('appolyE'(F,t,vars)=0),t);
    RETURN (select(oncurve,sols))        
  fi
end:  

`algcurve/patchsubs` := proc(L,r,sols) local s;
    seq(subs(r=s,L),s=sols)
end:
	
`algcurve/patchsolve` := proc(L) local Lup,r,poly,x;
    Lup := L:
    for r in indets(L,RootOf) do
        poly := op(1,r);
	poly := frontend( subs, [_Z=x,poly], [{`+`,`*`,`=`},{}] );
        if degree(poly,x)=2 then
	    Lup := map(`algcurve/patchsubs`,Lup,r,[solve(poly,x)]);
	fi;
    od;
    Lup
end:


# ___  ISREGULAR  ______________________________________________________

isregular := proc (F,pt)
local vars;
  if type([args],[anything,list,list(name)]) then
    vars:= args[3]
  elif type([args],[anything,list]) then
    vars:= getindets(F)
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  if not type(F,polynom(algebraic,vars)) then
    ERROR(`illegal first argument`)
  fi;
  if nops(vars) <> nops(pt) then
    ERROR(`dimension mismatch`)
  fi;  
  ( appolyE(F,pt,vars) = 0  or  indets(pt,name) <> {} )  and  
  not iszerolist(appoly0(grad(F,vars),pt,vars))
end:  

macro(isregular = algcurve['isregular']):




# ___  ISSINGULAR  ______________________________________________________

issingular := proc (F,pt)
local vars;
  if type([args],[anything,list,list(name)]) then
    vars:= args[3]
  elif type([args],[anything,list]) then
    vars:= getindets(F)
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  if not type(F,polynom(algebraic,vars)) then
    ERROR(`illegal first argument`)
  fi;
  if nops(vars) <> nops(pt) then
    ERROR(`dimension mismatch`)
  fi;  
  appolyE(F,pt,vars) = 0  and  iszerolist(appolyE(grad(F,vars),pt,vars))
end:  


# ___  CURVETAYLOR  ______________________________________________________

curvetaylor := proc (F,pt)
local vars,varset, tay, tmp,i;
  if type([args],[anything,list,list(name)]) then
    vars:= args[3]
  elif type([args],[anything,list]) then
    vars:= getindets(F)
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  if not type(F,polynom(algebraic,vars)) then
    ERROR(`illegal first argument`)
  fi;  
  if nops(vars) <> nops(pt) then
    ERROR(`dimension mismatch`)
  fi;
  varset:= convert(vars,set);  
  tay:= readlib(mtaylor) ( F, [ seq(vars[i]=pt[i],i=1..nops(pt)) ], 
                           degree(F,varset) + 1 );
  if indets(pt,name) <> {} then
    tmp:= subs( [ seq( vars[i]=vars[i]+pt[i], i=1..nops(pt) ) ], tay);
    if ldegree(tmp,varset) = 0 then
      RETURN(tay-tcoeff(tmp,varset))
    fi  
  fi;
  eval(tay)
end:

macro(curvetaylor = algcurve['curvetaylor']):




# ___  MULTIPLICITY  ______________________________________________________

multiplicity := proc (F,pt)
local vars,varset,tmp,i,k,lpt,isptlist,res;
  if type([args],[anything,list,list(name)]) then
    vars:= args[3]
  elif type([args],[anything,list]) then
    vars:= getindets(F)
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  if not type(F,polynom(algebraic,vars)) then
    ERROR(`illegal first argument`)
  fi;  
  if pt = [] then RETURN([]) fi;
  isptlist:= type(pt,listlist);
  if isptlist then lpt:= pt else lpt:= [pt] fi;
  if nops(lpt[1]) <> nops(vars) then
    ERROR(`dimension mismatch`)
  fi;
  res:= NULL;
  varset:= convert(vars,set);
  for k from 1 to nops(lpt) do
    tmp:= subs( [ seq( vars[i]=vars[i]+lpt[k][i], i=1..nops(lpt[k]) ) ],
                 curvetaylor(F,lpt[k],vars));
    res:= res, ldegree(expand(tmp),varset)
  od;
  if isptlist then RETURN([res]) else RETURN(eval(res)) fi 
end:




# ___  TANGENT  ______________________________________________________

singtangent := proc(f,pt,vars)
local tay,varset,lowterm,t,sfn,i;
  tay:= subs( [ seq( vars[i]=vars[i]+pt[i], i=1..nops(pt) ) ],
              curvetaylor(f,pt,vars));
  varset:= convert(vars,set);
  if not type(tay,`+`) then 
    lowterm:= tay 
  else 
    sfn:= unapply('evalb'('degree'(t,varset)=ldegree(tay,varset)),t);
    lowterm:= select(sfn,tay) 
  fi;
  subs( [ seq( vars[i]=vars[i]-pt[i], i=1..nops(pt) ) ],
        homxyfac(lowterm,vars) )
end:  
  	      
tangent := proc(F,pt)
local avars,vars, homcase,nr, gr,tmp,i;
  if type([args],[anything,list,list(name)]) then
    vars:= args[3]
  elif type([args],[anything,list]) then
    vars:= getindets(F)
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  if not type(F,polynom(algebraic,vars)) then
    ERROR(`illegal first argument`)
  fi;  
  if nops(vars) <> nops(pt) then
    ERROR(`dimension mismatch`)
  fi; 
  if not ( appolyE(F,pt,vars) = 0  or  indets(pt,name) <> {} ) then
    ERROR(`point is not on curve`)
  fi;
  homcase:= evalb( nops(vars) = 3  and type(F,projpoly(vars)) );
  if homcase and iszerolist(pt) then
    ERROR(`list was zero-tuple`)
  fi;  
  gr:= grad(F,vars);
  if not iszerolist(appolyE(gr,pt,vars)) then   # regular point
    if homcase then
      RETURN (dotprod ( vars, appoly(gr,pt,vars) ))
    else
      RETURN (dotprod ( [seq(vars[i]-pt[i],i=1..nops(vars))], 
                        appoly(gr,pt,vars) )) 
    fi
  fi;
  if nops(vars) > 2 and not homcase then RETURN('tangent(args)') fi;
  if homcase then
    for nr from 0 to nops(pt)-1 while pt[nr+1] = 0 do od;
    avars:= affvars;
    tmp:= singtangent(affpoly(F,vars,avars,nr),affpoint(pt,nr),avars);
    RETURN(hompoly(tmp,avars,vars,nr))
  else
    RETURN(singtangent(F,pt,vars))
  fi  
end:    

macro (tangent = algcurve['tangent']):



# ___  ISORDINARY  ______________________________________________________

isordinary := proc (F,pt)
local vars,varset, tay,i, sfn,t, lowterm;
  if type([args],[anything,list,list(name)]) then
    vars:= args[3]
  elif type([args],[anything,list]) then
    vars:= getindets(F)
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  if not type(F,polynom(algebraic,vars)) then
    ERROR(`illegal first argument`)
  fi;  
  if nops(vars) <> nops(pt) then
    ERROR(`dimension mismatch`)
  fi;  
  if isregular(F,pt) then RETURN(true)
  elif appolyE(F,pt,vars) <> 0 then RETURN(false) 
  else
    tay:= subs( [ seq( vars[i]=vars[i]+pt[i], i=1..nops(pt) ) ],
                curvetaylor(F,pt,vars));
    varset:= convert(vars,set);
    if not type(tay,`+`) then 
      lowterm:= tay 
    else 
      sfn:= unapply('evalb'('degree'(t,varset)=ldegree(tay,varset)),t);
      lowterm:= select(sfn,tay) 
    fi;
    not multcomp(lowterm,vars)
  fi
end:  




# ___  LINEARSYSTEM ______________________________________________________

unityvec := proc (n,vars)
local i,j,dim,vind,v;
  dim:= (n+1)*(n+2)/2; 
  vind:= 0;    
  v:= array([seq(0,i=1..dim)]);
  for i from n by -1 to 0 do
    for j from n-i by -1 to 0 do
      vind:= vind+1;
      v[vind]:= vars[1]^i*vars[2]^j*vars[3]^(n-i-j)
    od
  od;
  convert(v,list)
end:


filter := proc(pts,rest,mult) 
local resseq,complseq,i,lmult;
  if type (pts[1],[list,posint]) then
    lmult:= pts[1][2]
  else 
    lmult:= 1
  fi;  
  resseq:= NULL; complseq:= NULL;
  for i from 1 to nops(pts) do
    if type (pts[i],[list,posint]) then
      if pts[i][2] = lmult then
        resseq:= resseq,pts[i][1]
      else
      	complseq:= complseq,pts[i]
      fi	
    elif  lmult = 1 then
      resseq:= resseq,pts[i];
    else
      complseq:= complseq,pts[i]
    fi
  od;
  mult:= lmult;
  rest:= [complseq];
  RETURN([resseq])
end:       
  
processmult:= proc (pts,mult,unity,vars)
local lunity, s,len,tmp, i,j,k;
  len:= nops(pts);
  if mult = 1 then 
    RETURN([seq(appoly0(unity,pts[i],vars),i=1..len)])
  fi;  
  s:= mult-1;
  tmp:= NULL;
  for i from 0 to s do
    for j from 0 to s-i do
      lunity:= map(diff,unity,vars[1]$i,vars[2]$j,vars[3]$(s-i-j));
      tmp:= tmp,seq(appoly0(lunity,pts[k],vars),k=1..len)
    od
  od;
  [tmp]
end:




linearsystem := proc (pts,m)
local vars,unity,A,nspace,rowseq,restlist,mult,ptsofmult,i;
  if type([args],[list(list),posint,list(name)]) then
    vars:= args[3]
  elif type([args],[list(list),posint]) then 
    vars:= homvars
  else  
    ERROR(`wrong type or number of arguments`)
  fi;
  if nops(vars) <> 3 then ERROR(`dimension <> 3`) fi;
  for i from 1 to nops(pts) do
    if (type(pts[i],[list,posint]) and nops(pts[i][1]) <> 3) or
       (type(pts[i],list(algebraic)) and nops(pts[i]) <> 3) then
      ERROR(`dimension mismatch in point list`)
    fi
  od;
  unity:= unityvec(m,vars);
  rowseq:= NULL;
  restlist:= pts;
  while restlist <> [] do
    ptsofmult:= filter(restlist,'restlist','mult');
    rowseq:= rowseq,op(processmult(ptsofmult,mult,unity,vars))
  od;
  A:= linalg['matrix']([rowseq]);
  nspace:= map(convert,convert(nullspace(A),list),list);
  map(coord2poly,nspace,vars)
end:


# ___  RANDSYSPOLY  ______________________________________________________


randsyspoly := proc(L)
local vars,r,i;
  if type([args],[list,list(name)]) then
    vars:= args[2]
  elif type([args],[list]) then
    vars:= getindets(L)
  else  
    ERROR(`wrong type or number of arguments`)
  fi;
  r:= rand(-1..1);
  convert( [ seq( r()*L[i], i=1..nops(L) ) ], `+`)
end:




# ___  REGCURVEPTS  _________________________________________________________ 


iPoints := proc (f,oldptset,nrofpts,vars)
local ptset, sols,sols1,sols2, j,ii, isreg,t;
  ptset:= oldptset;
  isreg:= unapply('isregular'(f,t,vars),t);
  for ii from 0 to 50 while nops(ptset) < nrofpts do 
    sols1:= { isolve(subs(vars[1]=ii, f), vars[2]) };
    sols2:= { isolve(subs(vars[1]=-ii,f), vars[2]) };
    sols:= { seq([ ii,subs(sols1[j],vars[2])], j=1..nops(sols1)) } union
           { seq([-ii,subs(sols2[j],vars[2])], j=1..nops(sols2)) };
    sols1:= { isolve(subs(vars[2]=ii, f), vars[1]) };
    sols2:= { isolve(subs(vars[2]=-ii,f), vars[1]) };
    sols:= sols union { seq([subs(sols1[j],vars[1]), ii], j=1..nops(sols1)) } 
                union { seq([subs(sols2[j],vars[1]),-ii], j=1..nops(sols2)) };
    ptset:= ptset union select(isreg,sols)
  od;
  eval(ptset)
end:


cPoints := proc (f,oldptset,nrofpts,vars)
local ptset, sols,sols1,sols2, j,ii, isreg,t;
  ptset:= oldptset;
  isreg:= unapply('isregular'(f,t,vars),t);
  for ii from 0 to 10 while nops(ptset) < nrofpts do 
    sols1:= { solve(subs(vars[1]=ii, f), vars[2]) };
    sols2:= { solve(subs(vars[1]=-ii,f), vars[2]) };
    sols:= { seq([ ii,sols1[j] ], j=1..nops(sols1)) } union
           { seq([-ii,sols2[j] ], j=1..nops(sols2)) };
    sols1:= { solve(subs(vars[2]=ii, f), vars[1]) };
    sols2:= { solve(subs(vars[2]=-ii,f), vars[1]) };
    sols:= sols union { seq([sols1[j], ii], j=1..nops(sols1)) } 
                union { seq([sols2[j],-ii], j=1..nops(sols2)) };
    ptset:= ptset union select(isreg,sols)
  od;
  eval(ptset)
end:

getPoints := proc(F,hvars,avars,homcase,oldptset,nrofpts,pointproc)
local f,ptset,ptsetnr,nr,sel,sel0,t;
  ptset:= oldptset;
  sel0:= unapply('evalb'(t[1]=0),t );
  for nr from 0 to 2 while nops(ptset) < nrofpts do
    f:= affpoly(F,hvars,avars,nr);
    ptsetnr := map(affpoint,ptset,nr);
    ptsetnr := pointproc(f,ptsetnr,nrofpts,avars);
    ptsetnr := map(hompoint,ptsetnr,nr);
    if not homcase then ptsetnr := ptsetnr minus select(sel0,ptsetnr) fi;
    sel := unapply( 'evalb'(t[nr+1]=0),t );
    ptset:= select(sel,ptset) union ptsetnr
  od;
  eval(ptset)
end:


shorter := proc(t1,t2)
local l1,l2,test;
  l1:= length(t1); l2:= length(t2);
  if l1 < l2 then RETURN(true) fi;
  if l1 > l2 then RETURN(false) fi;
  test:= evalb( abs(t1[1]) <  abs(t2[1]) );
  if member(test,{true,false}) then
    if test then RETURN(true) fi;
    if abs(t1[1]) > abs(t2[1]) then RETURN(false) fi
  fi;
  test:= evalb(abs(t1[2]) < abs(t2[2]));
  if member(test,{true,false}) then RETURN(test) fi;
  false
end:


RegCurvePts := proc (G,nrofpts)
local homcase, ptset,ptlist, avars,hvars, F;
  if type([args],[anything,nonnegint,list(name)]) then
    avars:= args[3];
  elif type([args],[anything,nonnegint]) then
    avars:= `algcurve/getindets`(G);
  else 
    ERROR(`wrong type or number of arguments`)
  fi; 
  homcase:= evalb(nops(avars)=3);
  if homcase then
    hvars:= avars;
    if not type(G,projpoly(hvars)) then 
      ERROR(`first argument is not a homogeneous polynomial in`,hvars)
    fi;  
    avars:= affvars;
    F:= G
  else
    if nops(avars) <> 2 then ERROR(`number of affine variables <> 2`) fi;
    if not type(G,polynom(algebraic,avars)) then
      ERROR(`first argument ist not a polynomial in`,avars)
    fi;
    hvars:= homvars;
    F:= hompoly(G,avars,hvars,0)
  fi;
  ptset:= {};
  F:= standard(F,hvars);
  if type(F,polynom(rational,hvars)) then
    ptset:= getPoints(F,hvars,avars,homcase,ptset,nrofpts,iPoints)
  fi;
  if nops(ptset) < nrofpts then
    ptset:= getPoints(F,hvars,avars,homcase,ptset,nrofpts,cPoints)
  fi;
  ptlist:= convert(ptset,list);
  if homcase then
    sort(map(standard,ptlist),shorter)[1..min(nrofpts,nops(ptlist))]
  else
    ptlist:= affpoint(ptlist);
    sort(ptlist,shorter)[1..min(nrofpts,nops(ptlist))]
  fi
end: 



     

##
##    Title   :  algcurve,  the algebraic curves package  ( file 4/5)
##
##    Created :  Jul 24  1992
##
##    Author  :  Stefan Schwendimann
##               Institut fuer Informatik und 
##               angewandte Mathematik
##		 Laenggassstrasse 51 
##		 CH--3012 Bern
##
##		 <schwendi@iam.unibe.ch>
##
##    Documentation :   algcurve.tex (algcurve.ps)
##


macro ( 
  det         =  linalg['det'],
  hessian     =  linalg['hessian'],

  affvars     =  algcurve['affvars'],
  homvars     =  algcurve['homvars'],

# ---  alg1  ----
  iszerolist  =  `algcurve/iszerolist`,
  getindets   =  `algcurve/getindets`,
  dotprod     =  `algcurve/dotprod`,  
  hompoly     =  algcurve['hompoly'],
  hompoint    =  algcurve['hompoint'],
  affpoly     =  algcurve['affpoly'],
  affpoint    =  algcurve['affpoint'],
  standard    =  algcurve['standard'],
  grad        =  algcurve['grad'],

# ---  alg2  ----
  appoly0     =  `algcurve/appoly0`,
  appolyE     =  `algcurve/appolyE`,  
  appoly      =  algcurve['appoly'],
  commoncomp  =  algcurve['commoncomp'],

# ---  alg3  ----
  isregular   =  algcurve['isregular'],
  tangent     =  algcurve['tangent'],

# ---  alg4  ----
  intersection  =  `algcurve/intersection`,
  Polar         =  `algcurve/Polar`,
  tangentthrupt =  `algcurve/tangentthrupt`,
  Hessian       =  `algcurve/Hessian`,
  inflections   =  `algcurve/inflections`,
  intermultiplicity0 =  `algcurve/intermultiplicity0`,
  intermultiplicity  =  `algcurve/intermultiplicity`,
  newrootseq    =  `algcurve/newrootseq`,
  tryall        =  `algcurve/tryall`

):



# ___  INTERSECTION  ________________________________________________________

intersection := proc (F,G)
local vars,homcase, sols,i, sel,t;
  if type([args],[anything,anything,list(name)]) then
    vars:= args[3]
  elif type([args],[anything,anything]) then
    vars:= getindets([F,G])
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  if not type([F,G],list(polynom(algebraic,vars))) then
    ERROR(`illegal arguments`)
  fi;  
  if commoncomp(F,G,vars) then 
    ERROR(`polynomials have a common component`)
  fi;
  sols:= [solve({F,G},convert(vars,set))];
  sols:= `algcurve/patchsolve`(sols):
  sols:= convert( { seq(subs(sols[i],vars),i=1..nops(sols)) }, list);
  homcase:= evalb( nops(vars) = 3  and type(F,projpoly(vars) ));
  if homcase then
    sel:= unapply( 'evalb(not iszerolist(t))',t );
    sols:= select(sel,sols);
    RETURN (map(standard,sols))
  else
    RETURN (eval(sols))        
  fi
end:  


# ___  POLAR  ______________________________________________________

Polar:= proc(F,pt)
local vars, hvars, affcase, tmp;
  if type([args],[anything,list,list(name)]) then
    vars:= args[3]
  elif type([args],[anything,list]) then
    vars:= getindets(F)
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  if not type(F,polynom(algebraic,vars)) then
    ERROR(`illegal first argument`)
  fi;  
  if nops(vars) <> nops(pt) then
    ERROR(`dimension mismatch`)
  fi; 
  affcase:= evalb(nops(pt) = 2);
  if not affcase and not type(F,projpoly(vars)) then
    ERROR(`first argument is not a homogeneous polynomial in`,vars)
  fi;
  if affcase then
    hvars:= homvars;
    tmp:= dotprod(hompoint(pt,0),grad(hompoly(F,vars,hvars,0),hvars));
    RETURN ( affpoly(standard(tmp,hvars),hvars,vars,0) )
  else
    RETURN ( standard(dotprod(pt,grad(F,vars)),vars) )
  fi  
end:  




# ___  TANGENTTHRUPT  ________________________________________________________

tangentthrupt := proc (F,pt)
local vars,polsec,tmp,i,t,sel;
  if type([args],[anything,list,list(name)]) then
    vars:= args[3]
  elif type([args],[anything,list]) then
    vars:= getindets(F)
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  if not type(F,polynom(algebraic,vars)) then
    ERROR(`illegal first argument`)
  fi; 
  sel:= unapply('isregular'(F,t,vars),t);
  polsec := select(sel,intersection(F,Polar(F,pt,vars),vars));
  tmp:= NULL;
  for i from 1 to nops(polsec) do
    tmp:= tmp, standard(tangent(F,polsec[i],vars))
  od;
  [tmp]
end:


# ___  INTERMULTIPLICITY  ____________________________________________________

intermultiplicity0 := proc (f,g,pt,vars)
local lf,lg,k,fk,gk,fk0,gk0,v,R;
  if appolyE(f,pt,vars) <> 0  or  appolyE(g,pt,vars) <> 0 then  RETURN( 0 ) fi;
  v:= [vars[1]=vars[1]+pt[1], vars[2]=vars[2]+pt[2]];
  lf:= subs(v,f); 
  lg:= subs(v,g);
  v:=[];
  fk0:= simplify(expand(subs(vars[1]=0,lf)));  
  gk0:= simplify(expand(subs(vars[1]=0,lg)));
  fk0:= expand(fk0/vars[2]^ldegree(fk0,vars[2]));
  gk0:= expand(gk0/vars[2]^ldegree(gk0,vars[2]));
  R:= simplify(resultant(fk0,gk0,vars[2]));
  for k from 0 while R = 0 do
    v:= [vars[1]=k*vars[1]+vars[2], vars[2]=-vars[1]+ k*vars[2]];
    fk0:= simplify(expand(subs(v,vars[1]=0,lf))); 
    gk0:= simplify(expand(subs(v,vars[1]=0,lg)));
    fk0:= expand(fk0/vars[2]^ldegree(fk0,vars[2]));
    gk0:= expand(gk0/vars[2]^ldegree(gk0,vars[2]));
    R:= simplify(resultant(fk0,gk0,vars[2]));
  od;
  fk:= expand(subs(v,lf)); 
  gk:= expand(subs(v,lg));
  R:= simplify(resultant(fk,gk,vars[2]));
  ldegree(R,vars[1])
end:




intermultiplicity := proc(F,G,pt)
local vars,avars, f,g,lpt,actpt,i,nr,lastnr,isptlist,res;
  if type([args],[anything,anything,list,list(name)]) then
    vars:= args[4]
  elif type([args],[anything,anything,list]) then
    vars:= getindets([F,G])
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  isptlist:= type(pt,listlist);
  if isptlist then lpt:= pt else lpt:= [pt] fi;
  if nops(vars) <> nops(lpt[1]) then
    ERROR(`dimension mismatch`)
  fi;
  res:= NULL;
  if nops(vars) = 3 then
    if not type([F,G], list(projpoly(vars))) then
      ERROR(`arguments are not homogeneous polynomials in `,vars)
    fi;
    lastnr:= -1;
    avars:= affvars;
    for i from 1 to nops(lpt) do
      if iszerolist(lpt[i]) then ERROR(`zero coordinates`) fi;
      for nr from 0 to 2 while lpt[i][nr+1]=0 do od;
      if nr <> lastnr then
        f:= affpoly(F,vars,avars,nr); 
        g:= affpoly(G,vars,avars,nr); 
	lastnr:= nr
      fi;
      actpt:= affpoint(lpt[i],nr);
      res:= res, intermultiplicity0(f,g,actpt,avars)
    od
  elif nops(vars) <> 2 then
    ERROR(`illegal number of variables`)
  else
    if not type([F,G], list(polynom(algebraic,vars))) then
      ERROR(`arguments are not polynomials in`,vars)
    fi;
    f:= F;  g:= G; 
    res := seq(intermultiplicity0(f,g,lpt[i],vars), i=1..nops(lpt))
  fi;
  if isptlist then
    [res]
  else
    eval(res)
  fi
end:


      
# ___  HESSIAN  __________________________________________________

Hessian := proc(F)
local vars,hvars,affcase,lF,res;
  if type([args],[anything,list(name)]) then
    vars:= args[2]
  elif type([args],[anything]) then
    vars:= getindets(F)
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  affcase:= evalb(nops(vars)=2);
  if affcase then
    if not type(F,polynom(algebraic,vars)) then
      ERROR(`illegal first argument`)
    fi;
    hvars:= homvars;
    lF:= hompoly(F,vars,hvars,0)
  else
    if not type(F,projpoly(vars)) then
      ERROR(`first argument is not a homogeneous polynomial in`,vars)
    fi;    
    hvars:= vars;
    lF:= F
  fi;  
  res:= standard(det(hessian(lF,hvars)),hvars);
  if affcase then
    affpoly(res,hvars,vars,0)
  else
    eval(res)
  fi
end:        

macro(Hessian = algcurve['Hessian']):


# ___  INFLECTIONS  __________________________________________________


inflections := proc (F)
local vars,sols,t,sel,homcase;
  if type([args],[anything,list(name)]) then
    vars:= args[2]
  elif type([args],[anything]) then
    vars:= getindets(F)
  else
    ERROR(`wrong type or number of arguments`)
  fi;
  sols:= intersection(F,Hessian(F,vars),vars);
  homcase:= evalb( nops(vars) = 3  and type(F,projpoly(vars) ));
  sel:= unapply('isregular'(F,t,vars),t);
  if homcase then
    map(standard,select(sel,sols))
  else
    select(sel,sols)     
  fi
end:  



# ___  TRYALL  __________________________________________________

newrootseq := proc(eq)
local x,xeq,roots;
options `Copyright 1990 by the University of Waterloo`;
    xeq := subs(_Z = x,eq);
    _EnvExplicit := true;
    roots := [solve(xeq,x)];
    op(roots)
end:

tryall := proc(L)
local lL, sols,i, oldrootseq, optd, optsimpl;
global `allvalues/rootseq`;
  if nargs > 3 then ERROR(`wrong number of arguments`) fi;
  if not type(L,list) then
    ERROR(`wrong type of first argument`)
  fi;
  lL:= L;
  optd:= false; optsimpl:= false;
  if nargs > 1 then
    if nargs = 2 and member(args[2],{'d','simplified'}) then
      if args[2] = 'd' then optd:= true else optsimpl:= true fi
    elif nargs = 3 and {args[2..3]} = {'d','simplified'} then
      optd:= true; optsimpl:= true
    else
      ERROR(`illegal option(s)`)
    fi
  fi;
  readlib(allvalues);
  oldrootseq:= eval(`allvalues/rootseq`);
  `allvalues/rootseq`:= eval(newrootseq);
  if optd then
    sols:= {seq(allvalues(lL[i],'d'),i=1..nops(lL))}
  else
    sols:= {seq(allvalues(lL[i]),i=1..nops(lL))}
  fi;
  `allvalues/rootseq`:= eval(oldrootseq);
  if optsimpl then sols:= map(simplify,sols) fi;
  convert(sols,list)
end:





##
##    Title   :  algcurve,  the algebraic curves package  ( file 5/5)
##
##    Created :  Jul 24  1992
##
##    Author  :  Stefan Schwendimann
##               Institut fuer Informatik und 
##               angewandte Mathematik
##		 Laenggassstrasse 51 
##		 CH--3012 Bern
##
##		 <schwendi@iam.unibe.ch>
##
##    Documentation :   algcurve.tex (algcurve.ps)
##


macro ( 
  affvars     =  algcurve['affvars'],
  homvars     =  algcurve['homvars'],

# ---  alg1  ----
  iszerolist  =  `algcurve/iszerolist`,
  getindets   =  `algcurve/getindets`,
  hompoly     =  algcurve['hompoly'],
  hompoint    =  algcurve['hompoint'],
  affpoly     =  algcurve['affpoly'],
  affpoint    =  algcurve['affpoint'],
  standard    =  algcurve['standard'],

# ---  alg2  ----
  appoly0     =  `algcurve/appoly0`,
  appolyE     =  `algcurve/appolyE`, 
  appoly      =  algcurve['appoly'],

# ---  alg3  ----
  singularities     =  algcurve['singularities'],
  multiplicity =  algcurve['multiplicity'],
  linearsystem =  algcurve['linearsystem'],

# ---  alg5  ----
  implicit    =  `algcurve/implicit`,
  expldeg1    =  `algcurve/expldeg1`,
  explmultnminus1 = `algcurve/explmultnminus1`,
  regcurvept  =  `algcurve/regcurvept`,
  explicit0   =  `algcurve/explicit0`,
  explicit    =  `algcurve/explicit`, 
  quadtrans   =  `algcurve/quadtrans`,
  extractfactors = `algcurve/extractfactors`,
  plotcurve   =  `algcurve/plotcurve`,
  plotcurve0  =  `algcurve/plotcurve0`
):



# ___  IMPLICIT  ________________________________________________________

implicit := proc (V)
local v, var,vars,avars, f,g,affcase,nr;
  if type ([args],[anything,name,list(name)]) then
    var:= args[2];
    vars:= args[3]
  elif type ([args],[anything]) then  
    var:= explvar;
    if nops(V) = 2 then vars:= affvars else vars:= homvars fi
  else  
    ERROR(`wrong type or number of arguments`)
  fi;
  if not type(V, list(ratpoly(algebraic,var))) then
    ERROR(`illegal first argument`)
  fi;
  if nops(vars) < 2 or nops(vars) > 3 then 
    ERROR(`illegal number of variables`) 
  fi;  
  affcase:= evalb(nops(vars) = 2);
  if affcase then 
    v:= V;
    avars:= vars 
  else 
    avars:= affvars;
    if V[1] <> 0 then nr:=0 else nr:=1 fi;
    v:= affpoint(V,nr)
  fi;  
  f:= denom(v[1])*avars[1] - numer(v[1]);
  g:= denom(v[2])*avars[2] - numer(v[2]);
  if affcase then
    standard(resultant (f,g,var),avars)
  else
    standard(hompoly(resultant(f,g,var),avars,vars,nr),vars)
  fi  
end:



# ___  QUADTRANS  ________________________________________________________

quadtrans := proc (F)
local vars,hvars, G,lF, i,k,r, affcase;
  if type([args],[anything,list(name)]) then
    vars:= args[2]
  elif type([args],[anything]) then
    vars:= getindets(F)
  else  
    ERROR(`wrong type or number of arguments`)
  fi;
  if nops(vars) > 3 then ERROR(`Too many variables`) fi;
  affcase:= evalb(nops(vars)=2);
  if affcase then
    if not type(F,polynom(algebraic,vars)) then
      ERROR(`illegal first argument`)
    fi;  
    hvars:= homvars;
    lF:= hompoly(F,vars,hvars,0)
  else
    if not affcase and not type(F,projpoly(vars)) then
      ERROR(`first argument is not a homogeneous polynomial in`,vars)
    fi; 
    hvars:= vars;
    lF:= F
  fi;    
  G:= subs( [ hvars[1]=hvars[2]*hvars[3],hvars[2]=hvars[1]*hvars[3],
              hvars[3]=hvars[1]*hvars[2] ], lF);
  for i from 1 to 3 do
    for k from 0 do
      r:= rem(G,hvars[i],hvars[i],'G');
      if r <> 0 then  G:= expand(G*hvars[i])+r; break fi
    od;
  od;
  if affcase then
    affpoly(G,hvars,vars,0)
  else    	
    eval(G)	
  fi  
end:




# ___  EXPLICIT  ________________________________________________________


expldeg1 := proc (f,avars,hvars,nr, var,invfun)
local sol;
  if has(f,avars[2]) then
    sol:=  solve(f,avars[2]);
    invfun:= hompoly(avars[1],avars,hvars,nr);
    RETURN( [var , subs(avars[1]=var,sol) ] )
  else
    sol:=  solve(f,avars[1]);
    invfun:= hompoly(avars[2],avars,hvars,nr);
    RETURN( [ subs(avars[2]=var,sol) , var] )
   fi
end:     


explmultnminus1 := proc (f,pt,r,avars,hvars,var,invfun)
local tmpnr,apt,G1,G2,g,R,N,phi,i;
  for tmpnr from 0 to 2  while pt[tmpnr+1]=0 do od;
  apt := affpoint(pt,tmpnr);
  G1:= hompoly(avars[2] - apt[2],avars,hvars,tmpnr);
  G2:= hompoly(avars[1] - apt[1],avars,hvars,tmpnr);
  invfun:= G1/G2; 
  g:= affpoly(G1 - var*G2,hvars,avars,0);
  phi:= array([0,0]);
  for i from 1 to 2 do
    R:= expand(resultant(f,g,avars[i mod 2 + 1]));
    N:= degree(R,avars[i]);
    phi[i]:=  -coeff(R,avars[i],N-1)/coeff(R,avars[i],N);
    if simplify(pt[1]) <> 0 then 
      phi[i]:= phi[i] - r*pt[1+i] 
    fi
  od;
  convert(phi,list)
end:  



explicit0 := proc (F,f,hvars,avars,nr,var,invfun)
local n,nL,L,Lpts,spts,r,extrapts,nrofextra,g,k,i,phi,R,N;
  n:= degree(F,convert(hvars,set));
  if n = 1 then RETURN ( expldeg1(f,avars,hvars,nr, var,'invfun') ) fi;
  spts:= singularities(F,hvars);
  r:= multiplicity(F,spts,hvars); 
  if sum('r[i]*(r[i]-1)','i'=1..nops(spts)) <> (n-1)*(n-2) then
    ERROR(`requested precondition not fulfilled`)
  fi;  
  if nops(spts)=1 and r[1] = n-1 then
    RETURN (explmultnminus1(f,spts[1],n-1,avars,hvars,var,'invfun'))
  fi;
  if n = 2 then  
    nrofextra:=   1; nL:= 1  
  else  
    nrofextra:= n-3; nL:= n-2 
  fi;
  extrapts:= RegCurvePts(F,nrofextra,hvars);
  if nops(extrapts) < nrofextra then RETURN(FAIL) fi;
  Lpts:=  [ seq([spts[i],r[i]-1],i=1..nops(spts)),op(extrapts) ];
  L:= linearsystem(Lpts, nL, hvars);
  invfun:= L[1]/L[2];
  extrapts:= affpoint(extrapts,0);
  g:=  affpoly( L[1]-var*L[2], hvars,avars,0); ;
  phi:= array([0,0]);
  for i from 1 to 2 do
    R:= expand(resultant(f,g,avars[i mod 2 + 1]));
    N:= degree(R,avars[i]);
    phi[i]:= -coeff(R,avars[i],N-1)/coeff(R,avars[i],N);
    for k from 1 to nops(spts) do
      if expand(spts[k][1]) <> 0 then 
        phi[i]:= phi[i]-r[k]*(r[k]-1)*spts[k][1+i] 
      fi
    od;
    for k from 1 to nrofextra do phi[i]:= phi[i] - extrapts[k][i] od
  od; 
  convert(phi,list)
end:  
 

explicit := proc (ff,invfun)
local vars,avars,var,affcase,F,f,res,invfun0,nr;
  if type([args],[anything,name,list(name),name]) then
    vars:= args[3];
    var:= args[4]
  elif type([args],[anything,name]) then
    vars:= getindets(ff);
    var:= explvar;
  elif type([args],[anything]) then
    vars:= getindets(ff);
    var:= explvar;
  else  
    ERROR(`wrong type or number of arguments`)
  fi;
  if nops(vars) < 2  or  nops(vars) > 3 then
    ERROR(`illegal number of variables`)
  fi; 
  affcase:= evalb(nops(vars) = 2);   
  nr:= 0;
  if affcase then
    if not type(ff,polynom(algebraic,vars)) then
      ERROR(`illegal first argument`)
    fi;  
    avars:= vars;
    vars:= homvars;
    F:= hompoly(ff,avars,vars,nr);
    f:= ff
  else
    if not type(ff,projpoly(vars)) then
      ERROR(`polynomial is not homogeneous`)
    fi; 
    avars:= affvars;
    F:= ff; 
    if not (has(F,vars[2]) or has(F,vars[3])) then nr:=1 fi;
    f:= affpoly(F,vars,avars,nr);
  fi;  
  res:= explicit0(F,f,vars,avars,nr,var,'invfun0');
  if res <> FAIL then
    if affcase then
      if nargs > 1 then 
        invfun:= affpoly(numer(invfun0),vars,avars,nr) / 
                 affpoly(denom(invfun0),vars,avars,nr)
      fi;
      RETURN(map(normal,res))
    else
      if nargs > 1 then invfun:= invfun0 fi;
      RETURN(standard(hompoint(res,nr)))
    fi
  else
    RETURN(FAIL)
  fi        
end:  



# ___  PLOTCURVE  ________________________________________________________

extractfactors := proc(flist)
local fset,i,j,res;
  readlib(factors);
  fset:= { seq( frontend(factors,[flist[i]]) , i=1..nops(flist) ) };
  res:= NULL;
  for i from 1 to nops(fset) do
    res:= res, [seq(fset[i][2][j][1], j=1..nops(fset[i][2]))]
  od;  
  [res]
end:  


plotcurve0 := proc(curvelist,nrofpts,vars,minx,miny,incrx,incry,rx,ry)
local pts,spts,singlist, ff,xx,yy,sols, curvenr,factornr,i,j, warn;
  pts:= array(1..nops(curvelist));  
  for i from 1 to nops(curvelist) do pts[i]:= NULL od;
  warn:= false;
  xx:= minx; yy:= miny;
  for i from 1 to nrofpts+1 do  
    for curvenr from 1 to nops(curvelist) do  
      for factornr from 1 to nops(curvelist[curvenr]) do
        ff:= subs(vars[1]=xx,curvelist[curvenr][factornr]);
        if has(ff,vars[2]) then 
	  sols:= traperror(fsolve(ff,vars[2],ry));
	  if sols <> lasterror then
	    sols:= [sols];
            pts[curvenr]:= pts[curvenr], seq(op([xx,sols[j]]),j=1..nops(sols))
	  else
	    warn:= true  
	  fi  
        fi;
        ff:= subs(vars[2]=yy,curvelist[curvenr][factornr]);
        if has(ff,vars[1]) then
	  sols:= traperror(fsolve(ff,vars[1],rx));
	  if sols <> lasterror then
	    sols:= [sols];
            pts[curvenr]:= pts[curvenr], seq(op([sols[j],yy]),j=1..nops(sols))
	  else
	    warn:= true  
	  fi  
        fi
      od
    od;      
    xx:= xx+incrx; yy:= yy+incry
  od;
  spts:= array(1..nops(curvelist));
  for i from 1 to nops(curvelist) do spts[i]:= NULL od;
  for curvenr from 1 to nops(curvelist) do  
    for factornr from 1 to nops(curvelist[curvenr]) do
      sols:= traperror(singularities(curvelist[curvenr][factornr],vars));
      if sols <> lasterror then
        singlist:= map(convert,sols,list);
        for i from 1 to nops(singlist) do
          if not (has(singlist[i],RootOf)) then
            spts[curvenr]:= spts[curvenr], op(singlist[i])
          fi
        od
      else
        warn:= true	
      fi
    od  	
  od;    
  for curvenr from 1 to nops(curvelist) do
    if pts[curvenr] <> NULL then pts[curvenr] := [pts[curvenr]] fi;
    if spts[curvenr] <> NULL then  spts[curvenr]:= [spts[curvenr]] fi
  od;  
  pts:= op(convert(pts,list));  spts:= op(convert(spts,list));
  if warn then
    print(`WARNING: probably loss of some points`)
  fi;  
  RETURN ({pts,spts})
end:


plotcurve := proc (f)
local nrofpts, vars,varset, minx,miny,maxx,maxy, incrx,incry,i,
      drx,dry,rx,ry,curvelist, nonconstant,t ;
  if type([args],[anything,'range','range',posint,list(name)]) then
    nrofpts:= args[4];
    vars:= args[5]
  elif type([args],[anything,anything,anything,posint]) then
    nrofpts:= args[4];
    vars:= affvars
  elif type([args],[anything,anything,anything]) then
    nrofpts:= 40;
    vars:= affvars
  else
    ERROR(`wrong type or number of arguments`)
  fi;    
  
  if nops(vars) <> 2 then 
    ERROR(`illegal number of variables (<> 2)`)
  fi; 
  if type([args[2],args[3]],list('range')) then
    rx:= args[2]; ry:= args[3]
  elif type([args[2],args[3]],list(name='range')) then
    vars:= [ op(1,args[2]), op(1,args[3]) ];
    rx:= op(2,args[2]); ry:= op(2,args[3])
  else
    ERROR(`invalid ranges`)
  fi;
  if type(f,list) then curvelist:= f else curvelist:= [f] fi;  
  if not type(curvelist,list(polynom(algebraic,vars))) then
    ERROR(`illegal first argument`)
  fi;  
  
  minx:= evalf(op(1,rx)); maxx:= evalf(op(2,rx));
  miny:= evalf(op(1,ry)); maxy:= evalf(op(2,ry));
  if not type([minx,maxx],list(numeric)) then
    ERROR(`invalid horizontal range`)
  fi;
  if not type([miny,maxy],list(numeric)) then
    ERROR(`invalid vertical range`)
  fi;    
  drx:= maxx-minx; dry:= maxy-miny;
  rx:= (minx-0.05*drx)..(maxx+0.05*drx);
  ry:= (miny-0.05*dry)..(maxy+0.05*dry);
  incrx:= evalf((maxx-minx)/nrofpts); incry:= evalf((maxy-miny)/nrofpts);
  
  varset:= convert(vars,set);
  curvelist := extractfactors(curvelist);
  nonconstant:= unapply( 'evalb'('degree'( t, varset ) > 0), t );
  curvelist := [seq( select(nonconstant,t), t=curvelist )];
  #for i from 1 to nops(curvelist) do
  #  curvelist[i]:= select(nonconstant,curvelist[i])
  #od; 

  plot( plotcurve0(curvelist,nrofpts,vars,minx,miny,incrx,incry,rx,ry),
        vars[1]=rx,vars[2]=ry,style=POINT)
end:  

macro(
intermultiplicity0 = intermultiplicity0,
intermultiplicity = intermultiplicity,
scalarmul = scalarmul,
crossprod = crossprod,
getindets = getindets,
iszerolist = iszerolist,
hompoint = hompoint,
affpoint = affpoint,
poly2coord = poly2coord,
coord2poly = coord2poly,
standard = standard,
homequal = homequal,
genmatrix = genmatrix,
transpose = transpose,
multiply = multiply,
homtrans = homtrans,
afftrans = afftrans,
image2mat = image2mat,
mat2image = mat2image,
appimage = appimage,
NonCurvePt = NonCurvePt,
multcomp = multcomp,
commoncomp = commoncomp,
homxyfac = homxyfac,
isregular = isregular,
issingular = issingular,
isordinary = isordinary,
curvetaylor = curvetaylor,
singtangent = singtangent,
unityvec = unityvec,
processmult = processmult,
randsyspoly = randsyspoly,
getPoints = getPoints,
RegCurvePts = RegCurvePts,
inflections = inflections,
newrootseq = newrootseq,
implicit = implicit,
expldeg1 = expldeg1,
regcurvept = regcurvept,
explicit0 = explicit0,
explicit = explicit,
quadtrans = quadtrans,
plotcurve = plotcurve,
plotcurve0 = plotcurve0,
NonVertexTrans = NonVertexTrans,
singularities = singularities,
multiplicity = multiplicity,
linearsystem = linearsystem,
intersection = intersection,
tangentthrupt = tangentthrupt,
explmultnminus1 = explmultnminus1,
extractfactors = extractfactors,
affvars = affvars,
hompoly = hompoly,
homvars = homvars,
affpoly = affpoly,
grad = grad,
coldim = coldim,
inverse = inverse,
dotprod = dotprod,
appoly = appoly,
appoly0 = appoly0,
appolyE = appolyE,
appmat = appmat,
PtTrans = PtTrans,
tangent = tangent,
filter = filter,
iPoints = iPoints,
cPoints = cPoints,
shorter = shorter,
hessian = hessian,
Polar = Polar,
Hessian = Hessian,
tryall = tryall,
det = det);

#-----  library procedures ------------------------------------------------

algcurve[homvars] := [x0,x1,x2]:
algcurve[affvars] := [x,y]:
algcurve[explvar] := t:


#---  alg1  ---
algcurve[hompoly] := eval( `algcurve/hompoly` ):
algcurve[hompoint] := eval( `algcurve/hompoint` ):
algcurve[affpoly] := eval( `algcurve/affpoly` ):
algcurve[affpoint] := eval( `algcurve/affpoint` ):
algcurve[poly2coord] := eval( `algcurve/poly2coord` ):
algcurve[coord2poly] := eval( `algcurve/coord2poly` ):
algcurve[standard] := eval( `algcurve/standard` ):
algcurve[homequal] := eval( `algcurve/homequal` ):
algcurve[grad] := eval( `algcurve/grad` ):
  
  
#---  alg2  ---	          
algcurve[homtrans] := eval( `algcurve/homtrans` ):
algcurve[afftrans] := eval( `algcurve/afftrans` ):
algcurve[image2mat] := eval( `algcurve/image2mat` ):
algcurve[mat2image] := eval( `algcurve/mat2image` ):
algcurve[appoly] := eval( `algcurve/appoly` ):
algcurve[appimage] := eval( `algcurve/appimage` ):
algcurve[appmat] := eval( `algcurve/appmat` ):
algcurve[NonCurvePt] := eval( `algcurve/NonCurvePt` ):
algcurve[NonVertexTrans] := eval( `algcurve/NonVertexTrans` ):
algcurve[PtTrans] := eval( `algcurve/PtTrans` ):
algcurve[multcomp] := eval( `algcurve/multcomp` ):
algcurve[commoncomp] := eval( `algcurve/commoncomp` ):


    
#---  alg3  ---

algcurve[singularities] := eval( `algcurve/singularities` ):
algcurve[isregular] := eval( `algcurve/isregular` ):
algcurve[issingular] := eval( `algcurve/issingular` ):
algcurve[isordinary] := eval( `algcurve/isordinary` ):
algcurve[curvetaylor] := eval( `algcurve/curvetaylor` ):
algcurve[multiplicity] := eval( `algcurve/multiplicity` ):
algcurve[tangent] := eval( `algcurve/tangent` ):
algcurve[linearsystem] := eval( `algcurve/linearsystem` ):
algcurve[randsyspoly] := eval( `algcurve/randsyspoly` ):
algcurve[RegCurvePts] := eval( `algcurve/RegCurvePts` ):
  
#---  alg4  ---

algcurve[intersection] := eval( `algcurve/intersection` ):
algcurve[Polar] :=  eval( `algcurve/Polar` ):
algcurve[tangentthrupt] :=  eval( `algcurve/tangentthrupt` ):
algcurve[Hessian] :=  eval( `algcurve/Hessian` ):
algcurve[inflections] :=  eval( `algcurve/inflections` ):
algcurve[intermultiplicity] :=  eval( `algcurve/intermultiplicity` ):
algcurve[tryall] :=  eval( `algcurve/tryall` ):

#---  alg5  ---

algcurve[explicit] :=  eval( `algcurve/explicit` ):
algcurve[implicit] :=  eval( `algcurve/implicit` ):
algcurve[quadtrans] :=  eval( `algcurve/quadtrans` ):
algcurve[plotcurve] :=  eval( `algcurve/plotcurve` ):



#save `algcurve.m`;
#quit
