#
## <SHAREFILE=algebra/fields/fields.mpl >
## <DESCRIBE>
##       SEE ALSO:  algebra/fields/fields.mws, algebra/fields/fields.dvi (42K)
##
##                Let N be a field extension of L, a field extension of K,
##                such that N is finitely generated over K.
##                This package uses Grobner basis methods to calculate the
##                transcendence degree of N|L and the degree [N:L] if the field
##                extension is algebraic, and related questions.
##                The implementation works for K = Q or K = Q(alpha) a single
##                algebraic extension of Q.
##                AUTHOR: Gregor Kemper, kemper@kalliope.iwr.uni-heidelberg.de
## </DESCRIBE>


# FIELDS

# Implementation of an algorithm to calculate the transcendence degree of
# N|L and the degree [N=L] if the extension is finite, where N and L lie
# over the base field Q or an algebraic number field.

# Written by Gregor Kemper, 
# email kemper@kalliope.iwr.uni-heidelberg.de

# Aug 30, 1993

# This script defines the fields-package. If you want to install the package 
# right away,  set pathname:=<Directory where invar.m shall be put> 
# (with slash /).

macro(primdivs_in=fields[primdivs_in], extension=fields[extension], 
    normalform=fields[normalform], macaulay=fields[macaulay], 
    primdivs_in=fields[primdivs_in]);



# EXTENSION
# The main routine, calculating properties of extensions N|L lying over
# an algebraic number field K

extension:=proc(generators)
    local i,gens,rels,mipo,indep,spec,nonz,u,a,x,g,equ,vars,random,reorder,G,
        G_T,G_i,agent,trgrN,trgrL,deg,T,split,findarg,reduce;
    
    T:=time();
    
    split:=proc(M,V)
        local x,Y,N;
        Y:=NULL; N:=NULL;
        for x in M do if has(x,V) then Y:=Y,x else N:=N,x fi od;
        [[Y],[N]]
    end;
    
    findarg:=proc(token);
        (a,t)->
            type(a,`=`) and type(lhs(a),string) and substring(lhs(a),1..1)=t;
        select(",[args[2..nargs]],substring(token,1..1));
        if "=[] then 'FAIL' else rhs("[1]) fi
    end;
    
    reduce:=proc(baselist,vars)
        local i,j,n,last,span;
        n:=nops(baselist);
        last:=baselist[n];
        for i to n-1 do
            span:=grobner[gbasis]([baselist[i][],last[]],vars,plex);
            if span=baselist[i] then            # baselist[i] includes last
                RETURN([op(baselist[1..i-1]),op(baselist[i+1..n])])
            elif span=last then RETURN(baselist[1..n-1])
            fi
        od;
        baselist
    end;
        
    # Read and pre-process the arguments
    
    if type(generators,{list,set}) then G:=generators else G:=[generators] fi;
    gens:=NULL; i:=1;
    for x in G do
        if type(x,`=`) then gens:=gens,x
        else
            if assigned(g.i) then
                ERROR(``.(evaln(g.i)).` should be an unassigned name!`)
            fi;
            gens:=gens,g.i=x;
            i:=i+1
        fi
    od;
    gens:=[eval(gens)];
    x:=indets(map(rhs,gens)); g:=indets(map(lhs,gens));
    if not type(gens,list(name=ratpoly(rational))) or g intersect x <> {} 
                                                or nops(g)<>nops(gens) then
        ERROR(`Invalid first argument`)
    fi;
    
    rels:=findarg(rels,args);
    if rels<>'FAIL' then
        if not type(rels,{set,list}) then rels:=[rels] fi;
        rels:=[rels[]];
        x:=x union indets(rels);
        if not type(rels,list(polynom(rational))) or g intersect x <> {} 
            then ERROR(`Bad assignment for the relations`)
        fi;
        rels:=map(expand,rels);
    else rels:=[]
    fi;
    
    mipo:=findarg(mipo,args);
    if mipo<>'FAIL' then
        mipo:=expand(mipo);
        a:=indets(mipo)[];
        if nops([a])<>1 or not type(mipo,polynom(rational,a)) then
            ERROR(`Bad assignment for the minimal polynomial`)
        fi;
        x:=x minus {a}
    else mipo:=0
    fi;
    
    indep:=findarg(indep,args);
    if indep<>'FAIL' then
        if indep='all' then indep:=[$1..nops(g)]
        elif type(indep,nonnegint) and indep<=nops(g) then indep:=[$1..indep]
        elif type(indep,{set,list}) and {indep[]} minus {$1..nops(g)} ={} then
            indep:=[indep[]]
        else ERROR(
            `Bad assignment for what generators are supposed independent`)
        fi
    fi;
    
    spec:=findarg(spec,args);
    if spec<>'FAIL' then
        if spec='all' then spec:=[$1..nops(g)]
        elif type(spec,nonnegint) and spec<=nops(g) then spec:=[$1..spec]
        elif type(spec,{set,list}) and {spec[]} minus {$1..nops(g)} ={} then
            spec:=[spec[]]
        else ERROR(`Bad assignment for what generators to specialize`)
        fi
    fi;
    
    nonz:=findarg(nonz,args);
    if nonz<>'FAIL' then
        if not type(nonz,{set,list}) then nonz:=[nonz] fi;
        nonz:=[nonz[]];
        if not type(nonz,list(polynom(rational,[x[],a]))) then
            ERROR(`Bad assignment for non-zeroes`)
        fi;
        nonz:=map(expand,nonz)
    fi;
    
    agent:=findarg(agent,args);
    if agent<>'FAIL' and not member(agent,{'gbasis','Macaulay'}) then
            ERROR(`The "agent" must be 'gbasis' or 'Macaulay'`)
    fi;
    if agent='Macaulay' and not assigned(macaulay) then agent:='FAIL'
    elif agent='gbasis' then agent:=`'gbasis'`
    fi;
    
    if mipo<>0 then
        proc(e,m,a)
            subs(a=RootOf(m),rhs(e));
            evala(Normal("));
            lhs(e)=subs(RootOf(m)=a,")
        end
    else e->lhs(e)=normal(rhs(e))
    fi;
    gens:=map(",gens,mipo,a);
    if mipo<>0 then rels:=map(rem,rels,mipo,a) fi;
    
    `ext/out`(2,`Given Situation:`); `ext/out`(2);
    if mipo<>0 then `ext/out`(2,print,`K=Q(`.a.`) with`); 
    `ext/out`(2,print,mipo=`0,`)
    else `ext/out`(2,print,`K=Q,`)
    fi;
    `ext/out`(2);
    if rels<>[] then
        `ext/out`(2,print,
            cat(`N=K(`,x[1],seq(`,`.(x[i]),i=2..nops(x)),`) with`));
        `ext/out`(2,print,map(r->r=0,rels)[],``)
    else `ext/out`(2,print,cat(`N=K(`,x[1],seq(`,`.(x[i]),i=2..nops(x)),`),`))
    fi;
    `ext/out`(2);
    `ext/out`(2,print,cat(`L=K(`,g[1],seq(`,`.(g[i]),i=2..nops(g)),`) with`));
    `ext/out`(2,print,gens[]);
    `ext/out`(2);
    
    if nonz='FAIL' then
        # Calculate the prime divisors of the denominators of the g_i
        `ext/out`(
        `Calculating the prime divisors of the denominators of the g_i ...`);
        nonz:=primdivs_in(map(e->denom(rhs(e)),gens),mipo);
        `ext/out`(`... done`);
        `ext/out`(2,`They are:`);
        `ext/out`(2,print,nonz)
    else 
        `ext/out`(2,`Assumed non-zero:`);
        `ext/out`(2,print,nonz)
    fi;
    `ext/out`(2);
    
    equ:=[seq(nonz[i]*u[i]-1,i=1..nops(nonz)),rels[],
    seq(numer(rhs(gens[i]))-lhs(gens[i])*denom(rhs(gens[i])),i=1..nops(g))];
    if mipo<>0 then equ:=[equ[],mipo] fi;
    `ext/out`(3,`The equations are:`);
    `ext/out`(3,equ); `ext/out`(3);
    
    if indep<>'FAIL' then
        g:=g minus {seq(lhs(gens[indep[i]]),i=1..nops(indep))}
    fi;
    
    if spec<>'FAIL' then
        random:=rand(-100..100);
        [seq(lhs(gens[spec[i]])=random(),i=1..nops(spec))];
        equ:=subs(",equ);
        g:=g minus {map(lhs,"")[]};
        `ext/out`(`Specialize `.(nops(spec)).` of the generators of L`);
        `ext/out`(2,`Namely:`,"""[]); `ext/out`()
    fi;
    equ:=map(expand,equ);
    
    reorder:=readlib(`grobner/pplex/reorder`);
    x:=reorder(equ,x); if g<>{} then g:=reorder(equ,g) else g:=[] fi;
    if nops(nonz)>0 then reorder(equ,{seq(u[i],i=1..nops(nonz))}) else [] fi;
    vars:=["[],x[],g[]];
    map(convert,vars,string);
    `ext/out`(2,cat(`Take the term order with `,
        "[1],seq(` > `.("[i]), i=2..nops("))));
    `ext/out`(2);
    
    # Doing the Groebner basis calculation ...
    
    if agent<>'FAIL' then ``.agent.` c` else 'C' fi;
    `ext/out`(``.(").`alculating a Groebner basis of I ...`);
    if mipo<>0 then vars:=[vars[],a] fi;
    if agent='FAIL' then
        G:=grobner[gsolve](equ,vars)
    elif agent=`'gbasis'` then
        G:=grobner[gbasis](equ,vars,plex)
    else
        rand(31991); nextprime("());
        `ext/out`(`   (calculating modulo `.(").`)`);
        G:=macaulay(equ,",vars);
        nops(G);
        G:=[seq(G["-i],i=0.."-1)]
    fi;
    `ext/out`(`... done`); `ext/out`();
    
    if agent='FAIL' then
        i:=nops(G);
        if i=0 then G:=[[1]]
        elif i>1 then
            `ext/out`(`'gsolve' yielded `.i.` components!`);
            `ext/out`(3,`This is the result:`);
            `ext/out`(3,G); `ext/out`(3);
            `ext/out`(`Eliminating components that contain others ...`);
            do
                G:=reduce(G,vars);
                if nops(G)=i then break fi;     # No reduction was done
                i:=nops(G)
            od;
            if i=1 then 
                `ext/out`(
                `... Elimination was successful: Only one component left!`);
                `ext/out`()
            else
                `ext/out`(`... There are still multiple components left.`);
                `ext/out`(`Probably the ideal of relations isn't prime.`);
                `ext/out`(`Better rerun with 'agent=gbasis'!`);
                if type(args[nargs],name) then
                    `ext/out`(`Putting the bases into the last argument ...`);
                    assign(args[nargs],G)
                fi;
                RETURN(`MULTIPLE COMPONENTS`)
            fi
        fi;
        G:=G[1]
    fi;
    
    G:=select((p,m)->p<>m,G,mipo);
    `ext/out`(3,`The Groebner basis is`); `ext/out`(3,G); `ext/out`(3);
    
    if spec<>'FAIL' or agent='Macaulay' then `PROBABILISTIC ` else `` fi; 
    `ext/out`(``.(").`Results:`); `ext/out`();
    if G=[1] then
        if spec<>'FAIL' or indep<>'FAIL' or agent='Macaulay' then
            `ext/out`(
    `The supposedly algebraically independent g_i's were in fact dependent OR`)
        fi;
        `ext/out`(
            `Some power of the product of denominators of the g_i's lies`);
        `ext/out`(`    in the ideal spanned by the relations.`);
        RETURN('SINGULAR')
    fi;
    
    # Throw away equations for the u[i]
    G:=split(G,{seq(u[i],i=1..nops(nonz))})[2];
    
    # Calculate the G_i
    trgrN:=0;
    for i in x do
        split(G,i);
        G_i[i]:="[1]; G:=""[2];
        if G_i[i]=[] then trgrN:=trgrN+1; G_i[i]:=0
        else G_i[i]:=collect(G_i[i][nops(G_i[i])],i)
        fi
    od;
    if trgrN=0 then 
        deg:=product('degree(G_i[x[i]],x[i])', 'i'=1..nops(x));
        if deg=1 then `ext/out`(`L=N `)
        else `ext/out`(`N|L is algebraic of degree `.deg)
        fi
    else `ext/out`(`N|L has transcendence degree `.trgrN)
    fi;
    `ext/out`();
    `ext/out`(2,`Minimal polynomials:`); `ext/out`(2);
    for i in x do 
        if G_i[i]<>0 then `ext/out`(2,``.i.`:`); `ext/out`(2,print,G_i[i]) fi 
    od;
    
    G_T:=G;                             # What remains of G is G_T
    # The transcendence degree of L|K is calculated by splitting G_T
    trgrL:=nops(gens)-nops(g);
    for i in g do
        split(G,i);
        G:="[2]; if ""[1]=[] then trgrL:=trgrL+1 fi
    od:
    `ext/out`(`L|K has transcendence degree `.trgrL); `ext/out`();
    if G_T<>[] then
        `ext/out`(2,`Equations:`);
        `ext/out`(2,print,G_T[])
    fi;
    
    if type(args[nargs],name) then
        assign(args[nargs],[[G_T,g],[seq([G_i[x[i]],x[i]],i=1..nops(x))],mipo])
    fi;
    `ext/out`(2);
    if not member(agent,{'FAIL','Maple'}) then 
        ` sec (not counting the `.agent.`-time)`
    else ` sec`
    fi;
    `ext/out`(2,cat(`Time used: `,convert(time()-T,string),"));
    if trgrN=0 then RETURN(deg) else RETURN('TRANSCENDENT') fi
end:


# EXT/OUT
# Output of a piece of protocol

`ext/out`:=proc()
    local lev,narg,pl;
    
    if nargs>0 and type(args[1],integer) then
        lev:=args[1]; narg:=[args[2..nargs]]
    else lev:=1; narg:=[args]
    fi;
    if assigned(protocol_level) then pl:=protocol_level else pl:=1 fi;
    if pl<lev then RETURN() fi;
    if nops(narg)>0 and narg[1]=print then print(op(narg[2..nops(narg)]))
    else lprint(narg[])
    fi;
    RETURN()
end:


# PRIMDIVS_IN
# The (non-constant) prime divisors contained in a set of polynomials

primdivs_in:=proc(polys,minpol)
    local a,p,res;
    
    readlib(factors); res:={};
    if nargs<2 or minpol=0 then
        for p in polys do
            factors(p);
            res:=res union {map(x->x[1],"[2])[]}
        od
    else
        a:=indets(minpol)[];
        for p in polys do
            factors(subs(a=RootOf(minpol),p),RootOf(minpol));
            subs(RootOf(minpol)=a,");
            res:=res union {map(x->x[1],"[2])[]}
        od
    fi;
    [res[]]
end:


# NORMALFORM
# Having calculated the minimal polynomials of a chain of fields leading from
# L to N, calculate the basis representation of an expr relative to these
# minimal polynomials

normalform:=proc(expr,basis)
    local gbas,num,den,ex,x,mipo,a;
    
    if not type(expr,ratpoly(rational)) then 
        ERROR(`Invalid first argument`)
    fi;
    if not type(basis,[[list,list],listlist,polynom]) then 
        ERROR(`Invalid second argument`)
    fi;
    
    gbas:=map(x->x[1],basis[2]);
    x:=map(x->x[2],basis[2]);
    mipo:=basis[3];
    if mipo<>0 then
        a:=indets(mipo)[];
        gbas:=[gbas[],mipo];
        x:=[x[],a]
    fi;
    
    if has(expr,a) then
        evala(Normal(subs(a=RootOf(mipo),expr)));
        ex:=subs(RootOf(mipo)=a,")
    else ex:=normal(expr)
    fi;
    
    num:=numer(ex); den:=denom(ex);
    if has(den,x) then 
        grobner[normalf](den,gbas,x,plex);
        if basis[1][1]<>[] then 
            map(grobner[normalf],
                [numer("),denom(")],basis[1][1],basis[1][2],plex);
            "[1]/"[2]
        fi;
        den:="
    fi;
    if den=0 then RETURN(`The result is 1/0`) fi;
    grobner[normalf](num,gbas,x,plex);
    if basis[1][1]<>[] then
        map(grobner[normalf],[numer("),denom(")],basis[1][1],basis[1][2],plex);
        "[1]/"[2]
    fi;
    "/den;
    if mipo<>0 then
        evala(Normal(subs(a=RootOf(mipo),")));
        subs(RootOf(mipo)=a,")
    fi;
    normal(")
end:


# MACAULAY
# Groebner-basis of 'Ideal' modulo a prime 'P'. This routine provides a partial
# interface to the computer algebra system Macaulay!
# All arguments but 'Ideal' are optional. 'Ideal' must be a set or list of
# polynomials.
# If 'P' is not given or =0, 31991 is taken. 
# The variables are 'Vars', default is Vars=indets(Ideal). If 'Vars' is a set,
# the variables are reordered.
# In this case, the reordered variables are written into 'nord', if present.
# Let indets(Ideal) minus Vars = {p1,...,pn}; the 'macaulay' calculates over
# the field GF(P)(p1,...,pn).
# 'monorder' can be 'plex' or 'tdeg', 'plex' as default.
# If 'change' is given and a name, it is assigned the matrix of change between
# 'Ideal' and the resulting groebner-basis.
# If 'flag'=preserve, the directory 'Transfer_to_Macaulay' is not removed.
#
# CAVEATS: when calling 'macaulay' with `;', for some the result isn't printed!
#   The same for any procedure using 'macaulay'!
#   The Macaulay-program must be accessable by the command `Macaulay'!

macaulay:=proc(Ideal,P,Vars,monorder,nord,change,flag)
    local i,j,n,m,var,lvars,v,lideal,homo,prep,defcon,sub,c,path,hb,vars,ideal,
        pars,reorder,ord,p;
    
    prep:=proc(fname,path)              # change { -> [ and } -> ] in file
        system(`sed s/\\{/\\[/g `.path.fname.` > `.path.
                    `/temp; sed s/\\}/\\]/g `.path.`/temp > `.path.fname.
                                                `; echo \\; >>`.path.fname)
    end;
    
    defcon:=proc(fname,vname)
                # read an expression from a file and write it into vname
        read fname;
        vname:="
    end;
    
    if assigned(x) then ERROR(`x should be an unassigned name!`) fi;
    ideal:=map(expand,Ideal);
    if nargs>=2 and P>0 then p:=P else p:=31991 fi;
    if nargs>=3 then vars:=Vars else vars:=indets(ideal) fi;
    if nargs>=4 then ord:=monorder else ord:=plex fi;
    pars:=indets(ideal) minus {vars[]};
    if "={} then reorder:=readlib(`grobner/iplex/reorder`); pars:=[]
    else reorder:=readlib(`grobner/pplex/reorder`); pars:=reorder(ideal,pars)
    fi;
    if type(vars,list) then vars:=vars
    else
        vars:=reorder(ideal,vars);
        if nargs>=5 then nord:=vars fi
    fi;
    
    var:=[vars[],pars[]];
    map(type,ideal,polynom(rational,var));
    if member(false,") then
        ERROR(
        `Argument 'ideal' must be a set or list of rational polynomials in`,
                                                                        var[])
    fi;
    hb:=false;
    path:=`Transfer_to_Macaulay`;
    if assigned(pathname) then path:=cat(pathname,path) fi;
    if system(`mkdir `.path)<>0 then    # Maybe another maple is doing the same
        for i to 10 do
            if system(`mkdir `.path.i)=0 then path:=``.path.i; break fi
        od;
        if i=11 then ERROR(`Couldn't make the Transfer-directory`) fi
    fi;

    n:=nops(var);
    lvars:=[seq(x[i],i=1..n)];
                            # meet Macaulay's requirements for variable-names
    m:=nops(ideal);
    lideal:=subs([seq(var[i]=lvars[i],i=1..n)],ideal);

    # is 'ideal' homogeneous or not?
    homo:=proc(f,var) 
        expand(2**degree(f,var)*f-subs(map(x->x=2*x,var),f))
    end;
    homo:=evalb(map(homo,{lideal[]},{lvars[]})={0});

    # produce input for Macaulay ...
    writeto(cat(path,`/InputFromMaple`));
    lprint(`ring R`);                                       # define the ring
    lprint(p);                                              # characteristic
    lprint(n);                                              # n indeterminates
    for v in lvars do lprint(v) od;
    lprint();                                               # all weights=1
    if ord=plex then                            # purely lexicographic order!
        if pars=[] then  lprint(1$n)
        else lprint(1$nops(vars),nops(pars))        # pars with total degree!
        fi
    else 
        if pars=[] then lprint()                # lex. order with total degree!
        else lprint(nops(vars),nops(pars))
                            # lex. order with tot. degree, but vars > pars!
        fi
    fi;
    lprint(`ideal I`);                                      # define the ideal
    lprint(m);                                              # m generators
    for i from 1 to m do
        if type(lideal[i],monomial) then lprint(lideal[i])
        else            # split into parts, otherwise lines will be chopped!
            lprint(op(1,lideal[i]),`\\`);
            for j from 2 to nops(lideal[i])-1 do
                    lprint(``+op(j,lideal[i]),`\\`) od;
            lprint(``+op(nops(lideal[i]),lideal[i]))
        fi
    od;
    if homo then
        if nargs>=6 and type(change,name) then
            lprint(`lift-std I b`);     # matrix of changes required, too
            lprint(`putchange b c`);
            lprint(`stdpart b hb !`);
            if pars<>[] then
                lprint(`stdpart b b x[1]..x[`.(nops(vars)).`]`);
                lprint(`prmat hb >`.path.`/OutputForMaple_hb`);
                                            # output in Mathematica-format!
                hb:=true
            else lprint(`putstd b b`)
            fi
        else
            lprint(`std I b`);                          # only grobner-basis
            lprint(`stdpart b b x[1]..x[`.(nops(vars)).`]`)
        fi
    else
        if nargs>=6 and type(change,name) then 
            lprint(`<maple/Macau/inhom I b hb c`);      # "My" modified script
            if pars<>[] then
                lprint(`stdpart b b x[1]..x[`.(nops(vars)).`]`)
            fi;
            lprint(`stdpart hb hb !`);
            lprint(`prmat hb >`.path.`/OutputForMaple_hb`);
            hb:=true
        else
            lprint(`<inhomog_std I b`);     # standard-script or inhom.
            if pars<>[] then
                lprint(`stdpart b b x[1]..x[`.(nops(vars)).`]`)
            fi
        fi
    fi;
    lprint(`prmat b >`.path.`/OutputForMaple_b`);
    if nargs>=6 and type(change,name) then 
            lprint(`prmat c >`.path.`/OutputForMaple_c`) fi;
    lprint('exit');
    writeto(terminal);
    
    # call Macaulay!!
    system(`csh -c 'Macaulay <`.path.`/InputFromMaple > `.path.`/output'`);
                                # "csh", since Bond-shells don't use aliases!
    if "=256 then
        if nargs<7 or flag<>'preserve' then system(`'rm' -r `.path) fi;                 
                                                    # remove Transfer-directory
        ERROR(`I couldn't find the Macaulay-program`)
    fi;
    
    if system(`chmod 666 `.path.`/OutputForMaple_b`)<>0 then # file existent?
        if nargs<7 or flag<>'preserve' then system(`'rm' -r `.path) fi;                 
                                                    # remove Transfer-directory
        ERROR(`Computation by Macaulay wasn't completed!`)
    fi;
    prep(`/OutputForMaple_b`,path);                 # lists, not sets, please!
    if nargs>=6 and type(change,name) then prep(`/OutputForMaple_c`,path) fi;
    if hb then prep(`/OutputForMaple_hb`,path) fi;
    
    defcon(cat(path,`/OutputForMaple_b`),'MacaulayResult_b');
    if nargs>=6 and type(change,name) then
        defcon(cat(path,`/OutputForMaple_c`),'MacaulayResult_c')
    fi;
    if hb then 
        defcon(cat(path,`/OutputForMaple_hb`),'MacaulayResult_hb')
    fi;
    sub:=[seq(lvars[i]=var[i],i=1..n)];
    if nargs>=6 and type(change,name) then
        c:=subs(sub,MacaulayResult_c);                  # Matrix of changes
        c:=[seq([seq(c[j][i],j=1..nops(c))],i=1..nops(c[1]))];  # transpose
        if not hb then change:=c
        else        # pick out lines that actually entered the grobner basis
            v:=NULL;
            for i from 1 to nops(c) do
                if member(MacaulayResult_hb[1][i],{op(op(MacaulayResult_b))})
                                                            then v:=v,c[i] fi
            od;
            change:=[v]
        fi
    fi;
    if nargs<7 or flag<>'preserve' then system(`'rm' -r `.path) fi;
                                            # remove Transfer-directory
    subs(sub,op(MacaulayResult_b))
end:



macro(extension=extension, normalform=normalform, primdivs_in=primdivs_in, 
    macaulay=macaulay);

#    save fields,
#         `ext/out`,
#         `fields.m`:
#quit
