## <SHAREFILE=program/GRAD/GRAD >
## <DESCRIBE>
##         Differentiates functions in the form of Maple procedures.
##         FUNCTIONS:
##         GRAD:  Given a Maple procedure f, computes its gradient.
##                This is known as automatic differentiation.
##         HESSIAN: Given a procedure f, return a procedure which 
##                 computes the Hessian matrix of f.
##         JACOBI: Given an n-dimensional function f, return a procedure which
##                  which computes its jacobi matrix.
##         
##         AUTHOR:
## Note: replaces "autodiff"
## </DESCRIBE>
## <UPDATE=R4 >

macro(  MAKEINTREP = readlib(`intrep/makeintrep`),
	MAKEPROC =  readlib(`intrep/makeprocedure`),
	R = `tools/rename`,
	U = `tools/unrename`,
	FORGET = readlib(forget)):

GRAD:=proc()
	local m, pr, grname, pars, res, largs, eqargs, i, restype, funval;
	largs:=[seq(args[i],i=1..nargs)];
	eqargs:=select(type,largs,equation);

	readlib(R);
	pr:=select(type,largs,procedure);
	if nops(pr)<>1 then 
	   pr:=select(type,largs,specfunc(anything,R('Proc')));
	   if nops(pr)<>1 then
		ERROR(`You need to enter exactly one procedure.`)
	   fi
	fi;
	pr:=op(pr);

	pars:=select(type,largs,list); 
	if nops(pars)>1 then 
	   ERROR(`You can give only one list of the parameters w.r.t. `.
		 `which you wonna derive.`)
	elif nops(pars)=1 then 
	     pars:=op(pars);
	     if pars=[] then RETURN(eval(pr)) fi;
	     if not type(pars,list(name)) then
	   	ERROR(`The parameters w.r.t which you wonna derive `.
			`needs to be names.`)
	     fi
	fi;

	grname:=select(x->evalb(lhs(x)='gradname'),eqargs);
	if nops(grname)>1 then
 	   ERROR(`You can give at most one name.`)
	elif nops(grname)=1 then  grname:=rhs(op(grname));
	   if not type(grname,name) then
	      ERROR(`gradname has to be a name.`)
	   fi
	fi;

	m:=select(x->evalb(lhs(x)='mode'),eqargs);
	if nops(m)>1 then
	   ERROR(`You can enter at most one mode.`)
	elif nops(m)=1 then m:=rhs(op(m))
	fi;

	restype:=select(x->evalb(lhs(x)='result_type'),eqargs);
	if nops(restype)=1 then  restype:=rhs(op(restype))
	elif nops(restype)=0 then restype:=[]
	else ERROR(`You can enter at most one type of result.`)
	fi;

	funval:=select(x->evalb(lhs(x)='function_value'),eqargs);
	if nops(funval)=0 then funval:='false'
	elif nops(funval)=1 then funval:=rhs(op(funval))
	else ERROR(`You can enter at most one function_value.`)
	fi;

	if m='forward' or m='FORWARD' then
	   res:=`GRAD/forward`(pr,pars,grname,restype,funval);
	   FORGET(`tools/rename`); FORGET(U);
	   RETURN(eval(res))
	else 
	   res:=traperror(`GRAD/reverse`(pr,pars,grname,restype,funval));
	   if res=lasterror then
		lprint();
	   	lprint(`Can't apply the reverse mode.`);
	   	lprint(`Trying with the forward mode...`);
	   	res:=`GRAD/forward`(pr,pars,grname,restype,funval);
	   fi;
	   FORGET(`tools/rename`); FORGET(U);
	   RETURN(eval(res))
	fi
end:


###################
# FORWARD MODE:
###################


`GRAD/forward`:=proc(f,apars,grname,restype,funval)
	local	pr, pars, npars, parslist, x, statseq, i,
		loc, reslocals, resstat, NT, proclist, locpass,
		arraylist, prname, loclist, proctobediff, IT, typ,
		aname, constlist, localsnotarray, itsagrd, DN, lfnames;

	if type(f,'procedure') then   	pr:=R(MAKEINTREP(f))
	   elif op(0,f)=R('Proc') then	pr:=f
	   else  ERROR(`First argument must be a procedure`)
	fi;
	if has(pr,[R('diff'),R('int')]) then
	   ERROR(`The procedure cannot contain int or diff operators.`)
	fi;

	prname:=lhs(op(1,op(1,pr)));
	proclist:={prname};       
	loc:={op(op(5,pr))};
	loclist:=map(lhs,loc);
	pars:=op(2,pr);
	parslist:=op(map(lhs,pars));
	locpass:=loclist;
	if apars=[] then 
	   npars:=nops(pars);
	   x:=array(1..npars,[parslist])
	else
	   npars:=nops(apars);
	   x:=array(1..npars,R(apars))
	fi;
	parslist:={parslist};
	locpass:=locpass union parslist;
	if grname=[] then  
	  NT(prname):=`GRAD/givedname`(prname,locpass,'locpass')
	else  NT(prname):=R(grname); 
	      locpass:=locpass union {NT(prname)}
	fi;

 	for aname in loclist union parslist do
	  DN(aname):=`GRAD/givedname`(aname,locpass,'locpass');
	od;

	statseq:=op(7,pr);
	statseq:=`GRAD/intrepnoarray`(statseq,DN,NT,npars,{},
				      'arraylist',[],'reslocals',restype);
	statseq:=`GRAD/lastreturn`(statseq); 
	localsnotarray:=loclist minus arraylist;
	for i to nops(localsnotarray) do
	    IT(op(i,localsnotarray)):={}
	od;
	`GRAD/dependances`([],statseq,localsnotarray,IT,parslist);
	constlist:=`GRAD/findconst`(localsnotarray,IT,convert(x,'set'));

	resstat:=op(`GRAD/analyze`(statseq,x,parslist,npars,constlist,
			  NT,{prname},'proclist',arraylist,loclist,
			  {},'proctobediff',[],'itsagrd',DN,restype,{},
			  'lfnames',locpass,funval));
	if npars>1 then
	   resstat:=R('StatSeq')(seq(R('Assign')(DN(aname),
			                      R('array')(1..npars,[])),
		      aname=localsnotarray minus constlist minus proclist),
		 op(map(elem->R('Assign')(lhs(elem),R('array')(
		  	 op(1..nops(rhs(elem))-1,rhs(elem)),[])),
		 select(elem->evalb(op(0,rhs(elem))=R('List')),reslocals))),
		 resstat)
	else
	   resstat:=R('StatSeq')(
		 op(map(elem->R('Assign')(lhs(elem),R('array')(
		  	 op(1..nops(rhs(elem))-1,rhs(elem)),[])),
		 select(elem->evalb(op(0,rhs(elem))=R('List')),reslocals))),
		 resstat);
	fi;

	reslocals:=R('Locals')(
		    op(map(elem->elem..R('float'),localsnotarray)),
		    seq(DN(aname)..R('float'),aname=localsnotarray minus 
						    constlist),
		    op(reslocals),
		    op(map(elem->elem..R('float'),lfnames)));
	reslocals:=remove(proc(elem,parslist) 
			      member(lhs(elem),parslist) end,
			  reslocals, parslist);
	proctobediff:=proctobediff minus {prname};
	if proctobediff<>{} then
	  for i to nops(proctobediff) do
	    `GRAD/forward`(U(proctobediff[i]),[],U(NT(proctobediff[i])),
		'list',false)
	  od;
	  lprint();
	  lprint(`Attention: in the procedure are external procedures `);
	  lprint(proctobediff);
	  lprint(`of which the gradients have been computed and are now`.
		 ` available under the names: `);
	  lprint({seq(aname..NT(aname),aname=proctobediff)});
	fi;
	if itsagrd<>[] then
	   reslocals:=R('Locals')(op(reslocals),itsagrd)
	fi;

	typ:=rhs(op(1,op(1,pr)));
	if type(typ,specfunc(anything,R('List'))) then
	   typ:=`GRAD/expandList`(typ);
	   typ:=R('List')(op(1..nops(typ)-1,typ),1..npars,R('float'))
	else typ:=R('List')(1..npars,R('float'))
	fi;
	
	if type(f,'procedure') then
	   if grname<>[] then  assign(U(NT(prname)),
	      MAKEPROC(
	           'Proc'(U(R('Name')(NT(prname)..typ),pars,op(3,pr),
		    R('Description')( ),reslocals,
		    op(6,pr),resstat))) )
	   fi;
	   MAKEPROC(
	           'Proc'(U(R('Name')(NT(prname)..typ),pars,op(3,pr),
		    R('Description')( ),reslocals,
		    op(6,pr),resstat)) )
	elif op(0,f)=R('Proc') then  
		    R('Proc')(R('Name')(NT(prname)..typ),pars,
		    op(3,pr),R('Description')( ),reslocals,
		    op(6,pr),resstat)
	fi

end:



# Note: if you wonna differentiate a recursive function, then you shouldn't
#       say with respect to which parameters.
#       You have to assign to the derived procedure the name d<procname>
#	also or give that name explicitly with the option gradname=...





# `GRAD/intrepnoarray` transforms array declarations in the StatSeq of the 
# form Assign(name,array(bounds)) or Assign(name,array(bounds,list))
# or Assign(name,list) into explicit form Assign(aname[index],value).
# If a 'Return' assignment is a list, it is left so.
# `GRAD/intrepnoarray` works recursively on 'for' and 'if' statements.
# `GRAD/intrepnoarray` puts a 'Return' around every statement without a 
# label.


`GRAD/intrepnoarray`:=proc(statseq,DN,NT,npars,arraylist,arraylistpass,dloc,
		    dlocpass,restype)
	local   nstat, resstat, stat, arr, aname, arrayvalues, 
		indexlist, x, i, j, nopsstat, arraylistloc, dlocloc,
		bounds;
 
	nstat:=nops(statseq);
	resstat:=NULL;
	arraylistloc:=arraylist;
	dlocloc:=dloc;
	for i to nstat do
	  stat:=op(i,statseq); # e.g. stat=Assign(s,array(1..2,[1,2]))

				###  Assign  ###

	  if op(0,stat)=R('Assign') then
	    arr:=op(2,stat);	# e.g. arr=array(1..2,[1,2])
	    aname:=op(1,stat);	# e.g. aname=s
	    if op(0,arr)=R('array') then
		if type(op(nops(arr),arr),'list') then
		   arrayvalues:=op(nops(arr),arr);
		   indexlist:=`GRAD/arrayexpand`(arrayvalues);
		   if npars>1 then
		      NT(aname):=`GRAD/arraytolist`(DN(aname),
				    [op(1..nops(arr)-1,arr),1..npars]);
		      dlocloc:=[op(dlocloc), 
				aname..R('List')(op(1..nops(arr)-1,arr),
				R('float')),
			 	DN(aname)..
			 	R('List')(op(1..nops(arr)-1,arr),1..npars,
			 	R('float'))]

		   else NT(aname):=`GRAD/arraytolist`(DN(aname),
				    [op(1..nops(arr)-1,arr)]);
		        dlocloc:=[op(dlocloc), 
				aname..R('List')(op(1..nops(arr)-1,arr),
				R('float')),
			 	DN(aname)..
			 	R('List')(op(1..nops(arr)-1,arr),
			 	R('float'))]
		   fi;
		   stat:=seq(R('Assign')(aname[op(x)],
				arrayvalues[op(x)]), x=indexlist);
		else
		  if npars>1 then
		   dlocloc:=[op(dlocloc),
				aname..R('List')(op(1..nops(arr),arr),
				R('float')),
			 	DN(aname)..
			 	R('List')(op(1..nops(arr),arr),1..npars,
			 	R('float'))];
		   NT(aname):=`GRAD/arraytolist`(DN(aname),
				    [op(1..nops(arr),arr),1..npars])
		  else
		   dlocloc:=[op(dlocloc),
				aname..R('List')(op(1..nops(arr),arr),
				R('float')),
			 	DN(aname)..
			 	R('List')(op(1..nops(arr),arr),
			 	R('float'))];
		   NT(aname):=`GRAD/arraytolist`(DN(aname),
				    [op(1..nops(arr),arr)])
		  fi;
		  stat:=NULL
		fi;
		arraylistloc:=arraylistloc union {aname};
	    elif type(arr,'list') then
		if restype='array' then
		   stat:=`GRAD/correctarray`(`GRAD/listtoarray`(arr,[]),
						aname);
		   bounds:=`GRAD/findbounds`(stat);
		   if not member(aname,arraylistloc) then
		      if npars>1 then
		        dlocloc:=[op(dlocloc),
				 aname..R('List')(bounds,
				 R('float')),
				 DN(aname)..R('List')(bounds,1..npars,
				 R('float'))];
			NT(aname):=`GRAD/arraytolist`(DN(aname),
							[bounds,1..npars])
		      else
		        dlocloc:=[op(dlocloc),
				 aname..R('List')(bounds,
				 R('float')),
				 DN(aname)..R('List')(bounds,
				 R('float'))];
			NT(aname):=`GRAD/arraytolist`(DN(aname),[bounds])
		      fi
		   fi;
		   stat:=op(stat)
		else
		  if npars>1 then
		   NT(aname):=`GRAD/listtodlist`(DN(aname),
					`GRAD/listtonames`(aname,arr),
					npars);
		   dlocloc:=[op(dlocloc),aname..R('float'),
				DN(aname)..R('float')]
		  else
		   NT(aname):=subs(aname=DN(aname),
					`GRAD/listtonames`(aname,arr));
		   dlocloc:=[op(dlocloc),aname..R('float'),
				DN(aname)..R('float')]
		  fi
		fi;
		arraylistloc:=arraylistloc union {aname}
	    elif not ( type(arr,function) and op(0,arr)=R('Proc') ) then
		if not assigned(NT(aname)) then 
		   if type(aname,indexed) then
		      if npars>1 then
			NT(aname):=[seq(DN(op(0,aname))[op(aname),j],
					j=1..npars)]
		      else
			NT(aname):=[DN(op(0,aname))[op(aname)]]
		      fi
		   else if npars>1 then
			   NT(aname):=[seq(DN(aname)[j],j=1..npars)]
			else
			   NT(aname):=[DN(aname)]
			fi
		   fi
		fi
	    elif type(arr,function) and op(0,arr)=R('Proc') then
		NT(aname):=DN(aname)
	    fi
	     

				###    For    ###

	  elif op(0,stat)=R('For') then
	     if nops(stat)=4 then
		stat:=subsop(4=`GRAD/intrepnoarray`(op(4,stat),DN,NT,npars,
					arraylistloc,'arraylistloc',
					dlocloc,'dlocloc',restype),
					stat)
	     else 
		stat:=subsop(6=`GRAD/intrepnoarray`(op(6,stat),DN,NT,npars,
					arraylistloc,'arraylistloc',
					dlocloc,'dlocloc',restype),stat)
	     fi

				###    If     ###

	  elif op(0,stat)=R('If') then
	     nopsstat:=nops(stat);
	     for j from 2 by 2 to nopsstat do
		 stat:=subsop(j=`GRAD/intrepnoarray`(op(j,stat),DN,NT,npars,
					arraylistloc,'arraylistloc',
					dlocloc,'dlocloc',restype),stat)
	     od;
	     if type(nopsstat,'odd') then
		 stat:=subsop(nopsstat=
			      `GRAD/intrepnoarray`(op(nopsstat,stat),
					DN,NT,npars,
					arraylistloc,'arraylistloc',
					dlocloc,'dlocloc',restype),
			      stat)
	     fi

		      ###  Return  without 'Return' label  ###

	  elif not member(op(0,stat),R({'Return','print','Error','lprint',
					'printf','Break','Next'})) then
	       stat:=R('Return')(stat)

	  fi;
	  resstat:=resstat,stat
	od;
	arraylistpass:=arraylistloc;
	dlocpass:=dlocloc;
	R('StatSeq')(resstat)
end:



# `GRAD/differentiate` gives the gradient of a value (not of a list!)
# x: array of the parameters w.r.t. the gradient has to be computed
# NT: "table" (function) of the names of the gradients
# NT(a)=[da[1],da[2],...,da[n]]
# npars: number of parameters w.r.t. the gradient has to be computed
# constlist: set of the constant local variables
# proclist: set of the local procedures
# parslist: set of the parameters
# loclist: set of the local variables
# proctobediff: set of the external procedures of which the gradient will
# be computed later
# proctobediffpass: updated proctobediff
# DN: "tabel" (function) of the names of the arrays which represent
# the gradients: DN(a)=da
# dfunlist: set of the comparing functions in value
# locpass: local variables+parameters+new defined local variable (for the
# `GRAD/givedname`) 

`GRAD/differentiate`:=proc(value,x,NT,npars,constlist,parslist,proclist,
			   loclist,proctobediff,proctobediffpass,DN,
			   dfunlist,dfunlistpass,locpass)
	local	dvalue, actloc, j, dasswrtactloc, indlist, aname,
		indetslist,proctobediffloc, dfunlistloc;

	proctobediffloc:=proctobediff;
	dfunlistloc:=dfunlist;
	dvalue:=array(1..npars);
	indetslist:=select(type,indets(value),'name');
	indetslist:=indetslist minus parslist minus constlist;
	indlist:=NULL;
	for j to nops(indetslist) do
	    aname:=op(j,indetslist);
	    if type(aname,'indexed') then aname:=op(0,aname) fi;
	    if not member(aname,loclist) then
		indlist:=indlist,j=NULL
	    fi
	od;
	indetslist:=subsop(indlist,indetslist);

	for actloc in indetslist do
	    dasswrtactloc[actloc]:=diff(value,actloc);
	    if not assigned (NT(actloc)) then
		if type(actloc,'indexed') then  # for arrays
		   if npars>1 then
		      NT(actloc):=[seq(DN(op(0,actloc))
				    [op(actloc),j],
				    j=1..npars)]
		   else
		      NT(actloc):=[DN(op(0,actloc))[op(actloc)]]
		   fi
		fi
	     fi
	od;

	for j to npars do
	    dvalue[j]:=diff(value,x[j]);
	    for actloc in indetslist do
		dvalue[j]:=dvalue[j]+dasswrtactloc[actloc]*NT(actloc)[j]
	    od;
	    dvalue[j]:=`GRAD/difftoD`(dvalue[j]);
	    dvalue[j]:=`GRAD/Dtoname`(dvalue[j],proclist,NT,
				proctobediffloc,'proctobediffloc',
				dfunlistloc,'dfunlistloc',locpass)
	od;
	proctobediffpass:=proctobediffloc;
	dfunlistpass:=dfunlistloc;
	[seq(dvalue[j],j=1..npars)]
end:



# `GRAD/diffreturn` takes any kind of list and differentiates it
# It has been conceived for Return statements (no names)

`GRAD/diffreturn`:=proc(liste,x,NT,npars,constlist,parslist,proclist,
			loclist,proctobediff,proctobediffpass,DN,dfunlist,
			dfunlistpass,locpass)
	local res, i, proctobediffloc, dfunlistloc;
	proctobediffloc:=proctobediff;
	dfunlistloc:=dfunlist;
 	res := [];
  	for i to nops(liste) do
   	     if type(liste[i],'list') then
    	        res := [op(res),`GRAD/diffreturn`(liste[i],x,NT,npars,
     	           	constlist,parslist,proclist,loclist,
			proctobediffloc,'proctobediffloc',DN,dfunlistloc,
			'dfunlistloc',locpass)]
     	     else
       	     	   res := [op(res),`GRAD/differentiate`(liste[i],x,NT,
        	        npars,constlist,parslist,proclist,loclist,
			proctobediffloc,'proctobediffloc',DN,dfunlistloc,
			'dfunlistloc',locpass)]
             fi
    	od;
	proctobediffpass:=proctobediffloc;
	dfunlistpass:=dfunlistloc;
    	res
end:


# `GRAD/diffreturn1` takes any kind of list and differentiates it
# It has been written for the assignments, where if differentiation
# is carried out onlt w.r.t. one variable, differentiating a list
# will give a list of the same dimension.
# `GRAD/diffreturn` is for Return statement: no difference if you
# differentiate w.r.t. one or more var ==> the result is standard.
# `GRAD/diffreturn`([sin(x),x],x) -> [[cos(x)],[1]]
# `GRAD/diffreturn1`([sin(x),x],x) -> [cos(x),1]

`GRAD/diffreturn1`:=proc(liste,x,NT,npars,constlist,parslist,proclist,
			loclist,proctobediff,proctobediffpass,DN,dfunlist,
			dfunlistpass,locpass)
	local res, i, proctobediffloc, dfunlistloc;
	proctobediffloc:=proctobediff;
	dfunlistloc:=dfunlist;
 	res := [];
  	for i to nops(liste) do
   	     if type(liste[i],'list') then
    	        res := [op(res),`GRAD/diffreturn1`(liste[i],x,NT,npars,
     	           	constlist,parslist,proclist,loclist,
			proctobediffloc,'proctobediffloc',DN,dfunlistloc,
			'dfunlistloc',locpass)]
     	     else
		if npars>1 then
       	     	   res := [op(res),`GRAD/differentiate`(liste[i],x,NT,
        	        npars,constlist,parslist,proclist,loclist,
			proctobediffloc,'proctobediffloc',DN,dfunlistloc,
			'dfunlistloc',locpass)]
		else
       	     	   res := [op(res),op(`GRAD/differentiate`(liste[i],x,NT,
        	        npars,constlist,parslist,proclist,loclist,
			proctobediffloc,'proctobediffloc',DN,dfunlistloc,
			'dfunlistloc',locpass))]
		fi
             fi
    	od;
	proctobediffpass:=proctobediffloc;
	dfunlistpass:=dfunlistloc;
    	res
end:




# `GRAD/analyze` works recursively. It takes a statement sequence and gives
# back the derived statement sequence. The recursion is used in 'for'
# and 'if' statements. It is the main procedure of the forward mode.
# List of parameters:

# x: 	   	array of the parameters
# parslist:	list of the parameters of the procedure
# npars: 	number of parameters with respect to which it has to derive
# constlist: 	these local variables will not be derived
# NT: 		name table of the derivatives 
#     		(looks like NT(localvar)=[dlocalvar[1],...dlocalvar[npars]])
# proclist:	list of the internal procedures
# arraylist:	list of the arrays
# loclist:	list of the local variables
# proctobediff: external procedures found in assignments -> they will be
#		derived later
# proctobediffpass: updated proctobediff
# itsagrd:	name of the result as array (to be put in the local vars
# itsagrdpass:	updated itsagrd
# DN: 		table od the names of the derivatives of the local vars
#		e.g.: DN('a') = 'da'
# restype:	type of the result (list, seq or array)
# lfnames:	local function names
# locpass:	contains all the local variables included the new ones
# funval:	option for including the function value in the result
# funval:	option functionvalue = true or false: 

`GRAD/analyze`:=proc(statseq,x,parslist,npars,constlist,NT,proclist,
		     proclistpass,arraylist,loclist,proctobediff,
		     proctobediffpass,itsagrd,itsagrdpass,DN,restype,
		     lfnames,lfnamespass,locpass,funval)
	local  	i, j, nstat,assname, assvalue, dassvalue, proclistloc,
		stat, actloc, resstat, nopsstat, dasswrtactloc,  
		indetslist, proctobediffloc, aname, indlist, 
		RAssign, RReturn, Rarray, RFor, RIf, Rgrd, itsagrdloc,
		dfunlist, locandfun, rules, lfnamesfin, lfnamesact;

	RAssign:=R('Assign');
	RReturn:=R('Return');
	Rarray:=R('array');
	RFor:=R('For');
	RIf:=R('If');
	Rgrd:=`GRAD/findname`(R('grd'),loclist);

	itsagrdloc:=itsagrd;
	proclistloc:=proclist;
	proctobediffloc:=proctobediff;
	lfnamesfin:=lfnames;
	resstat:=NULL;
	nstat:=nops(statseq);
	dassvalue:=array(1..npars);
	for i to nstat do
	    stat:=op(i,statseq);

				###  Assign  ###

	    if op(0,stat)=RAssign then
	      assname:=op(1,stat);
	      assvalue:=op(2,stat);
	      if not member(assname,constlist) then 
	       if type(assvalue,function) and op(0,assvalue)=R('Proc') 
	       then  # internal procedure
		   proclistloc:=proclistloc union {assname};
		   stat:=RAssign(DN(assname),
				 `GRAD/forward`(assvalue,[],[],'list',
						false)),
			 stat;
	       elif type(assvalue,list) then
		stat:=RAssign(DN(assname),
	 			 `GRAD/diffreturn1`(assvalue,x,NT,npars,
				 	constlist,parslist,proclistloc,
					loclist,proctobediffloc,
					'proctobediffloc',DN,{},
					'dfunlist',locpass)),
		      stat;
		if dfunlist<>{} then
		   locandfun:=loclist;
		   indlist:=nops(dfunlist);
		   lfnamesact:=[seq(`GRAD/givename`(R(cat(''lf'',j)),
				   		    locandfun,'locandfun'),
				    j=1..indlist)];
		   rules:=[seq(dfunlist[j]=lfnamesact[j],j=1..indlist)];
		   lfnamesfin:=lfnamesfin union {op(lfnamesact)};
		   stat:=seq(RAssign(rhs(rules[j]),lhs(rules[j])),
			     j=1..indlist),
			 subs(rules,stat)	
		fi

	       else

		####### names of derivatives

		if not assigned (NT(assname)) then
		   if npars>1 then
		      NT(assname):=[seq(DN(op(0,assname))[op(assname),j],
					j=1..npars)]
		   else
		      NT(assname):=[DN(op(0,assname))[op(assname)]]
		   fi
		fi;

		####### end names of derivatives

		####### Differentiation

		indetslist:=select(type,indets(assvalue),'name');
		indetslist:=indetslist minus parslist minus constlist;
		indlist:=NULL;
		for j to nops(indetslist) do
		    aname:=op(j,indetslist);
		    if type(aname,'indexed') then aname:=op(0,aname) fi;
		    if not member(aname,loclist) then
			indlist:=indlist,j=NULL
	  	    fi
		od;
		indetslist:=subsop(indlist,indetslist);

		for actloc in indetslist do
		    dasswrtactloc[actloc]:=diff(assvalue,actloc);
		    if not assigned (NT(actloc)) then
			if npars>1 then
		       	   NT(actloc):=[seq(DN(op(0,actloc))[op(actloc),j],
					    j=1..npars)]
			else
		       	   NT(actloc):=[DN(op(0,actloc))[op(actloc)]]
			fi
	    	    fi
		od;

		dfunlist:={};
		for j to npars do
	    	    dassvalue[j]:=diff(assvalue,x[j]);
	            for actloc in indetslist do
			dassvalue[j]:=dassvalue[j]+
				      dasswrtactloc[actloc]*NT(actloc)[j]
	    	    od;
		    dassvalue[j]:=`GRAD/difftoD`(dassvalue[j]);
		    dassvalue[j]:=`GRAD/Dtoname`(dassvalue[j],proclistloc,
					NT,proctobediffloc,
					'proctobediffloc',dfunlist,
					'dfunlist',locpass)
		od;

		####### substitute the gradient functions with a loc var

		if dfunlist<>{} then
		   locandfun:=loclist;
		   indlist:=nops(dfunlist);
		   lfnamesact:=[seq(`GRAD/givename`(R(cat(''lf'',j)),
						   locandfun,'locandfun'),
				   j=1..indlist)];
		   rules:=[seq(dfunlist[j]=lfnamesact[j],j=1..indlist)];
		   lfnamesfin:=lfnamesfin union {op(lfnamesact)};
		   for j to npars do 
		       dassvalue[j]:=subs(rules,dassvalue[j])
		   od;

		   stat:=seq(RAssign(rhs(rules[j]),lhs(rules[j])),
			     j=1..indlist),
		         seq(RAssign(NT(assname)[j],dassvalue[j]),
			     j=1..npars), 
		         stat	
		else stat:=seq(RAssign(NT(assname)[j],dassvalue[j]),
			     j=1..npars), 
		           stat
		fi
	       fi
	      fi

				###  Return  ###
	
	    elif op(0,stat)=RReturn then
		assvalue:=op(stat);
		if `GRAD/typeseq`(assvalue) then
		   stat:=`GRAD/diffreturn`([assvalue],x,NT,npars,constlist,
				parslist,proclistloc,loclist,
				proctobediffloc,'proctobediffloc',DN,{},
				'dfunlist',locpass);
		   if restype=[] or restype='seq' then
		      if funval then
			stat:=RReturn(assvalue,`GRAD/listtoseq`(stat))
		      else
			stat:=RReturn(`GRAD/listtoseq`(stat))
		      fi
		   elif restype='array' then
		      itsagrdloc:=Rgrd;
		      stat:=`GRAD/correctarray`(
					`GRAD/listtoarray`(stat,[]),Rgrd);
		      # e.g. stat=[Assign(grd[1,1],x),Assign(grd[1,2],y)]
		      if funval then
			stat:=  RAssign(Rgrd,Rarray(
					    `GRAD/findbounds`(stat))),
				op(stat),
				RReturn(assvalue,Rgrd)
		      else
			stat:=	RAssign(Rgrd,Rarray(
					    `GRAD/findbounds`(stat))),
				op(stat),
				RReturn(Rgrd)
		      fi
		   else  # restype = list
		      if funval then
			stat:=RReturn(assvalue,stat)
		      else
			stat:=RReturn(stat)
		      fi
		   fi

		elif type(assvalue,'list') then
	   	   stat:=`GRAD/diffreturn`(assvalue,x,NT,npars,constlist,
				parslist,proclistloc,loclist,
				proctobediffloc,'proctobediffloc',DN,{},
				'dfunlist',locpass);
		   if restype='seq' then
		      if funval then
			stat:=RReturn(assvalue,`GRAD/listtoseq`(stat))
		      else
			stat:=RReturn(`GRAD/listtoseq`(stat))
		      fi
		   elif restype='array' then
		      itsagrdloc:=Rgrd;
		      stat:=`GRAD/correctarray`(
					`GRAD/listtoarray`(stat,[]),Rgrd);
		      # e.g. stat=[Assign(grd[1,1],x),Assign(grd[1,2],y)]
		      if funval then
			stat:=  RAssign(Rgrd,Rarray(
					    `GRAD/findbounds`(stat))),
				op(stat),
				RReturn(assvalue,Rgrd)
		      else
			stat:=	RAssign(Rgrd,Rarray(
					    `GRAD/findbounds`(stat))),
				op(stat),
				RReturn(Rgrd)
		      fi
		   else 
		      if funval then
			stat:=RReturn(assvalue,stat)
		      else
			stat:=RReturn(stat)
		      fi
		   fi
		elif type(assvalue,'indexed') then
		      if not assigned(NT(assvalue)) then
			if npars>1 then
		          NT(assvalue):=[seq(DN(op(0,assvalue))
					       [op(assvalue),j],
						j=1..npars)]
			else
		          NT(assvalue):=[DN(op(0,assvalue))[op(assvalue)]]
		    	fi
		      fi;
		      if restype='seq' then
			if funval then
			 stat:=RReturn(assvalue,op(NT(assvalue)))
			else
		         stat:=RReturn(op(NT(assvalue)))
			fi
		      elif restype='array' then
			 itsagrdloc:=Rgrd;
			if funval then
			 stat:= RAssign(Rgrd,Rarray(1..npars)),
				seq(RAssign(Rgrd[j],NT(assvalue)[j]),
				    j=1..npars),
				RReturn(assvalue,Rgrd)
			else
			 stat:= RAssign(Rgrd,Rarray(1..npars)),
				seq(RAssign(Rgrd[j],NT(assvalue)[j]),
				    j=1..npars),
				RReturn(Rgrd)
			fi
		      else
			if funval then
		         stat:=RReturn(assvalue,NT(assvalue))
			else
		         stat:=RReturn(NT(assvalue))
			fi
		      fi
		elif member(assvalue,arraylist) then
		   if restype='seq' then
			if funval then
			   stat:=RReturn(assvalue,
				      	`GRAD/listtoseq`(NT(assvalue)))
			else
			   stat:=RReturn(`GRAD/listtoseq`(NT(assvalue)))
			fi
		   elif restype='list' then
			if funval then
			   stat:=RReturn(assvalue,NT(assvalue))
			else stat:=RReturn(NT(assvalue))
			fi
		   else if funval then
		     	   stat:=RReturn(assvalue,DN(assvalue))
			else
		     	   stat:=RReturn(DN(assvalue))
			fi
		   fi
		else  # scalar
		   stat:=`GRAD/differentiate`( assvalue,x,NT,npars,
					constlist,parslist,proclistloc,
					loclist,proctobediffloc,
					'proctobediffloc',DN,{},'dfunlist',
					locpass);
		   if restype='seq' then
			if funval then
			   stat:=RReturn(assvalue,op(stat))
			else
			   stat:=RReturn(op(stat))
			fi
		   elif restype='array' then
			itsagrdloc:=Rgrd;
			if funval then
			  stat:=RAssign(Rgrd,Rarray(1..npars)),
				seq(RAssign(Rgrd[j],op(j,stat)),
				    j=1..npars),
				RReturn(assvalue,Rgrd)
			else
			  stat:=RAssign(Rgrd,Rarray(1..npars)),
				seq(RAssign(Rgrd[j],op(j,stat)),
				    j=1..npars),
				RReturn(Rgrd)
			fi
		   else if funval then
			  stat:=RReturn(assvalue,stat)
			else
			  stat:=RReturn(stat)
			fi
		   fi
		fi;
		if dfunlist<>{} then
		   locandfun:=loclist;
		   indlist:=nops(dfunlist);
		   lfnamesact:=[seq(`GRAD/givename`(R(cat(''lf'',j)),
				   		    locandfun,'locandfun'),
				    j=1..indlist)];
		   rules:=[seq(dfunlist[j]=lfnamesact[j],j=1..indlist)];
		   lfnamesfin:=lfnamesfin union {op(lfnamesact)};
		   for j to npars do 
		       dassvalue[j]:=subs(rules,dassvalue[j])
		   od;

		   stat:=seq(RAssign(rhs(rules[j]),lhs(rules[j])),
			     j=1..indlist),
			 subs(rules,stat)	
		fi


				###    For    ###

	    elif op(0,stat)=RFor then
		if nops(stat)=4 then
		   stat:=subsop(4=`GRAD/analyze`(op(4,stat),x,parslist,
				npars,constlist,NT,proclistloc,
				'proclistloc',arraylist,loclist,
				proctobediffloc,'proctobediffloc',
				itsagrdloc,'itsagrdloc',DN,restype,
				lfnamesfin,'lfnamesfin',locpass,funval),
				stat)
		else  stat:=subsop(6=`GRAD/analyze`(op(6,stat),x,parslist,
				npars,constlist,NT,proclistloc,
				'proclistloc',arraylist,
				loclist,proctobediffloc,'proctobediffloc',
				itsagrdloc,'itsagrdloc',DN,restype,
				lfnamesfin,'lfnamesfin',locpass,funval),
				stat)
		fi

				###    If     ###

	    elif op(0,stat)=RIf then
		nopsstat:=nops(stat);
		for j from 2 by 2 to nopsstat do
		    stat:=subsop(j=`GRAD/analyze`(op(j,stat),x,parslist,
				npars,constlist,NT,proclistloc,
				'proclistloc',arraylist,loclist,
				proctobediffloc,'proctobediffloc',
				itsagrdloc,'itsagrdloc',DN,restype,
				lfnamesfin,'lfnamesfin',locpass,funval),
			  	stat)
		od;
		if type(nopsstat,'odd') then
		   stat:=subsop(nopsstat=`GRAD/analyze`(op(nopsstat,stat),x,
				parslist,npars,constlist,NT,proclistloc,
				'proclistloc',arraylist,loclist,
				proctobediffloc,'proctobediffloc',
				itsagrdloc,'itsagrdloc',DN,restype,
				lfnamesfin,'lfnamesfin',locpass,funval),
			  	stat)
		fi
		
	    fi;
	    resstat:=resstat,stat
	od;
	itsagrdpass:=itsagrdloc;
	proclistpass:=proclistloc;
	proctobediffpass:=proctobediffloc;
	lfnamespass:=lfnamesfin;
	R('StatSeq')(resstat)
end:




# `GRAD/difftoD` substitutes every 'diff' command in an expression 
# with a 'D' command


`GRAD/difftoD`:=proc(expr)
	local	difflist, reslist, dfunc, func, x, i;
	difflist:=select(t->has(op(0,t),'diff'),indets(expr));
	reslist:=NULL;
	for dfunc in difflist do
	    func:=op(1,dfunc);
	    x:=op(2,dfunc);
	    i:=1;
	    while op(i,func)<>x do  i:=i+1  od;  
	    reslist:=reslist,dfunc=D[i](op(0,func))(op(func));
	od;
	subs({reslist},expr)
end:


# `GRAD/Dtoname` transforms D[i](f)(args) to df(args)[i] if that df 
# procedure is known. The external procedures are put in proctobediffpass
# so that their gradient will be computed later. dfunlist is a list with
# all the internal and external procedures (with their argument) comparing 
# in expr. They will be substituted with a local variable later.


`GRAD/Dtoname`:=proc(expr,proclist,NT,proctobediff,proctobediffpass,
		     dfunlist,dfunlistpass,locpass)
	local Dlist, Dfunc, func, justD, reslist, difffunc,proctobediffloc,
	      dfunlistloc, dlocfun, wasteloc;
	proctobediffloc:=proctobediff;
	dfunlistloc:=dfunlist;
	Dlist:=select(t->has(op(0,t),'D'),indets(expr));
	reslist:={};
	for Dfunc in Dlist do			# Dfunc=D[i](f)(args)
	    func:=op(1,op(0,Dfunc)); 		# func=f
	    if member(func,proclist) then
		dlocfun:=NT(func)(op(Dfunc));
		justD:=op(0,op(0,Dfunc));	# justD=D[i]
		if type(justD,'indexed') then
		   reslist:=reslist union {Dfunc=dlocfun[op(justD)]}
		else reslist:=reslist union {Dfunc=dlocfun[1]}
		fi;
		dfunlistloc:=dfunlistloc union {dlocfun}
	    elif type(U(func),'procedure') then
		difffunc:=`diff/`.U(func);
		if eval(difffunc,2)<>func then
		   if traperror(readlib(difffunc))=lasterror then
			if not assigned(NT(func)) then
			 NT(func):=`GRAD/givedname`(func,locpass,'wasteloc')
			fi;
			dlocfun:=NT(func)(op(Dfunc));
			justD:=op(0,op(0,Dfunc));
			if type(justD,'indexed') then
		 	  reslist:=reslist union {Dfunc=dlocfun[op(justD)]}
			else reslist:=reslist union {Dfunc=dlocfun[1]}
			fi;
		    	proctobediffloc:=proctobediffloc union {func};
			dfunlistloc:=dfunlistloc union {dlocfun}
		   fi
		fi
	    fi
	od;
	proctobediffpass:=proctobediffloc;
	dfunlistpass:=dfunlistloc;
	subs(reslist,expr)
end:


# `GRAD/givedname`(f,loc,locpass) gives df if df is not already a local
# variable of the entered procedure or a global variable, in which case 
# you'd have df<number>. The actualized loclist is then in locpass.



`GRAD/givedname`:=proc(aname,loc,locpass)
	local	i, dname, Uloc, dname1;
	Uloc:=U(loc);
	dname:=evaln(cat(''d'',evaln(U(aname))));
	dname1:=parse(cat(''d'',evaln(U(aname))));
	i:=0;
	while member(eval(dname1,1),Uloc) 
		or eval(dname1,1)<>eval(dname1,2) do
	      dname1:=parse(cat(dname,i));
	      i:=i+1
	od;
	dname1:=R(dname1);
	locpass:=loc union {dname1};
	dname1
end:



# The result of `GRAD/dependances` is to find in the table IT:
# IT(<local var>) is a list of every variable on which <local var> depends.
# `GRAD/dependances` sends only assignments to `GRAD/dependancesinIT` to find 
# those relationships. If there's a for loop then `GRAD/dependancesinIT` will
# repeat going through the assignment's sequence until the relationships
# don't change anymore.

`GRAD/dependances`:=proc(statseqinit,statseq,localsnotarray,IT,parslist)
	local asseq, i, k, numstat, stat, assvalue;
	asseq:=op(statseqinit);
	for i to nops(statseq) do
	    stat:=op(i,statseq);
	    if op(0,stat)=R('Assign') then
	      assvalue:=op(2,stat);
	      if type(assvalue,function) and op(0,assvalue)=R('Proc')
	      then IT(op(1,stat)):=parslist 
	      elif not type(assvalue,list) then
		   asseq:=asseq,stat
	      fi
	    elif op(0,stat)=R('If') then
		`GRAD/dependancesinIT`([asseq],localsnotarray,IT,'false');
		asseq:=NULL;
		numstat:=nops(stat);
		for k from 2 by 2 to numstat do
		    `GRAD/dependances`([],op(k,stat),localsnotarray,IT)
		od;
		if type(numstat,'odd') then
		    `GRAD/dependances`([],op(numstat,stat),
					localsnotarray,IT)
		fi
	    elif op(0,stat)=R('For') then
		`GRAD/dependancesinIT`([asseq],localsnotarray,IT,'false');
		asseq:=NULL;
		stat:=`GRAD/expandallfor`(op(nops(stat),stat));
		`GRAD/dependancesinIT`(stat,localsnotarray,IT,'true')
	    fi
	od;
	if asseq<>NULL then 
		`GRAD/dependancesinIT`([asseq],localsnotarray,IT,'false')
	fi
end:


`GRAD/dependancesinIT`:=proc(statseq,localsnotarray,IT,itsafor)
	local 	stat, aname, value, indlist, ind, changed, ITlength, i;
	for i to nops(statseq) do
	    stat:=op(i,statseq);
	    aname:=op(1,stat);
	    value:=op(2,stat);
	    indlist:=select(type,indets(value),'name');
	    if not type(aname,'indexed') then
	       IT(aname):=IT(aname) union indlist
	    fi;
	    IT(aname):=`union`(seq(IT(ind),
				   ind=indlist intersect localsnotarray))
			union IT(aname)
	od;
	if itsafor then
	   changed:='true';
	   while changed do
		changed:='false';
		for i to nops(statseq) do
		    aname:=op(1,op(i,statseq));
		    if not type(aname,'indexed') then
	   	       ITlength:=nops(IT(aname));
		       IT(aname):=`union`(seq(IT(ind),
				   ind=IT(aname) intersect localsnotarray))
				  union IT(aname);
		       if ITlength<>nops(IT(aname)) then changed:='true' fi
		    fi
		od
	   od
	fi
end:


# `GRAD/expandallfor` eliminates every for and if statements and gives back
# all the other statements it finds (included those which where inside
# the for and if statements).


`GRAD/expandallfor`:=proc(statseq)
	local	stat, i, k, numstat, resstat;
	resstat:=NULL;
	for i to nops(statseq) do
	    stat:=op(i,statseq);
	    if op(0,stat)=R('Assign') then
		resstat:=resstat,stat
	    elif op(0,stat)=R('For') then
	        resstat:=resstat,
			 op(`GRAD/expandallfor`(op(nops(stat),stat)))
	    elif op(0,stat)=R('If') then
		numstat:=nops(stat);	
		for k from 2 by 2 to numstat do
		    resstat:=resstat,op(`GRAD/expandallfor`(op(k,stat)))
		od;
		if type(numstat,'odd') then 
		    resstat:=resstat,
			     op(`GRAD/expandallfor`(op(numstat,stat)))
		fi
	    fi
	od;
	[resstat]
end:


# `GRAD/findconst` gives a list of the non constants local variables
# arrays and lists are assumed not to be constant


`GRAD/findconst`:=proc(localsnotarray,IT,parslist)
	local constlist, aname, i;
	constlist:={};
	for i to nops(localsnotarray) do
	    aname:=op(i,localsnotarray);
	    if IT(aname) intersect parslist = {} and
	       select(type,IT(aname),'indexed')={}  then
		  constlist:=constlist union {aname}
	    fi
	od;
	constlist
end:


# `GRAD/listtoarray`([x,[y,z]],[]) 
#  [[[1], x], [[2, 1], y], [[2, 2], z]]


`GRAD/listtoarray`:=proc(liste,ind)
	local res, indloc, elem, i;
	res:=NULL;
	for i to nops(liste) do
	    elem:=op(i,liste);
	    indloc:=op(ind),i;
	    if type(elem,'list') then
		res:=res,op(`GRAD/listtoarray`(elem,[indloc]))
	    else
		res:=res,[[indloc],elem]
	    fi
	od;
	[res]
end:


# `GRAD/listtonames`(a,[2,[5,1]]) -> [a[1],[a[2,1],a[2,2]]]

`GRAD/listtonames`:=proc(aname,liste)
	local	i,res, x;
	res:=[];
	for i to nops(liste) do
	    if type(liste[i],'list') then
		res:=[op(res),[seq(aname[i,op(x)],
				  x=`GRAD/listtonames`(aname,liste[i]))]]
	    else
		res:=[op(res),aname[i]]
	    fi
	od;
	res
end:


# `GRAD/listtodlist`(a,[a[1],[a[2,1],a[2,2]]],2) 
#   -> [[a[1,1], a[1,2]], [[a[2,1,1], a[2,1,2]], [a[2,2,1], a[2,2,2]]]]


`GRAD/listtodlist`:=proc(aname,liste,npars)
	local	i, j, res;
	res:=[];
	for i to nops(liste) do
	    if type(liste[i],list) then
		res:=[op(res),`GRAD/listtodlist`(aname,liste[i],npars)]
	    else
		res:=[op(res),[seq(aname[op(liste[i]),j],j=1..npars)]]
	    fi
	od;
	res
end:


# `GRAD/correctarray`([[[1], x], [[2, 1], y], [[2, 2], z]],grd)
#  -> [Assign(grd[1,1],x)  , Assign(grd[2,1],y)  , Assign(grd[2,2],z)]


`GRAD/correctarray`:=proc(liste,Rgrd)
	local i, maxlen, nliste, resliste, listei, j, RAssign;
	RAssign:=R('Assign');
	resliste:=NULL;
	nliste:=nops(liste);
	maxlen:=0;
	for i to nliste do
	    maxlen :=max(maxlen,nops(op(1,liste[i])))
	od;
	for i to nliste do
	    listei:=op(1,liste[i]);
	    if nops(listei)<>maxlen then
		resliste:=resliste,
			  RAssign(Rgrd[op(listei),
				       seq(1,j=1..maxlen-nops(listei))],
				  op(2,liste[i]))
	    else
		resliste:=resliste,
			  RAssign(Rgrd[op(listei)],op(2,liste[i]))
	    fi
	od;	
	[resliste]
end:


# `GRAD/findbounds`(
#	     [Assign(grd[1,1],x),Assign(grd[2,1],y),Assign(grd[2,2],z)])
#  -> 1..2, 1..2

`GRAD/findbounds`:=proc(asslist)
	local	maxbound, bound, i, j, ass;
	maxbound:=nops(op(1,asslist[1]));
	bound:=array(1..maxbound,sparse);
	for i to nops(asslist) do
	    ass:=op(1,asslist[i]);
	    for j to maxbound do
		bound[j]:=max(bound[j],op(j,ass))
	    od
	od;
	seq(1..bound[j],j=1..maxbound)
end:


# `GRAD/findname`(aname,loclist)  gives a name which is not in loclist and
# which doesn't exist globally.


`GRAD/findname`:=proc(aname,loclist)
	local	newname, Uloclist, i;
	newname:=U(aname);
	Uloclist:=U(loclist);
	i:=0;
	while member(newname,Uloclist) or eval(newname,1)<>eval(newname,2)
	do 	newname:=parse(cat(aname,i));
		i:=i+1
	od;
	R(newname)
end:


# `GRAD/arraytoseq`(a,[1..3,4..5]) -> a[1,4],a[1,5],a[2,4],a[2,5],...

`GRAD/arraytoseq`:=proc(aname,bounds)
	local	i, l, r;
	if bounds=[] then
		aname[args[3..nargs]]
	else
		l:=lhs(bounds[1]);	r:=rhs(bounds[1]);
	   	seq(`GRAD/arraytoseq`(aname,subsop(1=NULL,bounds),
				 args[3..nargs],i),
		     i=l..r)
	fi
end:


# `GRAD/arraytolist`(a,[1..3,4..5]) ->[[a[1,4],a[1,5]],[a[2,4],a[2,5]],...]

`GRAD/arraytolist`:=proc(aname,bounds)
	local	i, l, r;
	if bounds=[] then
		aname[args[3..nargs]]
	else
		l:=lhs(bounds[1]);	r:=rhs(bounds[1]);
	   	[seq(`GRAD/arraytolist`(aname,subsop(1=NULL,bounds),
				 args[3..nargs],i),
		     i=l..r)]
	fi
end:


 
# `GRAD/arrayexpand`(list) gives back an ordered list of the indices


`GRAD/arrayexpand`:=proc(liste)
	local  i, res, x;
	res:=[];
	for i to nops(liste) do
	    if type(liste[i],'list') then
		res:=[op(res), seq([i,op(x)], 
				   x=`GRAD/arrayexpand`(liste[i]))]
	    else	
		res:=[op(res),[i]]
	    fi
	od;
	RETURN(res)
end:


# `GRAD/listtoseq`([1,[2,3]]) ->  1,2,3


`GRAD/listtoseq`:=proc(liste)
	local i, res, elem;
	res:=NULL;
	for i to nops(liste) do
	    elem:=op(i,liste);
	    if type(elem,'list') then
	    	res:=res,`GRAD/listtoseq`(elem)
	    else res:=res,elem
	    fi
	od;
	res
end:



# `GRAD/correctarray`([[[1], x], [[2, 1], y], [[2, 2], z]],grd)
#  -> [Assign(grd[1,1],x)  , Assign(grd[2,1],y)  , Assign(grd[2,2],z)]


`GRAD/correctarray`:=proc(liste,Rgrd)
	local i, maxlen, nliste, resliste, listei, j, RAssign;
	RAssign:=R('Assign');
	resliste:=NULL;
	nliste:=nops(liste);
	maxlen:=0;
	for i to nliste do
	    maxlen :=max(maxlen,nops(op(1,liste[i])))
	od;
	for i to nliste do
	    listei:=op(1,liste[i]);
	    if nops(listei)<>maxlen then
		resliste:=resliste,
			  RAssign(Rgrd[op(listei),
				       seq(1,j=1..maxlen-nops(listei))],
				  op(2,liste[i]))
	    else
		resliste:=resliste,
			  RAssign(Rgrd[op(listei)],op(2,liste[i]))
	    fi
	od;	
	[resliste]
end:


# `GRAD/JAC` takes as argument(s) a list of procedures and gives back the 
# jacobian matrix of it
# It is possible to give as second argument a list of parameters
# with respect to which it will be derived.
# The type of the result can be choosed, too, by 
# result_type = list or array or seq.
# This is an old version, where for every procedure of the list, a gradient
# procedure is produced and put ( as an internal procedure ) in a unique
# procedure which gives the jacobi matrix.
# You'd better use the 'JACOBI' procedure, instead, where no internal 
# procedures are produced anymore. (see below)

`GRAD/JAC`:=proc()
	local   proclist, jac, nproclist, pars, pr, npars, i, result, 
		largs, dervars, restype, eargs, mo;


	largs:=[seq(args[i],i=1..nargs)];
	proclist:=select(type,largs,list(procedure));
	if nops(proclist)<>1 then ERROR(`You need to enter exactly one `.
				  `list of procedures.`)
	fi;
	proclist:=op(proclist);
	nproclist:=nops(proclist);

	pars:=[op(1,op(1,op(1,proclist)))];

	dervars:=[op({op(select(type,largs,list(name)))} minus {proclist})];
	if nops(dervars)>1 then
	   ERROR(`You can enter at most one liste of parameters w.r.t.`.
		 ` you wonna derive.`)
	elif nops(dervars)=1 then 
	   dervars:=op(dervars);
	   npars:=nops(dervars)
	else dervars:=pars;
 	     npars:=nops(pars)
	fi;
	
	eargs:=select(type,largs,equation);
	restype:=select(x->evalb(lhs(x)='result_type'),eargs);
	if nops(restype)>1 then
	   ERROR(`You can't give the option result_type more than once.`)
	elif nops(restype)=1 then
	   restype:=rhs(op(restype))
	else  restype:='list'
	fi;

	mo:=select(x->evalb(lhs(x)='mode'),eargs);
	if nops(mo)>1 then
	   ERROR(`You can't give the option mode more than once.`)
	else
	   mo:=op(mo)
	fi;

	if restype='array' then  result:='jac'(op(pars))
	   elif restype='seq' then 
		result:=Return(seq(jac[i](op(pars)),i=1..nproclist))
	   else	result:=[seq(jac[i](op(pars)),i=1..nproclist)]
	fi;

	pr:= 'Proc'('Name'('djac'..List(1..nproclist,1..npars,'float')),
		'Parameters'(seq(pars[i]..'float',i=1..nops(pars))),
		'Options'(),'Description'(),
		'Locals'('jac'),
		'Globals'(),
		'StatSeq'('Assign'('jac','array'(1..nproclist)),
		          seq('Assign'(jac[i],
				U(GRAD(R(MAKEINTREP(proclist[i])),
					dervars,mo,result_type='list'))),
			      i=1..nproclist),
		          result));
	MAKEPROC(pr)
end:




# JACOBI takes as input a list of procedures and return the jacobi matrix 
# of it. Optionally, you can enter as parameter a list of variable w.r.t. which
# you'd like it will be derivated. The mode can be given with
# mode = forward | reverse (default reverse) and the type of the result 
# with  result_type = list | seq | array.


JACOBI:=proc()
	local   proclist, Jac, nproclist, pars, pr, npars, i, j, 
		largs, eargs, dervars, restype, mo,  
		statseq, locals, globals, rets, resloc, resglob,
		RAssign, RReturn;

	RAssign:=R('Assign');
	RReturn:=R('Return');
	largs:=[seq(args[i],i=1..nargs)];
	proclist:=select(type,largs,list(procedure));
	if nops(proclist)<>1 then ERROR(`You need to enter exactly one `.
				  `list of procedures.`)
	fi;
	proclist:=op(proclist);
	nproclist:=nops(proclist);

	pars:=[op(1,op(1,op(1,proclist)))];

	dervars:=[op({op(select(type,largs,list(name)))} minus {proclist})];
	if nops(dervars)>1 then
	   ERROR(`You can enter at most one liste of parameters w.r.t.`.
		 ` you wonna derive.`)
	elif nops(dervars)=1 then 
	   dervars:=op(dervars);
	   npars:=nops(dervars)
	else dervars:=pars;
 	     npars:=nops(pars)
	fi;
	
	eargs:=select(type,largs,equation);
	restype:=select(x->evalb(lhs(x)='result_type'),eargs);
	if nops(restype)>1 then
	   ERROR(`You can't give the option result_type more than once.`)
	elif nops(restype)=1 then
	   restype:=rhs(op(restype))
	else  restype:='list'
	fi;

	mo:=select(x->evalb(lhs(x)='mode'),eargs);
	if nops(mo)>1 then
	   ERROR(`You can't give the option mode more than once.`)
	else
	   mo:=op(mo)
	fi;

	Jac:=R('jac');
	statseq:=array(1..nproclist);
	locals:=array(1..nproclist);
	globals:=array(1..nproclist);
	for i to nproclist do
	    proclist[i]:=R(MAKEINTREP(proclist[i]));
	    proclist[i]:=GRAD(proclist[i],dervars,mo,
			    result_type='list');
	    #### this gives the gradient of the i-th function in intrep
	    statseq[i]:=op(7,proclist[i]);
	    rets:=select(type,indets(statseq[i]),
			 specfunc(anything,RReturn));
	    rets:=map(proc(x,Jac,i,RAssign,npars)
			local j;
			x=seq(RAssign(Jac[i,j],op(j,op(x))),
			      j=1..npars)
		      end,
		      rets, Jac,i,RAssign,npars);
	    statseq[i]:=subs(rets,statseq[i]);
	    locals[i]:=op(5,proclist[i]);
	    globals[i]:=op(6,proclist[i])
	od;
	resloc:={seq(op(locals[i]),i=1..nproclist)};
	resloc:=map(lhs,resloc);
	resloc:=map(x->x..R('float'),resloc);
	resglob:={seq(op(globals[i]),i=1..nproclist)};
	resglob:=map(lhs,resglob);
	resglob:=map(x->x..R('float'),resglob);
	pars:=op(R(map(x->x..'float',pars)));
	rets:=RAssign(Jac,R('array')(1..nproclist,1..npars,[])),
	      seq(op(statseq[i]),i=1..nproclist);
	if restype='list' then
	   rets:=rets,[seq([seq(Jac[i,j],j=1..npars)],i=1..nproclist)]
	elif restype='seq' then
	   rets:=rets,RReturn(seq(seq(Jac[i,j],j=1..npars),i=1..nproclist))
	elif restype='array' then
	   rets:=rets,RReturn(Jac)
	fi;

	pr:=R('Proc')(R('Jacobi')..R('float'),
			R('Parameters')(pars),R('Options')(),
			R('Description')(),
			R('Locals')(op(resloc),
			 Jac..R('List')(1..nproclist,1..npars,R('float'))),
			R('Globals')(op(resglob)),
			R('StatSeq')(rets));
	MAKEPROC(U(pr))
end:


# HESSIAN(f::procedure,result_type=list | seq | array,
#	  mode=forward | reverse)
# gives the Hessian matrix of f with the desired result type and applying 
# desired mode. The two last arguments are optional.
# Defaults: result type = list
#	    mode = reverse 


HESSIAN:=proc()
	local 	restype, pr, largs, eqargs, m, i;

	largs:=[seq(args[i],i=1..nargs)];
	eqargs:=select(type,largs,equation);

	pr:=select(type,largs,procedure);
	if nops(pr)<>1 then 
	   ERROR(`You need to enter exactly one procedure.`)
	fi;
	pr:=op(pr);

	m:=select(x->evalb(lhs(x)='mode'),eqargs);
	if nops(m)>1 then
	   ERROR(`You can enter at most one mode.`)
	fi;
	m:=op(m);

	restype:=select(x->evalb(lhs(x)='result_type'),eqargs);
	if nops(restype)>1 then 
	   ERROR(`You can enter at most one type of result.`)
	fi;
	restype:=op(restype);

	GRAD(GRAD(pr,'result_type'='list',m),restype,m);
end:


##################
# REVERSE MODE
##################


`GRAD/reverse`:=proc(f,apars,grname,restype,funval)
	local	statseq, localslist, pr, pars, npars, x, n, prname, loc,
		arrlist, locandpars, NT, proctobediff, proclist, res, typ;

	if type(f,'procedure') then   	pr:=R(MAKEINTREP(f))
	   elif op(0,f)=R('Proc') then	pr:=f
	   else  ERROR(`First argument must be a procedure`)
	fi;
	if has(pr,[R('diff'),R('int')]) then
	   ERROR(`The procedure cannot contain int or diff operators.`)
	fi;

	pars:=op(2,pr);

	if apars=[] then 
	   npars:=nops(pars);
	   x:=array(1..npars,map(lhs,[op(pars)]))
	else
	   npars:=nops(apars);
	   x:=array(1..npars,R(apars))
	fi;
	prname:=lhs(op(1,op(1,pr)));
	if grname=[] then  
	   NT(prname):=R(cat(''d'',U(prname)))
	else  NT(prname):=R(grname)
	fi;

	statseq:=op(7,pr);
	statseq:=`GRAD/transformarray`(statseq,[],'arrlist');
	if not `GRAD/canreverse`(statseq,'statseq','localslist',NT(prname)) 
	then
		ERROR(`Procedure not reversable`)
	fi;
	locandpars:=convert(x,'set') union localslist;
	statseq:=`GRAD/dorev`([],statseq,x,npars,localslist,'localslist',
			locandpars,{},{},'proctobediff',{prname},'proclist',
			NT,restype,funval);
	proctobediff:=proctobediff minus {prname};
	for n in proctobediff do
		localslist:=localslist minus {NT(n)};
		assign(U(NT(n)),`GRAD/reverse`(U(n),[],[],'list',false))
	od;
	localslist:=localslist minus {NT(prname)} minus {op(pars)};
	if proctobediff<>{} then
	   lprint;
	   lprint(`Attention: in the procedure are external procedures `);
	   lprint(proctobediff);
	   lprint(`of which the gradients have been computed and are now`.
		  ` available under the names: `);
	   lprint({seq(n..NT(n),n=proctobediff)});
	fi;

	loc:=R('Locals')(seq(n..R('float'),n=localslist));

	typ:=rhs(op(1,op(1,pr)));
	if type(typ,specfunc(anything,R('List'))) then
	   typ:=`GRAD/expandList`(typ);
	   typ:=R('List')(op(1..nops(typ)-1,typ),1..npars,R('float'))
	else typ:=R('List')(1..npars,R('float'))
	fi;

	res:=MAKEPROC('Proc'(U(R('Name')(NT(prname)..typ),pars,
		             op(3,pr),R('Description')(),loc,op(6,pr),
		   	     statseq)));
	if grname<>[] then assign(U(NT(prname)),res) fi;
	if type(f,'procedure') then
	     eval(res)
	else R('Proc')(R('Name')(NT(prname)..typ),pars,
		        op(3,pr),R('Description')(),loc,op(6,pr),
		   	statseq)
	fi
end: 





# `GRAD/transformarray` transforms array declarations in the StatSeq of the 
# form Assign(name,array(bounds)) or Assign(name,array(bounds,list))
# or Assign(name,list) into explicit form Assign(aname[index],value).
# If a 'Return' assignment is a list, it is left so.
# `GRAD/transformarray` works recursively on 'for' and 'if' statements.
# `GRAD/transformarray` puts a 'Return' around every statement without a
# label.


`GRAD/transformarray`:=proc(statseq,arrlist,arrlistpass)
	local   nstat, resstat, stat, arr, aname, arrayvalues, 
		indexlist, x, i, j, nopsstat, arrlistloc,
		itsalist, hassubst;
 	arrlistloc:=arrlist;
	nstat:=nops(statseq);
	resstat:=NULL;
	for i to nstat do
	  stat:=op(i,statseq); # e.g. stat=Assign(s,array(1..2,[1,2]))

				###  Assign  ###

	  if op(0,stat)=R('Assign') then
	    arr:=op(2,stat);	# e.g. arr=array(1..2,[1,2])
	    aname:=op(1,stat);	# e.g. aname=s
	    if op(0,arr)=R('array') then
		if type(op(nops(arr),arr),'list') then
		   ######  case where arr = array(bounds,list)

		   arrayvalues:=op(nops(arr),arr);
		   indexlist:=`GRAD/arrayexpand`(arrayvalues);
		   stat:=seq(R('Assign')(aname[op(x)],
				arrayvalues[op(x)]), x=indexlist);
		   hassubst:='false';
		   # finds out if to that name was already assigned an array 
		   for j to nops(arrlistloc) do
			if op(1,op(j,arrlistloc))=aname then
		  	      arrlistloc:=subsop(j=[aname,
						 `GRAD/arraytolist`(aname,
						 [op(1..nops(arr)-1,arr)])],
						 arrlistloc);
			      hassubst:='true';
			      break
			fi
		   od;
		   if not hassubst then
			arrlistloc:=[op(arrlistloc), 
				     [aname,`GRAD/arraytolist`(
					aname,[op(1..nops(arr)-1,arr)])]]
		   fi;
		else stat:=NULL;  ## case where arr = array(bounds)
		   hassubst:='false';
		   for j to nops(arrlistloc) do
			if op(1,op(j,arrlistloc))=aname then
		  	      arrlistloc:=subsop(j=[aname,
						 `GRAD/arraytolist`(aname,
						 [op(1..nops(arr),arr)])],
						 arrlistloc);
			      hassubst:='true';
			      break
			fi
		   od;
		   if not hassubst then
			arrlistloc:=[op(arrlistloc), 
				     [aname,`GRAD/arraytolist`(
					aname,[op(1..nops(arr),arr)])]]
		   fi
		fi
	    elif type(arr,'list') then	  #  case where arr = list
		indexlist:=`GRAD/listtonames`(aname,arr);
		hassubst:='false';
		   for j to nops(arrlistloc) do
			if op(1,op(j,arrlistloc))=aname then
		  	      arrlistloc:=subsop(j=[aname,indexlist],
						 arrlistloc);
			      hassubst:='true';
			      break
			fi
		   od;
		   if not hassubst then
			arrlistloc:=[op(arrlistloc), [aname,indexlist]]
		   fi;

		stat:=seq(R('Assign')(aname[op(x)],
				arr[op(x)]), x=`GRAD/arrayexpand`(arr))
	    else
	        for j to nops(arrlistloc) do
		    if op(1,op(j,arrlistloc))=aname then
			arrlistloc:=subsop(j=NULL,arrlistloc); break
		    fi
	  	od
	    fi

				###    For    ###

	  elif op(0,stat)=R('For') then
	     if nops(stat)=4 then
		stat:=subsop(4=`GRAD/transformarray`(op(4,stat),arrlistloc,
						'arrlistloc'),
			     stat)
	     else stat:=subsop(6=`GRAD/transformarray`(op(6,stat),
						       arrlistloc,
						       'arrlistloc'),
			       stat)
	     fi

				###    If     ###

	  elif op(0,stat)=R('If') then
	     nopsstat:=nops(stat);
	     for j from 2 by 2 to nopsstat do
		 stat:=subsop(j=`GRAD/transformarray`(op(j,stat),arrlistloc,
							'arrlistloc'),
			      stat)
	     od;
	     if type(nopsstat,'odd') then
		 stat:=subsop(nopsstat=
			      `GRAD/transformarray`(op(nopsstat,stat),
						    arrlistloc,
						    'arrlistloc'),
			      stat)
	     fi

				### Return ###

	  elif op(0,stat)=R('Return') then
	     stat:=op(stat);
	     itsalist:='false';
	     for j to nops(arrlistloc) do
		if stat=op(1,op(j,arrlistloc)) then
			itsalist:='true'; break
		fi
	     od;
	     if itsalist then
		   stat:=R('Return')(op(2,op(j,arrlistloc)))
	     else  stat:=R('Return')(stat)
	     fi

		      ###  Return  without 'Return' label  ###

	  elif not member(op(0,stat),R({'print','Error','lprint',
					'printf','Break'})) then
	     itsalist:='false';
	     for j to nops(arrlistloc) do
		if stat=op(1,op(j,arrlistloc)) then
			itsalist:='true'; break
		fi
	     od;

	     if itsalist then
		   stat:=R('Return')(op(2,op(j,arrlistloc)))
	     else
	           stat:=R('Return')(stat)
	     fi
	  fi;
	  resstat:=resstat,stat
	od;
	arrlistpass:=arrlistloc;
	R('StatSeq')(resstat)
end:

# `GRAD/expandfor` takes a StatSeq() and gives it back with the for 
# statements expanded. It returns false if there's a while or if the 
# bounds aren ot constants. For statements of the form  "for x in list do" 
# are allowed when list is a list or a set of constants.


`GRAD/expandfor`:=proc(statseq)
	local	resstat, i, k, stat, intexpand, statloc, numstat;
	resstat:=NULL;
	for i to nops(statseq) do
	    stat:=op(i,statseq);
	    if op(0,stat)=R('For') then
		if nops(stat)=6 and op(5,stat)=R('true') and
		   type([op(2..4,stat)],list(numeric))  then
		     intexpand:=`GRAD/expandfor`(op(6,stat));
		     if intexpand='false' then RETURN('false')  fi;
	  	     statloc:=NULL;
		     for k from op(2,stat) by op(3,stat) to op(4,stat) do
		       statloc:=statloc,
				op(subs(op(1,stat)=k,intexpand))
		     od;
		     stat:=statloc
		elif nops(stat)=4 and op(3,stat)=R('true') and
		   (type(op(3,stat),list(numeric)) or 
		    type(op(3,stat),set(numeric))) then
		     intexpand:=`GRAD/expandfor`(op(4,stat));
		     if intexpand='false' then RETURN('false')  fi;
		     statloc:=NULL;
		     for k in op(2,stat) do
			statloc:=statloc,op(subs(op(1,stat)=k,intexpand))
		     od;
		     stat:=statloc
		else RETURN('false')
		fi
	     elif op(0,stat)=R('If') then
		numstat:=nops(stat);
		for k from 2 by 2 to numstat do
		    stat:=subsop(k=`GRAD/expandfor`(op(k,stat)),stat)
		od;
		if type(numstat,'odd') then
		   stat:=subsop(numstat=
				`GRAD/expandfor`(op(numstat,stat)),stat)
		fi
	     fi;
	     resstat:=resstat,stat
	od;
	R('StatSeq')(resstat)
end:


# `GRAD/Iftransform` takes a StatSeq not containing For statementsand does 
# the following transformations:
# One If:
# Iftransorm( A; if cond1 then a elif cond2 then b else c fi; B )
# -> A; if cond1 then a;B elif cond2 then b;B else c;B fi
#
# Composed If:
# Iftransorm( A; if cond1 then a1; if cond11 then b1 else b2 fi 
#    elif cond2 then a2 else a3 fi; B )
# -> A; if cond1 then a1;if cond11 then b1;B else b2;B fi
#       elif cond2 then a2;B else a3;B fi
#
# If there's no else then it puts one with the following statements
# If there's a RETURN and no If in a substatement of an If then
# it doesn't hang the following statements. E.g.
# `GRAD/Iftransform`(A; if cond1 then a;RETURN(a1) elif cond2 then b else c fi; B)
# -> A; if cond1 then a;RETURN(a1) elif cond2 then b;B else c;B fi


`GRAD/Iftransform`:=proc(statseq)
	local	i, k, resstat, stat, statafter, nstat, numstat;
	resstat:=NULL;
	nstat:=nops(statseq);
	for i from nstat by -1 to 1 do
	    stat:=op(i,statseq);
	    if op(0,stat)=R('If') then
		statafter:=resstat;
		numstat:=nops(stat);
		for k from 2 by 2 to numstat do
		    if has(op(k,stat),R('Return')) and
		       not has(op(k,stat),R('If')) then
		    	stat:=subsop(k=`GRAD/Iftransform`(op(k,stat)),stat)
		    else
		    	stat:=subsop(k=`GRAD/Iftransform`(R('StatSeq')(
				     op(op(k,stat)),statafter)),
			  	     stat)
		    fi
		od;
		if type(numstat,'odd') then
		    if has(op(numstat,stat),R('Return'))
		       and not has(op(numstat,stat),R('If')) then
	       	    	stat:=subsop(numstat=
				     `GRAD/Iftransform`(op(numstat,stat)),
			             stat)
		    else
	       	        stat:=subsop(numstat=`GRAD/Iftransform`(
				     R('StatSeq')(op(op(numstat,stat)),
				     		  statafter)),
			             stat)
		    fi
		elif not statafter=NULL then
		     stat:=R('If')(op(stat),R('StatSeq')(statafter))
		fi;
		resstat:=stat
	    else  resstat:=stat,resstat
	    fi
	od;
	R('StatSeq')(resstat)
end:


`GRAD/uniquenames`:=proc(statseq,localslist,localslistpass)
	local 	i, k, stat, numstat, nstat, newname, aname, nameslists,
		localslistloc, resstat;
	resstat:=statseq;
	nstat:=nops(statseq);
	localslistloc:=localslist;
	for i to nstat do
	    stat:=op(i,resstat);
	    if op(0,stat)=R('Assign') then
		aname:=op(1,stat);
		newname:=`GRAD/givename`(aname,localslistloc,
					 'localslistloc');
		resstat:=R('StatSeq')(op(1..i-1,resstat),
			 subsop(1=newname,stat),
			 op(subs(aname=newname,[op(i+1..nstat,resstat)])))
	    elif op(0,stat)=R('If') then
		numstat:=nops(stat);
		nameslists:=array(1..numstat);
		for k from 2 by 2 to numstat do
		  stat:=subsop(k=`GRAD/uniquenames`(
			       op(k,stat),localslistloc,'nameslists[k]'),
			       stat)
		od;
		if type(numstat,'odd') then
		  stat:=subsop(numstat=`GRAD/uniquenames`(
			       op(numstat,stat),localslistloc,
			       'localslistloc'),
			       stat)
		fi;
		for k from 2 by 2 to numstat do
		  localslistloc:=localslistloc union nameslists[k]
		od;
		resstat:=subsop(i=stat,resstat)
	    else resstat:=subsop(i=stat,resstat)
	    fi
	od;
	localslistpass:=localslistloc;
	resstat
end:


# `GRAD/givename`(aname,nameslist,nameslistpass) checks if aname has already
# been defined looking in nameslist and checks if it's not a global 
# variable. If so it will create a new name following the rule
# name -> name.r.<number>  (think of 'r' as "renamed")


`GRAD/givename`:=proc(aname,nameslist,nameslistpass)
	local	newname, newname1, Unameslist, num, len, k, subname;
	Unameslist:=U(nameslist);
	if type(aname,'indexed') then
	   newname:=parse(cat(op(0,aname),op(aname)));
	else newname:=aname
	fi;
	newname1:=U(newname);
	if member(newname1,Unameslist) or 
	   eval(newname1,1)<>eval(newname1,2) then
	   len:=length(newname1);
	   num:=parse(substring(newname1,len..len));
	   k:=len-1;
	   while type(num,'numeric') do
		num:=parse(substring(newname1,k..len));
	  	k:=k-1
	   od;
	   k:=k+1;
	   num:=parse(substring(num,2..len));
	   if substring(newname1,k..k)='r'  and k>1 and num<>NULL then
		subname:=substring(newname1,1..k);
		newname1:=parse(cat(subname,num+1));
		k:=num+1
	   else subname:=parse(cat(newname1,'r'));
		newname1:=parse(cat(subname,0));
		k:=1;
	   fi;
	   while member(newname1,Unameslist) or 		
		 eval(newname1,1)<>eval(newname1,2)  do
	 	      newname1:=parse(cat(subname,k));
		      k:=k+1
	   od
	fi;
	newname1:=R(newname1);
	nameslistpass:=nameslist union {newname1};
	newname1
end:


# `GRAD/canreverse` looks if the statement sequence is reversable
# If it is it gives back the transformed statement sequence in
# statseqpass. In localslistpass you'll have the local variables.


`GRAD/canreverse`:=proc(statseq,statseqpass,localslistpass,prname)
	local statseqloc;
	statseqloc:=`GRAD/expandfor`(statseq);
	if statseqloc='false' then RETURN('false') fi;
	statseqloc:=`GRAD/lastreturn`(statseqloc);
	statseqloc:=`GRAD/Iftransform`(statseqloc);
	statseqpass:=`GRAD/uniquenames`(statseqloc,{prname},
					'localslistpass');
	RETURN('true')
end:


# `GRAD/lastreturn` transforms the last assignment into a return statement
# No for loops are treated (since they have already been expanded.
# It works recursively on if stetements.

`GRAD/lastreturn`:=proc(statseq)
	local nstat, last, head, i, nopsif;
	nstat:=nops(statseq);
	last:=op(nstat,statseq);
	head:=op(0,last);
	if head=R('Return') then RETURN(statseq)
	elif head=R('Assign') then 
	     RETURN(subsop(nstat=R('Return')(op(2,last)),statseq))
	elif head=R('If') then 
	     nopsif:=nops(last);
	     for i from 2 by 2 to nopsif do
		last:=subsop(i=`GRAD/lastreturn`(op(i,last)),last)
	     od;
	     if type(nopsif,odd) then
	        last:=subsop(nopsif=`GRAD/lastreturn`(op(nopsif,last)),last)
	     fi;
	     RETURN(subsop(nstat=last,statseq))
	fi;
	statseq
end:


# `GRAD/revdiff` takes a statement sequence which has only assignments and
# a Return and applies the reverse mode on it.
# x is an array of the parameters w.r.t. which it will derive
# npars is the number of these parameters
# localslist is a list of the local variable which is used to assign
# new names
# localslistpass is the updated localslist
# arraydef is a list of the definitions of the new arrays 
# and arrayass is the assignment of them
# the result is a list (gradient of the Return statement)


`GRAD/revdiff`:=proc(statseq,x,npars,localslist,localslistpass,arraydef,
	      arraydefpass,arrayass,arrayasspass,proctobediff,
	      proctobediffpass,proclist,NT,dfunlist,dfunlistpass)
	local	ret, nstat, assname, assvalue, i, j, dff, dfname, dret,
		notconst, localslistloc, proctobediffloc, dfunlistloc;
	localslistloc:=localslist;
	proctobediffloc:=proctobediff;
	dfunlistloc:=dfunlist;
	nstat:=nops(statseq)-1;	
	ret:=op(op(nstat+1,statseq));
	dret:=array(1..npars);
	if nstat=0 then
	   for i to npars do
		dret[i]:=diff(ret,x[i]);
		dret[i]:=`GRAD/difftoD`(dret[i]);
	    	dret[i]:=`GRAD/Dtonamerev`(dret[i],proclist,NT,
				localslistloc,'localslistloc',
				proctobediffloc,'proctobediffloc',
				dfunlistloc,'dfunlistloc')
	   od;
	   localslistpass:=localslistloc;
	   proctobediffpass:=proctobediffloc;	
	   arraydefpass:=arraydef;
	   arrayasspass:=arrayass;
	   dfunlistpass:=dfunlistloc;
	   RETURN(convert(dret,'list'))
	fi;
	assname:=array(1..nstat);
	assvalue:=array(1..nstat);
	for i to nstat do
	    assname[i]:=op(1,op(i,statseq));	
	    assvalue[i]:=op(2,op(i,statseq));
	od;
	dff:=array(1..nstat);
	dfname:=`GRAD/givename`(R('df'),localslistloc,'localslistloc');
	notconst:=[];
	for i from nstat by -1 to 1 do
	    dff[i]:=convert([seq(dfname[j]*diff(assvalue[j],assname[i]),
				 j=notconst)],`+`) +
		    diff(ret,assname[i]);
	    if dff[i]<>0 then 
		notconst:=[op(notconst),i];
		dff[i]:=`GRAD/difftoD`(dff[i]);
		dff[i]:=`GRAD/Dtonamerev`(dff[i],proclist,NT,localslistloc,
				   'localslistloc',proctobediffloc,
				   'proctobediffloc',dfunlistloc,
				   'dfunlistloc')
	    fi 
	od;
	for i to npars do
	    dret[i]:=diff(ret,x[i])+convert([
		           seq(dfname[j]*diff(assvalue[j],x[i]),
			       j=notconst)],`+`);
	    dret[i]:=`GRAD/difftoD`(dret[i]);
	    dret[i]:=`GRAD/Dtonamerev`(dret[i],proclist,NT,localslistloc,
				'localslistloc',proctobediffloc,
				'proctobediffloc',dfunlistloc,'dfunlistloc')
	od;
	localslistpass:=localslistloc;
	proctobediffpass:=proctobediffloc;
	dfunlistpass:=dfunlistloc;
	arraydefpass:=[op(arraydef),R('Assign')(
					dfname,R('array')(1..nstat,[]))];
	arrayasspass:=[op(arrayass),seq(R('Assign')(dfname[i],
						    dff[i]),
					i=notconst)];
	convert(dret,'list')

end:


# `GRAD/revdifflist` applies recursively revdiff on any kind of list


`GRAD/revdifflist`:=proc(statseq,x,npars,localslist,localslistpass,arraydef,
		  arraydefpass,arrayass,arrayasspass,proctobediff,
		  proctobediffpass,proclist, NT,dfunlist,dfunlistpass)
	local	res, i, ret, statseq0, nstat, localslistloc,
		arrayassloc, arraydefloc, proctobediffloc, dfunlistloc;
	localslistloc:=localslist;
	proctobediffloc:=proctobediff;
	arrayassloc:=arrayass;
	arraydefloc:=arraydef;
	dfunlistloc:=dfunlist;
	nstat:=nops(statseq);
	statseq0:=op(1..nstat-1,statseq);
	ret:=op(op(nstat,statseq));
	res:=[];
	if `GRAD/typeseq`(ret) then ret:=[ret] fi;
	if type(ret,'list') then
	   for i to nops(ret) do
	       res:=[op(res),`GRAD/revdifflist`(R('StatSeq')(
		      statseq0,[ret[i]]),
		      x,npars,localslistloc,'localslistloc',arraydefloc,
		      'arraydefloc',arrayassloc,'arrayassloc',
		      proctobediffloc,'proctobediffloc',proclist,NT,
		      dfunlistloc,'dfunlistloc')]
	   od
	else   res:=`GRAD/revdiff`(statseq,x,npars,localslistloc,
			     	   'localslistloc',
			      	   arraydefloc,'arraydefloc',arrayassloc,
			    	   'arrayassloc',proctobediffloc,
			    	   'proctobediffloc',proclist,NT,
				   dfunlistloc,'dfunlistloc')
	fi;
	localslistpass:=localslistloc;	
	proctobediffpass:=proctobediffloc;
	arraydefpass:=arraydefloc;
	arrayasspass:=arrayassloc;
	dfunlistpass:=dfunlistloc;
	res
end:


# `GRAD/Isconst` controls if the expression depends only on the parameters
# If not then it returns 'true' (it's a constant)


`GRAD/Isconst`:=proc(stat,constlist,constlistpass,locandpars)
	local aname, value;
	aname:=op(1,stat);
	value:=op(2,stat);
	if  (select(type,indets(value),'name')
		minus constlist) intersect locandpars = {} then
		constlistpass:=constlist union {aname};
		RETURN('true')
	fi;
	constlistpass:=constlist;
	RETURN('false')
end:


# `GRAD/dorev` works recursively to form the right sequences to send to
# `GRAD/revdifflist` so that they will be derived.
# It treats only 'Assign', 'if' and 'Return' statements.


`GRAD/dorev`:=proc(statseqinit,statseq,x,npars,localslist,localslistpass,
	    locandpars,constlist,proctobediff,proctobediffpass,proclist,
	    proclistpass,NT,restype,funval)
	local	stattorev, stat, i, numstat, localslistloc,
		arraydef, arrayass, k, nameslist, constlistloc, Rgrd, 
		resstat, proclistloc, func, proctobediffloc, dfunlist,
		lfnamesact, ndfunlist, rules, retvalue;
	localslistloc:=localslist;
	proclistloc:=proclist;
	constlistloc:=constlist;
	proctobediffloc:=proctobediff;
	if has(statseq,R('Return')) then
	  stattorev:=op(statseqinit);
	  resstat:=NULL;
	  for i to nops(statseq) do
	    stat:=op(i,statseq);

				###  Assign  ###

	    if op(0,stat)=R('Assign') then
	       if op(0,op(2,stat))=R('Proc') then
		  func:=op(1,stat);
		  NT(func):=`GRAD/givename`(R(cat(''d'',U(func))),
					    localslistloc,'localslistloc');
		  proclistloc:=proclistloc union {func};
		  stat:=R('Assign')(NT(func),
			 	    `GRAD/reverse`(op(2,stat),[],[],
						   'list',false)),stat
	       elif not `GRAD/Isconst`(stat,constlistloc,'constlistloc',
				       locandpars)
	       then    stattorev:=stattorev,stat
	       fi

				###    If     ###

	    elif op(0,stat)=R('If') then
		numstat:=nops(stat);
		nameslist:=array(1..numstat);
		for k from 2 by 2 to numstat do
		    stat:=subsop(k=`GRAD/dorev`([stattorev],
				 op(k,stat),x,npars,localslistloc,
				 'nameslist[k]',locandpars,constlistloc,
				 proctobediffloc,'proctobediffloc',
				 proclistloc,'proclistloc',NT,restype,
				 funval),
				 stat)
		od;
		if type(numstat,'odd') then
		    stat:=subsop(numstat=`GRAD/dorev`([stattorev],
				 op(numstat,stat),x,npars,localslistloc,
				 'localslistloc',locandpars,constlistloc,
				 proctobediffloc,'proctobediffloc',
				 proclistloc,'proclistloc',NT,restype,
				 funval),
				 stat)
		fi;
		for k from 2 by 2 to numstat do
		    localslistloc:=localslistloc union nameslist[k]
		od

				###  Return  ###

	    elif op(0,stat)=R('Return') then
		retvalue:=op(stat);
		stat:=`GRAD/revdifflist`([stattorev,stat],x,npars,
				  localslistloc,'localslistloc',[],
				  'arraydef',[], 'arrayass',
				  proctobediffloc,'proctobediffloc',
				  proclistloc,NT,{},'dfunlist');
		if dfunlist<>{} then
		   ndfunlist:=nops(dfunlist);
		   lfnamesact:=[seq(`GRAD/givename`(R(cat(''lf'',k)),
				          localslistloc,'localslistloc'),
				     k=1..ndfunlist)];
		   rules:=[seq(dfunlist[k]=lfnamesact[k],k=1..ndfunlist)];
		   stat:=subs(rules,stat);
		   arrayass:=subs(rules,arrayass);
		   arrayass:=[seq(R('Assign')(rhs(rules[k]),lhs(rules[k])),
				  k=1..ndfunlist),
			      op(arrayass)]
		fi;

		if restype='array' then
		   Rgrd:=`GRAD/givename`(R('grd'),localslistloc,
				         'localslistloc');
		   stat:=`GRAD/correctarray`(`GRAD/listtoarray`(stat,[]),
					     Rgrd);
		   # e.g. stat=[Assign(grd[1,1],x),Assign(grd[1,2],y)]
		  if funval then
		   stat:=op(arraydef),op(arrayass),
			 R('Assign')(Rgrd,R('array')(
						`GRAD/findbounds`(stat),[])),
			 op(stat),
			 R('Return')(retvalue,Rgrd)
		  else
		   stat:=op(arraydef),op(arrayass),
			 R('Assign')(Rgrd,R('array')(
						`GRAD/findbounds`(stat),[])),
			 op(stat),
			 R('Return')(Rgrd)
		  fi

		elif restype='seq' or 
			(`GRAD/typeseq`(retvalue) and restype=[]) then
		  if funval then
		   stat:=op(arraydef),op(arrayass),
			  R('Return')(retvalue,`GRAD/listtoseq`(stat))
		  else
		   stat:=op(arraydef),op(arrayass),
			  R('Return')(`GRAD/listtoseq`(stat))
		  fi

		else
		  if funval then
	    	   stat:=op(arraydef),op(arrayass),
			    R('Return')(retvalue,stat)
		  else
	    	   stat:=op(arraydef),op(arrayass),R('Return')(stat)
		  fi
		fi;
		localslistpass:=localslistloc;
		proctobediffpass:=proctobediffloc;
		proclistpass:=proclistloc;
		RETURN(R('StatSeq')(resstat,stat))
	    fi;
	    resstat:=resstat,stat
	  od;
	  localslistpass:=localslistloc;
	  proctobediffpass:=proctobediffloc;
	  proclistpass:=proclistloc;
	  RETURN(R('StatSeq')(resstat))
	else	
	  localslistpass:=localslistloc;
	  proctobediffpass:=proctobediffloc;
	  proclistpass:=proclistloc;
	  RETURN(statseq)
	fi
end:





# `GRAD/Dtonamerev` transforms D[i](f)(args) to df(args)[i] if that df 
# procedure is known.


`GRAD/Dtonamerev`:=proc(expr,proclist,NT,loclist,loclistpass,proctobediff,
			proctobediffpass,dfunlist,dfunlistpass)
	local Dlist, Dfunc, func, justD, reslist, difffunc,proctobediffloc,
	      loclistloc, dfunlistloc, dlocfun;
	loclistloc:=loclist;
	proctobediffloc:=proctobediff;
	dfunlistloc:= dfunlist;
	Dlist:=select(t->has(op(0,t),'D'),indets(expr));
	reslist:={};
	for Dfunc in Dlist do			# Dfunc=D[i](f)(args)
	    func:=op(1,op(0,Dfunc)); 		# func=f
	    if member(func,proclist) then
		dlocfun:=NT(func)(op(Dfunc));
		justD:=op(0,op(0,Dfunc));	# justD=D[i]
		if type(justD,'indexed') then
		   reslist:=reslist union {Dfunc=dlocfun[op(justD)]}
		else reslist:=reslist union {Dfunc=dlocfun[1]}
		fi;
		dfunlistloc:=dfunlistloc union {dlocfun}

	    elif type(U(func),'procedure') then
		difffunc:=`diff/`.U(func);
		if eval(difffunc,2)<>func then
		   if traperror(readlib(difffunc))=lasterror then
			if not assigned(NT(func)) then
			   NT(func):=`GRAD/givename`(
						R(parse(cat('d',U(func)))),
					        loclistloc,'loclistloc')
			fi;
			dlocfun:=NT(func)(op(Dfunc));
			justD:=op(0,op(0,Dfunc));
			if type(justD,'indexed') then
		 	  reslist:=reslist union {Dfunc=dlocfun[op(justD)]}
			else reslist:=reslist union {Dfunc=dlocfun[1]}
			fi;
		    	proctobediffloc:=proctobediffloc union {func};
			dfunlistloc:=dfunlistloc union {dlocfun} 
		   fi
		fi
	    fi
	od;
	loclistpass:=loclistloc;
	proctobediffpass:=proctobediffloc;
	dfunlistpass:=dfunlistloc;
	subs(reslist,expr)
end:

`GRAD/typeseq`:=proc() evalb(nargs>1) end:

`GRAD/expandList`:=proc(typ)
	local op2;
	op2:=op(2,typ);
	if type(op2,specfunc(anything,R('List'))) then
	   subsop(2=op(`GRAD/expandList`(op2)),typ);
	else typ
	fi
end:

macro(  MAKEINTREP = MAKEINTREP,
	MAKEPROC = MAKEPROC,
	R = R,
	U = U,
	FORGET = FORGET):

