#
## <SHAREFILE=science/kinetics/kinetics.mpl >
## <DESCRIBE>
## (update)
##                SEE ALSO: science/kinetics//kinetics.tex
##                (routines: odes, laws, steady)
##                Given a reaction scheme these Maple procedures determine the
##                system of differential equations, the associated conservation
##                laws, and some of the species that have a zero steady state.
##		  SEE ALSO: Maple Technical Newsletter Vol. 1 No. 2, 1994.
##                AUTHOR: Mark Holmes, holmem@rpi.edu
## </DESCRIBE>
## <UPDATE=R4update >

#============================================================
#
#   Given a reaction scheme the following Maple procedures determine the
#   system of differential equations, the associated conservation laws,
#   and some of the species that have a zero steady state. The procedures 
#   can also plot the solutions of the kinetic differential equations. 
# 
#   This program works for chemical reaction schemes containing any number 
#   of reactants and products, and any number of reactions, as long as 
#   computer memory permits.
# 
#   The ideas underlying the algorithms used here can be found in the 
#   Maple Technical Newsletter articles 'An application of Maple to 
#   chemical kinetics,' by M. Holmes and J. Bell (Number 7), 'Maple as an
#   interface for numerically solving chemical kinetic equations,' by
#   J. W. Stayman and Mark H. Holmes (1994, v1, no 2), and the article 
#   'The application of symbolic computing to chemical kinetic reaction 
#   schemes,' by M. Holmes and J. Bell in the Journal of Computational 
#   Chemistry (Dec, 1991).
#
#   Thanks to Jon Bell for helping develop the theory on which the 
#   algorithms are based, and to Michael Monagan and Greg Fee for their 
#   suggestions concerning Maple syntax.
#  
#		by 
#  
#		Mark H. Holmes (holmes@rpi.edu)
#		Joseph (Web) Stayman
#		Yuklun Au
#		Dept of Mathematical Sciences, RPI, Troy, NY  12180
#   
#		last changed 8/30/94
#
#============================================================

kinetics := `See ?kinetics`;

#============================================================
#                procedure odes
#============================================================

odes:=proc()
local i,j,l,part,unit,sparse,t,num,vc,ii,jj,fnd:
global R,Mreacts,Mprods,eqU_subs,species_b,species_c,nreacts,
       species_a,nspecies,cou_nt,nzeros,zeros,ready,species:
options `Copyright 1994 by Mark H. Holmes`;

Mreacts:='Mreacts': 
Mprods:='Mprods':
k.i:='k.i':
rate.i:='rate.i':
eqU_subs:='eqU_subs':
species_b:='species_b':
species_c:='species_c':

# STEP 1:  Determine the variables (i.e., the species in the reaction scheme)

nreacts:=nops(R):
species_a:=indets(R):
nspecies:=nops(species_a):
cou_nt:=1:

# STEP 1.5: Rearrange the species in the order they appear in the reactions

for j from 1 to nspecies do
    species_c[j]:=0:
od:

if(nops(indets(R))=1) then
    species_b[cou_nt]:=op(1,indets(R)):
    cou_nt:=cou_nt+1:
else
    for i from 1 to nops(R) do
	if(nops(indets(R[i]))=1) then 
	    verify(op(1,indets(R[i]))):
	else
	    for j from 1 to nops(R[i]) do
		if(nops(indets(op(j,R[i])))=1) then 
		    verify(op(1,indets(op(j,R[i])))):
  		else
		    for l from 1 to nops(op(j,R[i])) do
			verify(op(1,indets(op(l,op(j,R[i]))))):
		    od:
		fi:
	    od:
	fi:
    od:
fi:
species:=[species_b[ii]$ii=1..nspecies]:

# STEP 2: Construct the Mapping Matrices:  Mreacts[nreacts,nspecies] and 
# 	  Mprods[nreacts,nspecies], whose entries are the stoichiometric 
#	  coefficients of the reactions

Mreacts:=array(sparse,1..nreacts,1..nspecies):
for i from 1 to nreacts do
    part:=op(2,R[i]):
    if ((nops(indets(part)))=1) then 
	if (nops(part)=1) then
	    putreacts(i,part,1):
        else
	    putreacts(i,op(2,part),op(1,part)):
        fi:
    else
        for l from 1 to nops(part) do
	    unit:=op(l,part):
	    if (nops(unit)=1) then
		putreacts(i,unit,1):
            else
		putreacts(i,op(2,unit),op(1,unit)):
	    fi:
        od:
    fi:
od:

Mprods:=array(sparse,1..nreacts,1..nspecies):
for i from 1 to nreacts do
    part:=op(1,R[i]):
    if ((nops(indets(part)))=1) then 
        if (nops(part)=1) then
            putprods(i,part,1):
        else
            putprods(i,op(2,part),op(1,part)):
        fi:
    else
        for l from 1 to nops(part) do
            unit:=op(l,part):
            if (nops(unit)=1) then
                putprods(i,unit,1):
            else
                putprods(i,op(2,unit),op(1,unit)):
            fi:
        od:
    fi:
od:
 
# STEP 3:  Determine the rate of each reaction;
# 	   the rate constant for the ith reaction is:  k.i 

for i from 1 to nreacts do
    rate.i:=k.i:
    for j from 1 to nspecies do
	if (Mreacts[i,j]<>0) then
	    rate.i:=rate.i*(species[j]^Mreacts[i,j]):
	fi:
    od:
od:

# STEP 3.5: Express each RHS as a linear expression (to be used
# 	    by "laws"). This is done by denoting each group of 
#	    species as u.num . To do this the different groups 
#	    must be identified and this is done below. In Step 4 
#	    the linear equations are constructed. 

num:=1:
vc[1]:=num:
for ii from 1 to nreacts do
    fnd:=0:
    for jj from 1 to ii-1 do
        if (rate.ii/k.ii=rate.jj/k.jj)and(fnd=0) then
            vc[ii]:=vc[jj]: fnd:=1:
        fi:
    od:
    if fnd=0 then
        vc[ii]:=num:
        num:=num+1:
    fi:
od: 

# STEP 4: Determine the ODE for Each Species
# 	  The ODE for the species is computed based upon the entries 
#         of the mapping matrices and the rate of the reactions.
#
# 	  Loop over all the species and over all the reactions

for j from 1 to nspecies do
    rhs.j:=0:
    eqU_subs[j]:=0:
    for i from 1 to nreacts do
	if ((Mprods[i,j]-Mreacts[i,j])<>0) then
	   rhs.j:=rhs.j+(Mprods[i,j]-Mreacts[i,j])*rate.i:
	   eqU_subs[j]:=eqU_subs[j]+(u.(vc[i]))*k.i*(Mprods[i,j]-Mreacts[i,j]):
	fi:
    od:
od:

# Print out the differential equations 

for i from 1 to nspecies do
    ode.i:=diff(species[i](t),t)=rhs.i:
od:

if (printlevel>0) then
print();
lprint(`The differential equations are:`);
print();
for i from 1 to nspecies do
    print(ode.i)
od:
print();
fi;

end:

# Procedures : putreacts and putprods
# 	       They are used to assist the entries of the stoichiometric 
#	       coefficients into the mapping matrices

putreacts:=proc(i,check,entry)
local j:
global Mreacts,Mprods,eqU_subs,species_b,species_c,nreacts,
       species_a,nspecies,cou_nt,nzeros,zeros,ready,species:
for j from 1 to nspecies do
    if(check=species[j]) then
	Mreacts[i,j]:=entry:
    fi:
od:
end:

putprods:=proc(i,check,entry)
local j:
global Mreacts,Mprods,eqU_subs,species_b,species_c,nreacts,
       species_a,nspecies,cou_nt,nzeros,zeros,ready,species:
for j from 1 to nspecies do
    if(check=species[j]) then
        Mprods[i,j]:=entry:
    fi:
od:
end:

# Procedure : verify
# 	      It is used in rearranging the species in the correct order.
# 	      It verifies whether a certain species has been arranged. 

verify:=proc(check)
local j:
global Mreacts,Mprods,eqU_subs,species_b,species_c,nreacts,
       species_a,nspecies,cou_nt,nzeros,zeros,ready,species:
for j from 1 to nspecies do
   if(check=species_a[j])and(species_c[j]=0) then
	species_c[j]:=1:
	species_b[cou_nt]:=species_a[j]:
	cou_nt:=cou_nt+1:
   fi:
od:
end:


#============================================================
#             procedure laws
#============================================================

laws:=proc()
local num,X,eqt,cof,i,ii,variables,vars:
global nlaws,Mreacts,Mprods,eqU_subs,species_b,species_c,nreacts,
       species_a,nspecies,cou_nt,nzeros,zeros,ready,species:

# Create the variables u1, u2, ..., u.nreacts for use in genmatrix()

for num from 1 to nreacts do
   vars[num]:=u.num:
od:

# Generate a matrix from the linear equations from odes().
# Find the kernel of the matrix and use the vectors found to
# produce the conservation laws.

eqt:=[eqU_subs[ii]$ii=1..nspecies]:
variables:=[vars[ii]$ii=1..nreacts]:
X:=linalg[transpose](linalg[genmatrix](eqt,variables)): 
cof:=linalg[kernel](X,'nlaws'):

# Generate a report on what conservation laws were found and
# assign conservation laws to law.i, i=1..nlaws

print();
if (nlaws=0) then
  lprint(`    There are no conservation laws.`);
  print();
else
  lprint(`    The conservation laws are:`);
  print();
fi:
for i from 1 to nlaws do
  law.i:=numer(simplify(linalg[dotprod](cof[i],species))):
  lprint(`   `,i*1.,` constant =`,law.i);
od:

end:
 

#============================================================
#             procedure steady
#============================================================

steady:=proc()
local i,ii,j,jj,k,chk,sp,nrhs,nzerosp,zerospos,num:
global Mreacts,Mprods,eqU_subs,species_b,species_c,nreacts,
       species_a,nspecies,cou_nt,nzeros,zeros,ready,species:

# Create a new set of equations to manipulate.

for jj from 1 to nspecies do
  nrhs.jj:=rhs.jj:
od:
nzerosp:=-1:  nzeros:=0:

# Find zero steady states by testing if there is a single 
# species in an equation.  Then replace these species with
# zero values.  Repeat until no more species are found.

while nzeros>nzerosp do
  for i from 1 to nspecies do
    sp:=0: chk:=0:
    for j from 1 to nspecies do
      if member(species[j],indets(nrhs.i))=true then
        sp:=j:  chk:=chk+1:
      fi:
    od:
    if chk=1 then
      zerospos[sp]:=0:
    fi:
  od:
  nzerosp:=nzeros:
  for ii from 1 to nspecies do
    nzeros:=0:
    for j from 1 to nspecies do
      if zerospos[j]=0 then
        nrhs.ii:=subs(species[j]=0,nrhs.ii):
        nzeros:=nzeros+1:
      fi:
    od:
  od:
od:

# Generate a report on what zeros were found and create the
# array zeros[] to hold the zero steady states.

num:=1:
print();
if nzeros=0 then
  lprint(`    No species were found to have a zero steady state.`): 
else
  lprint(`    There are`,nzeros,`species found to have a zero steady state`):
  lprint(`    They are:`):
  for k from 1 to nspecies do
    if zerospos[k]=0 then
      zeros[num]:=species[k]:  num:=num+1:
	lprint(`           `,species[k]):
    fi:
  od:
fi:
print();
end:

#============================================================
#                procedure ksolve
#============================================================

ksolve:=proc()
local eq,sp,c1,_check_,_ksub_:
global goback,Mreacts,Mprods,eqU_subs,species_b,species_c,nreacts,
       readyf,t,species_a,nspecies,cou_nt,nzeros,zeros,ready,species:

# Make the Diff Eqs that will be manipulated
for eq from 1 to nspecies do
   deq.eq:=rhs.eq:
   for sp from 1 to nspecies do
      deq.eq:=subs(species[sp]=species[sp](t),deq.eq):
   od:
   deq.eq:=diff(species[eq](t),t)=deq.eq:
od:
with(plots):
for sp from 1 to nspecies do
   spe.sp:=species[sp]:
od:
for c1 from 1 to nreacts do
   _ksub_.c1:=k.c1;
od:
_check_:=0:
ready:=0:
# Save species and kval information
save _ksub_.(1..nreacts),'kvals':
save spe.(1..nspecies),'species':
save ready,'readyf':
# Run the interface
system(`kpltint&`):
while _check_=0 do
# Wait State
   while ready=0 do
       system(`sleep 2`):
       read readyf:
   od:
# Plotting
   if ready=1 then
       ready:=0:
       save ready,'readyf':
       lprint(` Graphing...`);
       read goback:
       lprint(` Graph will appear momentarily.`);
       # (re)Make the Diff Eqs that will be manipulated
       for eq from 1 to nspecies do
           deq.eq:=rhs.eq:
           for sp from 1 to nspecies do
               deq.eq:=subs(species[sp]=species[sp](t),deq.eq):
           od:
           deq.eq:=diff(species[eq](t),t)=deq.eq:
       od:

   fi:
# Quit Condition
   if ready=2 then
       lprint(` Quitting.`):
       break:
   fi:
od:
# Remove those files we wrote.
system(`rm goback readyf kvals species`):

end:


#============================================================
#				The End
#============================================================





