#
## <SHAREFILE=system/maple2mif/maple2mif.mpl >
## <DESCRIBE>
##       The call maple2mif() will return a procedure that can then
##                be used to convert almost any kind of Maple expression into
##                FrameMaker MIF 4.0 code.  This MIF code can then be saved
##                into a file and opened from within FrameMaker to get a page
##                containing an equation object similar in appearance to
##                Maple's own typeset output.
##                SEE ALSO: ``Denotational Semantics Applied to the Typesetting
##                SEE ALSO: of Maple Expressions'', Reid Pinchback, MSWS '94
##                SEE ALSO: Proceedings, Birkhaueser, 1994.
##                AUTHOR: Reid M. Pinchback, reidmp@MIT.EDU
## </DESCRIBE>
## <UPDATE=R4 >

# =========================================================================
#
# maple2mif
#
# Version date 94/11/02
#
# Developed by Reid M. Pinchback
# Academic Computing Services
# MIT
#
# email: reidmp@mit.edu
# URL: http://web.mit.edu/afs/athena.mit.edu/user/r/e/reidmp/www/home.html
#
# =========================================================================
# Version history
#   Prior to 94/08/01   - The version presented in my MSWS '94 paper.
#   94/08/01            - First distributed version, with improved
#                         MIF page layout specifications.
#   94/11/02            - Added array support, fixed a bug in conversion
#                         of simple fractions, dramatically improved 
#                         output for complicated rational expressions,
#                         added automatic line breaks.  Output is now for
#                         MIF 4.0, not MIF 3.00 as was the case previously.
# =========================================================================
# This software is being provided to you, the LICENSEE, by the 
# Massachusetts Institute of Technology (M.I.T.) under the following 
# license.  By obtaining, using and/or copying this software, you agree 
# that you have read, understood, and will comply with these terms and 
# conditions:  
#
# Permission to use, copy, modify and distribute this software and its 
# documentation for any purpose and without fee or royalty is hereby 
# granted, provided that you agree to comply with the following 
# copyright notice and statements, including the disclaimer, and that 
# the same appear on ALL copies of the software and documentation, 
# including modifications that you make for internal use or for 
# distribution:
#
# Copyright 1994 by the Massachusetts Institute of Technology.  All 
# rights reserved.  
#
# THIS SOFTWARE IS PROVIDED "AS IS", AND M.I.T. MAKES NO 
# REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.  By way of 
# example, but not limitation, M.I.T. MAKES NO REPRESENTATIONS OR 
# WARRANTIES OF  MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE 
# OR THAT THE USE OF THE LICENSED SOFTWARE OR DOCUMENTATION WILL NOT 
# INFRINGE ANY THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER 
# RIGHTS.   
#
# The name of the Massachusetts Institute of Technology or M.I.T. may 
# NOT be used in advertising or publicity pertaining to distribution of 
# the software.  Title to copyright in this software and any associated 
# documentation shall at all times remain with M.I.T., and LICENSEE agrees 
# to preserve same.  
# =========================================================================
# This code is useable as it stands now, but is really intended to serve
# as a template that you may modify it to suit your needs.  My intention
# is to expand the features over time as I come to need them for working on
# the layout of the Maple Technical Newsletter.  I'd also like to add
# some Maple help pages on the next iteration.  Currently the MIF output
# creates one small equation on one page and uses FrameMaker defaults
# for almost everything.  The true FrameMaker hacker will want to
# expand the MIF prolog contained in the sfP routine to tailor the
# design of the output document.
#
# The design of this code as it stands now is documented in the paper
# I submitted to MSWS '94.  It is entitled:
#
#   Denotational Semantics Applied to the Typesetting of Maple Expressions
#
# and published (pages 22-28) in:
#
#   Maple V: Mathematics and Its Application
#   Proceedings of the Maple Summer Workshop and Symposium
#   Rensselaer Polytechnic Institute
#   Troy, NY
#   August 9-13, 1994
#
#   Robert J. Lopez, Editor
#   Birkhauser
#   ISBN: 0-8176-3791-5
#
# Copyright on the paper has been assigned to the publisher, so I don't
# distribute copies, whether printed or electronic.
#
# You don't need to know a lot about FrameMaker or denotational
# semantics to read the paper, but you should understand the basics of
# language parsing and Maple programming.
# =========================================================================

## Here is an example of how to use maple2mif:
#
# read `maple2mif.txt`;
# M := maple2mif();
# X := sin(a+b)^f(pi):
# interface(echo=0);writeto(file);
# printf(`%s`,M(X));
# writeto(terminal);interface(echo=1);
#
## In addition, if you are using a Motif-based interface to
## Maple, replace the line:
#
# printf(`%s`,M(X));
#
## with the following:
#
# print(printf(`%s`,M(X)));
#
## This works around a verified bug in the Motif interface of MVR2 and MVR3,
## which in this case would only result in an empty file being created.
#
## Note that there is a verified printf bug in the Windows version of MVR3
## that may cause Maple to create a General Protection Fault when used
## in this way.  The only workaround I know of at this time is to avoid
## using printf on that platform, and then hand edit the trailing "\"
## characters that will appear at the end of most lines in the output file.
#
## One FrameMaker oddity to note: FrameMaker may have difficulty recognizing
## that a file is a MIF file if Maple hasn't closed the file handle.  The
## easy solution is to either quit Maple or make a copy of the MIF file
## that you have written.  This may relate to how FrameMaker locks files
## opened for editing, but that is just a guess on my part.

maple2mif := proc()
  local env,sfP,sfQ,sfE,sfN,sfT,sfO,sfS,sfI,MIFargs,MIFid,MIFsymbol;
  options `Copyright 1994 by the Massachusetts Institute of Technology`;

  env := {'P'=sfP,'Q'=sfQ,'E'=sfE,'N'=sfN,'T'=sfT,'O'=sfO,'S'=sfS,'I'=sfI,
          'MIFARGS'=MIFargs,'MIFID'=MIFid,'MIFSYMBOL'=MIFsymbol};
      
  # semantic function P
  # defined over the syntactic domain of programs 
  sfP := subs(env, proc() 
    cat(`<MIFFile 4.0> # generated by maple2mif (version date 94/11/02)\n`,
        `<Document\n`,
          `<DBordersOn Yes >\n`,
        `> # end of Document\n`,
        `<AFrames\n`,
          `<Frame\n`,
            `<ID 1>\n`,
            `<BRect  1.0" 1.0" 6.0" 2.0">\n`,
            `<FrameType Below >\n`,
            `<AnchorAlign Center >\n`,
            `<Cropped No >\n`,
            `<Math\n`,
              `<MathFullForm ```,Q(args),`\'>\n`,
              `<MathLineBreak 6.0">\n`,
              `<MathOrigin 3.0" 1.0">\n`,
              `<MathAlignment Center >\n`,
              `<MathSize MathMedium>\n`,
            `> # end of Math\n`,
          `> # end of Frame\n`,
        `> # end of AFrames\n`,
        `<Page\n`,
          `<PageType BodyPage >\n`,
          `<PageSize  8.5" 11.0">\n`,
          `<PageOrientation Portrait >\n`,
        `> # end of Page\n`,
        `<TextFlow\n`,
          `<Para\n`,
            `<ParaLine\n`,
              `<AFrame 1>\n`,
            `> # end of ParaLine\n`,
          `> # end of Para\n`,
        `> # end of TextFlow\n`);
  end); # proc sfP

  # semantic function Q
  # defined over the syntactic domain of expression sequences
  sfQ := subs(env, proc()
    if nargs=0 then
      # NULL exprseq
      E(``);
    elif nargs=1 then
      # single expression
      E(args[1]);
    else # nargs > 1
      # multiple expressions
      cat(`comma[`,MIFARGS(op(map(E,[args]))),`]`);
    fi;
  end); # proc sfQ
  
  # semantic function E
  # defined over the syntactic domain of expressions
  sfE := subs(env, proc()
    local o;
    o := whattype(args);
    if member(o,{'string','function','indexed'}) then
      # the meaning of 'name' expressions
      N(args[1]);
    else
      T(args[1]);
    fi;
  end); # proc sfE
  
  # semantic function N
  # defined over the syntactic domain of typed name expressions 
  # where allowable name operators={string,function,indexed}
  sfN := subs(env, proc()
    local o;
    o := whattype(args[1]);
    if member(o,{`string`}) then
      O(o)(S(args[1]));
    elif member(o,{`function`,`indexed`}) then
      O(o)(N(op(0,args[1])),Q(op(args[1])));
    else
      ERROR(cat(`unsupported expression type, `,o,`, detected`));
    fi
  end); # proc sfN

  # semantic function T
  # defined over the syntactic domain of typed expressions
  # excluding name expressions and any unimplemented types
  sfT := subs(env, proc()
    local o,denom,numer,fract;
    o := whattype(args);
    if member(o,{'`^`','`=`','`..`','`<>`','`<`','`<=`','`and`','`or`'}) then
      # the meaning of expressions built from 
      # type operators : expression x expression -> expression
      O(o)(E(op(1,args[1])),E(op(2,args[1])));
    elif member(o,{'`not`'}) then
      # the meaning of expressions built from
      # type operators : expression -> expression
      O(o)(E(op(o,args[1])));
    elif member(o,{'`+`'}) then
      # the meaning of expressions built from
      # type operators : (expression, expression, ...) -> expression
      O(o)(op(map(E,[op(args[1])])));
    elif member(o,{'`*`'}) then
      # the meaning of expressions built from
      # type operators : (expression, expression, ...) -> expression
      O(o)(op(map(E,[op(args[1])])));

      # construct denominator items
      denom := map(x->op(1,x)^(-op(2,x)),
                   select(x->type(x,'`^`') and signum(op(2,x))=-1,
                          [op(args[1])]));
      if nops(denom)=0 then # do formatting of ordinary product
        O(o)(op(map(E,[op(args[1])])));
      else
        numer := select(x->not(type(x,'`^`') and signum(op(2,x))=-1),
                        [op(args[1])]);
        # separate fractions
        fract := select(x->type(x,'fraction'),numer);
        numer := select(x->not(type(x,'fraction')),numer);
        denom := convert(denom,'`*`');
        numer := convert(numer,'`*`'); # note that numer=[] becomes numer=1
        if type(denom,'`+`') then
          denom := MIFID(E(denom));
        else
          denom := E(denom);
        fi;
        if type(numer,'`+`') then
          numer := MIFID(E(numer));
        else
          numer := E(numer);
        fi;
        # I'm cheating a little bit here, using the fact that conversion
        # of Maple fractions (rational numbers) does part of the job I need
        # for conversion of arbitrary fractional expressions.
        if nops(fract)=0 then
          O('fraction')(numer,denom);
        else
          O(o)(op(map(E,fract)),O('fraction')(numer,denom));
        fi;
      fi;
    elif member(o,{'set','list'}) then
      # the meaning of expressions built from
      # type operators : expression sequence -> expression
      O(o)(Q(op(args[1])));
    elif member(o,{'`.`'}) then
      # the meaning of expressions built from
      # type operators : name x expression -> expression
      O(o)(N(op(1,args[1])),E(op(2,args[1])));
    elif member(o,{'float'}) then
      # the meaning of expressions built from
      # type operators : integer x integer -> expression
      O(o)(I(op(1,args[1])),I(op(2,args[1])));
    elif member(o,{'fraction'}) then
      # the meaning of expressions built from
      # type operators : integer x integer -> expression
      O(o)(E(I(op(1,args[1]))),E(I(op(2,args[1]))));
    elif member(o,{'integer'}) then
      # the meaning of expressions built from
      # type operators : integer -> expression
      O(o)(I(args[1]));
    elif member(o,{'array'})
      and type(args[1],{vector,matrix}) then
      # the meaning of expressions built from
      # type operators : array-structure x hash-table -> expression
      # (with implementation currently restricted to vectors and matrices)
      convert(args[1],matrix); # to change vectors into matrices
      O(o)(I(linalg[rowdim](")),
           I(linalg[coldim](")),
           op(map(E,map(op,convert(",listlist)))));
    else
      ERROR(cat(`unsupported expression type, `,o,`, detected`));
    fi;
  end); # proc sfT
  
  # semantic function O
  # defined over the syntactic domain of type operators 
  # where type operators=list+set+`+`+`*`+ (etc).
  sfO(`string`) := proc() 
    local found;
    if length(args[1])=1 then 
      cat(`char[`,args[1],`]`);
    elif MIFSYMBOL(args[1],'found') then  # to deal with symbol names like
      cat(`char[`,found,`]`);             # aleph, pi, Pi, PI, ...
    else
      cat(`string["`,args[1],`"]`);
    fi;
  end;
  sfO(`function`) := () -> cat(`function[`,MIFARGS(args[1],args[2]),`]`);
  sfO(`indexed`) := () -> cat(`indexes[0,1,`,MIFARGS(args[1],args[2]),`]`);
  sfO('`^`') := () -> cat(`power[`,MIFARGS(args[1],args[2]),`]`);
  sfO('`=`') := () -> () -> cat(`equal[`,MIFARGS(args[1],args[2]),`]`);
  sfO('`..`') := () -> 
    cat(`times[`,MIFARGS(args[1],O('string')('`..`'),args[2]),`]`);
  sfO('`<>`') := () -> cat(`notequal[`,MIFARGS(args[1],args[2]),`]`);
  sfO('`<`') := () -> cat(`lessthan[`,MIFARGS(args[1],args[2]),`]`);
  sfO('`<=`') := () -> cat(`leq[`,MIFARGS(args[1],args[2]),`]`);
  sfO('`and`') := () -> 
    cat(`times[`,MIFARGS(args[1],O('string')('`and`'),args[2]),`]`);
  sfO('`or`') := () ->  
    cat(`times[`,MIFARGS(args[1],O('string')('`or`'),args[2]),`]`);
  sfO('`not`') := () ->
    cat(`times[`,MIFARGS(O('string')('`not`'),args[1]),`]`);
  sfO('`+`') := () -> cat(`plus[`,MIFARGS(args),`]`);
  sfO('`*`') := () -> cat(`times[`,MIFARGS(args),`]`);
  sfO('set') := () -> cat(`id[(*i2i*)`,args[1],`]`); 
  sfO('list') := () -> cat(`id[(*i1i*)`,args[1],`]`);
  sfO('`.`') := () -> 
    cat(`times[`,MIFARGS(args[1],O('string')('`.`'),args[2]),`]`);
  sfO('fraction') := () -> cat(`over[`,MIFARGS(args[1],args[2]),`]`);
  sfO('float') := proc() 
    convert(Float(args[1],args[2]),string);
    cat(`num[`,",`,"`,",`"]`);
  end;
  sfO('integer') := proc() 
    convert(args[1],string);
    cat(`num[`,",`,"`,",`"]`);
  end;
  sfO('array') := proc() 
     cat(`matrix[`,convert(args[1],string),`,`,
                   convert(args[2],string),`,`,
                   MIFARGS(args[3..nargs]),
         `]`);
  end;
  sfO := subs(env, eval(sfO)); # proc sfO

  # semantic function S
  # defined over the syntactic domain of strings
  sfS := subs(env, proc()
    args[1];
  end); # proc sfS

  # semantic function I
  # defined over the syntactic domain of integers
  sfI := subs(env, proc()
    args[1];
  end); # proc sfI

  # MIFargs
  # used to constuct comma-separated MIF argument sequences
  MIFargs := proc()
    convert([args],'string');
    substring(",2..length(")-1);
  end; # proc MIFargs

  # MIFid
  # used to put parentheses around a MIF math object
  MIFid := proc()
    cat(`id[`,args[1],`]`);
  end; # proc MIFid

  # MIFsymbol tries to map Maple symbol 
  # names to corresponding MIF symbol names
  MIFsymbol := proc(s::string,found::name)
    if member(s,{aleph,alpha,beta,chi,delta,Delta,epsilon,eta,gamma,Gamma,
                 Im,iota,kappa,lambda,Lambda,mu,nu,omega,Omega,phi,Phi,
                 pi,Pi,psi,Psi,Re,rho,sigma,Sigma,tau,theta,Theta,
                 upsilon,Upsilon,xi,Xi,zeta}) then
      found := s;
      true;
    else
      false;
    fi
  end; # proc MIFsymbol
  
  # return meta-evaluator
  eval(sfP);

end: # proc maple2mif

