#
# Matrices for the reflection representation of a Coxeter group W.
#
# Let w=[i_1,i_2,...,i_l] be a list of integers (a word in W).
# Let R be the name of a root system for W.
# Let S be a list of base vectors for R.
# Let Basis be a list of independent vectors that span a vector space V
#  containing S.
#
# refl_rep(w,R,Basis) produces the representing matrix for the action of w
#  on V, using coordinates w.r.t. Basis
# refl_rep(w,S,Basis) does the same thing, using S as a base for R
# refl_rep(w,R) does the same thing, using Basis=base(R)
# refl_rep(w,S) does the same thing, using Basis=S
#
refl_rep:=proc(w) local vars,c,v0,col,basis,S,M,v,i;
  if type(args[2],'list') then S:=args[2] else S:=coxeter['base'](args[2]) fi;
  if nargs>2 then basis:=args[3] else basis:=S fi;
  v0:=convert([seq(c[i]*basis[i],i=1..nops(basis))],`+`);
  col:=[seq(c[i],i=1..nops(basis))];
  M:=NULL; vars:=[op(indets(basis))]; 
  for v in basis do;
    collect(v0-coxeter['reflect'](seq(S[i],i=w),v),vars);
    M:=M,subs(solve({coeffs(",vars)}),col);
  od;
  linalg['transpose'](array([M]));
end;
#
# Examples:
#  refl_rep([1,2,3],A3,[e1,e2,e3,e4]);
#  map(refl_rep,[[1],[2],[3]],A3);
#  refl_rep([1,2,3],base(H3,0),[e1,e2,e3]);
#
