with(share):readshare(summatn,algebra):
# file "summation.tst"
#
# Test file for the summation package
# (C) Wolfram Koepf, Gregor Stoelting, 1994, (koepf@zib-berlin.de)
# Version 1.0, December 30, 1994

#read `sum.m`;

TIME:=time():

# 1) Successful summations by the Gosper algorithm
# R. W. Gosper, Jr.:
# Decision procedure for indefinite hypergeometric summation,
# Proc. Nat. Acad. Sci. USA 75 (1978), 40-42.

r := gosper(k^20,k):
if simplify(r-subs(k=k-1,r)-
k^20
)=0
then ok else r fi;
r := gosper((6*k+3)/(4*k^4+8*k^3+8*k^2+4*k+3),k):
if simplify(r-subs(k=k-1,r)-
(6*k+3)/(4*k^4+8*k^3+8*k^2+4*k+3)
)=0
then ok else r fi;
r := gosper(2^k*(k^3-3*k^2-3*k-1)/(k^3*(k+1)^3),k):
if simplify(r-subs(k=k-1,r)-
2^k*(k^3-3*k^2-3*k-1)/(k^3*(k+1)^3)
)=0
then ok else r fi;
r := gosper(k*x^k,k):
if simplify(r-subs(k=k-1,r)-
k*x^k
)=0
then ok else r fi;
r := gosper(k*factorial(k),k):
if simplify_combinatorial(r-subs(k=k-1,r)-
k*factorial(k)
)=0
then ok else r fi;
r := gosper(1/(k^2-1),k):
if simplify(r-subs(k=k-1,r)-
1/(k^2-1)
)=0
then ok else r fi;
r := gosper((1+2*k)/((1+k^2)*(k^2+2*k+2)),k):
if simplify(r-subs(k=k-1,r)-
(1+2*k)/((1+k^2)*(k^2+2*k+2))
)=0
then ok else r fi;
r := gosper((k^2+4*k+1)*factorial(k),k):
if simplify_combinatorial(r-subs(k=k-1,r)-
(k^2+4*k+1)*factorial(k)
)=0
then ok else r fi;
r := gosper((4*k-3)*factorial(2*k-2)/factorial(k-1),k):
if simplify_combinatorial(r-subs(k=k-1,r)-
(4*k-3)*factorial(2*k-2)/factorial(k-1)
)=0
then ok else r fi;
r := gosper(GAMMA(k+n+2)*n/((k+n+1)*GAMMA(k+2)*GAMMA(n+1)),k):
if simplify_combinatorial(r-subs(k=k-1,r)-
GAMMA(k+n+2)*n/((k+n+1)*GAMMA(k+2)*GAMMA(n+1))
)=0
then ok else r fi;
r := gosper((3*(1+2*k))/((1+k^2)*(2+2*k+k^2)),k):
if simplify(r-subs(k=k-1,r)-
(3*(1+2*k))/((1+k^2)*(2+2*k+k^2))
)=0
then ok else r fi;
r := gosper((-25+15*k+18*k^2-2*k^3-k^4)/
(-23+479*k+613*k^2+137*k^3+53*k^4+5*k^5+k^6),k):
if simplify(r-subs(k=k-1,r)-
(-25+15*k+18*k^2-2*k^3-k^4)/
(-23+479*k+613*k^2+137*k^3+53*k^4+5*k^5+k^6)
)=0
then ok else r fi;
r := gosper(3^k*(2*k^4+4*k^3-7*k^2-k-4)/(k*(k+1)*(k^2+1)*((k+1)^2+1)),k):
if simplify_combinatorial(r-subs(k=k-1,r)-
3^k*(2*k^4+4*k^3-7*k^2-k-4)/(k*(k+1)*(k^2+1)*((k+1)^2+1))
)=0
then ok else r fi;
r := gosper(3^k*(4*k^2-2*k-3)/((2*k+3)*(2*k+1)*(k+1)*k),k):
if simplify(r-subs(k=k-1,r)-
3^k*(4*k^2-2*k-3)/((2*k+3)*(2*k+1)*(k+1)*k)
)=0
then ok else r fi;
r := gosper(2^k*(2*k^3+3*k^2-20*k-15)/
((2*k+3)*(2*k+1)*(k+5)*(k+4)*(k+1)*k),k):
if simplify(r-subs(k=k-1,r)-
2^k*(2*k^3+3*k^2-20*k-15)/
((2*k+3)*(2*k+1)*(k+5)*(k+4)*(k+1)*k)
)=0
then ok else r fi;
r := gosper(2^k*((k+11)^2*(k+1)^2-2*(k+10)^2*k^2)/
((k+11)^2*(k+10)^2*(k+1)^2*k^2),k):
if simplify(r-subs(k=k-1,r)-
2^k*((k+11)^2*(k+1)^2-2*(k+10)^2*k^2)/
((k+11)^2*(k+10)^2*(k+1)^2*k^2)
)=0
then ok else r fi;
r := gosper(2^k*((k+6)^2*(k+1)^2-2*(k+5)^2*k^2)/
((k+6)^2*(k+5)^2*(k+1)^2*k^2),k):
if simplify(r-subs(k=k-1,r)-
2^k*((k+6)^2*(k+1)^2-2*(k+5)^2*k^2)/
((k+6)^2*(k+5)^2*(k+1)^2*k^2)
)=0
then ok else r fi;
r := gosper(2^k*(k^4-14*k^2-24*k-9)/(k^2*(k+1)^2*(k+2)^2*(k+3)^2),k):
if simplify(r-subs(k=k-1,r)-
2^k*(k^4-14*k^2-24*k-9)/(k^2*(k+1)^2*(k+2)^2*(k+3)^2)
)=0
then ok else r fi;
r := gosper(((k^2-k-n^2-n-2)*GAMMA(k+n+2)*GAMMA(n+1))/
(2*(-1)^k*2^k*(k+n+1)*GAMMA(-(k-n-1))*GAMMA(k+2)),k):
if simplify_combinatorial(r-subs(k=k-1,r)-
((k^2-k-n^2-n-2)*GAMMA(k+n+2)*GAMMA(n+1))/
(2*(-1)^k*2^k*(k+n+1)*GAMMA(-(k-n-1))*GAMMA(k+2))
)=0
then ok else r fi;
r := gosper(1/(k+1)*binomial(2*k,k)/(n-k+1)*binomial(2*n-2*k,n-k),k):
if simplify_combinatorial(r-subs(k=k-1,r)-
1/(k+1)*binomial(2*k,k)/(n-k+1)*binomial(2*n-2*k,n-k)
)=0
then ok else r fi;
r := gosper(3^k*(4*k^2+2*a*k-4*k-2-a)/((2*k+2+a)*(2*k+a)*(k+1)*k),k):
if simplify(r-subs(k=k-1,r)-
3^k*(4*k^2+2*a*k-4*k-2-a)/((2*k+2+a)*(2*k+a)*(k+1)*k)
)=0
then ok else r fi;
r := gosper(2^k*(k^2-2*k-1)/(k^2*(k+1)^2),k):
if simplify(r-subs(k=k-1,r)-
2^k*(k^2-2*k-1)/(k^2*(k+1)^2)
)=0
then ok else r fi;
r := gosper((3*k^2+3*k+1)/(k^3*(k+1)^3),k):
if simplify(r-subs(k=k-1,r)-
(3*k^2+3*k+1)/(k^3*(k+1)^3)
)=0
then ok else r fi;
r := gosper((6*k+3)/(4*k^4+8*k^3+8*k^2+4*k+3),k):
if simplify(r-subs(k=k-1,r)-
(6*k+3)/(4*k^4+8*k^3+8*k^2+4*k+3)
)=0
then ok else r fi;
r := gosper((k^2+3*k+3)/(k^4+2*k^3-3*k^2-4*k+2),k):
if simplify(r-subs(k=k-1,r)-
(k^2+3*k+3)/(k^4+2*k^3-3*k^2-4*k+2)
)=0
then ok else r fi;
r := gosper(k^2*4^k/((k+1)*(k+2)),k):
if simplify(r-subs(k=k-1,r)-
k^2*4^k/((k+1)*(k+2))
)=0
then ok else r fi;
r := gosper(2^k*(k^3-3*k^2-3*k-1)/(k^3*(k+1)^3),k):
if simplify(r-subs(k=k-1,r)-
2^k*(k^3-3*k^2-3*k-1)/(k^3*(k+1)^3)
)=0
then ok else r fi;
r := gosper(3^k*(2*k^3+k^2+3*k+6)/((k^2+2)*(k^2+2*k+3)),k):
if simplify(r-subs(k=k-1,r)-
3^k*(2*k^3+k^2+3*k+6)/((k^2+2)*(k^2+2*k+3))
)=0
then ok else r fi;
r := gosper(4*(1-k)*(k^2-2*k-1)/(k^2*(k+1)^2*(k-2)^2*(k-3)^2),k):
if simplify(r-subs(k=k-1,r)-
4*(1-k)*(k^2-2*k-1)/(k^2*(k+1)^2*(k-2)^2*(k-3)^2)
)=0
then ok else r fi;
r := gosper(2^k*(k^4-14*k^2-24*k-9)/(k^2*(k+1)^2*(k+2)^2*(k+3)^2),k):
if simplify(r-subs(k=k-1,r)-
2^k*(k^4-14*k^2-24*k-9)/(k^2*(k+1)^2*(k+2)^2*(k+3)^2)
)=0
then ok else r fi;
r := gosper((1+k)/(1-k)+2/k,k):
if simplify(r-subs(k=k-1,r)-
((1+k)/(1-k)+2/k)
)=0
then ok else r fi;
r := gosper(1/(k*(k+2)),k):
if simplify(r-subs(k=k-1,r)-
1/(k*(k+2))
)=0
then ok else r fi;
r := gosper(1/(k*(k+1)*(k+2)),k):
if simplify(r-subs(k=k-1,r)-
1/(k*(k+1)*(k+2))
)=0
then ok else r fi;
r := gosper(pochhammer(k-n,n),k):
if simplify_combinatorial(r-subs(k=k-1,r)-
pochhammer(k-n,n)
)=0
then ok else r fi;
r := gosper((a+k-1)*pochhammer(a,k),k):
if simplify_combinatorial(r-subs(k=k-1,r)-
(a+k-1)*pochhammer(a,k)
)=0
then ok else r fi;
r := gosper((a-k-1)/pochhammer(a-k,k),k):
if simplify_combinatorial(r-subs(k=k-1,r)-
(a-k-1)/pochhammer(a-k,k)
)=0
then ok else r fi;
r := gosper(binomial(k,n),k):
if simplify_combinatorial(r-subs(k=k-1,r)-
binomial(k,n)
)=0
then ok else r fi;
r := gosper(1/binomial(k,n),k):
if simplify_combinatorial(r-subs(k=k-1,r)-
1/binomial(k,n)
)=0
then ok else r fi;
r := gosper(binomial(k-n,k),k):
if simplify_combinatorial(r-subs(k=k-1,r)-
binomial(k-n,k)
)=0
then ok else r fi;
r := gosper((-1)^k*binomial(n,k),k):
if simplify_combinatorial(r-subs(k=k-1,r)-
(-1)^k*binomial(n,k)
)=0
then ok else r fi;
r := gosper((-1)^k/binomial(n,k),k):
if simplify_combinatorial(r-subs(k=k-1,r)-
(-1)^k/binomial(n,k)
)=0
then ok else r fi;
r := gosper((-1)^(k+1)*(4*k+1)*factorial(2*k)/
(factorial(k)*4^k*(2*k-1)*factorial(k+1)),k):
if simplify_combinatorial(r-subs(k=k-1,r)-
(-1)^(k+1)*(4*k+1)*factorial(2*k)/
(factorial(k)*4^k*(2*k-1)*factorial(k+1))
)=0
then ok else r fi;

# 2) Examples for the Wilf-Zeilberger method:
# H. S. Wilf and D. Zeilberger:
# Rational functions certify combinatorial identities.
# J. Amer. Math. Soc. 3, 1990, 147-158.
# 
# Extension and examples from
# W. Koepf:
# Algorithms for the indefinite and definite summation.
# Konrad-Zuse-Zentrum Berlin (ZIB), Preprint SC 94-33, 1994.

# WZ:=proc(summand,k,n)
# local tmp,gos,m,l;
#    if nargs>3 then m:=args[4] else m:=1 fi;
#    if nargs>4 then l:=args[5] else l:=1 fi;
#    tmp:=summand-subs(n=n-m,summand);
#    gos:=extended_gosper(tmp,k,l):
#    RETURN(simplify_combinatorial(gos/summand))
# end:

# W. N. Bailey: 
# Generalized hypergeometric series. Cambridge University Press, England, 
# reprinted 1964 by Stechert-Hafner Service Agency, New York-London (1935).

# Bailey, p. 3:
# (Vandermonde-Gauss)

r := WZ(hyperterm([-n,b],[c],1,k)/hyperterm([c-b,1],[c],1,n),k,n):
if simplify(r-
(b+k)*(n-k)/n/(c+n-1)
)=0
then ok else r fi;

# Bailey, p. 9, (1):
# (Saalsch\"utz)

r := WZ(hyperterm([a,b,-n],[c,1+a+b-c-n],1,k)/
hyperterm([c-a,c-b,1],[c,c-a-b],1,n),k,n):
if simplify(r-
(b+k)*(n-k)*(a+k)/(1+a+b-c-n+k)/(c+n-1)/n
)=0
then ok else r fi;

# Bailey, p. 9, (1):
# (Kummer)

r := WZ(hyperterm([a,-n],[1+a+n],-1,k)/hyperterm([1+a,1],[1+a/2],1,n),k,n):
if simplify(r+
(a+k)*(n-k)/n/(2*n+a)
)=0
then ok else r fi;

# Bailey, p. 11, (2):
# (Gauss)

r := WZ(hyperterm([-n,b],[1/2*(1-n+b)],1/2,k)/
(GAMMA(1/2)*GAMMA(1/2-n/2+b/2))*GAMMA(1/2-n/2)*GAMMA(1/2+b/2),k,n,2):
if simplify(r+
(n-k)*(b+k)/(-1+n-b-2*k)/n
)=0
then ok else r fi;

# Bailey, p. 11, (3):
# (Bailey)

r := WZ(hyperterm([-n,1+n],[c],1/2,k)/
(GAMMA(c/2)*GAMMA(c/2+1/2))*GAMMA(c/2-n/2)*GAMMA(1/2+c/2+n/2),k,n,2):
if simplify(r-
(-1+2*n)*(n-k)/(c+n-1)/(k+n)
)=0
then ok else r fi;

# Bailey, p. 13 (1):
# (Dixon)

r := WZ(hyperterm([a,b,-n],[1+a-b,1+a+n],1,k)/
hyperterm([1+a,1+a/2-b,1],[1+a/2,1+a-b],1,n),k,n):
if simplify(r-
(a+k)*(n-k)*(b+k)/n/(2*n+a)/(a-b+n)
)=0
then ok else r fi;

# Bailey, p. 16 (1):
# (Watson)

r := WZ(hyperterm([-n,b,c],[1/2*(1-n+b),2*c],1,k)/
(GAMMA(1/2)*GAMMA(1/2+c)*GAMMA(1/2-n/2+b/2)*GAMMA(1/2+n/2-b/2+c))*
GAMMA(1/2-n/2)*GAMMA(1/2+b/2)*GAMMA(1/2+n/2+c)*GAMMA(1/2-b/2+c),k,n,2):
if simplify(r+
2*(n-k)*(b+k)*(c+k)/(-1+n-b-2*k)/(-1+n+2*c)/n
)=0
then ok else r fi;

# Bailey, p. 16 (1):
# (Whipple)

r := WZ(hyperterm([-n,1+n,c],[e,1+2*c-e],1,k)/(Pi*GAMMA(e)*GAMMA(1+2*c-e))*
2^(2*c-1)*GAMMA(-n/2+e/2)*GAMMA(-n/2+(1+2*c-e)/2)*GAMMA((1+n)/2+e/2)*
GAMMA((1+n)/2+(1+2*c-e)/2),k,n,2):
if simplify(r-
2*(-1+2*n)*(n-k)*(c+k)/(2*c-e+n)/(n-1+e)/(k+n)
)=0
then ok else r fi;

# Bailey, p. 25 (3):
# (Whipple)

r := WZ(hyperterm([a,1+a/2,c,d,-n],[a/2,1+a-c,1+a-d,1+a+n],1,k)/
hyperterm([1+a,1+a-c-d,1],[1+a-c,1+a-d],1,n),k,n):
if simplify(r-
(d+k)*(c+k)*(n-k)*(a+k)/(2*k+a)/n/(a-c+n)/(a-d+n)
)=0
then ok else r fi;

# Bailey, p. 26 (5):
# (Dougall)

#r := WZ(hyperterm([a,1+a/2,b,c,d,1+2*a-b-c-d+n,-n],
#[a/2,1+a-b,1+a-c,1+a-d,1+a-(1+2*a-b-c-d+n),1+a+n],1,k)/
#hyperterm([1+a,1+a-b-c,1+a-b-d,1+a-c-d,1],
#[1+a-b,1+a-c,1+a-d,1+a-b-c-d],1,n),k,n):

# Bailey, p. 27 (1):
# (Dixon-Dougall)

r := WZ(hyperterm([a,1+a/2,c,d,-n],[a/2,1+a-c,1+a-d,1+a+n],1,k)/
hyperterm([1+a-c-d,1+a,1],[1+a-c,1+a-d],1,n),k,n):
if simplify(r-
(d+k)*(c+k)*(n-k)*(a+k)/(2*k+a)/n/(a-c+n)/(a-d+n)
)=0
then ok else r fi;

# Bailey, p. 28 (3):

r := WZ(hyperterm([a,1+a/2,d,-n],[a/2,1+a-d,1+a+n],-1,k)/
hyperterm([1+a,1],[1+a-d],1,n),k,n):
if simplify(r+
(d+k)*(n-k)*(a+k)/(2*k+a)/n/(a-d+n)
)=0
then ok else r fi;

# Bailey, p. 30 (1.1):

r := WZ(hyperterm([a,1+a/2,-n],[a/2,w],1,k)/
hyperterm([w-a,1],[w],1,n)/(w-a-1-n)*(w-a-1+n),k,n):
if simplify(r-
(a^2+a*n-w*a+2*a-2*w+2+2*a*k-2*k*w+2*k+2*k*n)*(n-k)*(a+k)/(2*k+a)/(-w+a+n)/n/(w
+n-1)
)=0
then ok else r fi;

# Bailey, p. 30 (1.2):

r := WZ(hyperterm([a,b,-n],[1+a-b,1+2*b-n],1,k)/
hyperterm([a-2*b,1+a/2-b,-b,1],[1+a-b,a/2-b,-2*b],1,n),k,n):
if simplify(r+
(-2*b^2+a*b-2*b+2*b*n-1+n-k)*(a+k)*(n-k)*(b+k)/b/(-1-2*b+n-k)/(a-2*b+2*n-2)/(a
-b+n)/n
)=0
then ok else r fi;

# Bailey, p. 30 (1.3):

r := WZ(hyperterm([a,1+a/2,b,-n],[a/2,1+a-b,1+2*b-n],1,k)/
hyperterm([a-2*b,-b,1],[1+a-b,-2*b],1,n),k,n):
if simplify(r+
(a+k)*(n-k)*(b+k)*(2*b-n+1+a*b+2*b*k+k)/b/(2*k+a)/(-1-2*b+n-k)/(a-b+n)/n
)=0
then ok else r fi;

# Bailey, p. 30 (1.4):

r := WZ(hyperterm([a,1+a/2,b,-n],[a/2,1+a-b,2+2*b-n],1,k)/
hyperterm([a-2*b-1,1/2+a/2-b,-b-1,1],[1+a-b,a/2-b-1/2,-2*b-1],1,n),k,n):
if simplify(r+
(-4+6*b*n-2*a*b^2+a^2*b+2*a*b*n-4*b^2-8*b+6*n-2*n^2-a*b+4*b*n*k+4*k*n+2*a*b*k-
4*b^2*k-6*k-8*b*k-2*k^2)*(b+k)*(n-k)*(a+k)/b/(2*k+a)/(-2-2*b+n-k)/(a-2*b-3+2*n)
/(a-b+n)/n
)=0
then ok else r fi;

# I. Gessel, D. Stanton: 
# Strange evaluations of hypergeometric series. 
# Siam J. Math. Anal. 13 (1982), 295-308.

# Gessel-Stanton, p. 295 (1.1)

r := WZ(hyperterm([-n,n+3*a,a],[3/2*a,(3*a+1)/2],3/4,k)/
(factorial(n)*pochhammer(a+1,n/3)/(factorial(n/3)*pochhammer(3*a+1,n))),k,n,3):
if simplify(r-
3*(a+k)*(n-k)*(2*n-3+3*a)/(n+3*a+k-2)/(n+3*a+k-1)/n
)=0
then ok else r fi;

# Gessel-Stanton, p. 295 (1.2)

r := WZ(hyperterm([2*a,2*b,1-2*b,1+2*a/3,-n],
[a-b+1,a+b+1/2,2/3*a,1+2*a+2*n],1/4,k)/
hyperterm([a+1/2,a+1,1],[a+b+1/2,a-b+1],1,n),k,n):
if simplify(r+
1/2*(2*b+k)*(2*b-1-k)*(n-k)*(2*a+k)/(2*a+3*k)/(a-b+n)/(2*a+2*b-1+2*n)/n
)=0
then ok else r fi;

# Gessel-Stanton, p. 295 (1.3)

r := WZ(hyperterm([a,b,a+1/2-b,1+2*a/3,-n],
[2*a+1-2*b,2*b,2/3*a,1+a+n/2],4,k)/
(factorial(n)*2^(-n)/factorial(n/2))/
hyperterm([a+1,1],[a-b+1,b+1/2],1,n/2),k,n,2):
if simplify(r-
4*(n-k)*(2*a-2*b+2*k+1)*(a+k)*(b+k)/(2*a+3*k)/(2*b-1+n)/(2*a-2*b+n)/n
)=0
then ok else r fi;

# Gessel-Stanton, p. 295 (1.4), misprint.

r := WZ(hyperterm([1/2+3*a,1/2-3*a,-n],[1/2,-3*n],3/4,k)/
hyperterm([1/2-a,1/2+a,1],[1/3,2/3],1,n),k,n):
if simplify(r-
3/4*(6*a+2*k+1)*(6*a-2*k-1)*(n-k)/(3*n-k)/(3*n-1-k)/(3*n-2-k)
)=0
then ok else r fi;

# Gessel-Stanton, p. 295 (1.5)

r := WZ(hyperterm([1+3*a,1-3*a,-n],[3/2,-1-3*n],3/4,k)/
hyperterm([1+a,1-a,1],[2/3,4/3],1,n),k,n):
if simplify(r-
3*(3*a+1+k)*(3*a-k-1)*(n-k)/(3*n-k+1)/(3*n-k)/(3*n-1-k)
)=0
then ok else r fi;

# Gessel-Stanton, p. 295 (1.6)

r := WZ(hyperterm([2*a,1-a,-n],[2*a+2,-a-1/2-3/2*n],1,k)/
((n+1)*(2*a+1)/(2*a+n+1))/
hyperterm([(n+3)/2,1],[1+1/2*(n+2*a+1)],1,n),k,n,2):
if simplify(r-
2*(-2*a+2*a*n+9*n^2-10*n+1-8*n*k)*(-k+n)*(a-k-1)*(2*a+k)/(2*a+1+3*n-2*k)/(2*a
-1+3*n-2*k)/(2*a-3+3*n-2*k)/(-1+n)/n
)=0
then ok else r fi;

# Gessel-Stanton, p. 296 (1.7)

#r := WZ(hyperterm([2*a,2*b,1-2*b,1+2/3*a,a+d+n+1/2,a-d,-n],
#[a-b+1,a+b+1/2,2/3*a,-2*d-2*n, 2*d+1,1+2*a+2*n],1,k)/
#hyperterm([2*a+1,1],[2*d+1],1,2*n)/
#hyperterm([b+d+1/2,d-b+1,1],[a+b+1/2,a-b+1],1,n),k,n):

# Gessel-Stanton, p. 296 (1.8)

#r := WZ(hyperterm([a,b,a+1/2-b,1+2*a/3,1-2*d,2*a+2*d+n,-n],
#[2*a+1-2*b,2*b,2/3*a,a+d+1/2,1-d-n/2,1+a+n/2],1,k)/
#(factorial(n)*2^(-n)/factorial(n/2))/
#hyperterm([b+d,d-b+a+1/2,a+1,1],[b+1/2,a+d+1/2,d,a-b+1],1,n/2),k,n,2):

# Gessel-Stanton, p. 296 (1.9), works only for specified s

#r := seq(WZ(hyperterm([-s*b+s+1,b-1,-n],[b+1,s*(-n-b)-n],1,k)/
#(b*(n+1)/(b+n))/hyperterm([1+s+s*n,1],[1+s*b+s*n],1,n),k,n)
#,s=1..4):
#r := WZ(hyperterm([-1/2*b+1/2+1,b-1,-n],[b+1,1/2*(-n-b)-n],1,k)/
#(b*(n+1)/(b+n))/hyperterm([1+1/2+1/2*n,1],[1+1/2*b+1/2*n],1,n),k,n,2):

# Gessel-Stanton, p. 298 (3.7)

r := WZ(hyperterm([-n,-2*n-2/3],[4/3],-8,k)/
hyperterm([5/6,1],[3/2],-27,n),k,n):
if simplify(r-
4*(7*n-1-3*k)*(n-k)*(6*n+2-3*k)/(1+2*n)/(3*n+1)/n
)=0
then ok else r fi;

# Gessel-Stanton, p. 305, (5.21)

r := WZ(hyperterm([3*a+1/2,3*a+1,-n],[6*a+1,-n/3+2*a+1],4/3,k)/
hyperterm([1/3,2/3,1],[1+2*a,-2*a],1,n/3),k,n,3):
if simplify(r-
2*(n-k)*(6*a+1+2*k)*(3*a+1+k)/(-n+6*a+3+3*k)/(6*a+n)/n
)=0
then ok else r fi;

# Gessel-Stanton, p. 306 (5.22)

r := WZ(hyperterm([-n,1/2],[2*n+3/2],1/4,k)/
hyperterm([1/2,1],[2*n+3/2],27/4,n),k,n):
if simplify(r-
1/4*(5+6*k)*(2*k+1)*(n-k)/(6*n-1)/(6*n+1)/n
)=0
then ok else r fi;

# Gessel-Stanton, p. 306 (5.23)

r := WZ(hyperterm([-n,-1/3-2*n],[2/3],-8,k)/(-27)^n,k,n):
if simplify(r-
4*(6*n-3*k+1)*(n-k)*(21*n-7-9*k)/n/(3*n-1)/(6*n+1)
)=0
then ok else r fi;

# Gessel-Stanton, p. 306 (5.24)

r := WZ(hyperterm([-n,n/2+1],[4/3],8/9,k)/
hyperterm([1/2,1],[7/6],-1/3,n/2),k,n,2):
if simplify(r-
4*(n-k)/(1+3*n)
)=0
then ok else r fi;

# Gessel-Stanton, p. 306 (5.25)

r := WZ(hyperterm([-n,1/2],[(n+3)/2],4,k)/
hyperterm([1/2,3/2,1],[5/6,7/6],1,n/2),k,n,2):
if simplify(r-
4*(2*k+1)*(n-k)*(2+3*k)/(1+3*n)/(3*n-1)/n
)=0
then ok else r fi;

# Gessel-Stanton, p. 306 (5.27)

r := WZ(hyperterm([1/3-n,-n/2,(1-n)/2,22/21-3*n/7],
[5/6,4/3,1/21-3*n/7],-27,k)/((-8)^n/(1-9*n)),k,n):
if simplify(r-
81*(3*n-1-3*k)*(n-2*k-1)*(n-2*k)/(-1+9*n-21*k)/n/(3*n-1)
)=0
then ok else r fi;

# Gessel-Stanton, p. 307 (6.2)

r := WZ(hyperterm([a+1/2,a,b,1-b,-n,(2*a+1)/3+n,a/2+1],
[1/2,(2*a-b+3)/3,(2*a+b+2)/3,-3*n,2*a+1+3*n,a/2],1,k)/
hyperterm([(2*a+2)/3,2*a/3+1,(1+b)/3,(2-b)/3,1],
[(2*a-b)/3+1,(2*a+b+2)/3,2/3,1/3],1,n),k,n):
if simplify(r-
6*(a+3*n-1)*(a+k)*(2*a+2*k+1)*(n-k)*(b-1-k)*(b+k)/(2*k+a)/(3*n-k)/(3*n-1-k)/(3*
n-2-k)/(2*a+b-1+3*n)/(2*a-b+3*n)
)=0
then ok else r fi;

# Gessel-Stanton, p. 307 (6.3)

r := WZ(hyperterm([a+1/2,a,-n,(2*a+1)/3+n,a/2+1],
[1/2,-3*n,2*a+1+3*n,a/2],9,k)/
hyperterm([(2*a+2)/3,2*a/3+1,1],[1/3,2/3],1,n),k,n):
if simplify(r+
6*(a+3*n-1)*(a+k)*(2*a+2*k+1)*(n-k)/(2*k+a)/(3*n-k)/(3*n-1-k)/(3*n-2-k)
)=0
then ok else r fi;

# Gessel-Stanton, p. 307 (6.5)

r := WZ(hyperterm([-n,-n+1/4],[2*n+5/4],1/9,k)/pochhammer(5/4,2*n)/
hyperterm([1],[2/3,13/12],1,n)*3^(5*n)/2^(6*n),k,n):
if simplify(r+
1/27*(4*n-4*k-1)*(n-k)*(52*n^2-13*n-21+16*n*k-56*k-32*k^2)/(1+12*n)/(3*n-1)/(4
*n-1)/n
)=0
then ok else r fi;

# Gessel-Stanton, p. 307 (6.6)

r := WZ(hyperterm([-n,-n+1/4],[2*n+9/4],1/9,k)/pochhammer(9/4,2*n)/
hyperterm([1],[4/3,17/12],1,n)*3^(5*n)/2^(6*n),k,n):
if simplify(r+
1/27*(4*n-4*k-1)*(n-k)*(52*n^2+39*n-55+16*n*k-84*k-32*k^2)/(5+12*n)/(1+3*n)/(4
*n-1)/n
)=0
then ok else r fi;

r := WZ(binomial(n,k)*binomial(k,n)*(-1)^k/(-1)^n,k,n):
if simplify(r+
(n-k)/(n-1-k)
)=0
then ok else r fi;

r := WZ(binomial(n,k)*binomial(2*k,n)*(-1)^k/(-2)^n,k,n):
if simplify(r+
2*(2*k+1)*(n-k)/(n-2*k-1)/n
)=0
then ok else r fi;

r := WZ(binomial(n,k)*binomial(k/2,n)*(-1)^k*(-2)^n,k,n,1,2):
if simplify(r-
(n-k)*(n-1-k)/(-k+2*n-2)/(n-1)
)=0
then ok else r fi;

r := WZ(binomial(n,k)*binomial(k/2,n)*(-2)^n,k,n,1,2):
if simplify(r-
(n-k)*(n-1-k)/(-k+2*n-2)/(n-1)
)=0
then ok else r fi;

# Clausen

r := WZ(hyperterm([a,b,1/2-a-b-n,-n],[1/2+a+b,1-a-n,1-b-n],1,k)/
hyperterm([2*a,a+b,2*b,1],[2*a+2*b,a,b],1,n),k,n):
if simplify(r+
(2*a+3*n+2*b-3-2*k)*(2*a+2*n+2*b-1-2*k)*(a+k)*(n-k)*(b+k)/(-1+b+n-k)/(-1+a+n-k
)/(2*a+2*b+n-1)/n/(-1+2*a+2*b+2*n)
)=0
then ok else r fi;

# 4) Examples for which gosper decides that no hypergeometric term
# antidifference exists

r:='r':
r := gosper(factorial(k),k);
if r = 'r'
then ok else r fi;
r := gosper(factorial(2*k)/(factorial(k)*factorial(k+1)),k):
if r = 'r'
then ok else r fi;
r := gosper(1/(factorial(k)*(k^4+k^2+1)),k):
if r = 'r'
then ok else r fi;
r := gosper(binomial(A,k),k):
if r = 'r'
then ok else r fi;
r := gosper(1/k,k):
if r = 'r'
then ok else r fi;
r := gosper((1+k)/(1-k),k):
if r = 'r'
then ok else r fi;
r := gosper(3^k*(3*k^2+2*a*k-4*k-2-a)/((2*k+2+a)*(2*k+a)*(k+1)*k),k):
if r = 'r'
then ok else r fi;
r := gosper(factorial(k+n)*factorial(n)/
((-1)^k*factorial(n-k)*factorial(k)*2^k),k):
if r = 'r'
then ok else r fi;
r := gosper(1/(k*(k+1/2)),k):
if r = 'r'
then ok else r fi;
r := gosper(pochhammer(a,k),k):
if r = 'r'
then ok else r fi;
r := gosper(binomial(n,k),k):
if r = 'r'
then ok else r fi;

# 5) Finding recurrence equations for definite sums
# D. Zeilberger,
# A fast algorithm for proving terminating hypergeometric identities,
# Discrete Math. 80 (1990), 207-211.
#
# Extension and examples from
# W. Koepf:
# Algorithms for the indefinite and definite summation.
# Konrad-Zuse-Zentrum Berlin (ZIB), Preprint SC 94-33, 1994.

# Bailey, p. 3:
# (Vandermonde-Gauss)

r := sumrecursion(hyperterm([-n,b],[c],1,k)/hyperterm([c-b,1],[c],1,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([-n,b],[c],1,n):
if not(has(simplify(r/
(-(c-b+n-1)*summ(n-1)+(c+n-1)*summ(n))
),n))
then ok else r fi;
# r := Hypersum([-n,b],[c],1,n):
# if r = 
# Hyperterm([c-b, 1],[c],1,n)
# then ok else r fi;
r := hypersum([-n,b],[c],1,n):
if r = 
pochhammer(c-b,n)/pochhammer(c,n)
then ok else r fi;

# Bailey, p. 9, (1):
# (Saalsch\"utz)

r := sumrecursion(hyperterm([a,b,-n],[c,1+a+b-c-n],1,k)/
hyperterm([c-a,c-b,1],[c,c-a-b],1,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([a,b,-n],[c,1+a+b-c-n],1,n):
if not(has(simplify(r/
(-(c-b+n-1)*(-n+1+a-c)*summ(n-1)+summ(n)*(c+n-1)*(1+a+b-c-n))
),n))
then ok else r fi;
# r := Hypersum([a,b,-n],[c,1+a+b-c-n],1,n):
# if r = 
# Hyperterm([c-b, c-a, 1],[c-a-b, c],1,n)
# then ok else r fi;
r := hypersum([a,b,-n],[c,1+a+b-c-n],1,n):
if r =
pochhammer(c-b,n)*pochhammer(c-a,n)/pochhammer(c-a-b,n)/pochhammer(c,n)
then ok else r fi;

# Bailey, p. 9, (1):
# (Kummer)

r := sumrecursion(
hyperterm([a,-n],[1+a+n],-1,k)/hyperterm([1+a,1],[1+a/2],1,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([a,-n],[1+a+n],-1,n):
if not(has(simplify(r/
(-2*(n+a)*summ(n-1)+(2*n+a)*summ(n))
),n))
then ok else r fi;
# r := Hypersum([a,-n],[1+a+n],-1,n):
# if r = 
# Hyperterm([a+1, 1],[1+1/2*a],1,n)
# then ok else r fi;
r := hypersum([a,-n],[1+a+n],-1,n):
if r =
pochhammer(a+1,n)/pochhammer(1+1/2*a,n)
then ok else r fi;

# Bailey, p. 11, (2):
# (Gauss)

r := sumrecursion(hyperterm([-n,b],[1/2*(1-n+b)],1/2,k)/
(GAMMA(1/2)*GAMMA(1/2-n/2+b/2))*GAMMA(1/2-n/2)*GAMMA(1/2+b/2),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-2))
),n))
then ok else r fi;
r := sumrecursion(hyperterm([-n,b],[1/2*(1-n+b)],1/2,k),k,n):
if not(has(simplify(r/
(-(n-1)*summ(n-2)+(-b+n-1)*summ(n))
),n))
then ok else r fi;

# Bailey, p. 11, (3):
# (Bailey)

r := sumrecursion(hyperterm([-n,1+n],[c],1/2,k)/
(GAMMA(c/2)*GAMMA(c/2+1/2))*GAMMA(c/2-n/2)*GAMMA(1/2+c/2+n/2),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-2))
),n))
then ok else r fi;
r := hyperrecursion([-n,1+n],[c],1/2,n):
if not(has(simplify(r/
((n-c)*summ(n-2)+(c+n-1)*summ(n))
),n))
then ok else r fi;

# Bailey, p. 13 (1):
# (Dixon)

r := sumrecursion(hyperterm([a,b,-n],[1+a-b,1+a+n],1,k)/
hyperterm([1+a,1+a/2-b,1],[1+a/2,1+a-b],1,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([a,b,-n],[1+a-b,1+a+n],1,n):
if not(has(simplify(r/
(-(a-2*b+2*n)*(n+a)*summ(n-1)+summ(n)*(2*n+a)*(a-b+n))
),n))
then ok else r fi;
# r := Hypersum([a,b,-n],[1+a-b,1+a+n],1,n):
# if r = 
# Hyperterm([1+1/2*a-b, a+1, 1],[1+a-b, 1+1/2*a],1,n)
# then ok else r fi;
r := hypersum([a,b,-n],[1+a-b,1+a+n],1,n):
if r =
pochhammer(1+1/2*a-b,n)*pochhammer(a+1,n)/pochhammer(1+a-b,n)/pochhammer(1+
1/2*a,n)
then ok else r fi;

# Bailey, p. 16 (1):
# (Watson)

r := sumrecursion(hyperterm([-n,b,c],[1/2*(1-n+b),2*c],1,k)/
(GAMMA(1/2)*GAMMA(1/2+c)*GAMMA(1/2-n/2+b/2)*GAMMA(1/2+n/2-b/2+c))*
GAMMA(1/2-n/2)*GAMMA(1/2+b/2)*GAMMA(1/2+n/2+c)*GAMMA(1/2-b/2+c),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-2))
),n))
then ok else r fi;
r := hyperrecursion([-n,b,c],[1/2*(1-n+b),2*c],1,n):
if not(has(simplify(r/
(-(n-b+2*c-1)*(n-1)*summ(n-2)+summ(n)*(-b+n-1)*(n+2*c-1))
),n))
then ok else r fi;
# r := Hypersum([-n,b,c],[1/2*(1-n+b),2*c],1,n):
# r := hypersum([-n,b,c],[1/2*(1-n+b),2*c],1,n):
# ord = 2

# Bailey, p. 16 (1):
# (Whipple)

r := sumrecursion(
hyperterm([-n,1+n,c],[e,1+2*c-e],1,k)/(Pi*GAMMA(e)*GAMMA(1+2*c-e))*
2^(2*c-1)*GAMMA(-n/2+e/2)*GAMMA(-n/2+(1+2*c-e)/2)*GAMMA((1+n)/2+e/2)*
GAMMA((1+n)/2+(1+2*c-e)/2),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-2))
),n))
then ok else r fi;
r := hyperrecursion([-n,1+n,c],[e,1+2*c-e],1,n):
if not(has(simplify(r/
(-(n-e)*(n-1-2*c+e)*summ(n-2)+summ(n)*(-1+e+n)*(n+2*c-e))
),n))
then ok else r fi;
# r := Hypersum([-n,1+n,c],[e,1+2*c-e],1,n):
# r := hypersum([-n,1+n,c],[e,1+2*c-e],1,n):
# ord = 2

# Bailey, p. 26 (5):
# (Dougall)

r := sumrecursion(hyperterm([a,1+a/2,b,c,d,1+2*a-b-c-d+n,-n],
[a/2,1+a-b,1+a-c,1+a-d,1+a-(1+2*a-b-c-d+n),1+a+n],1,k)/
hyperterm([1+a,1+a-b-c,1+a-b-d,1+a-c-d,1],
[1+a-b,1+a-c,1+a-d,1+a-b-c-d],1,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([a,1+a/2,b,c,d,1+2*a-b-c-d+n,-n],
[a/2,1+a-b,1+a-c,1+a-d,1+a-(1+2*a-b-c-d+n),1+a+n],1,n):
if not(has(simplify(r/
(-(n+a)*(a-c-d+n)*(a-b-d+n)*(a-b-c+n)*summ(n-1)+(a-d+n)*(a-c+n)*(a-b+n)*(a-b-c
-d+n)*summ(n))
),n))
then ok else r fi;
# r := Hypersum([a,1+a/2,b,c,d,1+2*a-b-c-d+n,-n],
# [a/2,1+a-b,1+a-c,1+a-d,1+a-(1+2*a-b-c-d+n),1+a+n],1,n):
r := hypersum([a,1+a/2,b,c,d,1+2*a-b-c-d+n,-n],
[a/2,1+a-b,1+a-c,1+a-d,1+a-(1+2*a-b-c-d+n),1+a+n],1,n):
if r =
pochhammer(1+a-b-d,n)*pochhammer(1+a-b-c,n)*pochhammer(1+a-c-d,n)*pochhammer(
a+1,n)/pochhammer(1+a-b-c-d,n)/pochhammer(1+a-b,n)/pochhammer(1+a-c,n)/
pochhammer(1+a-d,n)
then ok else r fi;

# Bailey, p. 25 (3), p. 27 (1):
# (Dougall)

r := sumrecursion(hyperterm([a,1+a/2,c,d,-n],[a/2,1+a-c,1+a-d,1+a+n],1,k)/
hyperterm([1+a-c-d,1+a,1],[1+a-c,1+a-d],1,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([a,1+a/2,c,d,-n],[a/2,1+a-c,1+a-d,1+a+n],1,n):
if not(has(simplify(r/
(-(n+a)*(a-c-d+n)*summ(n-1)+summ(n)*(a-d+n)*(a-c+n))
),n))
then ok else r fi;
# r := Hypersum([a,1+a/2,c,d,-n],[a/2,1+a-c,1+a-d,1+a+n],1,n):
# if r = 
# Hyperterm([1+a-c-d, a+1, 1],[1+a-c, 1+a-d],1,n)
# then ok else r fi;
r := hypersum([a,1+a/2,c,d,-n],[a/2,1+a-c,1+a-d,1+a+n],1,n):
if r =
pochhammer(1+a-c-d,n)*pochhammer(a+1,n)/pochhammer(1+a-c,n)/pochhammer(1+a-d,
n)
then ok else r fi;

# Bailey, p. 28 (3):
# (Whipple)

r := sumrecursion(hyperterm([a,1+a/2,d,-n],[a/2,1+a-d,1+a+n],-1,k)/
hyperterm([1+a,1],[1+a-d],1,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([a,1+a/2,d,-n],[a/2,1+a-d,1+a+n],-1,n):
if not(has(simplify(r/
(-(n+a)*summ(n-1)+(a-d+n)*summ(n))
),n))
then ok else r fi;
# r := Hypersum([a,1+a/2,d,-n],[a/2,1+a-d,1+a+n],-1,n):
# if r = 
# Hyperterm([a+1, 1],[1+a-d],1,n)
# then ok else r fi;
r := hypersum([a,1+a/2,d,-n],[a/2,1+a-d,1+a+n],-1,n):
if r = 
pochhammer(a+1,n)/pochhammer(1+a-d,n)
then ok else r fi;
# another new identity:
# r := Hypersum([a,1+a/2,d,-n],[a/2,1+a-d,1+a+n],1,n):
# if r = 
# Hyperterm([-d+1/2+1/2*a, a+1, 1],[1/2*a+1/2, 1+a-d],1,n)
# then ok else r fi;
r := hypersum([a,1+a/2,d,-n],[a/2,1+a-d,1+a+n],1,n):
if r =
pochhammer(-d+1/2+1/2*a,n)*pochhammer(a+1,n)/pochhammer(1/2*a+1/2,n)/
pochhammer(1+a-d,n)
then ok else r fi;

# Bailey, p. 30 (1.1):

r := sumrecursion(hyperterm([a,1+a/2,-n],[a/2,w],1,k)/
hyperterm([w-a,1],[w],1,n)/(w-a-1-n)*(w-a-1+n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([a,1+a/2,-n],[a/2,w],1,n):
if not(has(simplify(r/
((-w+a+2-n)*(-w+a+1+n)*summ(n-1)+summ(n)*(w+n-1)*(-w+a+n))
),n))
then ok else r fi;
# r := Hypersum([a,1+a/2,-n],[a/2,w],1,n):
# if r = 
# Hyperterm([-w+a+2, w-a-1, 1],[-w+a+1, w],1,n)
# then ok else r fi;
r := hypersum([a,1+a/2,-n],[a/2,w],1,n):
if r =
pochhammer(-w+a+2,n)*pochhammer(w-a-1,n)/pochhammer(-w+a+1,n)/pochhammer(w,
n)
then ok else r fi;

# Bailey, p. 30 (1.2):

r := sumrecursion(hyperterm([a,b,-n],[1+a-b,1+2*b-n],1,k)/
hyperterm([a-2*b,1+a/2-b,-b,1],[1+a-b,a/2-b,-2*b],1,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([a,b,-n],[1+a-b,1+2*b-n],1,n):
if not(has(simplify(r/
(-(-b+n-1)*(a-2*b+2*n)*(a-2*b+n-1)*summ(n-1)+(a-2*b+2*n-2)*(a-b+n)*(-2*b+n-1
)*summ(n))
),n))
then ok else r fi;
# r := Hypersum([a,b,-n],[1+a-b,1+2*b-n],1,n):
# if r = 
# Hyperterm([-b, 1+1/2*a-b, a-2*b, 1],[1+a-b, -2*b, 1/2*a-b],1,n)
# then ok else r fi;
r := hypersum([a,b,-n],[1+a-b,1+2*b-n],1,n):
if r =
pochhammer(-b,n)*pochhammer(1+1/2*a-b,n)*pochhammer(a-2*b,n)/pochhammer(1+a-b
,n)/pochhammer(-2*b,n)/pochhammer(1/2*a-b,n)
then ok else r fi;

# Bailey, p. 30 (1.3):

r := sumrecursion(hyperterm([a,1+a/2,b,-n],[a/2,1+a-b,1+2*b-n],1,k)/
hyperterm([a-2*b,-b,1],[1+a-b,-2*b],1,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([a,1+a/2,b,-n],[a/2,1+a-b,1+2*b-n],1,n):
if not(has(simplify(r/
(-(-b+n-1)*(a-2*b+n-1)*summ(n-1)+summ(n)*(a-b+n)*(-2*b+n-1))
),n))
then ok else r fi;
# r := Hypersum([a,1+a/2,b,-n],[a/2,1+a-b,1+2*b-n],1,n):
# if r = 
# Hyperterm([-b, a-2*b, 1],[1+a-b, -2*b],1,n)
# then ok else r fi;
r := hypersum([a,1+a/2,b,-n],[a/2,1+a-b,1+2*b-n],1,n):
if r =
pochhammer(-b,n)*pochhammer(a-2*b,n)/pochhammer(1+a-b,n)/pochhammer(-2*b,n)
then ok else r fi;

# Bailey, p. 30 (1.4):

r := sumrecursion(hyperterm([a,1+a/2,b,-n],[a/2,1+a-b,2+2*b-n],1,k)/
hyperterm([a-2*b-1,1/2+a/2-b,-b-1,1],[1+a-b,a/2-b-1/2,-2*b-1],1,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([a,1+a/2,b,-n],[a/2,1+a-b,2+2*b-n],1,n):
if not(has(simplify(r/
(-(-b-2+n)*(a-2*b-2+n)*(a-2*b+2*n-1)*summ(n-1)+(-2*b-2+n)*(a-3-2*b+2*n)*(a-b+n
)*summ(n))
),n))
then ok else r fi;
# r := Hypersum([a,1+a/2,b,-n],[a/2,1+a-b,2+2*b-n],1,n):
# if r = 
# Hyperterm([1/2+1/2*a-b, -b-1, a-2*b-1, 1],[1+a-b, -2*b-1,
# 1/2*a-b-1/2],1,n)
# then ok else r fi;
r := hypersum([a,1+a/2,b,-n],[a/2,1+a-b,2+2*b-n],1,n):
if r =
pochhammer(1/2+1/2*a-b,n)*pochhammer(-b-1,n)*pochhammer(a-2*b-1,n)/
pochhammer(1+a-b,n)/pochhammer(-2*b-1,n)/pochhammer(1/2*a-b-1/2,n)
then ok else r fi;

# Gessel-Stanton, p. 295 (1.1)

r := sumrecursion(hyperterm([-n,n+3*a,a],[3/2*a,(3*a+1)/2],3/4,k)/
(factorial(n)*pochhammer(a+1,n/3)/(factorial(n/3)*pochhammer(3*a+1,n))),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-3))
),n))
then ok else r fi;
r := hyperrecursion([-n,n+3*a,a],[3/2*a,(3*a+1)/2],3/4,n):
if not(has(simplify(r/
(-(n-1)*(n-2)*summ(n-3)+summ(n)*(n-1+3*a)*(3*a-2+n))
),n))
then ok else r fi;
# r := Hypersum([-n,n+3*a,a],[3/2*a,(3*a+1)/2],3/4,n):
# r := hypersum([-n,n+3*a,a],[3/2*a,(3*a+1)/2],3/4,n):
# ord=3

# Gessel-Stanton, p. 295 (1.2)

r := sumrecursion(hyperterm([2*a,2*b,1-2*b,1+2*a/3,-n],
[a-b+1,a+b+1/2,2/3*a,1+2*a+2*n],1/4,k)/
hyperterm([a+1/2,a+1,1],[a+b+1/2,a-b+1],1,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([2*a,2*b,1-2*b,1+2*a/3,-n],
[a-b+1,a+b+1/2,2/3*a,1+2*a+2*n],1/4,n):
if not(has(simplify(r/
(-(n+a)*(-1+2*a+2*n)*summ(n-1)+summ(n)*(a-b+n)*(2*a-1+2*n+2*b))
),n))
then ok else r fi;
# r := 
# Hypersum([2*a,2*b,1-2*b,1+2*a/3,-n],[a-b+1,a+b+1/2,2/3*a,1+2*a+2*n],1/4,n):
# if r = 
# Hyperterm([a+1/2, a+1, 1],[1+a-b, a+b+1/2],1,n)
# then ok else r fi;
r := hypersum([2*a,2*b,1-2*b,1+2*a/3,-n],[a-b+1,a+b+1/2,2/3*a,1+2*a+2*n],1/4,n):
if r =
pochhammer(a+1/2,n)*pochhammer(a+1,n)/pochhammer(1+a-b,n)/pochhammer(a+b+1/
2,n)
then ok else r fi;

# Gessel-Stanton, p. 295 (1.3)

r := sumrecursion(hyperterm([a,b,a+1/2-b,1+2*a/3,-n],
[2*a+1-2*b,2*b,2/3*a,1+a+n/2],4,k)/
(factorial(n)*2^(-n)/factorial(n/2))/
hyperterm([a+1,1],[a-b+1,b+1/2],1,n/2),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-2))
),n))
then ok else r fi;
r := hyperrecursion([a,b,a+1/2-b,1+2*a/3,-n],
[2*a+1-2*b,2*b,2/3*a,1+a+n/2],4,n):
if not(has(simplify(r/
(-(n-1)*(a+1/2*n)*summ(n-2)+summ(n)*(2*b+n-1)*(a-b+1/2*n))
),n))
then ok else r fi;
# r := Hypersum([a,b,a+1/2-b,1+2*a/3,-n],[2*a+1-2*b,2*b,2/3*a,1+a+n/2],4,n):
# r := hypersum([a,b,a+1/2-b,1+2*a/3,-n],[2*a+1-2*b,2*b,2/3*a,1+a+n/2],4,n):
# ord=2

# Gessel-Stanton, p. 295 (1.4), misprint.

r := sumrecursion(hyperterm([1/2+3*a,1/2-3*a,-n],[1/2,-3*n],3/4,k)/
hyperterm([1/2-a,1/2+a,1],[1/3,2/3],1,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([1/2+3*a,1/2-3*a,-n],[1/2,-3*n],3/4,n):
if not(has(simplify(r/
(9*(1+2*a-2*n)*(-1+2*a+2*n)*summ(n-1)+4*summ(n)*(3*n-1)*(3*n-2))
),n))
then ok else r fi;
# r := Hypersum([1/2+3*a,1/2-3*a,-n],[1/2,-3*n],3/4,n):
# if r = 
# Hyperterm([a+1/2, 1/2-a, 1],[1/3, 2/3],1,n)
# then ok else r fi;
r := hypersum([1/2+3*a,1/2-3*a,-n],[1/2,-3*n],3/4,n):
if r =
pochhammer(a+1/2,n)*pochhammer(1/2-a,n)/pochhammer(1/3,n)/pochhammer(2/3,n)
then ok else r fi;

# Gessel-Stanton, p. 295 (1.5)

r := sumrecursion(hyperterm([1+3*a,1-3*a,-n],[3/2,-1-3*n],3/4,k)/
hyperterm([1+a,1-a,1],[2/3,4/3],1,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([1+3*a,1-3*a,-n],[3/2,-1-3*n],3/4,n):
if not(has(simplify(r/
(9*(a-n)*(n+a)*summ(n-1)+summ(n)*(3*n-1)*(3*n+1))
),n))
then ok else r fi;
# r := Hypersum([1+3*a,1-3*a,-n],[3/2,-1-3*n],3/4,n):
# if r = 
# Hyperterm([1-a, a+1, 1],[2/3, 4/3],1,n)
# then ok else r fi;
r := hypersum([1+3*a,1-3*a,-n],[3/2,-1-3*n],3/4,n):
if r =
pochhammer(1-a,n)*pochhammer(a+1,n)/pochhammer(2/3,n)/pochhammer(4/3,n)
then ok else r fi;

# Gessel-Stanton, p. 295 (1.6)

r := sumrecursion(hyperterm([2*a,1-a,-n],[2*a+2,-a-1/2-3/2*n],1,k)/
((n+1)*(2*a+1)/(2*a+n+1))/
hyperterm([(n+3)/2,1],[1+1/2*(n+2*a+1)],1,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-2))
),n))
then ok else r fi;
r := sumrecursion(hyperterm([2*a,1-a,-n],[2*a+2,-a-1/2-3/2*n],1,k)/
((n+1)*(2*a+1)/(2*a+n+1)),k,n):
if not(has(simplify(r/
(-3*(3*n-1)*(n-1)*(3*n+1)*(2*a+n+1)*summ(n-2)+summ(n)*(n+1)*(2*a-3+3*n)*(2*a+1
+3*n)*(2*a-1+3*n))
),n))
then ok else r fi;
r := hyperrecursion([2*a,1-a,-n],[2*a+2,-a-1/2-3/2*n],1,n):
if r = 
-3*(3*n-1)*(3*n+1)*(2*a+n-1)*summ(n-2)+summ(n)*(2*a+1+3*n)*(2*a-1+3*n)*(2*a-3
+3*n)
then ok else r fi;
# r := Hypersum([2*a,1-a,-n],[2*a+2,-a-1/2-3/2*n],1,n):
# r := hypersum([2*a,1-a,-n],[2*a+2,-a-1/2-3/2*n],1,n):
# ord=2

# Gessel-Stanton, p. 296 (1.7)

r := sumrecursion(hyperterm([2*a,2*b,1-2*b,1+2/3*a,a+d+n+1/2,a-d,-n],
[a-b+1,a+b+1/2,2/3*a,-2*d-2*n, 2*d+1,1+2*a+2*n],1,k)/
hyperterm([2*a+1,1],[2*d+1],1,2*n)/
hyperterm([b+d+1/2,d-b+1,1],[a+b+1/2,a-b+1],1,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([2*a,2*b,1-2*b,1+2/3*a,a+d+n+1/2,a-d,-n],
[a-b+1,a+b+1/2,2/3*a,-2*d-2*n, 2*d+1,1+2*a+2*n],1,n):
if not(has(simplify(r/
(-(-1+2*n+2*b+2*d)*(d-b+n)*(n+a)*(-1+2*a+2*n)*summ(n-1)+summ(n)*(d+n)*(2*d-1+2
*n)*(a-b+n)*(2*a-1+2*b+2*n))
),n))
then ok else r fi;
# r := Hypersum([2*a,2*b,1-2*b,1+2/3*a,a+d+n+1/2,a-d,-n],
# [a-b+1,a+b+1/2,2/3*a,-2*d-2*n, 2*d+1,1+2*a+2*n],1,n):
r := hypersum([2*a,2*b,1-2*b,1+2/3*a,a+d+n+1/2,a-d,-n],
[a-b+1,a+b+1/2,2/3*a,-2*d-2*n, 2*d+1,1+2*a+2*n],1,n):
if r =
pochhammer(a+1/2,n)*pochhammer(d-b+1,n)*pochhammer(a+1,n)*pochhammer(b+d+1/2,
n)/pochhammer(1+a-b,n)/pochhammer(d+1/2,n)/pochhammer(d+1,n)/pochhammer(a+b+1/2
,n)
then ok else r fi;

# Gessel-Stanton, p. 296 (1.8)

r := sumrecursion(hyperterm([a,b,a+1/2-b,1+2*a/3,1-2*d,2*a+2*d+n,-n],
[2*a+1-2*b,2*b,2/3*a,a+d+1/2,1-d-n/2,1+a+n/2],1,k)/
(factorial(n)*2^(-n)/factorial(n/2))/
hyperterm([b+d,d-b+a+1/2,a+1,1],[b+1/2,a+d+1/2,d,a-b+1],1,n/2),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-2))
),n))
then ok else r fi;
r := hyperrecursion([a,b,a+1/2-b,1+2*a/3,1-2*d,2*a+2*d+n,-n],
[2*a+1-2*b,2*b,2/3*a,a+d+1/2,1-d-n/2,1+a+n/2],1,n):
if not(has(simplify(r/
(-(n-1)*(b+d+1/2*n-1)*(a+1/2*n)*(2*a+n-1-2*b+2*d)*summ(n-2)+summ(n)*(d+1/2*n-1
)*(2*b+n-1)*(2*a+2*d+n-1)*(a-b+1/2*n))
),n))
then ok else r fi;
# r := Hypersum([a,b,a+1/2-b,1+2*a/3,1-2*d,2*a+2*d+n,-n],
# [2*a+1-2*b,2*b,2/3*a,a+d+1/2,1-d-n/2,1+a+n/2],1,n):
# r := hypersum([a,b,a+1/2-b,1+2*a/3,1-2*d,2*a+2*d+n,-n],
# [2*a+1-2*b,2*b,2/3*a,a+d+1/2,1-d-n/2,1+a+n/2],1,n):
# ord=2

# Gessel-Stanton, p. 296 (1.9), works only for specified s

# r := seq(
# sumrecursion(hyperterm([-s*b+s+1,b-1,-n],[b+1,s*(-n-b)-n],1,k)/
# (b*(n+1)/(b+n))/hyperterm([1+s+s*n,1],[1+s*b+s*n],1,n),k,n)
# ,s=1..4):
# r := sumrecursion(hyperterm([-1/2*b+1/2+1,b-1,-n],[b+1,1/2*(-n-b)-n],1,k)/
# (b*(n+1)/(b+n))/hyperterm([1+1/2+1/2*n,1],[1+1/2*b+1/2*n],1,n),k,n):
# r := seq(
# hyperrecursion([-s*b+s+1,b-1,-n],[b+1,s*(-n-b)-n],1,n)
# ,s=1..4):
# r := hyperrecursion([-1/2*b+1/2+1,b-1,-n],[b+1,1/2*(-n-b)-n],1,n):
# r := seq(
# Hypersum([-s*b+s+1,b-1,-n],[b+1,s*(-n-b)-n],1,n)
# ,s=1..4):
# r := seq(
# hypersum([-s*b+s+1,b-1,-n],[b+1,s*(-n-b)-n],1,n)
# ,s=1..4):

# Gessel-Stanton, p. 298 (3.7)

r := sumrecursion(hyperterm([-n,-2*n-2/3],[4/3],-8,k)/
hyperterm([5/6,1],[3/2],-27,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([-n,-2*n-2/3],[4/3],-8,n):
if not(has(simplify(r/
(9*(6*n-1)*summ(n-1)+(1+2*n)*summ(n))
),n))
then ok else r fi;
# r := Hypersum([-n,-2*n-2/3],[4/3],-8,n):
# if r = 
# Hyperterm([5/6, 1],[3/2],-27,n)
# then ok else r fi;
r := hypersum([-n,-2*n-2/3],[4/3],-8,n):
if r =
pochhammer(5/6,n)*n!/(1+2*n)!*(-108)^n
then ok else r fi;

# Gessel-Stanton, p. 299, (3.12)

r := sumrecursion(hyperterm([-n,-n/2+1/6],[2/3],-8,k)/3^((3*n-1)/2),k,n):
if not(has(simplify(r/
(summ(n)+summ(n-2))
),n))
then ok else r fi;
r := sumrecursion(hyperterm([-n,-n/2+1/6],[2/3],-8,k),k,n):
if not(has(simplify(r/
(summ(n)+27*summ(n-2))
),n))
then ok else r fi;
# r := Hypersum([-n,-n/2+1/6],[2/3],-8,n):
# r := hypersum([-n,-n/2+1/6],[2/3],-8,n):
# ord = 2

# Gessel-Stanton, p. 305, (5.21)

r := sumrecursion(hyperterm([3*a+1/2,3*a+1,-n],[6*a+1,-n/3+2*a+1],4/3,k)/
hyperterm([1/3,2/3,1],[1+2*a,-2*a],1,n/3),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-3))
),n))
then ok else r fi;
r := hyperrecursion([3*a+1/2,3*a+1,-n],[6*a+1,-n/3+2*a+1],4/3,n):
if not(has(simplify(r/
((n-1)*(n-2)*summ(n-3)+9*summ(n)*(2*a+1/3*n)*(-1/3*n+2*a+1))
),n))
then ok else r fi;
# r := Hypersum([3*a+1/2,3*a+1,-n],[6*a+1,-n/3+2*a+1],4/3,n):
# r := hypersum([3*a+1/2,3*a+1,-n],[6*a+1,-n/3+2*a+1],4/3,n):
# ord = 3

# Gessel-Stanton, p. 306 (5.22)

r := sumrecursion(hyperterm([-n,1/2],[2*n+3/2],1/4,k)/
hyperterm([1/2,1],[2*n+3/2],27/4,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([-n,1/2],[2*n+3/2],1/4,n):
if not(has(simplify(r/
(-9*(4*n+1)*(4*n-1)*summ(n-1)+4*summ(n)*(6*n-1)*(6*n+1))
),n))
then ok else r fi;
# r := Hypersum([-n,1/2],[2*n+3/2],1/4,n):
# if r = 
# Hyperterm([3/4, 5/4, 1],[5/6, 7/6],1,n)
# then ok else r fi;
r := hypersum([-n,1/2],[2*n+3/2],1/4,n):
if r = 
pochhammer(3/4,n)*pochhammer(5/4,n)/pochhammer(5/6,n)/pochhammer(7/6,n)
then ok else r fi;

# Gessel-Stanton, p. 306 (5.23)

r := sumrecursion(hyperterm([-n,-1/3-2*n],[2/3],-8,k)/(-27)^n,k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([-n,-1/3-2*n],[2/3],-8,n):
if not(has(simplify(r/
(summ(n)+27*summ(n-1))
),n))
then ok else r fi;
# r := Hypersum([-n,-1/3-2*n],[2/3],-8,n):
# if r = 
# Hyperterm([1],[],-27,n)
# then ok else r fi;
r := hypersum([-n,-1/3-2*n],[2/3],-8,n):
if r =
(-27)^n
then ok else r fi;

# Gessel-Stanton, p. 306 (5.24)

r := sumrecursion(hyperterm([-n,n/2+1],[4/3],8/9,k)/
hyperterm([1/2,1],[7/6],-1/3,n/2),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-2))
),n))
then ok else r fi;
r := hyperrecursion([-n,n/2+1],[4/3],8/9,n):
if not(has(simplify(r/
((n-1)*summ(n-2)+(3*n+1)*summ(n))
),n))
then ok else r fi;
# r := Hypersum([-n,n/2+1],[4/3],8/9,n):
# r := hypersum([-n,n/2+1],[4/3],8/9,n):
# ord = 2

# Gessel-Stanton, p. 306 (5.25)

r := sumrecursion(hyperterm([-n,1/2],[(n+3)/2],4,k)/
hyperterm([1/2,3/2,1],[5/6,7/6],1,n/2),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-2))
),n))
then ok else r fi;
r := hyperrecursion([-n,1/2],[(n+3)/2],4,n):
if not(has(simplify(r/
(-9*(n+1)*(n-1)*summ(n-2)+summ(n)*(3*n-1)*(3*n+1))
),n))
then ok else r fi;
# r := Hypersum([-n,1/2],[(n+3)/2],4,n):
# r := hypersum([-n,1/2],[(n+3)/2],4,n):
# ord=2

# Gessel-Stanton, p. 306 (5.27)

r := sumrecursion(hyperterm([1/3-n,-n/2,(1-n)/2,22/21-3*n/7],
[5/6,4/3,1/21-3*n/7],-27,k)/((-8)^n/(1-9*n)),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([1/3-n,-n/2,(1-n)/2,22/21-3*n/7],
[5/6,4/3,1/21-3*n/7],-27,n):
if not(has(simplify(r/
(8*(-10+9*n)*summ(n-1)+(-1+9*n)*summ(n))
),n))
then ok else r fi;
# r := Hypersum([1/3-n,-n/2,(1-n)/2,22/21-3*n/7],[5/6,4/3,1/21-3*n/7],-27,n):
# if r = 
# Hyperterm([-1/9, 1],[8/9],-8,n)
# then ok else r fi;
r := hypersum([1/3-n,-n/2,(1-n)/2,22/21-3*n/7],[5/6,4/3,1/21-3*n/7],-27,n):
if r =
pochhammer(-1/9,n)*(-8)^n/pochhammer(8/9,n)
then ok else r fi;

# Gessel-Stanton, p. 307 (6.2)

r := sumrecursion(hyperterm([a+1/2,a,b,1-b,-n,(2*a+1)/3+n,a/2+1],
[1/2,(2*a-b+3)/3,(2*a+b+2)/3,-3*n,2*a+1+3*n,a/2],1,k)/
hyperterm([(2*a+2)/3,2*a/3+1,(1+b)/3,(2-b)/3,1],
[(2*a-b)/3+1,(2*a+b+2)/3,2/3,1/3],1,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([a+1/2,a,b,1-b,-n,(2*a+1)/3+n,a/2+1],
[1/2,(2*a-b+3)/3,(2*a+b+2)/3,-3*n,2*a+1+3*n,a/2],1,n):
if not(has(simplify(r/
(-(3*n-2+b)*(3*n-1-b)*(2*a+3*n)*(2*a-1+3*n)*summ(n-1)+summ(n)*(3*n-1)*(3*n-2)*
(2*a+3*n-b)*(2*a-1+3*n+b))
),n))
then ok else r fi;
# r := Hypersum([a+1/2,a,b,1-b,-n,(2*a+1)/3+n,a/2+1],
# [1/2,(2*a-b+3)/3,(2*a+b+2)/3,-3*n,2*a+1+3*n,a/2],1,n):
# if r = 
# Hyperterm([2/3-1/3*b, 1/3+1/3*b, 2/3*a+2/3, 1+2/3*a,
# 1],[2/3*a+1/3*b+2/3, 1/3
# , 2/3, 2/3*a-1/3*b+1],1,n)
# then ok else r fi;
r := hypersum([a+1/2,a,b,1-b,-n,(2*a+1)/3+n,a/2+1],
[1/2,(2*a-b+3)/3,(2*a+b+2)/3,-3*n,2*a+1+3*n,a/2],1,n):
if r =
pochhammer(2/3-1/3*b,n)*pochhammer(1/3+1/3*b,n)*pochhammer(2/3*a+2/3,n)*
pochhammer(1+2/3*a,n)/pochhammer(2/3*a+1/3*b+2/3,n)/pochhammer(1/3,n)/
pochhammer(2/3,n)/pochhammer(2/3*a-1/3*b+1,n)
then ok else r fi;

# Gessel-Stanton, p. 307 (6.3)

r := sumrecursion(hyperterm([a+1/2,a,-n,(2*a+1)/3+n,a/2+1],
[1/2,-3*n,2*a+1+3*n,a/2],9,k)/
hyperterm([(2*a+2)/3,2*a/3+1,1],[1/3,2/3],1,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([a+1/2,a,-n,(2*a+1)/3+n,a/2+1],
[1/2,-3*n,2*a+1+3*n,a/2],9,n):
if not(has(simplify(r/
(-(2*a+3*n)*(2*a-1+3*n)*summ(n-1)+summ(n)*(3*n-1)*(3*n-2))
),n))
then ok else r fi;
# r := Hypersum([a+1/2,a,-n,(2*a+1)/3+n,a/2+1],[1/2,-3*n,2*a+1+3*n,a/2],9,n):
# if r = 
# Hyperterm([2/3*a+2/3, 1+2/3*a, 1],[1/3, 2/3],1,n)
# then ok else r fi;
r := hypersum([a+1/2,a,-n,(2*a+1)/3+n,a/2+1],[1/2,-3*n,2*a+1+3*n,a/2],9,n):
if r =
pochhammer(2/3*a+2/3,n)*pochhammer(1+2/3*a,n)/pochhammer(1/3,n)/pochhammer(2/
3,n)
then ok else r fi;

# Gessel-Stanton, p. 307 (6.5)

r := sumrecursion(hyperterm([-n,-n+1/4],[2*n+5/4],1/9,k)/pochhammer(5/4,2*n)/
hyperterm([1],[2/3,13/12],1,n)*3^(5*n)/2^(6*n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([-n,-n+1/4],[2*n+5/4],1/9,n):
if not(has(simplify(r/
(-16*(8*n-3)*(8*n+1)*summ(n-1)+27*summ(n)*(1+12*n)*(3*n-1))
),n))
then ok else r fi;
# r := Hypersum([-n,-n+1/4],[2*n+5/4],1/9,n):
# if r = 
# Hyperterm([5/8, 9/8, 1],[2/3, 13/12],256/243,n)
# then ok else r fi;
r := hypersum([-n,-n+1/4],[2*n+5/4],1/9,n):
if r =
pochhammer(5/8,n)*pochhammer(9/8,n)*(256/243)^n/pochhammer(2/3,n)/
pochhammer(13/12,n)
then ok else r fi;

# Gessel-Stanton, p. 307 (6.6)

r := sumrecursion(hyperterm([-n,-n+1/4],[2*n+9/4],1/9,k)/pochhammer(9/4,2*n)/
hyperterm([1],[4/3,17/12],1,n)*3^(5*n)/2^(6*n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([-n,-n+1/4],[2*n+9/4],1/9,n):
if not(has(simplify(r/
(-16*(8*n+1)*(8*n+5)*summ(n-1)+27*summ(n)*(5+12*n)*(3*n+1))
),n))
then ok else r fi;
# r := Hypersum([-n,-n+1/4],[2*n+9/4],1/9,n):
# if r = 
# Hyperterm([13/8, 9/8, 1],[17/12, 4/3],256/243,n)
# then ok else r fi;
r := hypersum([-n,-n+1/4],[2*n+9/4],1/9,n):
if r = 
pochhammer(13/8,n)*pochhammer(9/8,n)*(256/243)^n/pochhammer(17/12,n)/
pochhammer(4/3,n)
then ok else r fi;

r := sumrecursion(binomial(n,k)*binomial(k,n)*(-1)^k/(-1)^n,k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := sumrecursion(binomial(n,k)*binomial(k,n)/(-1)^n,k,n):
if not(has(simplify(r/
(summ(n)+summ(n-1))
),n))
then ok else r fi;
r := sumrecursion(binomial(n,k)*binomial(k,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := sumrecursion(binomial(n,k)*binomial(2*k,n)*(-1)^k/(-2)^n,k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := sumrecursion(binomial(n,k)*binomial(k/2,n)*(-1)^k*(-2)^n,k,n,1):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := sumrecursion(binomial(n,k)*binomial(k/2,n)*(-2)^n,k,n,1):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := sumrecursion(binomial(n,k)*binomial(k/2,n),k,n,1):
if not(has(simplify(r/
(2*summ(n)+summ(n-1))
),n))
then ok else r fi;

# Clausen

r := sumrecursion(hyperterm([a,b,1/2-a-b-n,-n],[1/2+a+b,1-a-n,1-b-n],1,k)/
hyperterm([2*a,a+b,2*b,1],[2*a+2*b,a,b],1,n),k,n):
if not(has(simplify(r/
(summ(n)-summ(n-1))
),n))
then ok else r fi;
r := hyperrecursion([a,b,1/2-a-b-n,-n],[1/2+a+b,1-a-n,1-b-n],1,n):
if not(has(simplify(r/
(-(2*b+n-1)*(2*a+n-1)*(a+b+n-1)*summ(n-1)+summ(n)*(b+n-1)*(a+n-1)*(2*a+2*b+n-1))
),n))
then ok else r fi;
# r := Hypersum([a,b,1/2-a-b-n,-n],[1/2+a+b,1-a-n,1-b-n],1,n):
# if r = 
# Hyperterm([a+b, 2*b, 2*a, 1],[a, 2*a+2*b, b],1,n)
# then ok else r fi;
r := hypersum([a,b,1/2-a-b-n,-n],[1/2+a+b,1-a-n,1-b-n],1,n):
if r =
pochhammer(a+b,n)*pochhammer(2*b,n)*pochhammer(2*a,n)/pochhammer(a,n)/
pochhammer(2*a+2*b,n)/pochhammer(b,n)
then ok else r fi;

# Apery recurrence

r := sumrecursion(binomial(n,k)^2*binomial(n+k,k)^2,k,n):
if not(has(simplify(r/
((n-1)^3*summ(n-2)-(2*n-1)*(17*n^2-17*n+5)*summ(n-1)+summ(n)*n^3)
),n))
then ok else r fi;
r := Sumtohyper(binomial(n,k)^2*binomial(n+k,k)^2,k):
if r =
Hypergeom([-n, -n, n+1, n+1],[1, 1, 1],1)
then ok else r fi;


# 6) Finding recurrence equations for orthogonal polynomials and
# special functions
# M. Abramowitz, I. A. Stegun:
# Handbook of Mathematical Functions. Dover Publ., New York (1964).
#
# W. Koepf:
# Identities for families of orthogonal polynomials and special functions.
# 1994 (in preparation).

# associated Legendre functions 

# Abramowitz-Stegun (8.1.2)
# LegendreP(n,m,x):=
# 1/GAMMA(1-m)*((z+1)/(z-1))^(m/2)*hypergeom([-n,n+1],[1-m],(1-x)/2);

r := sumrecursion(
1/GAMMA(1-m)*((z+1)/(z-1))^(m/2)*hyperterm([-n,n+1],[1-m],(1-x)/2,k),k,n):
if not(has(simplify(r/
((n-1+m)*summ(n-2)-x*(2*n-1)*summ(n-1)+(n-m)*summ(n))
),n))
then ok else r fi;

r := sumrecursion(
1/GAMMA(1-m)*((z+1)/(z-1))^(m/2)*hyperterm([-n,n+1],[1-m],(1-x)/2,k),k,m):
if not(has(simplify(r/
(-(n-1+m)*(n+2-m)*(z+1)*(-1+x)*summ(-2+m)+2*x*((z+1)/(z-1))^(1/2)*(z-1)*(-1+m)
*summ(-1+m)+(z-1)*(1+x)*summ(m))
),n))
then ok else r fi;

# Abramowitz-Stegun (8.1.3)
# LegendreQ(n,m,x):=
# exp(I*m*Pi)*2^(-n-1)*GAMMA(n+m+1)/GAMMA(n+3/2)*x^(-n-m-1)*(x^2-1)^(m/2)*
# hypergeom([1+n/2+m/2,1/2+n/2+m/2],[n+3/2],1/x^2);

r := sumrecursion(
exp(I*m*Pi)*2^(-n-1)*GAMMA(n+m+1)/GAMMA(n+3/2)*x^(-n-m-1)*(x^2-1)^(m/2)*
hyperterm([1+n/2+m/2,1/2+n/2+m/2],[n+3/2],1/x^2,k),k,n):
if not(has(simplify(r/
((n-1+m)*summ(n-2)-x*(2*n-1)*summ(n-1)+(n-m)*summ(n))
),n))
then ok else r fi;

r := sumrecursion(
exp(I*m*Pi)*2^(-n-1)*GAMMA(n+m+1)/GAMMA(n+3/2)*x^(-n-m-1)*(x^2-1)^(m/2)*
hyperterm([1+n/2+m/2,1/2+n/2+m/2],[n+3/2],1/x^2,k),k,m):
if not(has(simplify(r/
(-(E^(I*Pi))^2*(n-1+m)*(n+2-m)*summ(m-2)*((x-1)*(1+x))^(1/2)-2*E^(I*Pi)*x*(-1+
m)*summ(-1+m)+summ(m)*((x-1)*(1+x))^(1/2))
),n))
then ok else r fi;

# Bessel functions 

# Abramowitz-Stegun (9.1.69)
# BesselJ(n,x):=(x/2)^n/GAMMA(n+1)*hypergeom([],[n+1],-x^2/4);

r := sumrecursion((x/2)^n/GAMMA(n+1)*hyperterm([],[n+1],-x^2/4,k),k,n):
if not(has(simplify(r/
(-2*(n-1)*summ(n-1)+summ(n)*x+summ(n-2)*x)
),n))
then ok else r fi;

# BesselJ(n,x):=(x/2)^n*exp(I*x)/GAMMA(n+1)*hypergeom([n+1/2],[2*n+1],2*I*x);

r := sumrecursion(
(x/2)^n*exp(I*x)/GAMMA(n+1)*hyperterm([n+1/2],[2*n+1],2*I*x,k),k,n):
if not(has(simplify(r/
(-2*(n-1)*summ(n-1)+summ(n)*x+summ(n-2)*x)
),n))
then ok else r fi;

# Abramowitz-Stegun (9.6.47)
# BesselI(n,x):=(x/2)^n/GAMMA(n+1)*hypergeom([],[n+1],x^2/4);

r := sumrecursion((x/2)^n/GAMMA(n+1)*hyperterm([],[n+1],x^2/4,k),k,n):
if not(has(simplify(r/
(2*(n-1)*summ(n-1)+summ(n)*x-summ(n-2)*x)
),n))
then ok else r fi;

# BesselI(n,x):=(x/2)^n*exp(x)/GAMMA(n+1)*hypergeom([n+1/2],[2*n+1],2*x);

r := sumrecursion(
(x/2)^n*exp(x)/GAMMA(n+1)*hyperterm([n+1/2],[2*n+1],2*x,k),k,n):
if not(has(simplify(r/
(2*(n-1)*summ(n-1)+summ(n)*x-summ(n-2)*x)
),n))
then ok else r fi;

# Whittaker functions

# Abramowitz-Stegun (13.1.32)
# WhittakerM(n,m,x):=exp(-x/2)*x^(1/2+m)*hypergeom([1/2+m-n],[1+2*m],x);

r := sumrecursion(
exp(-x/2)*x^(1/2+m)*hyperterm([1/2+m-n],[1+2*m],x,k),k,n):
if not(has(simplify(r/
((-3-2*m+2*n)*summ(n-2)+2*(-2*n+x+2)*summ(n-1)+(2*n-1+2*m)*summ(n))
),n))
then ok else r fi;

r := sumrecursion(
exp(-x/2)*x^(1/2+m)*hyperterm([1/2+m-n],[1+2*m],x,k),k,m):
if not(has(simplify(r/
(16*m*x*(-1+m)*(-3+2*m)*(2*m-1)^2*summ(m-2)+16*(2*m-1)*(-1+m)*(-4*m^2+8*m+2*n*
x-3)*m*summ(-1+m)+x*(-3+2*m)*(2*n-1+2*m)*(1-2*m+2*n)*summ(m))
),n))
then ok else r fi;

# Jacobi polynomials

# Abramowitz-Stegun (22.5.42)
# JacobiP(n,a,b,x):=binomial(n+a,n)*hypergeom([-n,n+a+b+1],[a+1],(1-x)/2);

r := sumrecursion(
binomial(n+a,n)*hyperterm([-n,n+a+b+1],[a+1],(1-x)/2,k),k,n):
if not(has(simplify(r/
(2*(b+n-1)*(a+n-1)*(a+b+2*n)*summ(n-2)-(-1+a+b+2*n)*(a^2+x*a^2+4*n*x*a+2*x*a*b
-2*x*a-b^2+4*n^2*x+4*n*x*b-2*x*b+x*b^2-4*n*x)*summ(n-1)+2*summ(n)*(a+b+n)*(a-2+
b+2*n)*n)
),n))
then ok else r fi;

r := sumrecursion(
binomial(n+a,n)*hyperterm([-n,n+a+b+1],[a+1],(1-x)/2,k),k,a):
if not(has(simplify(r/
(-2*(a+n-1)*summ(a-2)-(x*a+x*b+2*n*x-3*a+2-b-2*n)*summ(-1+a)+summ(a)*(a+b+n)*(
x-1))
),n))
then ok else r fi;

r := sumrecursion(
binomial(n+a,n)*hyperterm([-n,n+a+b+1],[a+1],(1-x)/2,k),k,b):
if not(has(simplify(r/
(2*(b+n-1)*summ(b-2)-(-2+3*b+2*n+x*b+2*n*x+a+x*a)*summ(b-1)+summ(b)*(a+b+n
)*(1+x))
),n))
then ok else r fi;

# Abramowitz-Stegun (22.5.43)
# JacobiP(n,a,b,x):=binomial(2*n+a+b,n)*((x-1)/2)^n*
# hypergeom([-n,-n-a],[-2*n-a-b],2/(1-x));

r := sumrecursion(
binomial(2*n+a+b,n)*((x-1)/2)^n*hyperterm([-n,-n-a],[-2*n-a-b],2/(1-x),k),k,n):
if not(has(simplify(r/
(2*(b+n-1)*(a+n-1)*(a+b+2*n)*summ(n-2)-(-1+a+b+2*n)*(a^2+a^2*x+2*a*b*x-2*x*a+4
*n*a*x+4*n^2*x-4*n*x-b^2+4*n*b*x+b^2*x-2*x*b)*summ(n-1)+2*summ(n)*n*(a-2+2*n+b)
*(a+b+n))
),n))
then ok else r fi;

r := sumrecursion(
binomial(2*n+a+b,n)*((x-1)/2)^n*hyperterm([-n,-n-a],[-2*n-a-b],2/(1-x),k),k,a):
if not(has(simplify(r/
(-2*(a+n-1)*summ(-2+a)-(-3*a+x*a+x*b-b-2*n+2+2*n*x)*summ(a-1)+summ(a)*(a+b+n)*
(x-1))
),n))
then ok else r fi;

r := sumrecursion(
binomial(2*n+a+b,n)*((x-1)/2)^n*hyperterm([-n,-n-a],[-2*n-a-b],2/(1-x),k),k,b):
if not(has(simplify(r/
(2*(b+n-1)*summ(b-2)-(-2+3*b+2*n+x*b+2*n*x+a+x*a)*summ(b-1)+summ(b)*(a+b+n
)*(1+x))
),n))
then ok else r fi;

# Abramowitz-Stegun (22.5.44)
# JacobiP(n,a,b,x):=binomial(n+a,n)*((1+x)/2)^n*
# hypergeom([-n,-n-b],[a+1],(x-1)/(x+1));

r := sumrecursion(
binomial(n+a,n)*((1+x)/2)^n*hyperterm([-n,-n-b],[a+1],(x-1)/(x+1),k),k,n):
if not(has(simplify(r/
(2*(b+n-1)*(a+n-1)*(a+b+2*n)*summ(n-2)-(-1+a+b+2*n)*(a^2+x*a^2-2*x*a+2*a*b*x+4
*a*n*x+4*n*b*x+b^2*x+4*n^2*x-4*n*x-2*x*b-b^2)*summ(n-1)+2*summ(n)*n*(a+b+n)*(a-
2+b+2*n))
),n))
then ok else r fi;

r := sumrecursion(
binomial(n+a,n)*((1+x)/2)^n*hyperterm([-n,-n-b],[a+1],(x-1)/(x+1),k),k,a):
if not(has(simplify(r/
(-2*(a+n-1)*summ(a-2)-(x*b+x*a-2*n+2*n*x-3*a-b+2)*summ(a-1)+summ(a)*(a+b+n)*(x
-1))
),n))
then ok else r fi;

r := sumrecursion(
binomial(n+a,n)*((1+x)/2)^n*hyperterm([-n,-n-b],[a+1],(x-1)/(x+1),k),k,b):
if not(has(simplify(r/
(2*(b+n-1)*summ(b-2)-(-2+3*b+2*n+x*b+2*n*x+a+x*a)*summ(b-1)+summ(b)*(a+b+n
)*(1+x))
),n))
then ok else r fi;

# Abramowitz-Stegun (22.5.45)
# JacobiP(n,a,b,x):=binomial(n+b,n)*((x-1)/2)^n*
# hypergeom([-n,-n-a],[b+1],(x+1)/(x-1));

r := sumrecursion(
binomial(n+b,n)*((x-1)/2)^n*hyperterm([-n,-n-a],[b+1],(x+1)/(x-1),k),k,n):
if not(has(simplify(r/
(2*(b+n-1)*(a+n-1)*(a+b+2*n)*summ(n-2)-(-1+a+b+2*n)*(a^2+a^2*x-2*x*a+4*x*a*n+2
*x*a*b-4*n*x+x*b^2+4*n^2*x-2*x*b-b^2+4*x*b*n)*summ(n-1)+2*summ(n)*n*(a+b+n)*(b-
2+2*n+a))
),n))
then ok else r fi;

r := sumrecursion(
binomial(n+b,n)*((x-1)/2)^n*hyperterm([-n,-n-a],[b+1],(x+1)/(x-1),k),k,a):
if not(has(simplify(r/
(-2*(a+n-1)*summ(a-2)-(x*a-3*a-2*n+2+x*b+2*n*x-b)*summ(a-1)+summ(a)*(a+b+n)*(x
-1))
),n))
then ok else r fi;

r := sumrecursion(
binomial(n+b,n)*((x-1)/2)^n*hyperterm([-n,-n-a],[b+1],(x+1)/(x-1),k),k,b):
if not(has(simplify(r/
(2*(b+n-1)*summ(b-2)-(x*b+3*b+2*n*x+x*a+2*n+a-2)*summ(b-1)+summ(b)*(a+b+n)
*(1+x))
),n))
then ok else r fi;

# Gegenbauer polynomials

# Abramowitz-Stegun (22.5.46)
# GegenbauerC(b,a,x):=GAMMA(n+2*a)/(n!*GAMMA(2*a))*
# hypergeom([-n,n+2*a],[a+1/2],(1-x)/2);

r := sumrecursion(
GAMMA(n+2*a)/(n!*GAMMA(2*a))*hyperterm([-n,n+2*a],[a+1/2],(1-x)/2,k),k,n):
if not(has(simplify(r/
((n+2*a-2)*summ(n-2)-2*(a+n-1)*x*summ(n-1)+summ(n)*n)
),n))
then ok else r fi;

r := sumrecursion(
GAMMA(n+2*a)/(n!*GAMMA(2*a))*hyperterm([-n,n+2*a],[a+1/2],(1-x)/2,k),k,a):
if not(has(simplify(r/
(-(2*a+n-3)*(2*a+n-4)*summ(a-2)-2*(a-2)*(-4*a+2*x^2*a-2*x^2+5+2*n*x^2-2*n)*
summ(a-1)+4*(a-1)*(a-2)*(x-1)*(1+x)*summ(a))
),n))
then ok else r fi;

# Chebyshev polynomials

# Abramowitz-Stegun (22.5.47)
# ChebyshevT(n,x):=hypergeom([-n,n],[1/2],(1-x)/2);

r := sumrecursion(hyperterm([-n,n],[1/2],(1-x)/2,k),k,n):
if not(has(simplify(r/
(summ(n)-2*x*summ(n-1)+summ(n-2))
),n))
then ok else r fi;

# Abramowitz-Stegun (22.5.48)
# ChebyshevU(n,x):=(n+1)*hypergeom([-n,n+2],[3/2],(1-x)/2);

r := sumrecursion((n+1)*hyperterm([-n,n+2],[3/2],(1-x)/2,k),k,n):
if not(has(simplify(r/
(summ(n)-2*x*summ(n-1)+summ(n-2))
),n))
then ok else r fi;

# Legendre polynomials

# Abramowitz-Stegun (22.5.48)
# LegendreP(n,x):=hypergeom([-n,n+1],[1],(1-x)/2);

r := sumrecursion(hyperterm([-n,n+1],[1],(1-x)/2,k),k,n):
if not(has(simplify(r/
((n-1)*summ(n-2)-(2*n-1)*x*summ(n-1)+summ(n)*n)
),n))
then ok else r fi;

# Abramowitz-Stegun (22.5.49)
# LegendreP(n,x):=binomial(2*n,n)*((x-1)/2)^n*
# hypergeom([-n,-n],[-2*n],2/(1-x));

r := sumrecursion(
binomial(2*n,n)*((x-1)/2)^n*hyperterm([-n,-n],[-2*n],2/(1-x),k),k,n):
if not(has(simplify(r/
((n-1)*summ(n-2)-(2*n-1)*x*summ(n-1)+summ(n)*n)
),n))
then ok else r fi;


# Abramowitz-Stegun (22.5.50)
# LegendreP(n,x):=binomial(2*n,n)*(x/2)^n*
# hypergeom([-n/2,(1-n)/2],[1/2-n],1/x^2);

r := sumrecursion(
binomial(2*n,n)*(x/2)^n*hyperterm([-n/2,(1-n)/2],[1/2-n],1/x^2,k),k,n):
if not(has(simplify(r/
((n-1)*summ(n-2)-(2*n-1)*x*summ(n-1)+summ(n)*n)
),n))
then ok else r fi;

# Laguerre polynomials

# Abramowitz-Stegun (22.5.54)
# LaguerreL(n,a,x):=binomial(n+x,n)*hypergeom([-n],[a+1],x);

r := sumrecursion(binomial(n+x,n)*hyperterm([-n],[a+1],x,k),k,n):
if not(has(simplify(r/
((n+x)*(n+x-1)*summ(n-2)+(n+x)*(x-a+1-2*n)*summ(n-1)+summ(n)*n*(a+n))
),n))
then ok else r fi;

r := sumrecursion(binomial(n+x,n)*hyperterm([-n],[a+1],x,k),k,a):
if not(has(simplify(r/
(a*(a-1)*summ(a-2)-a*(x+a-1)*summ(a-1)+summ(a)*x*(a+n))
),n))
then ok else r fi;

# Discrete orthogonal polynomials

# A. F. Nikiforov, S. K. Suslov, V. B. Uvarov:
# Classical orthogonal polynomials of a discrete variable.
# Springer, Berlin-Heidelberg-New York, 1991.

# Charlier polynomials

# Nikiforov, Suslov, Uvarov (2.7.9)
# Charlier(n,mu,x):=hypergeom([-n,-x],[],-1/mu);

r := sumrecursion(hyperterm([-n,-x],[],-1/mu,k),k,n):
if not(has(simplify(r/
((n-1)*summ(n-2)+(-n+1-mu+x)*summ(n-1)+summ(n)*mu)
),n))
then ok else r fi;

r := sumrecursion(hyperterm([-n,-x],[],-1/mu,k),k,x):
if not(has(simplify(r/
((x-1)*summ(x-2)-(x-1+mu-n)*summ(x-1)+summ(x)*mu)
),n))
then ok else r fi;

# Nikiforov, Suslov, Uvarov (2.7.9)
# Charlier(n,mu,x):=
# (-1)^n/mu^n*pochhammer(x-n+1,n)*hypergeom([-n],[x-n+1],mu);

r := sumrecursion(
(-1)^n/mu^n*pochhammer(x-n+1,n)*hyperterm([-n],[x-n+1],mu,k),k,n):
if not(has(simplify(r/
((n-1)*summ(n-2)+(-mu-n+x+1)*summ(n-1)+summ(n)*mu)
),n))
then ok else r fi;

r := sumrecursion(
(-1)^n/mu^n*pochhammer(x-n+1,n)*hyperterm([-n],[x-n+1],mu,k),k,x):
if not(has(simplify(r/
((x-1)*summ(x-2)-(x-1+mu-n)*summ(x-1)+summ(x)*mu)
),n))
then ok else r fi;

# Krawtchouk polynomials

# Nikiforov, Suslov, Uvarov (2.7.11)
# Krawtchouk(n,N,p,x):=(-1)^n*p^n/n!*GAMMA(N-x+1)/GAMMA(N-x-n+1)*
# hypergeom([-n,-x],[N-x-n+1],-(1-p)/p);

r := sumrecursion((-1)^n*p^n/n!*GAMMA(N-x+1)/GAMMA(N-x-n+1)*
hyperterm([-n,-x],[N-x-n+1],-(1-p)/p,k),k,n):
if not(has(simplify(r/
(p*(-1+p)*(n-N-2)*summ(n-2)-(x-2*p-n+2*n*p-p*N+1)*summ(n-1)+summ(n)*n)
),n))
then ok else r fi;

r := sumrecursion((-1)^n*p^n/n!*GAMMA(N-x+1)/GAMMA(N-x-n+1)*
hyperterm([-n,-x],[N-x-n+1],-(1-p)/p,k),k,x):
if not(has(simplify(r/
((-1+p)*(x-1)*summ(x-2)-(2*x*p-x-p*N-2*p+n+1)*summ(x-1)+(-N+x-1)*p*summ(x))
),n))
then ok else r fi;

r := sumrecursion((-1)^n*p^n/n!*GAMMA(N-x+1)/GAMMA(N-x-n+1)*
hyperterm([-n,-x],[N-x-n+1],-(1-p)/p,k),k,N):
if not(has(simplify(r/
(-(-N+1+x)*summ(N-2)+(n+1+x-2*N+p*N)*summ(N-1)+summ(N)*(-1+p)*(-N+n))
),n))
then ok else r fi;

# Nikiforov, Suslov, Uvarov (2.7.11a)
# Krawtchouk(n,N,p,x):=(-1)^n*binomial(N,n)*p^n*hypergeom([-n,-x],[-N],1/p);

r := sumrecursion(
(-1)^n*binomial(N,n)*p^n*hyperterm([-n,-x],[-N],1/p,k),k,n):
if not(has(simplify(r/
(p*(-1+p)*(n-N-2)*summ(n-2)-(x-2*p-n+2*n*p-p*N+1)*summ(n-1)+summ(n)*n)
),n))
then ok else r fi;

r := sumrecursion(
(-1)^n*binomial(N,n)*p^n*hyperterm([-n,-x],[-N],1/p,k),k,x):
if not(has(simplify(r/
((-1+p)*(x-1)*summ(x-2)-(2*x*p-x-p*N-2*p+n+1)*summ(x-1)+(-N+x-1)*p*summ(x))
),n))
then ok else r fi;

r := sumrecursion(
(-1)^n*binomial(N,n)*p^n*hyperterm([-n,-x],[-N],1/p,k),k,N):
if not(has(simplify(r/
(-(-N+1+x)*summ(N-2)+(n+1+x-2*N+p*N)*summ(N-1)+summ(N)*(-1+p)*(-N+n))
),n))
then ok else r fi;

# Meixner polynomials

# Nikiforov, Suslov, Uvarov (2.7.12)
# Meixner(n,gamma,mu,x):=pochhammer(gamma,n)*hypergeom([-n,-x],[gamma],1-1/mu);

r := sumrecursion(
pochhammer(gamma,n)*hyperterm([-n,-x],[gamma],1-1/mu,k),k,n):
if not(has(simplify(r/
((n-1)*(gamma+n-2)*summ(n-2)-(n-1-x+mu*gamma-mu+x*mu+n*mu)*summ(n-1)+summ(n)*
mu)
),n))
then ok else r fi;

r := sumrecursion(
pochhammer(gamma,n)*hyperterm([-n,-x],[gamma],1-1/mu,k),k,x):
if not(has(simplify(r/
((x-1)*summ(x-2)-(x-1-mu-n+mu*gamma+n*mu+x*mu)*summ(x-1)+summ(x)*mu*(gamma
-1+x))
),n))
then ok else r fi;

r := sumrecursion(
pochhammer(gamma,n)*hyperterm([-n,-x],[gamma],1-1/mu,k),k,gamma):
if not(has(simplify(r/
(-(gamma-2+n)*summ(gamma-2)-(mu*gamma-mu+x*mu+n*mu-x-n+3-2*gamma)*summ(gamma-1
)+summ(gamma)*(mu-1)*(gamma-1+x))
),n))
then ok else r fi;

# Hahn polynomials

# Nikiforov, Suslov, Uvarov (2.7.17/18)
# Hahn(n,N,alpha,beta,x):=
# (-1)^n/n!*GAMMA(beta+n+x+1)*GAMMA(N-x)/(GAMMA(beta+x+1)*GAMMA(N-x-n))*
# hypergeom([-n,-x,alpha+N-x],[N-x-n,-beta-x-n],1);

r := sumrecursion(
(-1)^n/n!*GAMMA(beta+n+x+1)*GAMMA(N-x)/(GAMMA(beta+x+1)*GAMMA(N-x-n))*
hyperterm([-n,-x,alpha+N-x],[N-x-n,-beta-x-n],1,k),k,n):
if not(has(simplify(r/
(-(-1+alpha+n)*(n-1-N)*(beta+n-1)*(2*n+alpha+beta)*(n+alpha+beta-1+N)*summ(n-2
)-(2*n-1+beta+alpha)*(-n*alpha^2-2*n-beta+4*n*x*alpha+n*beta-2*x*beta+2*N*n+N*
beta+alpha^2+2*n^2+4*n^2*x-4*n*x+x*alpha^2-alpha-2*n*N*beta+3*n*alpha-alpha*N*
beta-2*alpha*N*n+2*alpha*x*beta-2*x*alpha+alpha*beta+alpha*N-N*beta^2+n*beta^2+
x*beta^2-alpha*n^2+n^2*beta-2*N*n^2+4*n*x*beta)*summ(n-1)+summ(n)*n*(2*n-2+
alpha+beta)*(n+alpha+beta))
),n))
then ok else r fi;

# r := sumrecursion(
# (-1)^n/n!*GAMMA(beta+n+x+1)*GAMMA(N-x)/(GAMMA(beta+x+1)*GAMMA(N-x-n))*
# hyperterm([-n,-x,alpha+N-x],[N-x-n,-beta-x-n],1,k),k,x):
# very difficult..., not yet done!
# System error, ran out of memory

r := sumrecursion(
(-1)^n/n!*GAMMA(beta+n+x+1)*GAMMA(N-x)/(GAMMA(beta+x+1)*GAMMA(N-x-n))*
hyperterm([-n,-x,alpha+N-x],[N-x-n,-beta-x-n],1,k),k,alpha):
if not(has(simplify(r/
(-(-1+alpha+n)*(n+alpha+beta-1+N)*summ(alpha-2)-(-1+3*n-2*alpha^2+N+2*beta+2*n
*x+3*alpha-n*beta+x*beta-2*N*n-N*beta-2*alpha*N-3*n*alpha-n^2-2*alpha*beta+x*
alpha)*summ(alpha-1)+summ(alpha)*(n+alpha+beta)*(-alpha+1-N+x))
),n))
then ok else r fi;

r := sumrecursion(
(-1)^n/n!*GAMMA(beta+n+x+1)*GAMMA(N-x)/(GAMMA(beta+x+1)*GAMMA(N-x-n))*
hyperterm([-n,-x,alpha+N-x],[N-x-n,-beta-x-n],1,k),k,beta):
if not(has(simplify(r/
((beta-1+n)*(n+alpha+beta-1+N)*summ(beta-2)-(1-n-2*beta+3*n*beta+x*beta+N*beta
-N+n^2+2*n*x-alpha+n*alpha+x*alpha+2*alpha*beta+2*beta^2)*summ(beta-1)+summ(
beta)*(n+alpha+beta)*(beta+x))
),n))
then ok else r fi;

# Nikiforov, Suslov, Uvarov (2.7.19)
# Hahn(n,N,alpha,beta,x):=
# (-1)^n/n!*pochhammer(N-n,n)*pochhammer(beta+1,n)*
# hypergeom([-n,alpha+beta+n+1,-x],[beta+1,1-N],1);

r := sumrecursion((-1)^n/n!*pochhammer(N-n,n)*pochhammer(beta+1,n)*
hyperterm([-n,alpha+beta+n+1,-x],[beta+1,1-N],1,k),k,n):
if not(has(simplify(r/
(-(-1+alpha+n)*(n-1-N)*(beta-1+n)*(2*n+alpha+beta)*(n+alpha+beta-1+N)*summ(n-2
)-(2*n+beta+alpha-1)*(-2*n-beta+4*n*x*alpha+x*alpha^2+n*beta-N*beta*alpha-2*N*n
*alpha-n^2*alpha-2*N*n^2-2*x*beta+2*N*n+N*beta+beta^2*x+alpha^2+2*n^2+4*n^2*x-4
*n*x-n*alpha^2-alpha-2*n*N*beta+3*n*alpha-2*x*alpha+alpha*beta+alpha*N-N*beta^2
+n*beta^2+n^2*beta+4*n*x*beta+2*x*alpha*beta)*summ(n-1)+n*(n+alpha+beta)*(2*n-2
+alpha+beta)*summ(n))
),n))
then ok else r fi;

r := sumrecursion((-1)^n/n!*pochhammer(N-n,n)*pochhammer(beta+1,n)*
hyperterm([-n,alpha+beta+n+1,-x],[beta+1,1-N],1,k),k,x):
if not(has(simplify(r/
((-alpha-1+x-N)*(x-1)*summ(x-2)-(2*x^2-2*x+x*beta-2*x*N-x*alpha+n*alpha+1+
n^2+n+N+alpha+n*beta-N*beta)*summ(x-1)+summ(x)*(x-N)*(beta+x))
),n))
then ok else r fi;

r := sumrecursion((-1)^n/n!*pochhammer(N-n,n)*pochhammer(beta+1,n)*
hyperterm([-n,alpha+beta+n+1,-x],[beta+1,1-N],1,k),k,alpha):
if not(has(simplify(r/
(-(-1+alpha+n)*(n+alpha+beta-1+N)*summ(alpha-2)-(-1+3*n-2*alpha^2+N+2*beta+2*n
*x+3*alpha-n*beta+x*beta-2*N*n-N*beta-2*alpha*N-3*n*alpha-n^2-2*alpha*beta+x*
alpha)*summ(alpha-1)+summ(alpha)*(n+alpha+beta)*(-alpha+1-N+x))
),n))
then ok else r fi;

r := sumrecursion((-1)^n/n!*pochhammer(N-n,n)*pochhammer(beta+1,n)*
hyperterm([-n,alpha+beta+n+1,-x],[beta+1,1-N],1,k),k,beta):
if not(has(simplify(r/
((beta-1+n)*(n-1+beta+N+alpha)*summ(beta-2)-(n^2+3*n*beta+2*n*x-n+n*alpha+2*
beta^2-2*beta+x*beta+x*alpha+N*beta-N-alpha+1+2*alpha*beta)*summ(beta-1)+summ(
beta)*(n+alpha+beta)*(beta+x))
),n))
then ok else r fi;

r := sumrecursion((-1)^n/n!*pochhammer(N-n,n)*pochhammer(beta+1,n)*
hyperterm([-n,alpha+beta+n+1,-x],[beta+1,1-N],1,k),k,N):
if not(has(simplify(r/
(-(n-1+beta+N+alpha)*(2+x-N)*summ(N-2)+(-3-2*x+n+2*beta+n*beta+2*x*N+x*beta-N*
beta+x*alpha-2*N^2+n^2+3*alpha+n*alpha+5*N-2*alpha*N)*summ(N-1)+summ(N)*(n-N+1)
*(-alpha+1-N+x))
),n))
then ok else r fi;

# Nikiforov, Suslov, Uvarov (2.4)
# Chebyshevt(n,N,x):=Hahn(n,N,0,0,x):=
# (-1)^n*pochhammer(N-n,n)*hypergeom([-n,n+1,-x],[1,1-N],1);

r := 
sumrecursion((-1)^n*pochhammer(N-n,n)*hyperterm([-n,n+1,-x],[1,1-N],1,k),k,n):
if not(has(simplify(r/
(-(n-1)*(n-1+N)*(n-1-N)*summ(n-2)-(2*n-1)*(1+2*x-N)*summ(n-1)+summ(n)*n)
),n))
then ok else r fi;

r := 
sumrecursion((-1)^n*pochhammer(N-n,n)*hyperterm([-n,n+1,-x],[1,1-N],1,k),k,x):
if not(has(simplify(r/
((-N+x-1)*(x-1)*summ(x-2)-(n^2+n-2*x+1+N-2*x*N+2*x^2)*summ(x-1)+summ(x)*x*(-N+
x))
),n))
then ok else r fi;

r := 
sumrecursion((-1)^n*pochhammer(N-n,n)*hyperterm([-n,n+1,-x],[1,1-N],1,k),k,N):
if not(has(simplify(r/
(-(n-1+N)*(2+x-N)*summ(N-2)+(5*N-2*N^2+n-2*x-3+2*x*N+n^2)*summ(N-1)+summ(N
)*(n-N+1)*(-N+1+x))
),n))
then ok else r fi;

# discrete Laguerre:
# Laguerrel(n,rho,alpha,x):=rho^n/n!*pochhammer(1+alpha,n)*
# hypergeom([-n,-x+1],[alpha+1],1-1/rho);

r := sumrecursion(rho^n/n!*pochhammer(1+alpha,n)*
hyperterm([-n,-x+1],[alpha+1],1-1/rho,k),k,n):
if not(has(simplify(r/
((alpha+n-1)*rho*summ(n-2)-(n+rho*alpha+n*rho+x*rho-x-rho)*summ(n-1)+summ(n)*n)
),n))
then ok else r fi;

r := sumrecursion(rho^n/n!*pochhammer(1+alpha,n)*
hyperterm([-n,-x+1],[alpha+1],1-1/rho,k),k,x):
if not(has(simplify(r/
((x-2)*summ(x-2)-(x+x*rho-2+n*rho-n-rho+rho*alpha)*summ(x-1)+summ(x)*rho*(
alpha+x-1))
),n))
then ok else r fi;

r := sumrecursion(rho^n/n!*pochhammer(1+alpha,n)*
hyperterm([-n,-x+1],[alpha+1],1-1/rho,k),k,alpha):
if not(has(simplify(r/
(-(alpha+n-1)*summ(alpha-2)-(rho*alpha+2+x*rho-2*alpha-x+n*rho-rho-n)*summ(
alpha-1)+summ(alpha)*(rho-1)*(alpha+x-1))
),n))
then ok else r fi;

#time()-TIME;

#quit;


