#
## <SHAREFILE=algebra/coxeter/coxeter.mpl>
## <DESCRIBE>
## ALSO INCLUDES: weyl package
##       SEE ALSO: algebra/coxeter/examples/*, algebra/coxeter/coxeter.tex
##
##               The coxeter package contains 30 basic procedures for studying
##              roots systems and finite Coxeter groups.  The weyl package is a
##                supplement to the coxeter package that contains 7 procedures
##                for manipulating weight vectors and computing multiplicities
##                for irreducible representations of semisimple Lie algebras.
##                AUTHOR: John Stembridge, jrs@math.lsa.umich.edu
## </DESCRIBE>

# COPYLEFT NOTICE:
# Copyleft (c) 1991 by John R. Stembridge.
#  
# Permission is granted to anyone to to use, modify, or redistribute this
# software freely, subject to the following restrictions:
# 
# 1. The author accepts no responsibility for any consequences of this
# software and makes no guarantee that the software is free of defects.
# 2. The origin of this software must not be misrepresented, either by
# explicit claim or by omission.
# 3. This notice and the copyleft must be included in all copies or
# modified versions of this software.
# 4. This software may not be included or redistributed as part of any
# package to be sold for profit unless the author has given explicit written
# permission to do so.
# 
# John Stembridge
# Department of Mathematics
# University of Michigan
# Ann Arbor, MI 48109-1003
# Internet:  jrs@math.lsa.umich.edu
#
###############################################################################
#
# base(R) returns a base for the root system R
# If a second argument is present, and R is not crystallographic, then base(R)
# returns floating-point coordinates. (Otherwise, non-crystal R will generate
# an error message).
#
`coxeter/base`:=proc() local sys,R,L,i,j,res,r,new;
  if args[1]=1 then RETURN([]) fi;
  sys:=sort([op(indets(args[1]))],`coxeter/cox_order`);
  sys:=[seq(R$degree(args[1],R),R=sys)];
  i:=0; res:=[];
  for R in sys do;
    r:=coxeter['rank'](R); 
    if not type(R,'indexed') then L:=substring(R,1..1) else L:=`I2` fi;
    if R=`A1` then
      new:=[e.(i+1)]
    elif L=`A` then 
      new:=[seq(e.(j+1)-e.j,j=i+1..i+r)]; i:=i+1;
    elif L=`B` then
      new:=[e.(i+1),seq(e.(j+1)-e.j,j=i+1..i+r-1)]
    elif L=`C` then
      new:=[2*e.(i+1),seq(e.(j+1)-e.j,j=i+1..i+r-1)]
    elif L=`D` then 
      new:=[e.(i+1)+e.(i+2),seq(e.(j+1)-e.j,j=i+1..i+r-1)]
    elif L=`E` then
     new:=[(e.(i+1)-e.(i+2)-e.(i+3)-e.(i+4)-e.(i+5)-e.(i+6)-e.(i+7)+e.(i+8))/2,
        e.(i+1)+e.(i+2),seq(e.(j+1)-e.j,j=i+1..i+r-2)]; i:=i+8-r;
    elif R=F4 then 
      new:=[(-e.(i+1)-e.(i+2)-e.(i+3)+e.(i+4))/2,
        e.(i+1),e.(i+2)-e.(i+1),e.(i+3)-e.(i+2)]
    elif R=G2 then 
      new:=[e.(i+2)-e.(i+1),e.(i+1)-2*e.(i+2)+e.(i+3)]; i:=i+1
    elif R=H3 or R=H4 or L=`I2` then 
      if nargs>1 then new:=`coxeter/base/kludge`(L,R,r,i)
        else ERROR(`This section of code still under construction`) fi;
    else
      ERROR(`Not a legal system`)
    fi;
    i:=i+r; res:=[op(res),op(new)];
  od;
  res;
end:
#
`coxeter/base/kludge`:=proc(L,R,r,i) local a,new;
  if L=`I2` then
      a:=op(R); new:=[e.(i+1),-cos(Pi/a)*e.(i+1)-sin(Pi/a)*e.(i+2)];
  elif L=`H` then
    a:=(1+sqrt(5))/2;
    new:=[a*e.(i+1)-e.(i+2)+(a-1)*e.(i+3),-a*e.(i+1)+e.(i+2)+(a-1)*e.(i+3),
      e.(i+1)+(a-1)*e.(i+2)-a*e.(i+3)];
    if r=4 then new:=[op(new),-e.(i+1)-a*e.(i+2)+(a-1)*e.(i+4)] fi;
  fi;
  evalf(new);
end:
#
#
`coxeter/char_poly`:=proc(w) local S,z,vars,A,v,n,i;
  if type(args[2],'list') then S:=args[2] else S:=coxeter['base'](args[2]) fi;
  vars:=[op(indets(S))]; A:=NULL; n:=nops(vars);
  for v in vars do;
    coxeter['reflect'](seq(S[i],i=w),v);
    A:=A,[seq(coeff(",vars[i]),i=1..n)];
  od;
  if n=0 then RETURN(1) fi;
  A:=array(1..n,1..n,[A]);
  linalg['matadd'](linalg['band']([1],n),A,1,-z);
  linalg['det'](")/(1-z)^(n-nops(S));
  expand(normal("));
  if nargs>2 then subs(z=args[3],") else subs(z=q,") fi;
end:
#
#
# class_rep(w,R) returns a canonical representative for the conjugacy class
#   of w.
# class_rep(R) returns the list of canonical representatives of all classes.
#
`coxeter/class_rep`:=proc() local n,R,N,res,new,sys,i,j,k,w,L,T;
  R:=args[nargs]; N:=0;
  sys:=sort([op(indets(R))],`coxeter/cox_order`);
  sys:=[seq(i$degree(R,i),i=sys)];
  if nargs=1 then res:=[[]];
    for R in sys do;
      n:=coxeter['rank'](R);
      if type(R,'indexed') then L:=op(0,R) else L:=substring(R,1..1) fi;
      new:=`coxeter/class_rep/list`(L,n,R);
      res:=[seq(seq([op(i),seq(k+N,k=j)],i=res),j=new)];
      N:=N+n;
    od;
  else res:=[];
    for R in sys do;
      n:=coxeter['rank'](R);
      w:=map(proc(x,y,z) if x>y and x<=z then x-y fi end,args[1],N,N+n);
      if type(R,'indexed') then L:=op(0,R) else L:=substring(R,1..1) fi;
      new:=`coxeter/mytype`(w,L,n,R);
      if L=`C` then L:=`B` fi;
      if member(L,[`A`,`B`,`D`]) then new:=`coxeter/par2`.L(new)
        elif member(L,[`E`,`F`,`H`]) then 
          #new:=readlib(evaln(`coxeter/class_rep/`.R),
          #  ``.HomeLib.`/coxeter/lib/`.R.`_classes.m`)[new]
	  T:=(`coxeter/class_rep/`.R); new := T[new]
        else new:=`coxeter/class_rep/list`(L,n,R)[new]
      fi;
      res:=[op(res),seq(i+N,i=new)]; N:=N+n;
    od;
  fi;
  res;
end:
#
`coxeter/class_rep/list`:=proc(L,n,R) local i,j,mu,nu,m;
  if L=`A` then
    [seq(`coxeter/par2A`(mu),mu=combinat['partition'](n+1))]
  elif L=`B` or L=`C` then
    [seq(seq(seq(`coxeter/par2B`(mu,nu),mu=combinat['partition'](n-i)),
      nu=combinat['partition'](i)),i=0..n)]
  elif L=`D` then
    [seq(seq(seq(`coxeter/par2D`(mu,nu),mu=combinat['partition'](n-i)), 
      nu=combinat['partition'](i)),i=0..n)];
    if modp(n,2)=0 then [op("),seq(`coxeter/par2D`(map(x->(2*x),mu),[],-1),
      mu=combinat['partition'](n/2))] else " fi;
  elif member(L,[`E`,`F`,`H`]) then
    #readlib(evaln(`coxeter/class_rep/`.R),
    #  ``.HomeLib.`/coxeter/lib/`.R.`_classes.m`);
    `coxeter/class_rep/`.R
  elif R=G2 then [[],[1],[2],[1,2],[1,2,1,2],[1,2,1,2,1,2]]
  elif L=`I2` then m:=op(R);
    if modp(m,2)=0 then
      [[],[1],[2],seq([seq(1+modp(i,2),i=0..2*j-1)],j=1..m/2)]
    else
      [[],[1],seq([seq(1+modp(i,2),i=0..2*j-1)],j=1..(m-1)/2)]
    fi;
  else
    ERROR(`Not a finite Coxeter group`)
  fi;
end: 
#
#
# class_size(w,R) returns the size of the conjugacy class of w.
# class_size(R) returns the sizes of all conjugacy classes.
#
`coxeter/class_size`:=proc() local n,R,N,res,new,sys,i,j,w,L,T;
  R:=args[nargs];
  sys:=sort([op(indets(R))],`coxeter/cox_order`);
  sys:=[seq(i$degree(R,i),i=sys)];
  if nargs=1 then res:=[1];
    for R in sys do;
      n:=coxeter['rank'](R);
      if type(R,'indexed') then L:=op(0,R) else L:=substring(R,1..1) fi;
      new:=`coxeter/class_size/list`(L,n,R);
      res:=[seq(seq(i*j,i=res),j=new)];
    od;
  else
    N:=0; res:=1;
    for R in sys do;
      n:=coxeter['rank'](R);
      w:=map(proc(x,y,z) if x>y and x<=z then x-y fi end,args[1],N,N+n);
      if type(R,'indexed') then L:=op(0,R) else L:=substring(R,1..1) fi;
      new:=`coxeter/mytype`(w,L,n,R);
      if L=`A` then `coxeter/par2A`(new,0) 
        elif L=`B` or L=`C` then `coxeter/par2B`(new,0) 
        elif L=`D` then `coxeter/par2D`(op(1..2,[new]),0)
        elif member(L,[`E`,`F`,`H`]) then 
          #readlib(evaln(`coxeter/class_size/`.R),
          #  ``.HomeLib.`/coxeter/lib/`.R.`_sizes.m`)[new]
          T := `coxeter/class_size/`.R; new := T[new]
        else `coxeter/class_size/list`(L,n,R)[new]
      fi;
      res:=res*"; N:=N+n;
    od;
  fi;
  res;
end:
#
`coxeter/class_size/list`:=proc(L,n,R) local i,mu,nu,m;
  if L=`A` then
    [seq(`coxeter/par2A`(mu,0),mu=combinat['partition'](n+1))]
  elif L=`B` or L=`C` then
    [seq(seq(seq(`coxeter/par2B`(mu,nu,0),mu=combinat['partition'](n-i)),
      nu=combinat['partition'](i)),i=0..n)]
  elif L=`D` then
    [seq(seq(seq(`coxeter/par2D`(mu,nu,0),mu=combinat['partition'](n-i)), 
      nu=combinat['partition'](i)),i=0..n)];
    if modp(n,2)=0 then [op("),seq(`coxeter/par2D`(map(x->(2*x),mu),[],0),
      mu=combinat['partition'](n/2))] else " fi
  elif member(L,[`E`,`F`,`H`]) then
    #readlib(evaln(`coxeter/class_size/`.R),
    #  ``.HomeLib.`/coxeter/lib/`.R.`_sizes.m`)
    `coxeter/class_size/`.R
  elif R=G2 then [1,3,3,2,2,1]
  elif L=`I2` then m:=op(R);
    if modp(m,2)=0 then [1,m/2,m/2,2$(m/2-1),1]
      else [1,m,2$((m-1)/2)] fi;
  else
    ERROR(`Not a finite Coxeter group`)
  fi;
end: 
#
#
#  This is H/I-compatible, but in a kludgey way.
#
`coxeter/cox_matrix`:=proc() local S,n,M,i,j,len,m;
  if type(args[1],'list') then S:=args[1]
    else S:=coxeter['base'](args[1],0) fi;     #Here's the Kludge
  # Patch MBM Feb/93 - array([]) doesn't work in Release 3
  if S=[] then RETURN(array(1..1)) fi;
  n:=nops(S); M:=array(symmetric,1..n,1..n);
  len:=[seq(coxeter['iprod'](S[i],S[i]),i=1..n)];
  for i to n do;
    M[i,i]:=1;
    for j to i-1 do;
      -coxeter['iprod'](S[i],S[j])/sqrt(len[i]*len[j]);
      m:=evalf(Pi/arccos("),10); M[i,j]:=round(m);
      if abs(M[i,j]-m)>.001 then ERROR(`Not a Coxeter system`) fi;
    od;
  od;
  op(M);
end:
#
#
# cox_order() is the ordering relation for irreducible root systems
#
`coxeter/cox_order`:=proc(R1,R2) local L1,L2;
  if not type(R1,'indexed') then L1:=substring(R1,1..1) 
    elif op(0,R1)='I2' then L1:=`I2` else ERROR(`Not a Coxeter system`) fi;
  if not type(R2,'indexed') then L2:=substring(R2,1..1) 
    elif op(0,R2)='I2' then L2:=`I2` else ERROR(`Not a Coxeter system`) fi;
  if L1<>L2 then lexorder(L1,L2)
    elif L1=`I2` then evalb(op(R1)<=op(R2))
    elif length(R1)=length(R2) then lexorder(R1,R2)
    else evalb(length(R1)<length(R2))
  fi;
end:
#
#
#  size(R)=size of the Coxeter group of R
#  exponents(R)=list of exponents of R
#  num_refl(R)=number of reflections in the Coxeter group of R
#  cox_number(R)=order of the Coxeter element of W(R)
#  degrees(R)=degrees of the fundamental invariants of W(R)
#
`coxeter/size`:=proc(R) convert(`coxeter/degrees`(R),`*`) end:
`coxeter/exponents`:=proc(R) map(proc(x) x-1 end,`coxeter/degrees`(R)) end:
`coxeter/num_refl`:=proc(R) convert(`coxeter/exponents`(R),`+`) end:
`coxeter/cox_number`:=proc(R) local r;
  ilcm(seq(2*`coxeter/num_refl`(r)/coxeter['rank'](r),r=indets(R)));
end:
#
`coxeter/degrees`:=proc() local R,L,r,k,i;
  if args[1]=1 then RETURN([]) fi;
  R:=indets(args[1])[1]; 
  k:=degree(args[1],R); r:=coxeter['rank'](R);
  if not type(R,'indexed') then L:=substring(R,1..1) else L:=`I2` fi;
  if L=`A` then [$2..(r+1)]
    elif L=`B` or L=`C` then [seq(2*i,i=1..r)]
    elif L=`D` then [seq(2*i,i=1..r-1),r]
    elif R=E6 then [2,5,6,8,9,12]
    elif R=E7 then [2,6,8,10,12,14,18]
    elif R=E8 then [2,8,12,14,18,20,24,30]
    elif R=F4 then [2,6,8,12]
    elif R=G2 then [2,6]
    elif R=H3 then [2,6,10]
    elif R=H4 then [2,12,20,30]
    elif L=`I2` then [2,op(R)]
    else
      ERROR(`Not a legal system`)
  fi;
  map(op,["$k,`coxeter/degrees`(args[1]/R^k)]);
  sort(");
end:
#
#
# descent_gf(R,q) returns the generating function for w according to the
#   number of descents. (=the h-polynomial of the Coxeter complex of R).
# descent_gf(R,[x1,x2,...,x.n]) returns the generating function for w
#   according to the descent set of w.
# descent_gf(R) is an abbreviation for descent_gf(R,q).
#
`coxeter/descent_gf`:=proc(R) local n,vars,res,M,J,R0,i;
  n:=coxeter['rank'](R);
  if nargs=1 then vars:=[q$n]
    elif type(args[2],'list') then vars:=args[2] 
    else vars:=[args[2]$n]
  fi;
  M:=coxeter['cox_matrix'](R); res:=0;
  for J in combinat['powerset']([$1..n]) do:
    if J=[] then [] else linalg['submatrix'](M,J,J) fi;
    R0:=coxeter['name_of'](");
    convert(subsop(seq(i=1-vars[i],i=J),vars),`*`);
    res:=res+"/coxeter['size'](R0);
  od:
  expand(coxeter['size'](R)*res);
end:
#

#
#  diagram(R) prints a Dynkin diagram for R, with the nodes numbered in the
#   way used by base(R). The diagrams for G2, H3-4, and I2 aren't pretty.
#
`coxeter/diagram`:=proc() local sys,R,L,i,r,j;
  lprint(); j:=1;
  sys:=sort([op(indets(args[1]))],`coxeter/cox_order`);
  sys:=[seq(R$degree(args[1],R),R=sys)];
  for R in sys do;
    r:=coxeter['rank'](R); 
    if type(R,'indexed') then L:=`I2` else L:=substring(R,1..1)  fi;
    if L=`A` then 
      lprint(cat(`   `.j,seq(`---`.i,i=j+1..j+r-1)))
    elif L=`B` then
      lprint(cat(`   `.j.`=<=`,j+1,seq(`---`.i,i=j+2..j+r-1)))
    elif L=`C` then
      lprint(cat(`   `.j.`=>=`,j+1,seq(`---`.i,i=j+2..j+r-1)))
    elif L=`D` then 
      lprint(cat(`      `,` `$length(j),j+1));
      lprint(cat(`      `,` `$length(j),`|`));
      lprint(cat(`   `.j,seq(`---`.i,i=j+2..j+r-1)))
    elif L=`E` then
      lprint(cat(`         `,` `$(length(j)+length(j+2)),j+1));
      lprint(cat(`         `,` `$(length(j)+length(j+2)),`|`));
      lprint(cat(`   `.j,seq(`---`.i,i=j+2..j+r-1)))
    elif R=F4 then 
      lprint(cat(`   `,j,`---`,j+1,`=<=`,j+2,`---`,j+3))
    elif R=G2 then 
      lprint(cat(`   `,j,`=<=`,j+1,`  [G2]`))
    elif R=H3 or R=H4 then 
      lprint(cat(`   `,j,`-*-`,j+1,seq(`---`.i,i=j+2..j+r-1),`  [5]`))
    elif L=`I2` then 
      lprint(cat(`   `,j,`-*-`,j+1,`  [`,op(R),`]`))
    else
      ERROR(`Not a legal system`)
    fi;
    j:=j+r; lprint();
  od;
end:
#

#
# This procedure rejects the input if R is reducible. It gives a spurious
#  answer if R is not crytallographic--highest roots are not defined in
#  such cases. 
#
`coxeter/highest_root`:=proc() local S,v,l,ll,j;
  if type(args[1],'list') then S:=args[1] else
    if degree(args[1])>1 then ERROR(`root system not irreducible`) fi;
    S:=coxeter['base'](args[1])
  fi;
  ll:=0; v:=0;
  for j to nops(S) do;
    l:=coxeter['iprod'](S[j],S[j]);
    if l>ll then v:=S[j]; ll:=l fi;
  od;
  coxeter['vec2fc'](v,S);
end:
#

#
# procedure to produce an interior point for the positive orthant defined
# by a list of independent hyperplanes. The coordinates must be real.
#  interior_pt(R) will produce a point in the interior of the fundamental
# chamber defined by base(R). If a second argument is present, it allows
# the kludge for non-crystallographic R.
# 
`coxeter/interior_pt`:=proc() local i,v,c,S,eqns;
  S:=args[1]; 
  if not type(S,'list') then S:=coxeter['base'](args) fi;
  v:=convert([seq(c[i]*S[i],i=1..nops(S))],`+`);
  v:=collect(v,[op(indets(S))]);
  eqns:={seq(coxeter['iprod'](S[i],v)=1,i=1..nops(S))};
  subs(solve(eqns),v);
end:
#

#
`coxeter/length_gf`:=proc(R) local d,z;
  [seq((1-z^d)/(1-z),d=coxeter['degrees'](R))];
  expand(normal(convert(",`*`)));
  if nargs=2 then subs(z=args[2],") else subs(z=q,") fi;
end:
#
#
# Let S be a set of equations of the form {s1=[perm1],s2=[perm2],...}
# where the perms are permutations in disjoint cycle notation.
# If w=[w1,w2,w3,...] is a sequence of integers, then
# multperm(w,S) computes the product of s.w1,s.w2,... in disjoint cycle form.
# 
`coxeter/multperm`:=proc(w,S) local n,i,pi,c,j,left,res,cyc,Done;
  n:=max(op(map(x->max(op(map(op,op(2,x)))),S)));
  for i in {op(w)} do;
    pi[i]:=table([$1..n]);
    for c in subs(S,s.i) do;
      pi[i][c[nops(c)]]:=c[1];
      for j to nops(c)-1 do pi[i][c[j]]:=c[j+1] od;
    od;
  od;
  left:={$1..n}; res:=[];
  while nops(left)>0 do;
    i:=min(op(left)); cyc:=[i]; Done:=evalb(1=0);
    while not Done do;
      for j in w do i:=pi[j][i] od;
      if i=cyc[1] then Done:=not Done else cyc:=[op(cyc),i] fi;
    od;
    left:=left minus {op(cyc)};
    if nops(cyc)>1 then res:=[op(res),cyc] fi;
  od;
  res;
end:
#
#
# generate the label for the conjugacy class of w.
#
`coxeter/mytype`:=proc(w,L,n,R) local S,i,j,mu,nu,pi,sig,Q,w0,m;
  if L=`A` then
    mu:=map(nops,coxeter['multperm'](w,{seq(s.i=[[i,i+1]],i=1..n)}));
    [1$(n+1-convert(mu,`+`)),op(sort(mu))];
  elif L=`B` or L=`C` then
    {s1=[[1,2]],seq(s.i=[[2*i-3,2*i-1],[2*i-2,2*i]],i=2..n)};
    pi:=coxeter['multperm'](w,"); mu:=[]; nu:=[];
    for S in pi do;
      j:=min(op(S)); 
      if modp(j,2)=1 then
        if member(j+1,S) then nu:=[op(nu),nops(S)/2]
        else mu:=[op(mu),nops(S)] fi;
      fi;
    od;
    mu:=[1$(n-convert([op(mu),op(nu)],`+`)),op(sort(mu))];
    mu,sort(nu);
  elif L=`D` then
    {s1=[[1,4],[2,3]],seq(s.i=[[2*i-3,2*i-1],[2*i-2,2*i]],i=2..n)};
    pi:=coxeter['multperm'](w,"); mu:=[]; nu:=[]; sig:=1;
    for S in pi do; 
      j:=min(op(S));  
      if modp(j,2)=1 then
        sig:=sig*(-1)^(convert(map(modp,S,2),`+`));
        if member(j+1,S) then nu:=[op(nu),nops(S)/2]
        else mu:=[op(mu),nops(S)] fi;
      fi; 
    od; 
    nu:=sort(nu); mu:=[1$(n-convert([op(mu),op(nu)],`+`)),op(sort(mu))];
    if nu=[] and convert(map(modp,mu,2),`+`)=0 then mu,nu,sig else mu,nu fi;
  elif L=`E` then
    coxeter['char_poly'](w,R,Q); [seq(coeff(",Q,i),i=0..n)];
    #readlib(evaln(`coxeter/mytype/`.R),
    #  ``.HomeLib.`/coxeter/lib/`.R.`_types.m`);
    `coxeter/mytype/`.R;
    if not member("",",'j') then ERROR()
    elif n=7 and member(j,[10,21,23,40,51,53]) then 
      pi:=map(nops,coxeter['multperm'](w,op(2,coxeter['perm_rep'](R))));
      if (j=10 or j=40) and nops(pi)=28 then j:=j+1
        elif (j=21 or j=51) and nops(pi)=18 then j:=j+1
        elif (j=23 or j=53) and convert(pi,`+`)=56 then j:=j+1
      fi;
    elif n=8 and member(j,[5,20,23,49,65]) then
      pi:=map(nops,coxeter['multperm'](w,op(2,coxeter['perm_rep'](R,120))));
      if j=5 and nops(pi)=56 then j:=j+1
        elif j=20 and nops(pi)=36 then j:=j+1
        elif j=23 and nops(pi)=32 then j:=j+1
        elif j=49 and nops(pi)=26 then j:=j+1
        elif j=49 and convert(pi,`+`)=118 then j:=j+2
        elif j=65 and nops(pi)=16 then j:=j+1
      fi;
    fi; j
  elif R=F4 then
    sig:=convert(subs({1=-1,2=-1,3=1,4=1},w),`*`);
    sig*coxeter['char_poly'](w,R,Q);
    [seq(coeff(",Q,i),i=0..n)];    
    #readlib('`coxeter/mytype/F4`',``.HomeLib.`/coxeter/lib/F4_types.m`);
    `coxeter/mytype/F4`;
    if not member("",",'j') then ERROR() fi;
    if j=4 then
      S:=op(2,coxeter['perm_rep'](F4));
      if nops(coxeter['multperm'](w,S))=8 then j:=j+1 fi;
    elif j=20 then
      S:=op(2,coxeter['perm_rep'](F4));
      if nops(coxeter['multperm'](w,S))=6 then j:=j+1 fi;
    fi; j
  elif L=`H` then
    coxeter['char_poly'](w,coxeter['base'](R,0),Q);
    [seq(round(2*coeff(",Q,i)),i=1..n)];
    #readlib(evaln(`coxeter/mytype/`.R),
    #  ``.HomeLib.`/coxeter/lib/`.R.`_types.m`);
    `coxeter/mytype/`.R;
    if member("",",'j') then j else ERROR() fi;
  elif L=`I2` or R=G2 then
    w0:=coxeter['reduce'](w,R);
    if R=G2 then m:=6 else m:=op(R) fi;
    if w0=[] then 1
      elif modp(nops(w0),2)=1 then
        if modp(m,2)=1 then 2 else 1+w0[(nops(w0)+1)/2] fi
      else nops(w0)/2+3-modp(m,2)
    fi
  else
    ERROR(`Not a finite Coxeter group`)
  fi;
end:
#
# name_of(S) produces the name of the Coxeter group generated by the simple
#   roots S.
# name_of(M) does the same for any Coxeter matrix M. Note that the matrix
#   M cannot distinguish between types B and C. 
# name_of(S,'pi') does the same but also assigns to 'pi' a permutation of
#   [1,...,n] (n=nops(S)) indicating the proper ordering of the roots S
#   agreeing with the convention used by base().
# name_of(M,'pi') does the same, but produces the ordering used by
#   cox_matrix().
#
`coxeter/name_of`:=proc()
  local X,Y,res,M,i,j,n,m,S,bonds,edge,e,CC,nghbr,R,ord,tord;
  if type(args[1],'matrix') then
      M:=args[1]; n:=linalg['rowdim'](M); S:=0;
    elif type(args[1],'table') then S:=[]; n:=0
    else S:=args[1]; n:=nops(S); M:=coxeter['cox_matrix'](S)
  fi;
  X:=[$1..n]; bonds:=table([3$n]);
  res:=[]; e:=0; tord:=[];
  for i to n do;
    nghbr[i]:=NULL;
    for j to i-1 do;
      if M[i,j]=2 then next fi;
      if M[i,j]>3 then
        if max(bonds[X[i]],bonds[X[j]])=3 then
          bonds[X[j]]:=M[i,j]; edge[X[j]]:={i,j} 
          else ERROR(`Not a finite Coxeter group`)
        fi;
      fi;
      X:=subs(X[i]=X[j],X); e:=e+1;
      nghbr[i]:=nghbr[i],j; nghbr[j]:=nghbr[j],i
    od;
  od;
  CC:={op(X)}; Y:=table([seq((i)=NULL,i=CC)]);
  if nops(CC)<>n-e then ERROR(`Not a finite Coxeter group`) fi;
  for i to n do Y[X[i]]:=Y[X[i]],i od;
  for i in CC do;
    m:=nops([Y[i]]);
    if nargs=1 then
      R:=`coxeter/name_of/type`(nghbr,[Y[i]],bonds[i],edge[i])
    else
      R:=`coxeter/name_of/type`(nghbr,[Y[i]],bonds[i],edge[i],'ord');
      if S<>0 and member(R,[B2,G2,F4]) and
        `coxeter/name_of/longer`(S[ord[1]],S[ord[m]])
        then ord:=[seq(ord[m-j],j=0..m-1)]
      fi;
      tord:=[op(tord),ord];
    fi;
    if S<>0 and R=B.m and m>2 then
      if nops([nghbr[edge[i][1]]])=2 then j:=edge[i][2] else j:=edge[i][1] fi;
      if `coxeter/name_of/longer`(S[j],S[nghbr[j]]) then R:=C.m fi;
    fi;
    res:=[op(res),R];
  od;
  if nargs>1 then res:=sort([seq([res[i],tord[i]],i=1..nops(CC))],
    proc(x,y) `coxeter/cox_order`(x[1],y[1]) end);
    assign(args[2],map(x->op(x[2]),res));
    res:=map(x->x[1],res);
  fi;
  convert(res,`*`);
end:
#
`coxeter/name_of/type`:=proc(N,X,b,e) local n,i,k,fork,leafs,J,K;
  n:=nops(X);
  if n<3 then
    if nargs=5 then assign(args[5],X) fi;
    if n=1 then RETURN(A1)
      elif b=3 then RETURN(A2)
      elif b=4 then RETURN(B2)
      elif b=6 then RETURN(G2)
      else RETURN(I2[b])
    fi;
  fi;
  leafs:={}; fork:=NULL;
  for i in X do;
    k:=nops([N[i]]);
    if k=1 then leafs:={op(leafs),i} fi;
    if k=3 then fork:=i fi;
  od;
  if nops(leafs)=3 then
    if b>3 then ERROR(`Not a finite Coxeter group`) fi;
    J:=leafs intersect {N[fork]};
    if nops(J)>1 then
      if nargs=5 then 
        `coxeter/name_of/path`(fork,op(leafs minus {J[1],J[2]}),N);
        assign(args[5],[J[1],J[2],op(")])
      fi;
      RETURN(D.n)
    elif nops(J)=1 and n<9 then
      K:={N[fork]} intersect {seq(N[i],i=leafs)};
      if K={} then
        ERROR(`Not a finite Coxeter group`)
      elif nargs=5 then
        k:=op(leafs intersect {N[K[1]]});
        `coxeter/name_of/path`(fork,op(leafs minus {k,J[1]}),N);
        assign(args[5],[k,J[1],K[1],op(")]);
      fi;
      RETURN(E.n);
    fi;
  elif nops(leafs)=2 then
    if nargs=5 then assign(args[5],`coxeter/name_of/path`(op(leafs),N)) fi;
    if b=3 then RETURN(A.n) fi;
    J:=leafs intersect e;
    if nops(J)=0 then if n=4 and b=4 then RETURN(F4)
      else ERROR(`Not a finite root system`) fi fi;
    if nargs=5 and leafs[2]=J[1] then
      assign(args[5],`coxeter/name_of/path`(leafs[2],leafs[1],N)) fi; 
    if b=4 then RETURN(B.n) fi;
    if b=5 and (n=3 or n=4) then RETURN(H.n) fi;
  fi;
  ERROR(`Not a finite Coxeter group`);
end:
#
# Find the unique path from vertex i to leaf j
#
`coxeter/name_of/path`:=proc(i,j,N) local path,k,old;
  path:=[j]; k:=j; old:={};
  while k<>i do;
    op({N[k]} minus old); old:={k}; k:=""; path:=[k,op(path)];
  od;
  path;
end:
#
`coxeter/name_of/longer`:=proc(u,v)
  evalb(coxeter['iprod'](u,u)>coxeter['iprod'](v,v))
end:
#

#
# orbit(v,R) will return the list of all distinct images of the vector v
# under the action of the Weyl group of R.
#
`coxeter/orbit`:=proc(v) local S,res,v0,sat,r,vv;
  if type(args[2],'list') then S:=args[2] else S:=coxeter['base'](args[2]) fi;
  res:=[v]; sat:=0;
  while sat<nops(res) do;
    v0:=res[sat+1];
    for r in S do;
      vv:=coxeter['reflect'](r,v0);
      if not member(vv,res) then res:=[op(res),vv] fi;
    od;
    sat:=sat+1;
  od;
  res;
end:
#
#
# orbit_size(v,R) will produce the size of the orbit of the vector v under the
#   action of the Weyl group of R.
# orbit_size(v,R,-1) will do the same, but for the group generated by -1 and
#   the Weyl group of R.
# R can also be a base S.
#
`coxeter/orbit_size`:=proc() local v,R,S,res,R0,u;
  if type(args[2],'list') then S:=args[2]; R:=coxeter['name_of'](S)
    else R:=args[2]; S:=coxeter['base'](R) fi;
  v:=coxeter['vec2fc'](args[1],S);
  map(proc(x,y) if coxeter['iprod'](x,y)=0 then x fi end,S,v);
  R0:=coxeter['name_of'](");
  res:=coxeter['size'](R)/coxeter['size'](R0);
  if nargs=2 then RETURN(res) fi;
  u:=coxeter['vec2fc'](-args[1],S);
  if u=v then res else 2*res fi;
end:
#
#
# par2A(mu)   = canonical rep of type mu
# par2A(mu,*) = size of class indexed by mu
#
`coxeter/par2A`:=proc(mu) local sz,rep,mul,j,i;
  if nargs=1 then if mu=[] then RETURN([]) fi;
    j:=mu[nops(mu)]; rep:=[$1..j-1];
    for i from nops(mu)-1 by -1 to 1 do;
      rep:=[op(rep),$j+1..j+mu[i]-1]; j:=j+mu[i];
    od; rep;
  else if mu=[] then RETURN(1) fi;
    sz:=mu[1]; mul:=1;
    for i from 2 to nops(mu) do;
      if mu[i]=mu[i-1] then mul:=mul+1 else mul:=1 fi;
      sz:=sz*mul*mu[i];
    od; convert(mu,`+`)!/sz;
  fi;
end:
#
# par2B(mu,nu)   = canonical rep of type (mu,nu)
# par2B(mu,nu,*) = size of class indexed by (mu,nu)
#
`coxeter/par2B`:=proc(mu,nu) local m,n,rep,j,i,rep1,k;
  n:=convert(nu,`+`); m:=convert(mu,`+`);
  if nargs=2 then
    rep1:=[seq(i+n+1,i=`coxeter/par2A`(mu))];
    if nu=[] then RETURN(rep1) fi;
    j:=nu[nops(nu)]+1; rep:=[$1..j-1];
    for i from nops(nu)-1 by -1 to 1 do;
      rep:=[op(rep),seq(-k,k=-j..-1),$2..j+nu[i]-1]; j:=j+nu[i];
    od; [op(rep),op(rep1)]; 
  else
    2^(m+n-nops(mu)-nops(nu))*binomial(m+n,m);
    "*`coxeter/par2A`(mu,0)*`coxeter/par2A`(nu,0);
  fi;
end:
#
# par2D(mu,nu)    = canonical rep of type (mu,nu) (or NULL if nops(nu) odd)
# par2D(mu,[],+1) = canonical rep of type (mu,+), if all mu even
# par2D(mu,[],-1) = canonical rep of type (mu,-), if all mu even
# par2D(mu,nu,0)  = size of class(es) indexed by (mu,nu) (NULL if nops(nu) odd)
#
#
`coxeter/par2D`:=proc(mu,nu) local res,Odd,j;
  if modp(nops(nu),2)=1 then NULL 
  elif nargs=2 then
    res:=`coxeter/par2B`(mu,nu); Odd:=evalb(1=0);
    for j from nops(res) by -1 to 1 do;
      if res[j]=1 then res:=subsop(j=NULL,res); Odd:=not Odd
        elif res[j]=2 and Odd then res:=subsop(j=1,res) fi;
    od; res
  elif args[3]=1 then   
    `coxeter/par2B`(mu,nu)
  elif args[3]=-1 then
    subs(2=1,`coxeter/par2B`(mu,nu))
  else
    `coxeter/par2B`(mu,nu,0);
    if nu=[] and convert([seq(modp(j,2),j=mu)],`+`)=0 then "/2 else " fi
  fi;
end:
#
#
# R should be a root system, S0 a list of roots of R that form a simple
# subsystem. The output will be a list of character values on the various
# conjugacy classes on W(R).
#
`coxeter/perm_char`:=proc(S0,R) local S,w,gens,R0,pi,cc,cc0,sz,sz0,res,v,i,j;
  R0:=coxeter['name_of'](S0,pi);
  S:=coxeter['base'](R,0); gens:=[];
  v:=coxeter['interior_pt'](S);
  for j to nops(S0) do;
    coxeter['vec2fc'](coxeter['reflect'](S0[pi[j]],v),S,'w');
    gens:=[op(gens),w];
  od;
  cc:=coxeter['class_rep'](R);
  sz:=coxeter['class_size'](R);
  res:=[0$nops(sz)];
  cc0:=coxeter['class_rep'](R0);
  sz0:=coxeter['class_size'](R0);
  for i to nops(sz0) do;
    w:=map(op,[seq(gens[j],j=cc0[i])]);
    w:=coxeter['class_rep'](w,R);
    if not member(w,cc,'j') then ERROR() fi;
    res:=subsop(j=res[j]+sz0[i],res);
  od;
  coxeter['size'](R)/coxeter['size'](R0);
  [seq(res[i]*"/sz[i],i=1..nops(res))];
end:
#
#
`coxeter/perm_rep`:=proc(R) local g,n,sg,i;
  if type(eval(permrep),`string`) then with(group,permrep) fi;
  if nargs=2 and args[2]=120 and R=E8 then 
    #readlib( evaln(`coxeter/perm_rep/E8_120`),
    #``.HomeLib.`/coxeter/lib/E8_repn_120.m`)
    `coxeter/perm_rep/E8_120`
  elif member(R,[F4,E6,E7,E8,H4]) then
    #readlib(evaln(`coxeter/perm_rep/`.R),
    #``.HomeLib.`/coxeter/lib/`.R.`_repn.m`)
    `coxeter/perm_rep/`.R
  else
    g:=`coxeter/presentation`(R);
    n:=nops(op(1,g));
    sg:=subgrel({seq(s.i=[s.i],i=1..n-1)},g);
    if n>0 then permrep(sg) else permgroup(1,{}) fi;
  fi;
end:
#
`coxeter/presentation`:=proc(R) local n,M,i,j,gens,rels;
  M:=coxeter['cox_matrix'](R);
  if type(M,'matrix') then n:=linalg['rowdim'](M) else n:=0 fi;
  gens:={seq(s.i,i=1..n)};
  rels:={seq([s.i,s.i],i=1..n),
    seq(seq(map(op,[[s.j,s.i]$M[i,j]]),j=1..i-1),i=2..n)};
  grelgroup(gens,rels);
end:
#

#
# pos_roots(R) returns the list of all positive roots of R.
# pos_roots(S) does the same, but uses base S.
#
`coxeter/pos_roots`:=proc() local S,res,v,v0,sat,r;
  if type(args[1],'list') then S:=args[1]
    else S:=coxeter['base'](args[1]) fi;
  res:=S; sat:=0;
  while sat<nops(res) do;
    v0:=res[sat+1];
    for r in S do;
      if r=v0 then next fi;
      v:=coxeter['reflect'](r,v0);
      if not member(v,res) then res:=[op(res),v] fi;
    od;
    sat:=sat+1;
  od;
  res;
end:
#
 
#
# rank(R) returns the rank of root system R. R is assumed to be a 
# monomial in the irreducible root systems.
#
`coxeter/rank`:=proc() local R,k;
  if args[1]=1 then RETURN(0) fi;
  R:=indets(args[1])[1]; 
  k:=degree(args[1],R);
  if type(R,'indexed') then 2*k+`coxeter/rank`(args[1]/R^k) 
    else k*`coxeter/rank/number`(R)+`coxeter/rank`(args[1]/R^k) fi;
end:
#  
`coxeter/rank/digits`:=[`0`,`1`,`2`,`3`,`4`,`5`,`6`,`7`,`8`,`9`]:
#
`coxeter/rank/number`:=proc(R) local s,n,i,j;
  n:=0; s:=substring(R,2..length(R));
  for i to length(s) do;
    member(substring(s,i..i),`coxeter/rank/digits`,'j');
    n:=10*n+j-1;
  od; n;
end:
#
#
# reduce([i_1,...,i_r],S) returns a reduced expression representing the
#  same element of the Coxeter group as [i_1,...,i_r].
# reduce([i1,i2,...,i_l],R) does the same, using base(R).
#  If a third argument is present, this procedure returns a reduced _subword_ 
#  using the (quadratic) exchange property.
#
`coxeter/reduce`:=proc(w) local S,w0,i,v;
  if type(args[2],'list') then S:=args[2]
    else S:=coxeter['base'](args[2],0) fi;
  if nargs>2 then
    `coxeter/reduce/subwd`(w,S) 
  else
    v:=coxeter['reflect'](seq(S[i],i=w),coxeter['interior_pt'](S));
    coxeter['vec2fc'](v,S,'w0'); w0
  fi;
end:
#
`coxeter/reduce/subwd`:=proc(w,S) local w0,r,i;
  if nops(w)<2 then RETURN(w) fi;
  w0:=`coxeter/reduce/subwd`(w[1..nops(w)-1],S);
  r:=S[w[nops(w)]];
  for i from nops(w0) by -1 to 1 do;
    if S[w0[i]]=r then RETURN(subsop(i=NULL,w0))
      else r:=coxeter['reflect'](S[w0[i]],r) fi;
  od;
  [op(w0),w[nops(w)]];
end:
#
#
#  reflect(r,v) applies the reflection through r to vector v.
#  reflect(r1,r2,...,v) iterates the application of the reflections
#    r1,r2,... to vector v.
#
#  iprod(v1,v2) computes the standard inner product of two vectors,
#    expressed as a linear combo of the standard basis e1,e2,e3,...
#
#  WARNING: the coefficients of v1 MUST be rational (not symbolic,
#  not irrational).
#
`coxeter/reflect`:=proc() local i,v,r;
  v:=args[nargs];
  for i from nargs-1 by -1 to 1 do;
    r:=args[i];
    v:=v-2*(`coxeter/iprod`(r,v)/`coxeter/iprod`(r,r))*r;
  od;
  v;
end:
#
`coxeter/iprod`:=proc(u,v) local x;
  [seq(coeff(u,x)*coeff(v,x),x=indets(u))];
  convert(",`+`);
end:
#

#
# root_coords(v,R) returns the coordinates of v with respect to the
#  base vectors of R.
# root_coords(v,S), where S is the base of R, does the same.
#
`coxeter/root_coords`:=proc(v) local S,i,c;
  if type(args[2],'list') then S:=args[2] else S:=coxeter['base'](args[2]) fi;
  convert([seq(c[i]*S[i],i=1..nops(S))],`+`);
  coeffs(expand("-v),indets(S));
  subs(solve({"}),[seq(c[i],i=1..nops(S))]);
end:
#

#
# vec2fc(v,R) produces the unique vector v0 in the orbit of v that belongs to
#   the fundamental chamber of R.
# vec2fc(v,R,'w') does the same thing, but also assigns to w the shortest
#   element of W(R) with the property that w.v0 =v. The form of w will be
#   [i1,i2,...,i.m], where the terms index the simple reflections of base(R).
# vec2fc(v,S) and vec2fc(v,S,'w') are similar but instead use the vectors of
#   the list S as a base, rather than base(R).
#
`coxeter/vec2fc`:=proc() local v,w,S,i,rv,fund;
  if type(args[2],'list') then S:=args[2] else S:=coxeter['base'](args[2]) fi;
  w:=[]; v:=args[1]; fund:=evalb(1=0);
  while not fund do;
    for i to nops(S) do;
      rv:=coxeter['iprod'](S[i],v);
      if rv<0 then w:=[op(w),i]; v:=coxeter['reflect'](S[i],v); break fi;
    od;
    if i>nops(S) then fund:=not fund fi;
  od;
  if nargs>2 then assign(args[3],w) fi;
  v;
end:
#
`coxeter/longest_elt`:=proc(R) local w,S;
  if type(R,'list') then S:=R else S:=coxeter['base'](R,0) fi;
  `coxeter/vec2fc`(-coxeter['interior_pt'](S),S,'w'); w;
end: 
#
#
#
# Root Systems/Reflection Groups Package
#
# Calling sequence:    coxeter[<funcname>](<arguments>)
#
coxeter:='coxeter':
`coxeter/reader`:=proc(x) eval(`coxeter/`.x) end:
#
coxeter[base]:=`coxeter/reader`(base):
coxeter[diagram]:=`coxeter/reader`(diagram):
coxeter[degrees]:=`coxeter/reader`(degrees):
coxeter[exponents]:=`coxeter/reader`(exponents,degrees):
coxeter[num_refl]:=`coxeter/reader`(num_refl,degrees):
coxeter[class_rep]:=`coxeter/reader`(class_rep):
coxeter[class_size]:=`coxeter/reader`(class_size):
coxeter[cox_number]:=`coxeter/reader`(cox_number,degrees):
coxeter[cox_matrix]:=`coxeter/reader`(cox_matrix):
coxeter[descent_gf]:=`coxeter/reader`(descent_gf):
coxeter[size]:=`coxeter/reader`(size,degrees):
coxeter[char_poly]:=`coxeter/reader`(char_poly):
coxeter[highest_root]:=`coxeter/reader`(highest_root):
coxeter[interior_pt]:=`coxeter/reader`(interior_pt):
coxeter[length_gf]:=`coxeter/reader`(length_gf):
coxeter[multperm]:=`coxeter/reader`(multperm):
coxeter[name_of]:=`coxeter/reader`(name_of):
coxeter[orbit]:=`coxeter/reader`(orbit):
coxeter[orbit_size]:=`coxeter/reader`(orbit_size):
coxeter[perm_char]:=`coxeter/reader`(perm_char):
coxeter[perm_rep]:=`coxeter/reader`(perm_rep,perm_rep):
coxeter[presentation]:=`coxeter/reader`(presentation,perm_rep):
coxeter[rank]:=`coxeter/reader`(rank):
coxeter[reduce]:=`coxeter/reader`(reduce):
coxeter[iprod]:=`coxeter/reader`(iprod,reflect):
coxeter[reflect]:=`coxeter/reader`(reflect):
coxeter[pos_roots]:=`coxeter/reader`(pos_roots):
coxeter[root_coords]:=`coxeter/reader`(root_coords):
coxeter[vec2fc]:=`coxeter/reader`(vec2fc):
coxeter[longest_elt]:=`coxeter/reader`(longest_elt,vec2fc):

#save `coxeter.m`;
#
# COPYLEFT NOTICE:
# Copyleft (c) 1991 by John R. Stembridge.
#  
# Permission is granted to anyone to to use, modify, or redistribute this
# software freely, subject to the following restrictions:
# 
# 1. The author accepts no responsibility for any consequences of this
# software and makes no guarantee that the software is free of defects.
# 2. The origin of this software must not be misrepresented, either by
# explicit claim or by omission.
# 3. This notice and the copyleft must be included in all copies or
# modified versions of this software.
# 4. This software may not be included or redistributed as part of any
# package to be sold for profit unless the author has given explicit written
# permission to do so.
# 
# John Stembridge
# Department of Mathematics
# University of Michigan
# Ann Arbor, MI 48109-1003
# Internet:  jrs@math.lsa.umich.edu
#
###############################################################################
#
# rho(R) computes half the sum of the positive roots of R without
#  actually computing the positive roots themselves.
# rho(S) does the same, using S as the list of base vectors.
#
`weyl/rho`:=proc() local rho,S,i,c;
  if type(args[1],'list') then S:=args[1] else S:=coxeter['base'](args[1]) fi;
  rho:=[seq(c[i]*S[i],i=1..nops(S))];
  rho:=collect(convert(rho,`+`),indets(S));
  {seq(2*coxeter['iprod'](i,rho)=coxeter['iprod'](i,i),i=S)};
  subs(solve("),rho);
end:
#
#
`weyl/store`:=proc(R) local S,n,len,r,j,roots;
  S:=coxeter['base'](R); n:=nops(S);
  len:=[seq(coxeter['iprod'](r,r),r=S)];
  roots:=coxeter['pos_roots'](R);
  `weyl/weights/`.R:=
    [seq([seq(2*coxeter['iprod'](r,S[j])/len[j],j=1..n)],r=roots)];
  coxeter['pos_roots'](R):=roots;
  NULL;
end:
#
#
# weight_coords(v,R) returns the coordinates of v with respect to the
#  fundamental weights. 
# weight_coords(v,S), where S is the base of R, does the same.
#
`weyl/weight_coords`:=proc(v) local S,s;
  if type(args[2],'list') then S:=args[2] else S:=coxeter['base'](args[2]) fi;
  [seq(2*coxeter['iprod'](v,s)/coxeter['iprod'](s,s),s=S)];
end:
#
#
# weight_mults(v,R) produces the dimensions of the weight spaces of the
#   irreducible rep'n of highest weight v. The output is a sum of the form
#   c1*M[w1]+c2*M[w2]+..., where w1,w2,... are the coordinates of the
#   dominant weights in the weight system of v, and c1,c2,... are their
#   multiplicities.
#
# weight_mults(v,u,R) computes the multiplicity of the weight u in the
#   rep'n of highest weight v. The weight u need not be dominant (but v must).
# 
#  u and v are assumed to be in the standard coordinates.
#
`weyl/weight_mults`:=proc(v0) local mults,R,S,Wt,Rp,rho,n,wR,i,j,j0,k,
    wts,v,v1,c,m,u,wl,J,dom,orb;
  if nargs>2 and args[1]=args[2] then RETURN(1) fi;
  R:=args[nargs]; S:=coxeter['base'](R); Wt:=weyl['weights'](R); 
  if nargs>2 then 
    v1:=coxeter['vec2fc'](args[2],S);
    for v in Wt do if coxeter['iprod'](v0-v1,v)<0 then RETURN(0) fi od;
  fi;
  Rp:=coxeter['pos_roots'](R); rho:=convert(Rp,`+`)/2;
  v:=coxeter['interior_pt'](S);
  wts:=weyl['weight_sys'](v0,R,'wl','wR');
  wts:=[seq([wts[i],wl[i],coxeter['iprod'](wts[i],v)],i=1..nops(wts))];
  wts:=sort(wts,proc(x,y) evalb(x[3]>=y[3]) end);
  wl:=map(x->op(2,x),wts); wts:=map(x->op(1,x),wts);
  if nargs>2 then member(weyl['weight_coords'](v1,S),wl,'n')
    else n:=nops(wts) fi;
  mults:=[1];
  for i from 2 to n do;
    v:=wts[i]; m:=0;
    J:=map(proc(x,y) if y[x]=0 then x fi end,[$1..nops(S)],wl[i]);
    c:=coxeter['iprod'](v0+rho,v0+rho)-coxeter['iprod'](v+rho,v+rho);
    for k to nops(Rp) do;
      dom:=evalb(0=0);
      for j in J while dom do if wR[k][j]<0 then dom:=evalb(1=0) fi od;
      if not dom then next else u:=v+Rp[k] fi;
      coxeter['vec2fc'](u,S);
      if not member(",wts,'j') then next fi;
      orb:=coxeter['orbit_size'](Rp[k],[seq(S[j0],j0=J)],-1);
      do;
        m:=m+mults[j]*orb*coxeter['iprod'](u,Rp[k]);
        u:=u+Rp[k]; coxeter['vec2fc'](u,S);
        if not member(",wts,'j') then break fi;
      od;
    od;
    mults:=[op(mults),m/c];
  od;
  if nargs>2 then mults[n] else
    convert([seq(mults[i]*M[op(wl[i])],i=1..nops(wts))],`+`) fi;
end: 
#
#
# weight_sys(v,R) produces the list of all dominant weights that are
#  "less" than the dominant weight v, w.r.t. the partial ordering in
#  which v1<v2 if v2-v1 is a sum of positive roots.
# weight_sys(v,R,'wl') will do the same, but also assigns to 'wl' the
#   coordinates of the weight vectors with respect to the fundamental
#   weights.
# weight_sys(v,R,'wl','wR') will do the above, as well as assign the
#   weight coordinates of the positive roots to 'wR'.
# To save time, one can prestore the positive roots and 'wR' via store(R).
#
`weyl/weight_sys`:=proc(v0,R) local n,S,Rplus,i,j,len,wR,res,wres,v,sat,wv;
  S:=coxeter['base'](R); n:=nops(S);
  Rplus:=coxeter['pos_roots'](R);
  if type(`weyl/weights/`.R,'list') then
    wR:=`weyl/weights/`.R
  else
    len:=[seq(coxeter['iprod'](i,i),i=S)];
    wR:=[seq([seq(2*coxeter['iprod'](i,S[j])/len[j],j=1..n)],i=Rplus)]
  fi;
  wres:=[weyl['weight_coords'](v0,S)];
  res:=[v0]; sat:=0;
  while sat<nops(res) do;
    for i to nops(Rplus) do;
      v:=res[sat+1]-Rplus[i];
      if member(v,res) then next fi;
      wv:=[seq(wres[sat+1][j]-wR[i][j],j=1..n)];
      if min(op(wv))<0 then next fi;
      res:=[op(res),v]; wres:=[op(wres),wv];
    od;
    sat:=sat+1;
  od;
  if nargs>2 then assign(args[3],wres) fi;
  if nargs>3 then assign(args[4],wR) fi;
  res;
end:
#
#
# weights(R) returns the list of fundamental weights for R.
# weights(S) does the same, using S as an ordered list of simple roots.
#
`weyl/weights`:=proc(R) local S,n,A,i,j,B;
  if type(R,'list') then S:=R else S:=coxeter['base'](R) fi;
  n:=nops(S); A:=array(symmetric,1..n,1..n);
  for i to n do;
    for j from i to n do;
      A[i,j]:=coxeter['iprod'](S[i],S[j]);
    od;
  od;
  B:=linalg['inverse'](A);
  [seq(A[i,i]*convert([seq(B[i,j]*S[j],j=1..n)],`+`)/2,i=1..n)];
end:
#
#
# Note that this algorithm is inefficient if several computations of
# weyl_dim()'s are required for the same choice of R. To increase 
# efficiency in such cases one should store the pos_roots(R) in the 
# remember table: coxeter['pos_roots'](R):=pos_roots(R);
#
`weyl/weyl_dim`:=proc(v,R) local r,roots,rho,q,res;
  roots:=coxeter['pos_roots'](R);
  rho:=convert(roots,`+`)/2;
  if nargs=2 then
    [seq(1+coxeter['iprod'](r,v)/coxeter['iprod'](r,rho),r=roots)];
    res:=convert(",`*`);
  else
    q:=args[3]; res:=q^(-coxeter['iprod'](rho,v));
    for r in roots do;
      res:=res*(q^(coxeter['iprod'](r,v+rho))-1)
        /(q^(coxeter['iprod'](r,rho))-1);
    od;
  fi;
  res;
end:
#
#
#
# Supplement to Coxeter for manipulating Weyl characters
#
# Calling sequence:    weyl[<funcname>](<arguments>)
#
weyl:='weyl':
`weyl/reader`:=proc(x) local y; eval(`weyl/`.x) end:
#
weyl[rho]:=`weyl/reader`(rho):
weyl[store]:=`weyl/reader`(store):
weyl[weights]:=`weyl/reader`(weights):
weyl[weight_coords]:=`weyl/reader`(weight_coords):
weyl[weight_mults]:=`weyl/reader`(weight_mults):
weyl[weight_sys]:=`weyl/reader`(weight_sys):
weyl[weyl_dim]:=`weyl/reader`(weyl_dim):

#save `weyl.m`;
#quit
#
# CEF: this was moved into the .hlp file by mistake:
`coxeter/class_size/E6` := [1, 36, 270, 240, 1440, 540, 1620, 2160, 5184, 3240
, 480, 1440, 45, 5184, 6480, 540, 1440, 4320, 4320, 720, 80, 4320, 540, 5760, 
1440]:
`coxeter/class_rep/E6` := [[], [1], [1, 2], [1, 3], [1, 2, 3], [1, 2, 5], [1, 3
, 4], [1, 2, 3, 5], [1, 2, 3, 4], [1, 2, 4, 5], [1, 3, 5, 6], [2, 3, 4, 5], [2
, 3, 4, 2, 3, 4, 5, 4, 2, 3, 4, 5], [1, 2, 3, 4, 6], [1, 2, 3, 4, 5], [2, 4, 2
, 3, 4, 5], [1, 2, 3, 5, 6], [1, 3, 5, 4, 6], [1, 2, 3, 5, 4, 6], [1, 2, 3, 1, 
4, 5, 4, 2, 3, 6, 5, 4], [1, 2, 3, 1, 4, 5, 4, 2, 3, 1, 4, 3, 5, 6, 5, 4, 2, 3
, 1, 4, 3, 5, 4, 6], [1, 2, 3, 4, 3, 5, 4], [3, 4, 2, 3, 1, 4, 3, 5, 4, 2, 3, 4
, 5], [1, 3, 4, 2, 6, 5, 4, 2], [1, 2, 3, 4, 2, 3, 5, 4, 2, 3, 4, 6, 5, 4]]:
`coxeter/mytype/E6` := [[1, -6, 15, -20, 15, -6, 1], [1, -4, 5, 0, -5, 4, -1], 
[1, -2, -1, 4, -1, -2, 1], [1, -3, 3, -2, 3, -3, 1], [1, -1, -1, 0, 1, 1, -1], 
[1, 0, -3, 0, 3, 0, -1], [1, -2, 1, 0, -1, 2, -1], [1, 1, -1, -2, -1, 1, 1], [1
, -1, 0, 0, 0, -1, 1], [1, 0, -1, 0, -1, 0, 1], [1, 0, 0, -2, 0, 0, 1], [1, -1
, -1, 2, -1, -1, 1], [1, 2, -1, -4, -1, 2, 1], [1, 1, 0, 0, 0, -1, -1], [1, 0, 
-1, 0, 1, 0, -1], [1, -2, 3, -4, 3, -2, 1], [1, 2, 2, 0, -2, -2, -1], [1, 0, 0
, 0, 0, 0, -1], [1, 1, 0, -1, 0, 1, 1], [1, -1, 2, -1, 2, -1, 1], [1, 3, 6, 7, 
6, 3, 1], [1, -1, 1, 0, -1, 1, -1], [1, 2, 1, 0, -1, -2, -1], [1, 0, 0, 1, 0, 0
, 1], [1, 2, 2, 2, 2, 2, 1]]:
`coxeter/perm_rep/E6` := permgroup(27,{s1 = [[5, 7], [8, 9], [10, 11], [12, 14]
, [15, 17], [26, 27]], s2 = [[4, 6], [5, 8], [7, 9], [18, 20], [21, 22], [23, 
24]], s4 = [[3, 4], [8, 10], [9, 11], [16, 18], [19, 21], [24, 25]], s3 = [[4, 
5], [6, 8], [11, 13], [14, 16], [17, 19], [25, 26]], s6 = [[1, 2], [12, 15], [
14, 17], [16, 19], [18, 21], [20, 22]], s5 = [[2, 3], [10, 12], [11, 14], [13, 
16], [21, 23], [22, 24]]}):
`coxeter/class_size/E7` := [1, 96768, 48384, 672, 145152, 63, 40320, 13440, 
10080, 315, 3780, 945, 90720, 11340, 207360, 120960, 3780, 161280, 60480, 7560
, 45360, 7560, 40320, 120960, 30240, 60480, 90720, 10080, 20160, 2240, 1, 96768
, 48384, 672, 145152, 63, 40320, 13440, 10080, 3780, 315, 945, 90720, 11340, 
207360, 120960, 3780, 161280, 60480, 7560, 7560, 45360, 120960, 40320, 30240, 
60480, 90720, 10080, 20160, 2240]:
`coxeter/class_rep/E7` := [[], [1, 2, 3, 4, 6, 7], [1, 2, 3, 4], [2, 4], [2, 3
, 4, 5, 6, 7], [2, 3, 4, 3, 2, 4, 5, 4, 2, 3, 4, 5, 6, 5, 4, 3, 2, 4, 5, 6, 7, 
6, 5, 4, 2, 3, 4, 5, 6, 7], [2, 4, 5, 4, 2, 3, 4, 5, 6, 7], [1, 3, 5, 6], [1, 3
, 1, 6, 5, 4, 3, 2, 4, 1, 3, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 1, 3, 4], [2, 3, 4, 
3, 2, 4, 5, 4, 2, 3, 4, 5], [1, 2, 5, 7], [1, 2], [2, 4, 5, 6, 5, 4, 2, 3, 4, 5
, 6, 7], [2, 4, 2, 3, 4, 5, 6, 4, 3, 2, 4, 5, 7, 6, 4, 2, 3, 4, 5, 6], [1, 3, 4
, 5, 6, 7], [1, 2, 3, 4, 5, 6], [2, 4, 2, 3, 4, 5], [1, 3, 4, 2, 6, 5, 4, 2], [
4, 3, 2, 5, 4, 3, 2, 6, 5, 4, 3, 1, 7, 6, 5, 4, 3, 2, 4, 5, 6, 1, 3, 4, 5, 2], 
[3, 4, 2, 5, 4, 3, 1, 4, 2, 6, 5, 4, 3, 2, 4, 5, 6, 1, 4, 5, 4, 3, 7, 6, 5, 4, 
3, 2, 4, 5, 6, 7], [1, 2, 4, 5], [2, 4, 5, 7], [1, 2, 3, 4, 2, 3, 5, 4, 2, 3, 4
, 6, 5, 4], [1, 2, 4, 5, 6, 7], [1, 2, 3, 5], [1, 2, 3, 5, 6, 7], [1, 2, 3, 4, 
5, 7], [2, 3, 4, 5], [1, 2, 3, 1, 4, 5, 4, 2, 3, 6, 5, 4], [1, 2, 3, 1, 4, 5, 4
, 2, 3, 1, 4, 3, 5, 6, 5, 4, 2, 3, 1, 4, 3, 5, 4, 6], [1, 2, 3, 1, 4, 2, 3, 1, 
4, 3, 5, 4, 2, 3, 1, 4, 3, 5, 4, 2, 6, 5, 4, 2, 3, 1, 4, 3, 5, 4, 2, 6, 5, 4, 3
, 1, 7, 6, 5, 4, 2, 3, 1, 4, 3, 5, 4, 2, 6, 5, 4, 3, 1, 7, 6, 5, 4, 2, 3, 4, 5
, 6, 7], [2, 6, 5, 4, 3, 4, 5, 2, 6, 4, 3, 1, 7, 5, 4, 2, 3, 4, 6, 5, 4, 3, 7, 
6, 2, 4, 5], [5, 2, 4, 5, 6, 7, 3, 4, 2, 5, 4, 3, 1, 5, 4, 6, 5, 4, 2, 3, 4, 5
, 6], [3, 4, 5, 6, 5, 2, 4, 3, 5, 4, 5, 7, 6, 5, 4, 2, 3, 1, 4, 3, 5, 4, 2, 6, 
5, 4, 7, 6, 5, 4, 2], [1, 2, 3, 4, 6], [1], [1, 2, 3, 5, 6], [6, 5, 4, 2, 3, 1
, 4, 5, 6, 4, 3, 5, 4, 2, 1, 7, 6, 5, 4, 3, 1, 4, 3, 5, 2, 6, 4, 3, 5, 4, 2], [
1, 2, 3], [1, 2, 5], [2, 5, 7], [2, 3, 4, 3, 2, 4, 5, 4, 2, 3, 4, 5, 7], [5, 4
, 6, 7, 6, 5, 3, 4, 2, 3, 1, 4, 5, 3, 4, 6, 5, 4, 2, 3, 4, 6, 7, 5, 1, 3, 4, 2
, 6, 5, 4, 3, 1], [2, 4, 2, 3, 4, 5, 7], [4, 5, 7, 6, 5, 4, 3, 4, 2, 1, 3, 4, 5
, 4, 3, 6, 2, 4, 3, 5, 4, 1, 7, 6, 3], [4, 5, 3, 1, 4, 2, 4, 3, 1, 6, 5, 4, 7, 
6, 5, 2, 3, 1, 4, 3, 5, 2, 6, 5, 4, 1, 7, 6, 5], [2, 4, 3, 1, 5, 4, 2, 6, 5, 4
, 3, 1, 4, 5, 2, 6, 5, 4, 3, 1, 7, 6, 5, 4, 2, 3, 1, 4, 3, 5, 4, 6, 5, 4, 3, 1
, 7, 6, 2, 3, 4], [1, 2, 3, 4, 5, 6, 7], [1, 3, 4, 6, 7], [1, 3, 4], [3, 4, 2, 
3, 1, 4, 3, 5, 4, 2, 3, 4, 5], [1, 2, 4, 5, 7], [1, 3, 4, 5, 6], [2, 4, 5, 6, 7
], [2, 3, 4, 5, 7], [1, 2, 3, 4, 3, 5, 4], [1, 2, 3, 4, 5], [1, 2, 3, 5, 7], [4
, 5, 2, 7, 1, 4, 3, 5, 4, 2, 1, 3, 5, 4, 6, 5, 4, 3, 1, 7, 6, 5, 4, 2, 3, 4, 3
, 5, 4, 2, 6, 3, 1, 7, 6, 5, 4, 3, 4, 5, 6], [4, 2, 4, 5, 4, 1, 7, 6, 5, 3, 4, 
1, 2, 5, 6, 7, 4, 3, 4, 5, 6]]:
`coxeter/mytype/E7` := [[1, -7, 21, -35, 35, -21, 7, -1], [1, 1, 1, 0, 0, -1, -
1, -1], [1, -2, 1, 0, 0, -1, 2, -1], [1, -4, 6, -5, 5, -6, 4, -1], [1, 0, -1, 0
, 0, 1, 0, -1], [1, 5, 9, 5, -5, -9, -5, -1], [1, -1, 0, 2, -2, 0, 1, -1], [1, 
-1, 0, -2, 2, 0, 1, -1], [1, 2, 0, -1, 1, 0, -2, -1], [1, 1, -3, -3, 3, 3, -1, 
-1], [1, 1, -3, -3, 3, 3, -1, -1], [1, -3, 1, 5, -5, -1, 3, -1], [1, -1, 1, -1
, 1, -1, 1, -1], [1, 1, 1, 1, -1, -1, -1, -1], [1, 0, 0, 0, 0, 0, 0, -1], [1, 0
, -1, -1, 1, 1, 0, -1], [1, -3, 5, -7, 7, -5, 3, -1], [1, -1, 0, 1, -1, 0, 1, -
1], [1, 0, 0, 1, -1, 0, 0, -1], [1, 3, 3, 1, -1, -3, -3, -1], [1, -1, -1, 1, -1
, 1, 1, -1], [1, -1, -1, 1, -1, 1, 1, -1], [1, 1, 0, 0, 0, 0, -1, -1], [1, 1, 0
, 0, 0, 0, -1, -1], [1, 0, -2, -1, 1, 2, 0, -1], [1, 2, 2, 1, -1, -2, -2, -1], 
[1, 1, -1, -1, 1, 1, -1, -1], [1, -2, 0, 3, -3, 0, 2, -1], [1, -2, 3, -3, 3, -3
, 2, -1], [1, 2, 3, 1, -1, -3, -2, -1], [1, 7, 21, 35, 35, 21, 7, 1], [1, -1, 1
, 0, 0, 1, -1, 1], [1, 2, 1, 0, 0, 1, 2, 1], [1, 4, 6, 5, 5, 6, 4, 1], [1, 0, -
1, 0, 0, -1, 0, 1], [1, -5, 9, -5, -5, 9, -5, 1], [1, 1, 0, -2, -2, 0, 1, 1], [
1, 1, 0, 2, 2, 0, 1, 1], [1, -2, 0, 1, 1, 0, -2, 1], [1, -1, -3, 3, 3, -3, -1, 
1], [1, -1, -3, 3, 3, -3, -1, 1], [1, 3, 1, -5, -5, 1, 3, 1], [1, 1, 1, 1, 1, 1
, 1, 1], [1, -1, 1, -1, -1, 1, -1, 1], [1, 0, 0, 0, 0, 0, 0, 1], [1, 0, -1, 1, 
1, -1, 0, 1], [1, 3, 5, 7, 7, 5, 3, 1], [1, 1, 0, -1, -1, 0, 1, 1], [1, 0, 0, -
1, -1, 0, 0, 1], [1, -3, 3, -1, -1, 3, -3, 1], [1, 1, -1, -1, -1, -1, 1, 1], [1
, 1, -1, -1, -1, -1, 1, 1], [1, -1, 0, 0, 0, 0, -1, 1], [1, -1, 0, 0, 0, 0, -1
, 1], [1, 0, -2, 1, 1, -2, 0, 1], [1, -2, 2, -1, -1, 2, -2, 1], [1, -1, -1, 1, 
1, -1, -1, 1], [1, 2, 0, -3, -3, 0, 2, 1], [1, 2, 3, 3, 3, 3, 2, 1], [1, -2, 3
, -1, -1, 3, -2, 1]]:
`coxeter/perm_rep/E7` := permgroup(56,{s5 = [[3, 4], [10, 13], [12, 15], [14, 
17], [25, 28], [26, 31], [27, 30], [29, 32], [39, 43], [42, 45], [44, 46], [53
, 54]], s1 = [[7, 8], [9, 11], [10, 12], [13, 15], [16, 18], [19, 22], [37, 40]
, [38, 41], [42, 44], [45, 46], [47, 48], [49, 50]], s2 = [[5, 6], [7, 9], [8, 
11], [20, 23], [25, 27], [26, 29], [28, 30], [31, 32], [33, 36], [47, 49], [48
, 50], [51, 52]], s7 = [[1, 2], [16, 19], [18, 22], [21, 24], [25, 26], [27, 29
], [28, 31], [30, 32], [34, 35], [37, 38], [40, 41], [55, 56]], s3 = [[5, 7], [
6, 9], [12, 14], [15, 17], [18, 21], [22, 24], [34, 37], [35, 38], [39, 42], [
43, 45], [48, 51], [50, 52]], s4 = [[4, 5], [9, 10], [11, 12], [17, 20], [21, 
25], [24, 26], [30, 34], [32, 35], [36, 39], [45, 47], [46, 48], [52, 53]], s6
 = [[2, 3], [13, 16], [15, 18], [17, 21], [20, 25], [23, 27], [31, 33], [32, 36
], [35, 39], [38, 42], [41, 44], [54, 55]]}):
`coxeter/class_size/E8` := [1, 23224320, 120, 11612160, 3150
, 113400, 3780, 37800, 37800, 3780, 2240, 4480, 89600, 268800, 15120, 37800, 
37800, 45360, 45360, 151200, 2721600, 453600, 680400, 5443200, 907200, 907200, 
453600, 907200, 580608, 1161216, 2240, 4480, 80640, 80640, 89600, 100800, 
100800, 268800, 268800, 268800, 403200, 604800, 604800, 806400, 806400, 1209600
, 1209600, 1612800, 1612800, 3225600, 14515200, 2419200, 2419200, 2419200, 
1612800, 4838400, 4838400, 2419200, 24883200, 1814400, 1814400, 3628800, 
5443200, 5443200, 10886400, 43545600, 10886400, 6451200, 12902400, 580608, 
1161216, 5806080, 5806080, 8709120, 8709120, 1209600, 1209600, 1209600, 1209600
, 2419200, 2419200, 2419200, 3628800, 3628800, 4838400, 4838400, 7257600, 
7257600, 9676800, 9676800, 14515200, 14515200, 29030400, 24883200, 24883200, 
24883200, 11612160, 23224320, 6451200, 12902400, 19353600, 19353600, 17418240, 
17418240, 34836480, 14515200, 14515200, 29030400, 11612160, 11612160, 120, 1]:
`coxeter/class_rep/E8` := [[], [1, 2, 3, 4, 5, 6, 7, 8], [6], [3, 2, 1, 4, 2, 5
, 4, 3, 2, 4, 5, 1, 6, 5, 4, 3, 2, 4, 5, 6, 7, 6, 8], [3, 4, 3, 2, 4, 3, 5, 4, 
3, 2, 4, 5], [8, 5, 3, 2], [5, 6, 5, 3, 4, 5, 6, 3, 4, 5, 7, 6, 5, 4, 3, 2, 4, 
5, 6, 7, 3, 4, 5, 6, 2, 4, 5, 3, 4, 2], [8, 3, 2], [4, 5, 4, 2, 4, 5, 7, 3, 4, 
5, 2, 4, 3], [2, 8], [7, 6], [1, 4, 3, 2, 1, 5, 4, 3, 2, 4, 5, 1, 3, 4, 6, 5, 4
, 3, 2, 4, 5, 6, 1, 3, 4, 2, 7, 6, 5, 4, 3, 2, 4, 5, 6, 1, 3, 4, 5, 8, 7, 6, 5
, 4, 3, 2, 4, 5, 6, 7, 8, 1, 3, 4, 5, 6, 7, 2, 4, 5, 6, 3, 4, 5, 2, 4, 3, 1, 3
, 4, 5, 6, 7, 8, 2, 4, 5, 6, 7, 3], [1, 2, 3, 1, 4, 5, 4, 2, 3, 1, 4, 3, 5, 6, 
5, 4, 2, 3, 1, 4, 3, 5, 4, 6], [1, 3, 5, 6], [1, 2, 4, 3, 5, 4, 3, 2, 4, 5, 1, 
6, 5, 4, 3, 2, 4, 5, 1, 3, 4, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 1, 3, 4, 5, 6, 2, 4
, 3, 1, 8, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 8, 1, 3, 4, 5, 6, 2, 4, 3], [2, 4, 2, 
3, 4, 5], [3, 4, 3, 2, 1, 3, 5, 4, 3, 2, 4, 1, 3, 6, 5, 4, 3, 2, 4, 5, 6, 7, 6
, 5, 4, 3, 2, 4, 5, 6, 7, 1, 3, 4, 5, 6, 2, 4, 3, 1, 8, 7, 6, 5, 4, 3, 2, 4, 5
, 6, 7, 8, 1, 3, 4, 5, 6, 7, 2, 4, 5, 6, 3, 4, 5, 2, 4, 1, 3, 4, 5, 6, 7, 8, 2
, 4, 5, 6], [1, 3, 4], [3, 4, 5, 3, 6, 4, 3, 2, 4, 5, 3, 4, 2, 7, 6, 5, 4, 3, 4
, 5, 6, 8, 7, 6, 5, 4, 2, 4, 5, 6, 7], [3, 4, 2, 3, 1, 4, 3, 5, 4, 2, 3, 4, 5]
, [1, 2, 4, 5, 7], [2, 4, 2, 3, 4, 5, 7], [2, 4, 2, 3, 4, 5, 6, 4, 3, 2, 4, 5, 
7, 6, 4, 2, 3, 4, 5, 6], [1, 8, 7, 6, 4, 3], [2, 4, 2, 3, 4, 5, 6, 5, 4, 3, 2, 
4, 5, 6, 7, 8, 7, 6, 5, 4, 2, 3, 4, 5, 6, 7, 8], [1, 2, 4, 5], [2, 4, 3, 1, 5, 
4, 2, 6, 5, 4, 3, 1, 4, 5, 2, 6, 5, 4, 3, 1, 7, 6, 5, 4, 2, 3, 1, 4, 3, 5, 4, 6
, 5, 4, 3, 1, 7, 6, 2, 3, 4], [2, 4, 2, 3, 4, 5, 4, 3, 2, 4, 5, 6, 5, 4, 2, 3, 
4, 5, 6, 8], [1, 2, 3, 4], [3, 1, 4, 2, 5, 4, 3, 1, 6, 5, 4, 3, 2, 4, 5, 7, 6, 
5, 4, 3, 2, 4, 5, 1, 3, 8, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 8, 1, 3, 4, 5, 6, 7, 2
, 5, 6, 3, 4, 5, 2, 4, 3, 1, 3, 4, 5, 6, 7, 2, 4, 5, 6, 3, 4, 5, 2], [1, 3, 2, 
4, 3, 2, 4, 5, 4, 3, 2, 4, 5, 6, 5, 4, 3, 2, 4, 5, 6, 7, 6, 5, 4, 3, 2, 4, 5, 6
, 7, 8, 1, 3, 4, 5, 6, 7, 2, 4, 5, 6, 3, 4, 5, 2, 4, 3, 1, 3, 4, 5, 6, 7, 2, 4
, 5, 6, 3, 4, 5, 2, 4, 3], [1, 4, 3, 2, 1, 5, 4, 3, 2, 4, 5, 1, 3, 6, 5, 4, 3, 
2, 4, 5, 6, 1, 7, 6, 5, 4, 3, 8, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 1, 3], [3, 5, 1]
, [3, 4, 5, 2, 6, 5, 4, 3, 5, 1, 4, 5, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 3, 4, 5, 6
, 2, 4, 5, 4, 2], [2, 1, 4, 3, 1, 5, 4, 3, 6, 5, 4, 2, 4, 5, 6, 3, 4, 7, 6, 5, 
4, 3, 2, 4, 1, 3, 4, 2, 8, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 1, 3, 4, 5, 6, 4, 3], 
[2, 3, 4, 3, 2, 4, 5, 4, 2, 3, 4, 5, 7, 8], [2, 5, 4, 3], [3, 5, 4, 3, 2, 4, 5
, 1, 6, 5, 4, 3, 2, 4, 5, 6, 1, 3, 4, 2, 3, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 1, 3
, 4, 5, 2, 4, 3, 8, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 8, 1, 3, 4, 5, 6, 7, 2, 4, 5
, 6, 3, 4, 5, 2, 1, 3, 4, 5, 6, 7], [3, 2, 1, 4, 3, 2, 6, 5, 4, 3, 2, 4, 5, 6, 
1, 3, 4, 5, 2, 4, 3, 7, 6, 5, 4, 3, 2, 4, 5, 6, 1, 3, 4, 2, 8, 7, 6, 5, 4, 3, 2
, 4, 5, 6, 7, 8, 1, 3, 4, 5, 6, 7, 2, 4, 5, 6, 3, 4, 5, 2, 4, 3, 1, 3, 5, 6, 7
, 2, 4], [3, 2, 1, 5, 4, 2, 6, 5, 7, 4, 3, 2, 6, 5, 4, 3, 2, 5, 1, 3, 4], [2, 4
, 3, 2, 4, 5, 4, 3, 2, 4, 1, 6, 5, 4, 3, 2, 4, 5, 6, 1, 3, 4, 5, 2, 4, 3, 7, 6
, 5, 4, 3, 2, 4, 5, 6, 7, 1, 8, 7, 6, 5, 4, 3, 1], [1, 2, 3, 5], [1, 3, 1, 6, 5
, 4, 3, 2, 4, 1, 3, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 1, 3, 4], [1, 2, 3, 1, 4, 5, 
4, 2, 3, 6, 5, 4], [3, 2, 1, 4, 3, 1, 5, 4, 3, 2, 6, 5, 4, 3, 2, 4, 5, 6, 1, 3
, 4, 5, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 1, 3, 4, 5, 6, 2, 4, 5, 3, 4, 2, 1, 3, 4
, 5, 6, 7, 8, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 1, 3, 4, 5, 6, 2, 4, 5, 3, 4, 2, 1
, 3, 4, 5, 6, 7, 8], [8, 5, 4, 2, 3], [8, 7, 3, 5, 2], [6, 5, 4, 2, 3, 1, 4, 5
, 6, 4, 3, 5, 4, 2, 1, 7, 6, 5, 4, 3, 1, 4, 3, 5, 2, 6, 4, 3, 5, 4, 2], [1, 2, 
3, 4, 2, 3, 5, 4, 2, 3, 4, 6, 5, 4], [2, 4, 5, 4, 2, 3, 4, 5, 7, 8], [1, 2, 4, 
5, 6, 7], [2, 4, 5, 4, 2, 3, 4, 5, 6, 7], [4, 3, 4, 1, 6, 5, 4, 3, 2, 4, 5, 6, 
1, 7, 6, 5, 4, 3, 2, 4, 5, 6, 1, 3, 4, 5, 2, 4, 5, 3, 1, 8, 7, 6, 5, 4, 3, 2, 4
, 5, 6, 7, 8], [7, 5, 3, 2, 1, 8], [1, 2, 3, 5, 6], [3, 4, 5, 4, 3, 2, 4, 1, 6
, 5, 4, 3, 2, 4, 5, 6, 1, 7, 8, 6, 5, 4, 3, 2, 4, 5, 6, 7, 1, 3, 4, 5, 6, 2, 4
, 5, 3, 4, 2, 1, 3, 4, 5, 6, 7], [1, 3, 4, 5, 6], [4, 5, 2, 7, 1, 4, 3, 5, 4, 2
, 1, 3, 5, 4, 6, 5, 4, 3, 1, 7, 6, 5, 4, 2, 3, 4, 3, 5, 4, 2, 6, 3, 1, 7, 6, 5
, 4, 3, 4, 5, 6], [1, 3, 4, 5, 6, 7], [2, 4, 5, 6, 5, 4, 2, 3, 4, 5, 6, 7, 6, 5
, 4, 3, 2, 4, 5, 6, 7, 8, 7, 6, 5, 4, 2, 3, 4, 5, 6, 7, 8], [1, 2, 3, 4, 5], [1
, 3, 4, 3, 5, 4, 2, 4, 6, 5, 4, 3, 4, 5, 6, 1, 7, 6, 5, 4, 3, 8, 7, 6, 5, 4, 3
, 2, 4, 5, 6, 7, 8, 1, 3, 4, 5, 6, 7, 2, 4, 5, 3, 4, 1, 3], [4, 2, 5, 4, 3, 2, 
5, 1, 6, 5, 4, 3, 2, 4, 5, 6, 1, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 1, 3, 4, 5, 2, 4
, 3, 1, 8, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 1, 3, 4, 5, 6, 2, 4, 5, 3, 4, 2, 1, 3
, 4, 5, 6, 7, 8], [2, 4, 5, 6, 5, 4, 2, 3, 4, 5, 6, 7], [5, 4, 6, 7, 6, 5, 3, 4
, 2, 3, 1, 4, 5, 3, 4, 6, 5, 4, 2, 3, 4, 6, 7, 5, 1, 3, 4, 2, 6, 5, 4, 3, 1], [
1, 3, 4, 5, 6, 7, 8], [1, 2, 3, 4, 5, 7], [1, 3, 4, 2, 6, 5, 4, 2], [3, 2, 1, 4
, 3, 6, 5, 4, 3, 2, 4, 5, 6, 1, 3, 4, 2, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 1, 8, 7
, 6, 5, 4, 3, 2, 4, 5, 6, 7, 8, 1, 3, 4, 5, 2, 4, 3, 1], [3, 4, 2, 4, 5, 3, 4, 
6, 5, 3, 2, 4, 5, 6, 1, 3, 4, 5, 2, 4, 3, 1, 7, 6, 5, 4, 3, 2, 4, 5, 6, 8, 7, 6
, 5, 4, 3, 2, 4, 5, 6, 7, 8, 1, 3, 4, 5, 6, 7, 2, 4, 5, 6, 3, 4, 5, 2, 4, 1, 3
, 4, 5, 6, 7, 8, 2, 4, 5, 6, 7], [2, 4, 3, 2, 1, 5, 4, 6, 5, 3, 7, 6, 5, 4, 3, 
2, 4, 5, 6, 7, 1, 3, 4, 5, 6, 2, 4, 5, 3, 4, 1, 3, 8, 7, 6, 5, 4, 3, 2, 4, 5, 6
, 7, 8, 1, 3, 4, 5], [1, 2, 3, 4, 6], [3, 4, 5, 2, 7, 6, 5, 4, 3, 4, 1, 8, 7, 6
, 5, 4, 3, 2, 4, 5, 6, 7, 8], [2, 3, 5, 6, 7, 8], [2, 3, 4, 5, 6, 7], [2, 4, 2
, 3, 4, 5, 7, 8], [1, 4, 3, 4, 1, 3, 5, 4, 3, 2, 6, 5, 4, 2, 7, 6, 5, 4, 3, 2, 
4, 5, 6, 7, 1, 3, 8, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 1, 3, 4, 5, 6, 2, 4, 5, 3, 4
, 2, 1, 3, 4, 5, 6, 7, 8], [1, 2, 3, 4, 3, 5, 4], [3, 1, 5, 4, 3, 2, 4, 6, 5, 4
, 3, 2, 4, 5, 6, 1, 3, 4, 5, 2, 4, 3, 1, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 8, 7, 6
, 5, 4, 3, 2, 4, 5, 6, 7, 1, 3, 4, 5, 6, 2, 4, 5, 3, 4, 2, 1, 3, 4, 5, 6, 7, 8]
, [3, 4, 2, 5, 4, 3, 2, 4, 5, 6, 1, 4, 7, 5, 4, 3, 2, 4, 5, 6, 1, 3, 4, 5, 2, 4
, 8, 7, 6, 5, 3, 2, 4, 5, 6, 7], [4, 3, 2, 1, 5, 4, 3, 1, 6, 5, 4, 3, 2, 4, 5, 
6, 1, 3, 4, 2, 7, 6, 5, 4, 3, 8], [2, 4, 3, 2, 4, 1, 3, 5, 4, 3, 2, 4, 6, 5, 4
, 3, 1, 7, 8, 6, 5, 4, 3, 2, 4, 5, 6, 7], [1, 3, 4, 6, 7], [2, 4, 5, 4, 2, 3, 4
, 5, 6, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 8, 7, 6, 5, 4, 2, 3, 4, 5, 6, 7, 8], [1, 
2, 3, 4, 5, 6], [2, 4, 3, 2, 4, 5, 4, 3, 2, 4, 1, 3, 6, 5, 7, 6, 4, 3, 2, 4, 5
, 6, 1, 3, 4, 5, 2, 4, 3, 1, 8, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 8], [2, 4, 5, 4, 
2, 3, 4, 5, 6, 8], [1, 2, 3, 5, 6, 7], [1, 4, 3, 5, 4, 3, 2, 4, 5, 6, 5, 4, 3, 
7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 1, 3, 4, 5, 2, 4, 3, 1, 8, 7, 6, 5, 4, 3, 2, 4, 5
, 6, 7, 1, 3, 4, 5, 6, 2, 4, 5, 3, 4, 2, 1, 3, 4, 5, 6, 7, 8], [1, 3, 4, 2, 5, 
4, 3, 2, 4, 1, 3, 6, 5, 4, 3, 2, 4, 5, 6, 1, 3, 4, 5, 7, 8, 6, 5, 4, 3, 2, 4, 5
, 6, 7, 1, 3, 4, 5, 6, 2, 4, 5, 3, 4, 2, 1, 3, 4, 5, 6, 7], [4, 5, 3, 1, 4, 2, 
4, 3, 1, 6, 5, 4, 7, 6, 5, 2, 3, 1, 4, 3, 5, 2, 6, 5, 4, 1, 7, 6, 5], [1, 2, 3
, 4, 5, 6, 8], [2, 4, 5, 6, 7, 8, 7, 6, 5, 4, 2, 3, 4, 5, 6, 7, 8], [4, 5, 7, 6
, 5, 4, 3, 4, 2, 1, 3, 4, 5, 4, 3, 6, 2, 4, 3, 5, 4, 1, 7, 6, 3], [3, 1, 4, 5, 
4, 3, 2, 4, 5, 6, 7, 8, 5, 4, 3, 2, 4, 5, 6, 7, 1, 3, 4, 5, 6, 2, 4, 5, 3, 4, 2
, 1, 3, 4, 5, 6], [1, 2, 4, 5, 6, 7, 8], [1, 2, 3, 4, 6, 7], [3, 4, 2, 1, 5, 4
, 6, 7, 5, 3, 4, 5, 6, 1, 3, 8, 7, 6, 5, 4, 3, 2, 4, 1], [2, 1, 4, 3, 5, 4, 3, 
2, 4, 1, 3, 6, 5, 4, 3, 2, 4, 5, 7, 6, 5, 4, 8, 7, 6, 5, 3, 2, 4, 5, 6, 7], [3
, 2, 1, 4, 5, 4, 3, 6, 5, 4, 3, 2, 7, 6, 5, 4, 3, 1, 8, 7, 6, 5, 4, 3, 2, 4, 5
, 6, 7, 8], [3, 2, 1, 4, 2, 5, 4, 3, 2, 4, 5, 1, 6, 5, 4, 3, 7, 6, 5, 4, 1, 8, 
7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 3, 4, 5, 6, 2, 4, 5, 3, 4, 2, 1, 3, 4, 5, 6], [1
, 2, 3, 4, 5, 6, 7], [1, 2, 3, 4, 6, 7, 8], [1, 3, 4, 1, 3, 5, 4, 2, 4, 5, 6, 7
, 8], [3, 2, 1, 4, 3, 1, 5, 4, 2, 6, 5, 4, 3, 4, 5, 7, 6, 5, 4, 3, 2, 4, 5, 6, 
7, 1, 3, 4, 5, 8], [2, 4, 5, 6, 5, 4, 2, 3, 4, 5, 6, 7, 8], [1, 2, 3, 4, 5, 7, 
8], [3, 4, 5, 3, 2, 4, 6, 5, 3, 2, 4, 1, 3, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 1, 3
, 4, 5, 8], [2, 4, 3, 5, 4, 3, 2, 4, 1, 3, 6, 5, 4, 3, 2, 4, 5, 6, 7, 6, 5, 4, 
3, 1, 8, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 8], [1, 2, 3, 5, 6, 7, 8], [4, 5, 4, 6, 
5, 4, 2, 4, 5, 6, 3, 4, 5, 6, 2, 4, 5, 3, 4, 2, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7, 1
, 3, 4, 5, 6, 7, 2, 4, 5, 6, 3, 4, 5, 2, 4, 3, 1, 3, 4, 5, 6, 7, 2, 4, 5, 6, 3
, 4, 5, 2, 4, 3, 1], [1, 3, 2, 1, 4, 3, 2, 4, 1, 3, 5, 4, 3, 2, 4, 5, 1, 3, 4, 
2, 6, 5, 4, 3, 2, 4, 5, 6, 1, 3, 4, 5, 2, 4, 3, 1, 7, 6, 5, 4, 3, 2, 4, 5, 6, 7
, 1, 3, 4, 5, 6, 2, 4, 5, 3, 4, 2, 1, 3, 4, 5, 6, 7, 8, 7, 6, 5, 4, 3, 2, 4, 5
, 6, 7, 8, 1, 3, 4, 5, 6, 7, 2, 4, 5, 6, 3, 4, 5, 2, 4, 3, 1, 3, 4, 5, 6, 7, 8
, 2, 4, 5, 6, 7, 3, 4, 5, 6, 2, 4, 5, 3, 4, 2, 1, 3, 4, 5, 6, 7, 8]]:
`coxeter/mytype/E8` := [[1, -8, 28, -56, 70, -56, 28, -8, 1], [1, 1, 0, -1, -1
, -1, 0, 1, 1], [1, -6, 14, -14, 0, 14, -14, 6, -1], [1, -2, 2, -1, 0, 1, -2, 2
, -1], [1, 0, -4, 0, 6, 0, -4, 0, 1], [1, 0, -4, 0, 6, 0, -4, 0, 1], [1, 4, 4, 
-4, -10, -4, 4, 4, 1], [1, -2, -2, 6, 0, -6, 2, 2, -1], [1, 2, -2, -6, 0, 6, 2
, -2, -1], [1, -4, 4, 4, -10, 4, 4, -4, 1], [1, -5, 10, -11, 10, -11, 10, -5, 1
], [1, 4, 10, 16, 19, 16, 10, 4, 1], [1, 1, 1, -2, -2, -2, 1, 1, 1], [1, -2, 1
, -2, 4, -2, 1, -2, 1], [1, 0, 4, 0, 6, 0, 4, 0, 1], [1, -4, 8, -12, 14, -12, 8
, -4, 1], [1, 4, 8, 12, 14, 12, 8, 4, 1], [1, -4, 6, -4, 0, 4, -6, 4, -1], [1, 
4, 6, 4, 0, -4, -6, -4, -1], [1, 0, -2, 0, 0, 0, 2, 0, -1], [1, 0, -2, 0, 0, 0
, 2, 0, -1], [1, -2, 2, -2, 0, 2, -2, 2, -1], [1, 0, 0, 0, -2, 0, 0, 0, 1], [1
, 0, 0, 0, -2, 0, 0, 0, 1], [1, 0, 2, 0, 0, 0, -2, 0, -1], [1, -2, 0, 2, -2, 2
, 0, -2, 1], [1, 2, 2, 2, 0, -2, -2, -2, -1], [1, 2, 0, -2, -2, -2, 0, 2, 1], [
1, -3, 3, -1, 0, -1, 3, -3, 1], [1, 2, 3, 4, 5, 4, 3, 2, 1], [1, 5, 10, 11, 10
, 11, 10, 5, 1], [1, -4, 10, -16, 19, -16, 10, -4, 1], [1, -3, 2, 1, 0, -1, -2
, 3, -1], [1, 3, 2, -1, 0, 1, -2, -3, -1], [1, -1, 1, 2, -2, 2, 1, -1, 1], [1, 
3, 2, -3, -6, -3, 2, 3, 1], [1, -3, 2, 3, -6, 3, 2, -3, 1], [1, 2, 1, 2, 4, 2, 
1, 2, 1], [1, 3, 5, 4, 0, -4, -5, -3, -1], [1, -3, 5, -4, 0, 4, -5, 3, -1], [1
, 0, 2, 0, 3, 0, 2, 0, 1], [1, -1, -2, 1, 2, 1, -2, -1, 1], [1, 1, -2, -1, 2, -
1, -2, 1, 1], [1, -3, 5, -6, 6, -6, 5, -3, 1], [1, 3, 5, 6, 6, 6, 5, 3, 1], [1
, -1, -2, 3, 0, -3, 2, 1, -1], [1, 1, -2, -3, 0, 3, 2, -1, -1], [1, 0, -1, 2, 0
, -2, 1, 0, -1], [1, 0, -1, 0, 0, 0, -1, 0, 1], [1, 0, -1, 0, 0, 0, -1, 0, 1], 
[1, 0, -1, 0, 0, 0, -1, 0, 1], [1, -2, 1, 2, -4, 2, 1, -2, 1], [1, -1, 1, 0, 0
, 0, -1, 1, -1], [1, 2, 1, -2, -4, -2, 1, 2, 1], [1, 0, -1, -2, 0, 2, 1, 0, -1]
, [1, 2, 1, 0, 0, 0, -1, -2, -1], [1, -2, 1, 0, 0, 0, -1, 2, -1], [1, 1, 1, 0, 
0, 0, -1, -1, -1], [1, -1, 0, 0, 0, 0, 0, -1, 1], [1, 2, 0, -2, 0, 2, 0, -2, -1
], [1, -2, 0, 2, 0, -2, 0, 2, -1], [1, 0, 0, 0, 2, 0, 0, 0, 1], [1, 2, 2, 2, 2
, 2, 2, 2, 1], [1, -2, 2, -2, 2, -2, 2, -2, 1], [1, 0, 0, 0, 0, 0, 0, 0, -1], [
1, 0, 0, 0, 0, 0, 0, 0, -1], [1, 0, -2, 0, 2, 0, -2, 0, 1], [1, -2, 1, 1, -2, 1
, 1, -2, 1], [1, 1, 1, 1, 1, 1, 1, 1, 1], [1, 3, 3, 1, 0, 1, 3, 3, 1], [1, -2, 
3, -4, 5, -4, 3, -2, 1], [1, -1, -1, 1, 0, -1, 1, 1, -1], [1, 1, -1, -1, 0, 1, 
1, -1, -1], [1, 1, -1, -1, 0, -1, -1, 1, 1], [1, -1, -1, 1, 0, 1, -1, -1, 1], [
1, -1, 2, -3, 2, -3, 2, -1, 1], [1, 3, 4, 3, 0, -3, -4, -3, -1], [1, -3, 4, -3
, 0, 3, -4, 3, -1], [1, 1, 2, 3, 2, 3, 2, 1, 1], [1, 0, -2, 0, 3, 0, -2, 0, 1]
, [1, -2, 2, 0, -1, 0, 2, -2, 1], [1, 2, 2, 0, -1, 0, 2, 2, 1], [1, -1, 0, -1, 
0, 1, 0, 1, -1], [1, 1, 0, 1, 0, -1, 0, -1, -1], [1, -1, -1, 0, 2, 0, -1, -1, 1
], [1, 1, -1, 0, 2, 0, -1, 1, 1], [1, -1, 0, 1, -2, 1, 0, -1, 1], [1, 1, 0, -1
, -2, -1, 0, 1, 1], [1, 0, 1, 0, 0, 0, 1, 0, 1], [1, 0, 1, 0, 0, 0, -1, 0, -1]
, [1, -1, -1, 2, 0, -2, 1, 1, -1], [1, 1, -1, -2, 0, 2, 1, -1, -1], [1, 0, -1, 
0, 0, 0, 1, 0, -1], [1, -1, 0, 0, 0, 0, 0, 1, -1], [1, 1, 0, 0, 0, 0, 0, 1, 1]
, [1, 1, 0, 0, 0, 0, 0, -1, -1], [1, 0, 0, -1, 0, -1, 0, 0, 1], [1, -1, 0, 1, -
1, 1, 0, -1, 1], [1, 2, 1, -1, -2, -1, 1, 2, 1], [1, -1, 1, -1, 1, -1, 1, -1, 1
], [1, 0, -1, 1, 0, -1, 1, 0, -1], [1, 0, -1, -1, 0, 1, 1, 0, -1], [1, 1, 1, 1
, 0, -1, -1, -1, -1], [1, -1, 1, -1, 0, 1, -1, 1, -1], [1, 0, -1, 0, 1, 0, -1, 
0, 1], [1, -1, 0, 1, 0, -1, 0, 1, -1], [1, 1, 0, -1, 0, 1, 0, -1, -1], [1, 0, 0
, 0, -1, 0, 0, 0, 1], [1, 0, 0, 1, 0, 1, 0, 0, 1], [1, 2, 2, 1, 0, -1, -2, -2, 
-1], [1, 6, 14, 14, 0, -14, -14, -6, -1], [1, 8, 28, 56, 70, 56, 28, 8, 1]]:
`coxeter/perm_rep/E8` := permgroup(240,{s2 = [[6, 7], [8, 9], [10, 12], [24, 28
], [27, 32], [30, 35], [31, 36], [33, 38], [34, 40], [37, 43], [39, 44], [42, 
48], [47, 53], [69, 76], [74, 81], [77, 83], [79, 87], [82, 89], [84, 91], [85
, 93], [88, 95], [90, 97], [92, 99], [94, 101], [96, 103], [100, 107], [102, 
109], [108, 115], [116, 127], [124, 126], [125, 137], [129, 142], [136, 144], [
138, 140], [139, 154], [141, 145], [143, 151], [147, 159], [150, 161], [152, 
156], [155, 167], [158, 163], [160, 164], [165, 171], [187, 194], [193, 199], [
197, 202], [198, 204], [201, 207], [203, 208], [205, 210], [206, 211], [209, 
214], [213, 217], [229, 231], [232, 233], [234, 235]], s1 = [[8, 10], [9, 12], 
[11, 14], [13, 16], [15, 19], [18, 22], [21, 25], [46, 52], [51, 58], [55, 62]
, [57, 64], [61, 68], [63, 70], [66, 73], [67, 75], [69, 77], [72, 80], [74, 82
], [76, 83], [78, 86], [79, 88], [81, 89], [85, 94], [87, 95], [92, 100], [93, 
101], [99, 107], [106, 114], [113, 123], [121, 122], [125, 138], [136, 139], [
137, 140], [143, 152], [144, 154], [150, 155], [151, 156], [153, 157], [160, 
165], [161, 167], [162, 169], [164, 171], [166, 172], [168, 173], [175, 178], [
176, 180], [177, 183], [179, 185], [188, 190], [189, 195], [216, 220], [219, 
223], [222, 226], [225, 228], [227, 230], [229, 232], [231, 233]], s4 = [[5, 6]
, [9, 11], [12, 14], [20, 24], [23, 27], [26, 30], [29, 33], [36, 41], [40, 45]
, [43, 49], [44, 50], [48, 54], [53, 59], [63, 69], [67, 74], [70, 77], [72, 79
], [75, 82], [78, 85], [80, 88], [86, 94], [91, 98], [97, 104], [99, 106], [103
, 110], [107, 114], [108, 116], [109, 117], [115, 124], [121, 125], [122, 138]
, [126, 127], [128, 129], [131, 141], [144, 153], [146, 147], [149, 158], [151
, 162], [154, 157], [156, 169], [161, 166], [164, 175], [167, 172], [171, 178]
, [182, 187], [186, 193], [191, 197], [192, 198], [196, 201], [200, 205], [208
, 212], [211, 215], [214, 218], [217, 221], [227, 229], [230, 232], [235, 236]]
, s3 = [[6, 8], [7, 9], [14, 17], [16, 20], [19, 23], [22, 26], [25, 29], [41, 
46], [45, 51], [49, 55], [50, 57], [54, 61], [56, 63], [59, 66], [60, 67], [65
, 72], [71, 78], [77, 84], [82, 90], [83, 91], [88, 96], [89, 97], [94, 102], [
95, 103], [100, 108], [101, 109], [106, 113], [107, 115], [114, 121], [122, 123
], [124, 125], [126, 137], [129, 136], [141, 143], [142, 144], [145, 151], [147
, 150], [157, 170], [158, 160], [159, 161], [163, 164], [169, 174], [172, 181]
, [173, 182], [178, 184], [180, 186], [183, 191], [185, 192], [190, 196], [195
, 200], [212, 216], [215, 219], [218, 222], [221, 225], [224, 227], [232, 234]
, [233, 235]], s5 = [[4, 5], [11, 13], [14, 16], [17, 20], [27, 31], [30, 34], 
[32, 36], [33, 37], [35, 40], [38, 43], [50, 56], [54, 60], [57, 63], [59, 65]
, [61, 67], [64, 70], [66, 72], [68, 75], [73, 80], [85, 92], [93, 99], [94, 
100], [98, 105], [101, 107], [102, 108], [104, 111], [109, 115], [110, 118], [
117, 128], [124, 129], [125, 136], [126, 142], [130, 131], [133, 146], [137, 
144], [138, 139], [140, 154], [148, 149], [162, 168], [166, 176], [169, 173], [
172, 180], [174, 182], [175, 177], [178, 183], [181, 186], [184, 191], [198, 
203], [201, 206], [204, 208], [205, 209], [207, 211], [210, 214], [221, 224], [
225, 227], [228, 230], [236, 237]], s6 = [[3, 4], [13, 15], [16, 19], [20, 23]
, [24, 27], [28, 32], [34, 39], [37, 42], [40, 44], [43, 48], [45, 50], [49, 54
], [51, 57], [55, 61], [58, 64], [62, 68], [65, 71], [72, 78], [79, 85], [80, 
86], [87, 93], [88, 94], [95, 101], [96, 102], [103, 109], [105, 112], [110, 
117], [111, 119], [118, 130], [128, 131], [129, 141], [132, 133], [135, 148], [
136, 143], [139, 152], [142, 145], [144, 151], [153, 162], [154, 156], [157, 
169], [170, 174], [176, 179], [177, 188], [180, 185], [183, 190], [186, 192], [
191, 196], [193, 198], [197, 201], [199, 204], [202, 207], [209, 213], [214, 
217], [218, 221], [222, 225], [226, 228], [237, 238]], s8 = [[1, 2], [18, 21], 
[22, 25], [26, 29], [30, 33], [34, 37], [35, 38], [39, 42], [40, 43], [44, 48]
, [45, 49], [50, 54], [51, 55], [56, 60], [57, 61], [58, 62], [63, 67], [64, 68
], [69, 74], [70, 75], [76, 81], [77, 82], [83, 89], [84, 90], [91, 97], [98, 
104], [105, 111], [112, 119], [120, 134], [132, 135], [133, 148], [146, 149], [
147, 158], [150, 160], [155, 165], [159, 163], [161, 164], [166, 175], [167, 
171], [172, 178], [176, 177], [179, 188], [180, 183], [181, 184], [185, 190], [
186, 191], [192, 196], [193, 197], [198, 201], [199, 202], [203, 206], [204, 
207], [208, 211], [212, 215], [216, 219], [220, 223], [239, 240]], s7 = [[2, 3]
, [15, 18], [19, 22], [23, 26], [27, 30], [31, 34], [32, 35], [36, 40], [41, 45
], [42, 47], [46, 51], [48, 53], [52, 58], [54, 59], [60, 65], [61, 66], [67, 
72], [68, 73], [74, 79], [75, 80], [81, 87], [82, 88], [89, 95], [90, 96], [97
, 103], [104, 110], [111, 118], [112, 120], [119, 132], [130, 133], [131, 146]
, [134, 135], [141, 147], [143, 150], [145, 159], [151, 161], [152, 155], [156
, 167], [162, 166], [168, 176], [169, 172], [173, 180], [174, 181], [182, 186]
, [187, 193], [188, 189], [190, 195], [194, 199], [196, 200], [201, 205], [206
, 209], [207, 210], [211, 214], [215, 218], [219, 222], [223, 226], [238, 239]]
}):
`coxeter/perm_rep/E8_120` := permgroup(120,{s3 = [[6, 45], [7, 8], [10, 11], [
12, 47], [17, 18], [23, 30], [24, 25], [32, 33], [34, 60], [39, 40], [48, 49], 
[56, 58], [57, 59], [69, 98], [71, 80], [74, 88], [75, 96], [76, 92], [77, 120]
, [79, 85], [82, 106], [86, 114], [87, 119], [91, 113], [93, 97], [94, 107], [
104, 112], [108, 109]], s8 = [[1, 2], [20, 61], [21, 62], [22, 64], [23, 94], [
24, 108], [25, 109], [26, 111], [27, 73], [28, 70], [30, 107], [31, 66], [32, 
98], [33, 69], [34, 79], [35, 100], [36, 115], [37, 116], [38, 83], [39, 96], [
40, 75], [41, 105], [52, 53], [54, 55], [56, 57], [58, 59], [60, 85], [102, 110
]], s4 = [[5, 6], [8, 44], [9, 10], [12, 13], [16, 17], [22, 23], [25, 26], [31
, 32], [34, 35], [38, 39], [49, 50], [54, 56], [55, 57], [64, 94], [65, 97], [
66, 98], [67, 119], [71, 90], [72, 74], [76, 118], [77, 101], [79, 100], [83, 
96], [89, 112], [95, 106], [99, 114], [103, 113], [109, 111]], s5 = [[4, 5], [
10, 47], [11, 12], [15, 16], [21, 22], [26, 27], [32, 60], [33, 34], [37, 38], 
[43, 44], [50, 51], [52, 54], [53, 55], [62, 64], [67, 81], [68, 72], [69, 79]
, [71, 93], [73, 111], [76, 120], [77, 92], [78, 89], [80, 97], [83, 116], [84
, 95], [85, 98], [86, 113], [91, 114]], s1 = [[8, 9], [10, 44], [18, 19], [20, 
58], [21, 60], [22, 32], [23, 31], [24, 102], [40, 41], [42, 48], [43, 47], [45
, 46], [59, 61], [62, 85], [63, 88], [64, 98], [66, 94], [67, 77], [71, 95], [
75, 105], [78, 86], [81, 92], [84, 93], [89, 113], [90, 106], [101, 119], [103
, 112], [108, 110]], s7 = [[2, 3], [15, 52], [16, 54], [17, 56], [18, 58], [19
, 20], [28, 29], [37, 51], [38, 50], [39, 49], [40, 48], [41, 42], [62, 84], [
63, 110], [64, 95], [65, 100], [66, 90], [68, 73], [69, 80], [71, 98], [72, 111
], [74, 109], [79, 97], [82, 107], [85, 93], [88, 108], [94, 106], [115, 117]]
, s6 = [[3, 4], [12, 49], [13, 50], [14, 15], [20, 21], [27, 28], [34, 56], [35
, 54], [36, 37], [42, 43], [47, 48], [55, 100], [57, 79], [58, 60], [59, 85], [
61, 62], [63, 81], [70, 73], [71, 112], [72, 118], [74, 76], [80, 104], [82, 91
], [88, 92], [89, 90], [95, 103], [106, 113], [115, 116]], s2 = [[6, 7], [8, 45
], [9, 46], [13, 14], [15, 50], [16, 51], [23, 24], [25, 30], [31, 102], [35, 
36], [37, 54], [38, 52], [53, 83], [55, 116], [63, 90], [65, 117], [66, 110], [
67, 78], [74, 82], [76, 91], [77, 86], [81, 89], [88, 106], [92, 113], [94, 108
], [100, 115], [107, 109], [114, 120]]}):
#
`coxeter/class_size/F4` := [1, 12, 12, 32, 32, 72, 36, 96, 96, 96, 96, 96, 18, 
72, 72, 144, 16, 12, 12, 32, 32, 12, 36, 16, 1]:
`coxeter/class_rep/F4` := [[], [1], [3], [1, 2], [3, 4], [1, 3], [2, 3], [1, 2
, 3], [1, 2, 4], [1, 3, 4], [2, 3, 4], [1, 2, 3, 4], [2, 3, 2, 3], [1, 2, 3, 2
, 3], [2, 3, 2, 3, 4], [1, 2, 3, 2, 3, 4], [1, 2, 1, 3, 2, 3, 4, 3], [1, 2, 1, 
3, 2, 1, 3, 2, 3], [2, 3, 2, 3, 4, 3, 2, 3, 4], [1, 2, 1, 3, 2, 1, 3, 2, 3, 4]
, [4, 3, 4, 2, 3, 4, 2, 3, 2, 1], [1, 2, 1, 3, 2, 1, 3, 4, 3, 2, 3, 4], [1, 2, 
1, 3, 2, 1, 3, 2, 3, 4, 3, 2, 3, 4], [1, 2, 1, 3, 2, 1, 3, 4, 3, 2, 1, 3, 2, 3
, 4, 3], [1, 2, 1, 3, 2, 1, 3, 2, 3, 4, 3, 2, 1, 3, 2, 3, 4, 3, 2, 1, 3, 2, 3, 
4]]:
`coxeter/mytype/F4` := [[1, -4, 6, -4, 1], [-1, 2, 0, -2, 1], [1, -2, 0, 2, -1]
, [1, -1, 0, -1, 1], [1, -1, 0, -1, 1], [-1, 0, 2, 0, -1], [-1, 2, -2, 2, -1], 
[1, -1, 0, 1, -1], [1, 1, 0, -1, -1], [-1, -1, 0, 1, 1], [-1, 1, 0, -1, 1], [1
, 0, -1, 0, 1], [1, 0, -2, 0, 1], [-1, 0, 0, 0, 1], [1, 0, 0, 0, -1], [-1, 0, 0
, 0, -1], [1, -2, 3, -2, 1], [1, 2, 0, -2, -1], [-1, -2, 0, 2, 1], [1, 1, 0, 1
, 1], [1, 1, 0, 1, 1], [1, 0, 2, 0, 1], [-1, -2, -2, -2, -1], [1, 2, 3, 2, 1], 
[1, 4, 6, 4, 1]]:
`coxeter/perm_rep/F4` := permgroup(24,{s1 = [[4, 6], [5, 7], [8, 10], [16, 20]
, [17, 19], [18, 21]], s2 = [[3, 4], [7, 9], [10, 12], [14, 17], [15, 16], [21
, 22]], s3 = [[2, 3], [4, 5], [6, 7], [9, 11], [12, 15], [13, 14], [17, 18], [
19, 21], [22, 23]], s4 = [[1, 2], [5, 8], [7, 10], [9, 12], [11, 13], [14, 15]
, [16, 17], [19, 20], [23, 24]]}):
#
`coxeter/class_size/H3` := [1, 15, 12, 15, 20, 12, 12, 20, 12, 1]:
`coxeter/class_rep/H3` := [[], [1], [1, 2], [1, 3], [2, 3], [1, 2, 3], [1, 2, 1
, 2], [1, 2, 1, 2, 3], [1, 2, 1, 2, 3, 2, 1, 2, 3], [1, 2, 1, 2, 1, 3, 2, 1, 2
, 1, 3, 2, 1, 2, 3]]:
`coxeter/mytype/H3` := [[-6, 6, -2], [-2, -2, 2], [-3, 3, -2], [2, -2, -2], [0
, 0, -2], [-1, -1, 2], [1, -1, -2], [0, 0, 2], [3, 3, 2], [6, 6, 2]]:
#
`coxeter/class_size/H4` := [1, 60, 144, 450, 400, 720, 720, 
1200, 1800, 144, 480, 1200, 720, 720, 480, 720, 1200, 24, 288, 60, 144, 480, 
720, 40, 400, 24, 288, 480, 60, 24, 144, 40, 24, 1]:
`coxeter/class_rep/H4` := [[], [1], [1, 2], [1, 3], [2, 3], [1, 2, 3], [1, 2, 4
], [1, 3, 4], [2, 3, 4], [1, 2, 1, 2], [1, 2, 3, 4], [1, 2, 1, 2, 3], [1, 2, 1
, 2, 4], [1, 2, 1, 2, 3, 4], [1, 2, 1, 2, 3, 2, 4, 3], [1, 2, 1, 2, 3, 2, 1, 2
, 3], [1, 2, 1, 2, 3, 2, 1, 2, 3, 4], [1, 2, 1, 2, 1, 3, 2, 1, 2, 3, 4, 3], [1
, 2, 1, 2, 1, 3, 2, 1, 2, 1, 3, 2, 4, 3], [1, 2, 1, 2, 1, 3, 2, 1, 2, 1, 3, 2, 
1, 2, 3], [1, 2, 1, 2, 1, 3, 2, 1, 2, 1, 3, 2, 1, 2, 3, 4], [1, 2, 1, 2, 3, 2, 
1, 2, 3, 4, 3, 2, 1, 2, 3, 4], [1, 2, 1, 2, 1, 3, 2, 1, 2, 1, 3, 4, 3, 2, 1, 2
, 3, 4], [1, 2, 1, 2, 1, 3, 2, 1, 2, 1, 3, 2, 1, 4, 3, 2, 1, 2, 3, 4], [1, 2, 1
, 2, 1, 3, 2, 1, 2, 1, 3, 2, 1, 2, 3, 4, 3, 2, 1, 2, 3, 4], [1, 2, 1, 2, 1, 3, 
2, 1, 2, 1, 3, 4, 3, 2, 1, 2, 1, 3, 2, 1, 2, 3, 4, 3], [1, 2, 1, 2, 1, 3, 2, 1
, 2, 1, 3, 2, 1, 4, 3, 2, 1, 2, 1, 3, 2, 1, 2, 3, 4, 3], [1, 2, 1, 2, 1, 3, 2, 
1, 2, 3, 4, 3, 2, 1, 2, 1, 3, 2, 1, 2, 3, 4, 3, 2, 1, 2, 3, 4], [1, 2, 1, 2, 1
, 3, 2, 1, 2, 1, 3, 2, 1, 2, 4, 3, 2, 1, 2, 1, 3, 2, 1, 4, 3, 2, 1, 2, 3, 4], [
1, 2, 1, 2, 1, 3, 2, 1, 2, 1, 3, 2, 4, 3, 2, 1, 2, 1, 3, 2, 1, 2, 3, 4, 3, 2, 1
, 2, 1, 3, 2, 1, 2, 3, 4, 3], [1, 2, 1, 2, 1, 3, 2, 1, 2, 1, 3, 2, 1, 2, 3, 4, 
3, 2, 1, 2, 1, 3, 2, 1, 2, 3, 4, 3, 2, 1, 2, 1, 3, 2, 1, 2, 3, 4], [1, 2, 1, 2
, 1, 3, 2, 1, 2, 1, 3, 2, 1, 2, 4, 3, 2, 1, 2, 1, 3, 2, 1, 2, 4, 3, 2, 1, 2, 1
, 3, 2, 1, 4, 3, 2, 1, 2, 3, 4], [1, 2, 1, 2, 1, 3, 2, 1, 2, 1, 3, 2, 1, 4, 3, 
2, 1, 2, 1, 3, 2, 1, 2, 3, 4, 3, 2, 1, 2, 1, 3, 2, 1, 2, 3, 4, 3, 2, 1, 2, 1, 3
, 2, 1, 2, 3, 4, 3], [1, 2, 1, 2, 1, 3, 2, 1, 2, 1, 3, 2, 1, 2, 3, 4, 3, 2, 1, 
2, 1, 3, 2, 1, 2, 3, 4, 3, 2, 1, 2, 1, 3, 2, 1, 2, 3, 4, 3, 2, 1, 2, 1, 3, 2, 1
, 2, 3, 4, 3, 2, 1, 2, 1, 3, 2, 1, 2, 3, 4]]:
`coxeter/mytype/H4` := [[-8, 12, -8, 2], [-4, 0, 4, -2], [-5, 6, -5, 2], [0, -4
, 0, 2], [-2, 0, -2, 2], [-3, 0, 3, -2], [-1, 0, 1, -2], [2, 0, -2, -2], [0, 0
, 0, -2], [-1, -2, -1, 2], [-1, -1, -1, 2], [-2, 0, 2, -2], [3, 0, -3, -2], [0
, -3, 0, 2], [-3, 3, -3, 2], [1, 0, -1, -2], [0, -2, 0, 2], [-6, 9, -6, 2], [-2
, 2, -2, 2], [4, 0, -4, -2], [1, -2, 1, 2], [1, -1, 1, 2], [0, 1, 0, 2], [-4, 6
, -4, 2], [2, 0, 2, 2], [-2, 5, -2, 2], [2, 2, 2, 2], [3, 3, 3, 2], [0, 4, 0, 2
], [2, 5, 2, 2], [5, 6, 5, 2], [4, 6, 4, 2], [6, 9, 6, 2], [8, 12, 8, 2]]:
`coxeter/perm_rep/H4` := permgroup(120,{s2 = [[3, 4], [5, 6], [9, 12], [11, 15]
, [13, 14], [16, 17], [19, 25], [22, 28], [23, 24], [26, 27], [29, 30], [32, 33
], [35, 36], [37, 41], [39, 45], [42, 52], [43, 48], [44, 46], [50, 51], [53, 
54], [55, 56], [57, 58], [59, 65], [62, 70], [63, 64], [66, 67], [68, 69], [71
, 72], [73, 93], [74, 117], [77, 99], [79, 82], [80, 86], [84, 85], [87, 97], [
89, 90], [91, 100], [94, 95], [96, 102], [98, 101], [103, 104], [108, 109], [
111, 112], [114, 120], [118, 119]], s1 = [[4, 5], [6, 7], [8, 9], [10, 11], [12
, 13], [15, 16], [18, 19], [24, 32], [25, 26], [28, 29], [30, 31], [33, 34], [
36, 44], [38, 39], [40, 42], [41, 43], [45, 50], [46, 47], [48, 49], [52, 53], 
[55, 59], [56, 57], [58, 69], [64, 111], [65, 66], [67, 68], [70, 71], [72, 110
], [73, 81], [74, 85], [75, 80], [76, 120], [78, 87], [79, 105], [82, 99], [83
, 84], [86, 103], [88, 89], [90, 119], [92, 101], [93, 100], [95, 97], [98, 102
], [108, 114], [112, 113]], s3 = [[2, 3], [6, 8], [7, 9], [14, 20], [15, 18], [
16, 19], [17, 23], [21, 22], [24, 25], [26, 32], [27, 35], [30, 38], [31, 39], 
[33, 36], [34, 44], [40, 41], [42, 43], [48, 56], [49, 57], [51, 60], [52, 55]
, [53, 59], [54, 63], [61, 62], [64, 65], [66, 111], [67, 79], [68, 105], [72, 
78], [75, 84], [76, 90], [80, 83], [82, 112], [87, 110], [91, 109], [92, 93], [
94, 115], [96, 118], [98, 108], [99, 113], [100, 101], [102, 114], [104, 106], 
[116, 117], [119, 120]], s4 = [[1, 2], [8, 10], [9, 11], [12, 15], [13, 16], [
14, 17], [20, 21], [22, 23], [24, 28], [29, 32], [30, 33], [31, 34], [35, 37], 
[36, 41], [38, 40], [39, 42], [43, 44], [45, 52], [46, 48], [47, 49], [50, 53]
, [51, 54], [60, 61], [62, 63], [64, 70], [71, 111], [72, 112], [73, 79], [74, 
119], [77, 91], [78, 92], [81, 105], [82, 93], [83, 88], [84, 89], [85, 90], [
87, 101], [94, 96], [95, 102], [97, 98], [99, 100], [106, 107], [110, 113], [
115, 116], [117, 118]]}):
