#
## <SHAREFILE=analysis/gfun/gfun.mpl >
## <DESCRIBE>
##                SEE ALSO: calculus/gfun.ps
##        (update - version 2)
##                The gfun package  has been designed as  an help for the
##                manipulation and discovery  of generating functions.
##                Given the first  terms of the sequence,  the package
##                contains functions  that will help  conjecture what the
##                generating  function is. In some  cases, this  answer will
##                be ``explicit''. In most cases though, such an explicit
##                expression  will not exist, and the answer will be
##                an equation (either differential or algebraic) satisfied by
##                the generating function.
##                The gfun package  also  provides tools  to compute with
##                generating functions defined  by  equations. For  instance,
##                given two  generating functions defined  by  linear
##                differential equations  with polynomial coefficients,
##                there   is  a  procedure  to  compute   the differential
##                equation satisfied by their product.
##                AUTHORS: Bruno.Salvy@inria.fr, Paul.Zimmermann@loria.fr
## </DESCRIBE>
## <UPDATE=R4update >


##    -*-Maple-*-
##    Title: 	gfun package (for Generating FUNctions)
##    Created:	Wed Mar  4 15:13:42 1992
##    Authors: 	Bruno Salvy <Bruno.Salvy@inria.fr>
##		Paul Zimmermann <Paul.Zimmermann@inria.fr>
##
## Description: converts implicit equations into differential equations,
##              differential equations into recurrences and vice-versa,
##              ordinary into exponential recurrences.
##		also converts lists to linear recurrences or differential
##		equations.
##
##  Some of the ideas used in this file are due to S. Plouffe and F. Bergeron.
##
## Modifications:
##   formatdiffeq, formatrec added.		 		B.S. Dec 92
##   rectoproc, diffeqtorec, rectodiffeq, rectoproc,
##   borel, invborel rewritten.					B.S. Dec 92
##   2.0 new version						B.S. May-Nov 93
##   2.01 a few extra checks, clarified help for listtolist     B.S. May 94
##   2.05 guessing part rewritten to use pade2 for rational	B.S. Jun 94
##	series, differential and algebraic equations
##	plus a bug fixed in the initial conditions of a product 
##   2.06 faster reversion of power series			B.S. Jul 94
##   2.07 heavy testing, plus a refinement of the criterion for	B.S. Aug 94
##      rejecting overdetermined results from pade2
##   2.1 fixed bugs reported by Ch. Mallinger			P.Z. Nov 94
##   2.11 fixed bug in algebraicsubs				B.S. Dec 94
##   2.12 reduced the number of RETURN statements		B.S. Dec 94
##   2.13 diffeq*diffeq did not always find all the initial	B.S. Jan 95
##        conditions. This is fixed by giving an extra arg
##	  to goodinitvalues/diffeq. Same for goodinitvalues/rec.
##   2.14 diffeq+diffeq was returning a non-formatted equation	B.S. Jan 95
##          when called with two rational functions.
##	  formatdiffeq reinforced to forbid y(0) in the coefficients.
##	  rectodiffeq now returns a homogeneous equation when
##	    the inhomogeneous one contains (D@@k)(y)(0).
##   2.15 rectoproc improved in the non-remember case		B.S. Jan 95
##	    following an idea of PZ.
##	  removed the macros MAPLE5.2 and MAPLE5.3
##	    (switching did not work anymore)
##	  added a case in diffeqtorec where the order of the   
##	    recurrence can be lowered by adding an inhomogeneous part
##	  added Laplace as an alias to invborel
##	  fixed a weakness in rectodiffeq
##   2.16 diffeq+diffeq had a local variable leak when the	B.S. Feb 95
##	   initial conditions involved a linear dependency
##         (like D(y)(0)=y(0))
##   2.17 rec+rec used diffeq+diffeq, but the order of  	B.S. Mar 95
##	   the recurrence was not always optimal. Rewritten
##	   to compute directly on recurrences.
##   2.18 formatrec did not complain on empty recurrences	B.S. Apr 95
##	   plus a bug fix in systomatrix for equations with
##         parameters.
##   2.19 algeqtodiffeq did not allow functions that were	B.S. Apr 95
##	   singular at the origin.
##	  further polishing of goodinitvalues/diffeq both in
##	   regular and singular case.
##        same modification as version 2.13 for diffeq+diffeq
##   2.2  macro MAPLE5.4					B.S. Apr 95
##	  New version for the share library.

##########################################################################
## CONVENTION: a recurrence is an expression of the form
##      sum(p[i]*u(n+i),i=0..d)                   (E)
## or
##    { sum(p[i]*u(n+i),i=0..d), u(0)=a0,..., u(k)=ak }.
##
## The p[i]'s are polynomials in n. The sequence(s) represented by such
## a recurrence are solutions of (E) for n>=k, where k is the largest 
## positive integer solution to p[d](n-d)=0, or 0 if this does not cancel.
## For n<k, the values of the sequence are given by the initial conditions,
## and it is part of the convention that u(i)=0 for i<0.
##########################################################################

macro(
MAPLE5.4	=  false,
algebraicsubs	=  gfun[algebraicsubs],
algeqtodiffeq 	=  gfun[algeqtodiffeq],
cauchyproduct   =  gfun[cauchyproduct],
cheapgausselim	= `gfun/gausselim`,
`diffeq+diffeq` =  gfun[`diffeq+diffeq`],
`diffeq*diffeq` =  gfun[`diffeq*diffeq`],
diffeqtorec	=  gfun[diffeqtorec],
firstnonzero	= `gfun/firstnonzero`,
formatdiffeq	= `gfun/formatdiffeq`,
formatpoleq	= `gfun/formatpoleq`,
formatrec	= `gfun/formatrec`,
gensolvelin	= `gfun/gensolvelin`,
`goodinitvalues/diffeq` = `gfun/goodinitvalues/diffeq`,
`goodinitvalues/rec` = `gfun/goodinitvalues/rec`,
guesseqn	=  gfun[guesseqn],
guessgf		=  gfun[guessgf],
hadamardproduct	=  gfun[hadamardproduct],
inicond		= `gfun/inicond`,
lindep 		= `gfun/lindep`,
listprimpart	= `gfun/listprimpart`,
listtoalgeq	=  gfun[listtoalgeq],
listtodiffeq	=  gfun[listtodiffeq],
listtohypergeom =  gfun[listtohypergeom],
listtolist	=  gfun[listtolist],
listtoratpoly	=  gfun[listtoratpoly],
listtorec	=  gfun[listtorec],
listtoseries	=  gfun[listtoseries],
`l2r/l2r`	= `gfun/listtorec/listtorec`,
`l2h/l2h`	= `gfun/listtohypergeom/listtohypergeom`,
makediffeq	= `gfun/makediffeq`,
makediffeqdiff	= `gfun/makediffeqdiff`,
makerec		= `gfun/makerec`,
maxdegcoeff	=  gfun['maxdegcoeff'],
maxdegeqn	=  gfun['maxdegeqn'],
maxindex	= `gfun/maxindex`,
maxordereqn	=  gfun['maxordereqn'],
mindegcoeff	=  gfun['mindegcoeff'],
mindegeqn	=  gfun['mindegeqn'],
minindex	= `gfun/minindex`,
minordereqn	=  gfun['minordereqn'],
mygcdex		= `gfun/gcdex`,
myisolve	= `gfun/isolve`,
optionsgf	=  gfun['optionsgf'],
ratpolytocoeff	=  gfun[ratpolytocoeff],
`ratpolytocoeff/elmt`=`gfun/ratpolytocoeff/elmt`,
ratsolvelin	= `gfun/ratsolvelin`,
`rec+rec`       =  gfun[`rec+rec`],
`rec*rec`       =  gfun[`rec*rec`],
rectodiffeq	=  gfun[rectodiffeq],
rectoproc	=  gfun[rectoproc],
remove		= `gfun/remove`,
`s2d/s2d`	= `gfun/seriestodiffeq/seriestodiffeq`,
`s2a/s2a`	= `gfun/seriestoalgeq/seriestoalgeq`,
seriestoalgeq	=  gfun[seriestoalgeq],
seriestodiffeq	=  gfun[seriestodiffeq],
seriestohypergeom =gfun[seriestohypergeom],
seriestolist	=  gfun[seriestolist],
seriestoratpoly	=  gfun[seriestoratpoly],
seriestorec	=  gfun[seriestorec],
seriestoseries	=  gfun[seriestoseries],
systomatrix	= `gfun/systomatrix`,
typecheck	= `gfun/typecheck`,
W		= LambertW
);

######################### Global Constants ##############################

`gfun/version`:=2.2:
optionsgf:=['ogf','egf']:
maxordereqn:=3: # default 3rd order
minordereqn:=1: # default 1st order
maxdegcoeff:=4: # default degree 4 coefficients
mindegcoeff:=0: # default constant coefficients
maxdegeqn:=3:	# default maximum 3th degree
mindegeqn:=2:	# default minimum 2nd degree

######################### The pade2 package ##########################
# This should not be there. It just simplifies the distribution for this
# version.
## -*-Maple-*-
##
##    Title: 	pade2
##    Created:	Sep 1993
##    Author: 	Harm Derksen <hderksen@sci.kun.nl>
##
## Description: pade-hermite approximants.
##
##    Modified: Oct 1993
##    Author: 	Bruno Salvy <Bruno.Salvy@inria.fr>
##    Modification: rewritten for efficiency
##
##    June 94. Added the option 'easy'. BS.

macro(BIG=1000); # this is only used with option easy

# This does the interface
pade2:=proc(functionlist::list(algebraic),
	    point::{name,name=algebraic},
	    accuracy::{integer,list(nonnegint)},
	    optional::identical(easy))
local x, a, n, l, i, acc, m, easy, result;
    if type(point,`=`) then a:=op(2,point); x:=op(1,point)
    else a:=0; x:=point fi;
    n:=nops(functionlist);
    if type(accuracy,list) then
	m:=max(op(accuracy));
	l:=[seq(m-i,i=accuracy)];
	acc:=convert(accuracy,`+`)+n-1
    else
	l:=[0$n];
	acc:=accuracy
    fi;
    easy:=evalb(nargs=4);
    result:=`pade2/doit`(map(taylor,subs(x=x+a,functionlist),x,acc),
	x,l,acc-1,easy); # it has to be taylor and not series
    if result=FAIL then FAIL else subs(x=x-a,result) fi
end: # pade2


#  pade2/exmin
#  Extended minimum.
# Input: a list, a boolean function, and an optional name.
# Output: the minimum of the list according to the order
# the name being assigned the index of the first occurrence of
# the minimum in the list.
# 
# No attempt at efficiency has been made, since this should really be in
# the kernel, with sort.
`pade2/exmin`:=proc(l,order,aux)
local res;
    res:=op(1,sort(l,order));
    if nargs=3 then member(res,l,aux) fi;
    res
end: # `pade2/exmin`

# `pade2/doit`
# Input: a list of series in the variable x, the variable x, a list of degree
#	bounds and a bound on the number of iterations. This latter bound 
#	should be at most the order of the series.
# Output: a list of polynomial coefficients for the series, such that 
#	the scalar product of this list with the input has zero Taylor
#	coefficients up to a large order.
`pade2/doit`:=proc(ll,x,degs,nbiter,easy)
local y, i, L, n, j, l, ind, l2, totorder, pivot, vars, leadcoeff, k, 
	t, normalize, lk;
    l:=ll;
    n:=nops(l);
    vars:=[seq(y[i],i=1..n),x];
    L:=[seq(y[i]*x^degs[i],i=1..n)];
    if indets(l)={x} then normalize:=x->x else normalize:=normal fi;
    totorder:=subs([_tosubs=[seq(i=i/t,i=vars)],_t=t,_vars=vars,
	_svars={op(vars)}], proc(p1,p2) local lterm, u, _t, i, d1, d2;
	d1:=degree(p1,_svars); d2:=degree(p2,_svars);
	if d1<>d2 then evalb(d1<d2) else
	lterm:=collect(op(1,series(subs(_tosubs,u*p1+p2),_t,2)),
	    _vars,distributed);
	for i in _vars while nops(series(coeff(lterm,i),u,infinity))<>2 do od;
	    not has(coeff(lterm,i),u) fi end);
    for j from 0 to nbiter while not (
	member(0,l,'k') or member(O(1),[seq(op(1,i),i=l)],'k')) do
	userinfo(3,'pade2',`iteration number`,j);
	leadcoeff:=map(coeff,l,x,j);
	l2:=select(proc(i,l) evalb(op(i,l)<>0) end,[$1..n],leadcoeff);
	if l2=[] then next fi;
	`pade2/exmin`([seq(L[i],i=l2)],totorder,'ind');
	ind:=l2[ind];
	pivot:=1/leadcoeff[ind];
	if pivot=1/O(1) then k:=ind; break fi;
	L:=[seq(L[i]-leadcoeff[i]*pivot*L[ind],i=1..ind-1),collect(x*L[ind],x)
		,seq(L[i]-leadcoeff[i]*pivot*L[ind],i=ind+1..n)];
	if easy and min(op(map(length,L)))>BIG*n then RETURN(FAIL) fi;
	userinfo(5,'pade2',`current list`,
	    lprint(map(collect,L,vars,distributed,normalize)));
	l:=map(normalize,map(series,
	    [seq(l[i]-leadcoeff[i]*pivot*l[ind],i=1..ind-1),x*l[ind],
	    seq(l[i]-leadcoeff[i]*pivot*l[ind],i=ind+1..n)],x,infinity))
    od;
    if j=nbiter+1 and not 
	(member(0,l,'k') or member(O(1),[seq(op(1,i),i=l)],'k')) then
	`pade2/exmin`([seq(op(2,i),i=l)],numeric,'k') fi;
    lk:=collect(L[k],vars,distributed,normalize);
    map(expand,[seq(coeff(lk,y[i])*x^(-degs[i]),i=1..n)])
end: # `pade2/doit`


######################### Type Checking ##############################

`type/gfun/identity`:=proc(x) type(x,`=`) and op(1,x)=op(2,x) end:
`type/gfun/free`:=proc(x,y) not has(x,y) end:

# type(y(0),initeq(y))		-> true
# type(D(y)(0),initeq(y))	-> true
# type((D@@k)(y)(0),initeq(y))	-> true
# otherwise			-> false
`type/gfun/initeq` := proc(expr,y)
local f;
    if not type(expr,function(0)) then
	false
    else
	f := op(0,expr);
	f=y or f='D(y)' or (type(f,function(identical(y)))) and 
	    type(op(0,f),`@@`(identical(D),integer))
    fi
end: # `type/gfun/initeq`

# This procedure avoids several type checks of the same expression.
# Besides, it handles the defaults.
typecheck:=proc (n)
local i;
    if n=1 then		# l, x, <met>
	if nargs>2 and type([args[2..3]],[list,name]) then
	    if nargs=3 then RETURN('stamped',args[2..3],op(1,optionsgf))
	    elif nargs=4 and type(gfun[cat(`listtoseries/`,args[4])],procedure)
		then RETURN('stamped',args[2..4])
	    elif nargs>4 then ERROR(`too many arguments`)
	    else ERROR(`invalid argument`,args[4])
	    fi
	elif nargs<3 then ERROR(`too few parameters`)
	elif not type(args[2],list) then ERROR(`not a list`,args[2])
	elif not type(args[3],name) then ERROR(`not a name`,args[3])
	else ERROR(`invalid arguments`)
	fi
    elif n=2 then	# l, y(x), <[met]>
	if nargs>2 and type([args[2..3]],[list,function(name)]) then
	    if nargs=3 then RETURN('stamped',args[2..3],optionsgf)
	    elif nargs=4 and type(args[4],list) and
		type([seq(gfun[cat(`listtoseries/`,i)],i=args[4])],
		    list(procedure)) then RETURN('stamped',args[2..4])
	    elif nargs>4 then ERROR(`too many arguments`)
	    else ERROR(`invalid argument`,args[4])
	    fi
	elif nargs<3 then ERROR(`too few arguments`)
	elif not type(args[2],list) then ERROR(`not a list`,args[2])
	elif not type(args[3],function(name)) then
	    ERROR(`invalid unknown function`,args[3])
	else ERROR(`invalid arguments`)
	fi
    elif n=3 then	# l, x, [<met>]
	if nargs>2 and type([args[2..3]],[list,name]) then
	    if nargs=3 then RETURN('stamped',args[2..3],optionsgf)
	    elif nargs=4 and type(args[4],list) and
		type([seq(gfun[cat(`listtoseries/`,i)],i=args[4])],
		    list(procedure)) then RETURN('stamped',args[2..4])
	    elif nargs>4 then ERROR(`too many arguments`)
	    elif nargs=4 then ERROR(`invalid argument`,args[4])
	    fi
	elif nargs<3 then ERROR(`too few parameters`)
	elif not type(args[2],list) then ERROR(`not a list`,args[2])
	elif not type(args[3],name) then ERROR(`not a name`,args[3])
	else ERROR(`invalid argments`)
	fi
    elif n=4 then	# s, x, y
	if nargs=4 and type([args[2..4]],[series,name,name]) then
	    RETURN('stamped',args[2..4])
	elif nargs<>4 then ERROR(`wrong number of arguments`)
	elif not type(args[2],series) then ERROR(`not a series`,args[2])
	elif not type(args[3],name) then ERROR(`not a name`,args[3])
	elif not type(args[4],name) then ERROR(`not a name`,args[4])
	else ERROR(`invalid arguments`)
	fi
    elif n=5 then	# l, y(x)
	if nargs=3 and type([args[2..3]],[list,function(name)]) then
	    RETURN('stamped',args[2..3])
	elif nargs<>3 then ERROR(`wrong number of arguments`)
	elif not type(args[2],list) then ERROR(`not a list`,args[2])
	elif not type(args[3],function(name)) then
	    ERROR(`invalid unknown function`,args[3])
	else ERROR(`invalid arguments`)
	fi
    elif n=6 then	# s, y(x), <[met]>
	if nargs>2 and type([args[2..3]],[series,function(name)]) then
	    if nargs=3 then RETURN('stamped',args[2..3],optionsgf)
	    elif nargs=4 and type(args[4],list) and
		type([seq(gfun[cat(`listtoseries/`,i)],i=args[4])],
		    list(procedure)) then RETURN('stamped',args[2..4])
	    elif nargs>4 then ERROR(`too many arguments`)
	    else ERROR(`invalid argument`,args[4])
	    fi
	elif nargs<3 then ERROR(`too few arguments`)
	elif not type(args[2],series) then ERROR(`not a series`,args[2])
	elif not type(args[3],function(name)) then
	    ERROR(`invalid unknown function`,args[3])
	else ERROR(`invalid arguments`)
	fi
    elif n=7 then 	# s, <[met]>
	if nargs>1 and type(args[2],series) then
	    if nargs=2 then RETURN('stamped',args[2],optionsgf)
	    elif nargs=3 and type(args[3],list) and 
		type([seq(gfun[cat(`listtoseries/`,i)],i=args[3])],
		    list(procedure)) then RETURN('stamped',args[2..3])
	    elif nargs>3 then ERROR(`too many arguments`)
	    else ERROR(`invalid argument`,args[3])
	    fi
	elif nargs=1 then ERROR(`too few arguments`)
	elif not type(args[2],series) then ERROR(`not a series`,args[2])
	else ERROR(`invalid arguments`)
	fi
    elif n=8 then 	# s, <met>
	if nargs>1 and type(args[2],series) then
	    if nargs=2 then RETURN('stamped',args[2],'ogf')
	    elif nargs=3 and type(gfun[cat(`listtoseries/`,args[3])],procedure)
		then RETURN('stamped',args[2..3])
	    elif nargs>3 then ERROR(`too many arguments`)
	    else ERROR(`invalid argument`,args[3])
	    fi
	elif nargs=1 then ERROR(`too few arguments`)
	elif not type(args[2],series) then ERROR(`not a series`,args[2])
	else ERROR(`invalid arguments`)
	fi
    elif n=9 then 	# l, <met>
	if nargs>1 and type(args[2],list) then
	    if nargs=2 then RETURN('stamped',args[2],'ogf')
	    elif nargs=3 and type(gfun[cat(`listtoseries/`,args[3])],procedure)
		then RETURN('stamped',args[2..3])
	    elif nargs>3 then ERROR(`too many arguments`)
	    else ERROR(`invalid argument`,args[3])
	    fi
	elif nargs=1 then ERROR(`too few arguments`)
	elif not type(args[2],list) then ERROR(`not a list`,args[2])
	else ERROR(`invalid arguments`)
	fi
    elif n=10 then	# recurrence, function(name), function(name)
	if nargs>4 then ERROR(`too many arguments`)
	elif nargs<4 then ERROR(`too few arguments`)
	elif not type([args[3..4]],[function(name),function(name)]) then
	    ERROR(`invalid arguments`)
	fi
    else ERROR(`should not happen`)
    fi;
end: # typecheck

################## Modifications of existing Maple code ############

## Adapted from solve/linear/integer
ratsolvelin := proc (equations, unknowns)
local eqn, eqns, vars, sol, sols, i, k, l, pivot, var, c, j;
    # Suppress trivial equations and
    # normalize: convert from Q to Z and divide out by the content
    eqns := {seq(l/icontent(l)*sign(l), l=  equations minus {0})};
    # Solve the remaining system using a sparse implementation
    for k while eqns <> {} do
	l:=map(length,[op(eqns)]);
	member(min(op(l)),l,'i');
	eqn := eqns[i];
	vars := indets(eqn) intersect unknowns;
	var := vars[1];
	pivot := coeff(eqn,var,1);
	for j from 2 to nops(vars) do
	    # Eliminate the unknown with the smallest coefficient
	    c := coeff(eqn,vars[j],1);
	    if length(c) < length(pivot) then pivot := c; var := vars[j] fi
	od;
	eqns:=subs(var=-1/pivot*subs(var=0,eqn),subsop(i=NULL,eqns)) minus {0};
	sol[k]:=var,eqn,pivot
    od;
    sols := {};j:=0;
    for i from k-1 by -1 to 1 do
	eqn[i] := - subs( sol[i][1] = 0, sols, sol[i][2] );
	if eqn[i]<>0 then j:=j+1;l:=subsop(j=i,l) fi;
	sols := sols union {sol[i][1] = eqn[i] / sol[i][3]}
    od;
    # Parameterize an infinite solution space
    vars := {op(unknowns)} minus {seq(op(1,i),i=sols)};
    if vars={} then sols
    elif nops(vars)=1 then {op(subs(op(vars)=1,sols)),op(vars)=1}
    elif select(type,{seq(op(2,i),i=sols)},rational) minus {0} <> {} then
	{op(subs(seq(i=0,i=vars),sols)),seq(i=0,i=vars)}
    else FAIL # we do not want to treat this
    fi
end: # ratsolvelin

# A simple interface to solve/linear
gensolvelin:=proc ()
local s, t;
    s:=readlib(`solve/linear`)(args);
    t:=select(type,s,`gfun/identity`);
    if nops(t)<>1 then FAIL
    else subs(op(1,op(t))=1,(1=1)=(op(1,op(t))=1),s)
    fi
end: # gensolvelin


# this procedure does not do exactly the same as gcdex, but is much faster
# it returns g' and assign s,t such that s*a+t*b=g'
# and g'=lambda gcd(a,b) with lambda a rational fraction independent of y
###################################################################
##########  ONE CAN PROBABLY IMPROVE IT MORE   ####################
###################################################################
mygcdex := proc(a,b,y,s,t)
local q,r,u,v,du,dv,alpha,beta,oldalpha,oldbeta,dr,lr;
    u:=collect(a,y); v:=collect(b,y);
    if u=0 then 
	b
    elif v=0 then a
    else
	oldalpha:=1; oldbeta:=0;
	alpha:=0; beta:=1;
	# u = oldalpha*a + oldbeta*b
	# v = alpha*a + beta*b
	du:=degree(u,y);
	dv:=degree(v,y);
	do
	    if du<dv then
		dv; dv:=du; du:="";
		v; v:=u; u:="";
		alpha; alpha:=oldalpha; oldalpha:="";
		beta; beta:=oldbeta; oldbeta:="";
	    fi;
	    userinfo(3,'gfun',`degrees of the polynomials`,du,dv);
	    userinfo(5,'gfun','polynomials',u,v);
	    q:=lcoeff(u,y)/lcoeff(v,y)*y^(du-dv);
	    r:=u-collect(q*v,y);
	    do
		dr:=degree(r,y);
		lr:=coeff(r,y,dr);
		if normal(lr)=0 then
		    r:=subs(y^dr=0,r);
		    if degree(r,y)=dr then break fi
		else break
		fi
	    od;
	    if r=0 then break fi;
	    oldalpha-q*alpha; oldalpha:=alpha; alpha:="";
	    oldbeta-q*beta; oldbeta:=beta; beta:="";
	    u:=v;du:=dv;
	    v:=r;dv:=degree(v,y);
	od;
	if dv=0 then
	    if nargs=4 then s:=rem(alpha/v,b,y) fi;
	    if nargs=5 then t:=rem(beta/v,a,y) fi;
	    1
	else
	    if nargs=4 then s:=rem(alpha,b,y) fi;
	    if nargs=5 then t:=rem(beta,a,y) fi;
	    v
	fi
    fi
end: # mygcdex

# This is really needed because MapleV's isolve is terrible
# eqn is a polynomial in n
myisolve:=proc (eqn, n)
local sol, i, d, f;
    if not has(eqn,n) then {}
    # These two lines are due to M. Monagan:
    elif type(eqn,polynom(rational,n)) then
	select(type,{seq(i[1],i=roots(eqn))},integer)
    else
	f:=collect(eqn,n);
	d:=degree(f,n);
	if d=1 then
	    f:=-coeff(f,n,0)/coeff(f,n,1);
	    if type(f,integer) then {f}
	    elif not has(f,RootOf) and not has(f,`&RootOf`) then {}
	    else f:=evala(Normal(f));
		if type(f,integer) then {f} else {} fi
	    fi
	elif d=2 then
	    f:=[seq(coeff(f,n,i),i=0..2)];
	    d:=sqrt(op(2,f)^2-4*op(3,f)*op(1,f));
	    select(type,map(normal,
		    {(d-op(2,f))/2/op(3,f),-(d+op(2,f))/2/op(3,f)}),integer)
	elif type(eqn,`*`) then `union`(op(map(myisolve,[op(eqn)],n)))
	else
	    sol:=traperror(isolve(expand(eqn),n));
	    if sol=lasterror then {}
	    else {seq(op(2,i),i=select(type,map(op,[sol]),
		identical(n)=integer))}
	    fi
	fi
    fi
end: # myisolve

######################### Various Utilities ###################

firstnonzero:=proc(pol,n) max(-1,op(myisolve(pol,n)))+1 end:

# returns the smallest i such that u(n+i) appears in a recurrence
minindex := proc(rec,u,n)
    min(op(map(op,indets(rec,u(linear(n))))))-n
end: # minindex

# returns the largest i such that u(n+i) appears in a recurrence
maxindex := proc(rec,u,n)
    max(op(map(op,indets(rec,u(linear(n))))))-n
end: # maxindex

# the opposite of select
# this should be in the kernel
remove:=proc(f,a) map(subs(_f=f,
    proc(x) if not _f(args) then x else NULL fi end),
    args[2..nargs]) end:

# systomatrix
#  Input: a system of homogeneous linear equations and a list of variables V,
#         and the name of a vector B.
#  Output: a matrix A such that the system is equivalent to A.V=B.
# Almost like linalg[genmatrix], but more suitable for our purpose.
# Also, if sys is a list instead of a set, then the order will be preserved.
systomatrix:=proc (sys, V, B)
local a, i, j, eqn, zero;
    if sys={} or sys=[] then
	B:=array(1..1,[0]);
	array(1..1,1..nops(V),[[0$nops(V)]])
    else
	zero:=[seq(i=0,i=V)];
	a:=array(1..nops(sys),1..nops(V),sparse);
	B:=array(1..nops(sys));
	for i to nops(sys) do
	    eqn:=op(i,sys);
	    if type(eqn,`=`) then eqn:=op(1,eqn)-op(2,eqn) fi;
	    eqn:=collect(eqn,V,distributed);
	    for j to nops(V) do a[i,j]:=coeff(eqn,V[j]) od;
	    B[i]:=-subs(zero,eqn) od;
	op(a)
    fi
end: # systomatrix

#listprimpart
# Input: a list of polynomials and a variable
# Output: the list where common factors have been removed.
listprimpart:=proc (l, z)
local g, i, T, q;
    g:=content(convert([seq(op(i,l)*T^(i-1),i=1..nops(l))],`+`),T,'q');
    if has(g,z) then [seq(coeff(q,T,i),i=0..nops(l)-1)]
    else l fi
end: # listprimpart

# This is used to find a linear dependency.
# The result is the last element of the last line which is not 0.
# The pivoting must not be done as gausselim does it.
# This is a fraction free gausselim using the input matrix for the intermediate
# computations. Also, no type checking is performed.
cheapgausselim:=proc (A, nlin, ncol)
local c, i, j, k, def;
    def:=0;
    for c to nlin do
	for k from c+def to ncol-1 while normal(A[c,k])=0 do A[c,k]:=0 od;
	if k=ncol then RETURN(A[c,ncol]) fi;
	for k from c to nlin while normal(A[k,c+def])=0 do A[k,c+def]:=0 od;
	if k=nlin+1 then def:=def+1; c:=c-1; next fi;
	for i in [seq(i,i=c..k-1),seq(i,i=k+1..nlin)] do
	    if A[i,c+def]=0 then next fi;
	    for j from c+def+1 to ncol do
		A[i,j]:=A[k,c+def]*A[i,j]-A[i,c+def]*A[k,j] od;
	    A[i,c+def]:=0  # this line and the next commented line add some
	od;
	userinfo(5,'gfun',`line `,k,` eliminated`);
	# speedup for very large matrices:
	if k=c then for i from c to ncol-1 do A[c,i]:=0 od fi;
	userinfo(6,'rsolve',`remaining matrix `,print(op(A)));
    od;
    FAIL
end: # cheapgausselim

#lindep
# Input:  a vector u = [u1 , ... , uk] and a matrix A = array(1..k,1..l)
#         such that u[i] = sum(A[i,j] e[j]) for some e[j].
#         The coefficients are rational functions in x.
# Output: a linear dependency relation between the u[i] if there is one,
#	  FAIL otherwise
lindep := proc(A, u, x)
local k, i, l, v, rel, unk, j;
    k := nops(u);
    if linalg[rowdim](A)<>k then ERROR(`incorrect number of rows`) fi;
    l := linalg[coldim](A);
    userinfo(2,'gfun',`looking for a linear dependency in a`,k,' x ',l,'matrix');
    unk:=[seq(v[i],i=1..k)];
    rel := cheapgausselim(
	linalg[augment](A,linalg[matrix](k,1,unk)),k,l+1);
    if rel=FAIL then FAIL
    else
	rel:=primpart(numer(convert([seq(normal(subs([i=1,seq(j=0,
	    j=subs(i=NULL,unk))],rel))*i,i=unk)],`+`)),unk);
	convert([seq(subs([i=1,seq(j=0,j=subs(i=NULL,unk))],rel)
	    *u[op(i)],i=unk)],`+`)
    fi
end: # lindep

#formatdiffeq
# Input:  a list [diffeq,y(x)] containing a linear differential equation
#            with polynomial coefficients and its unknown
#         (optional) Y and X 
#             names to be assigned the unknown function and its variable
#         (optional) iniconds to be assigned the initial conditions
# Output: a list of polynomials in x [u(x),p_0(x),...,p_d(x)] meaning
#                                              (d)
#            eqn = p_0(x) y(x) + ... + p_d(x) y   (x) + u(x)
#
#   This is where the type checking is done.
#
formatdiffeq:=proc (l,Y,X,iniconds)
local r, y, x, i, difford, j, lvar;
    if nops(l)<>2 then ERROR(`wrong number of arguments`) fi;
    if not type(op(2,l),function(name)) then
	ERROR(`invalid unknown`,op(2,l)) fi;
    x:=op(op(2,l)); y:=op(0,op(2,l)); if nargs>1 then X:=x; Y:=y fi;
    if type(op(1,l),set)  then
	r:=select(has,op(1,l),x);
	if nops(r)>1 then ERROR(`invalid differential equation`) fi;
	if nops(r)=0 then
	    ERROR(`the unknown variable does not appear in the equation`) fi;
	if nargs>3 then	iniconds:=op(1,l) minus r fi;
	r:=op(r)
    else r:=op(1,l); if nargs>3 then iniconds:={} fi fi;
    if type(r,`=`) then r:=op(1,r)-op(2,r) fi;
    if select(has,indets(r,y(anything)),x) minus {y(x)} <> {} then 
	ERROR(`invalid differential equation`) fi;
    r:=expand(numer(normal(convert(r,D))));
    lvar:=select(proc(u,y) has(u,D) and has(u,`@@`) and has(u,y) end,
	indets(r,function),y);
    if lvar<>{} and map(op,lvar)<>{x} then
	ERROR(`invalid differential equation`) fi;
    lvar:= {seq(op(2,op(0,op(0,i))),i=lvar)};
    if lvar<>{} then difford:=max(op(lvar))
    elif has(r,D(y)(x)) then difford:=1
    else difford:=0 fi;
    D(y):=D(y);
    r:=[subs([seq((D@@j)(y)(x)=0,j=0..difford)],r),
	seq(coeff(r,(D@@j)(y)(x),1),j=0..difford)];
    if not type(r,list(polynom(anything,x))) then
	ERROR(`non-polynomial coefficients`) fi;
    listprimpart(r,x)
end: # formatdiffeq

# makediffeq
# Input:  a differential equation in the format returned by formatdiffeq
#         the variables y and x, meaning the unknown function is y(x)
#         (optional) a set of initial conditions
# Output: the corresponding differential equation, which is in a set if
#         there are initial conditions.
makediffeq:=proc (deq, y, x, ini)
local r, i;
    r:=convert([seq(deq[i+2]*(D@@i)(y)(x),i=0..nops(deq)-2),deq[1]],`+`);
    if nargs=3 or ini={} then r
    else {r,op(ini)} fi
end: # makediffeq

# makediffeqdiff
# Input:  a differential equation in the format returned by formatdiffeq
#         the variables y and x, meaning the unknown function is y(x)
#         (optional) a set of initial conditions
# Output: the corresponding differential equation, which is in a set if
#         there are initial conditions. The difference with makediffeq
#	  is that here diff is used instead of D.
makediffeqdiff:=proc (deq, y, x, ini)
local r, i;
    r:=convert([deq[1],deq[2]*y(x),seq(deq[i+2]*diff(y(x),x$i),
	i=1..nops(deq)-2)],`+`);
    if nargs=3 or ini={} then r
    else {r,op(ini)} fi
end: # makediffeqdiff

#formatrec
# Input:  a list [rec,u(n)] containing a linear recurrence with 
#            polynomial coefficients and its unknown
#         u, n names to be assigned the unknown function and variable
#         (optional) iniconds to be assigned the initial conditions
#         (optional) one: boolean saying whether one is interested in 
#                         only one solution.
# Ouput:  a list of polynomials in n: [b(n),p_0(n),...,p_d(n)] meaning
#
#                  rec=p_0(n)u(n)+...+p_d(n)u(n+d)+b(n)
#
#  This is where the type checking is done.
#
formatrec:=proc (l, u, n, iniconds, one)
local r, i, U, N, mi, ma, locini;
    if nops(l)<2 or nops(l)>5 then
	ERROR(`rsolve: wrong number of arguments`) fi;
    if not type(op(2,l),function(name)) then
	ERROR(`invalid unknown`,op(2,l)) fi;
    N:=op(op(2,l)); U:=op(0,op(2,l)); if nargs>1 then n:=N; u:=U fi;
    if type(op(1,l),set) then
	r:=select(has,op(1,l),N);
	if nops(r)>1 then ERROR(`invalid recurrence`) fi;
	if r={} then ERROR(`empty recurrence`) fi;
	if nargs>3 then locini:=op(1,l) minus r; iniconds:=locini;
	    if not type(locini,set(U(integer)=anything)) then
		ERROR(`invalid initial conditions`,locini)
	    fi fi;
	r:=op(r)
    else r:=op(1,l); if nargs>3 then iniconds:={} fi
    fi;
    if nargs>4 then if nops(l)>4 and type(op(5,l),boolean) then one:=op(5,l)
    else one:=false fi fi;
    if type(r,`=`) then r:=op(1,r)-op(2,r) fi;
    mi:=minindex(r,U,N); ma:=maxindex(r,U,N);
    r:=collect(r,[seq(U(N+i),i=mi..ma)],normal);
    if mi<>0 then r:=subs(N=N-mi,r) fi;
    if has(map(denom,{op(r)}),N) then
	r:=collect(numer(normal(r)),[seq(U(N+i),i=mi..ma)],normal) fi;
    r:=[subs([seq(U(N+i)=0,i=0..ma-mi)],r),seq(coeff(r,U(N+i),1),i=0..ma-mi)];
    if has(r,U) or not type(r,list(polynom(anything,n))) then 
	ERROR(`invalid recurrence or unknown`) fi;
    listprimpart(r,N)
end: # formatrec

# makerec
# Input:  a recurrence in the format returned by formatrec
#         the variables u and n, meaning the unknown sequence is u(n)
#         (optional) a set of initial conditions
# Output: the corresponding recurrence, which is in a set if there are
#         initial conditions.
makerec:=proc (rec, u, n, ini)
local r, i;
    r:=convert([seq(rec[i+2]*u(n+i),i=0..nops(rec)-2),rec[1]],`+`);
    if nargs=3 or ini={} then r
    else {r,op(ini)} fi
end: # makerec

# formatpoleq
# Input:  a list [p, y(z)] containing a polynomial in two variables and
#          an unknown function and possibly initial conditions.
#	  y, z names to be assigned the unknown function and variable
#	  (optional) iniconds to be assigned the initial conditions.
# Output: the polynomial, type checked and without its inital values
formatpoleq:=proc (l, y, z, iniconds)
local Y, Z, P;
    if nops(l)<2 or nops(l)>4 then 
	ERROR(`formatpoleq: wrong number of arguments`) fi;
    if not type(op(2,l),function(name)) then
	ERROR(`invalid unknown`,op(2,l)) fi;
    Z:=op(1,op(2,l));Y:=op(0,op(2,l)); if nargs>1 then y:=Y; z:=Z fi;
    if type(op(1,l),`=`) then P:=op(1,op(1,l))-op(2,op(1,l))
    else P:=op(1,l) fi;
    if not type(P,polynom(anything,[Y,Z])) then ERROR(`invalid argument`,P) fi;
    if nargs=4 then
	if nops(l)>2 then
	    if type(op(3,l),set) then iniconds:=op(3,l)
	    else ERROR(`invalid argument`,op(3,l)) fi;
	else iniconds:={} fi fi;
    P
end: # formatpoleq

# `goodinitvalues/rec`
# Input:  a recurrence in the format returned by formatrec
#         the unknown sequence and its variable
#         (optional) some initial conditions
#	  (optional) an integer p
# Output: a set of equalities u(k)=v_k, from which all the other values
#         can be deduced by solving the recurrence for its maximal index.
#	  This set is continued up to the pth term is p is given.
#         These equalities are of the type u(k)=u(k) when this value is 
#         arbitrary.
#         The result is an ERROR when no initial condition can be found.
`goodinitvalues/rec`:=proc (rec, u, n, ini, p)
local n0, order, i, inds, minind, maxind, sys, r, sol, b, a,minind2,lmin, j, k;
    order:=nops(rec)-2;
    maxind:=order-1;
    if nargs>3 then
	if type(ini,set) then inds:=map(op,indets(ini,u(integer)))
	else inds:={}; maxind:=max(maxind,p) fi;
	if nargs=5 then maxind:=max(maxind,p) fi fi;
    n0:=firstnonzero(subs(n=n-order,rec[nops(rec)]),n);
    maxind:=max(maxind,op(inds),n0-1)-order;
    minind:=min(op(inds),max(0,n0-order-1));
    minind2:=max(order-1,n0-1,op(inds))-order+1;
    if minind2>minind then lmin:=[minind,minind2] else lmin:=[minind] fi;
    r:=makerec(rec,u,n);
    for j to nops(lmin) do
	sys:={op(ini),seq(subs(n=i,r),i=op(j,lmin)..maxind)};
	if sys={} then RETURN({seq(u(i)=u(i),i=0..minind-1)}) fi;
	a:=systomatrix(sys,[seq(u(i),i=minind..maxind+order)],'b');
	sol:=linalg[linsolve](a,b);
	if sol=NULL and j=nops(lmin) then ERROR(`no valid initial conditions`)
	elif sol<>NULL then break fi
    od;
    sol:=convert(sol,list);
    for i in indets(sol,_t[anything]) do
	if member(i,sol,'k') then sol:=subs(i=u(minind+k-1),sol) fi od;
    {seq(u(i)=u(i),i=0..minind-1),
            seq(u(i)=sol[i-minind+1],i=minind..minind+nops(sol)-1)}
end: # `goodinitvalues/rec`

# `goodinitvalues/diffeq`
# Input:  a differential equation in the format returned by formatdiffeq
#         the unknown function and its variable
#         (optional) some initial conditions
#	  (optional) an integer p
# Output: a set of equalities (D@@k)(y)(0)=v_k, from which all the others
#	   can be computed. This set is continued up to order p if p is given.
#         The result is an ERROR when no initial condition can be found,
#	  except in the case when the origin is a singular point. Then it 
#	  returns {}.
`goodinitvalues/diffeq`:=proc (deq, y, z, ini, p)
local u, init, i, sol, maxorder, ord;
    maxorder:=nops(deq)-3;
    if nargs=5 then maxorder:=max(maxorder,p) fi;
    ord:=maxorder;
    if nargs>3 and type(ini,set) then
	init:=ini;
	maxorder:=max(maxorder,seq(op(2,op(0,op(0,i))),
	    i=indets(init,`gfun/initeq`(y)) minus {y(0),D(y)(0)}));
	if maxorder=0 and has(init,D(y)(0)) then maxorder:=1 fi
    else init:={} fi;
    u:=series(eval(subs(y(z)=convert([seq((D@@i)(y)(0)*z^i/i!,
	i=0..maxorder),O(1)*z^(maxorder+1)],`+`),op(init),
	makediffeqdiff(deq,y,z))),z,infinity);
    u:={seq(coeff(u,z,i),i=0..maxorder)} minus {O(1)};
    sol:=solve(u,{seq((D@@i)(y)(0),i=0..maxorder)}
	intersect indets(u,function(0)));
    if subs(z=0,op(nops(deq),deq))<>0 then
	init:=select(proc(x,y) member(op(1,x),y) end,init,[seq((D@@i)(y)(0),
	    i=0..maxorder)]) fi;
    if sol=NULL then ERROR(`no valid initial conditions`) fi;
    init:=remove(type,sol union init,`gfun/identity`);
    if subs(z=0,deq[nops(deq)])<>0 then
	remove(has,init,{seq((D@@i)(y)(0),i=ord+1..maxorder)})
    elif {seq(op(2,i),i=init)}={0} then {}
    else select(type,init,anything=`gfun/free`(y)) fi
end: # `goodinitvalues/diffeq`

######################### Conversion Routines ###################

seriestolist:=proc()
local s, meth, l, x, i;
    if args[1]<>'stamped' then RETURN(seriestolist(typecheck(8,args))) fi;
    s:=args[2]; meth:=args[3]; x:=op(0,s);
    if op(nops(s)-1,s)=O(1) then l:=[seq(coeff(s,x,i),i=0..op(nops(s),s)-1)]
    else l:=[seq(coeff(s,x,i),i=0..op(nops(s),s))] fi;
    if meth='ogf' then l
    else seriestolist('stamped',listtoseries('stamped',l,x,meth),'ogf')
    fi
end: # seriestolist

listtolist:=proc()
local x;
    if args[1]<>'stamped' then listtolist(typecheck(9,args))
    elif args[3]='ogf' then args[2]
    else seriestolist('stamped',
	listtoseries('stamped',args[2],x,args[3]),'ogf')
    fi
end: #listtolist

seriestoseries:=proc ()
    if args[1]<>'stamped' then seriestoseries(typecheck(8,args))
    else listtoseries('stamped',seriestolist('stamped',args[2],'ogf'),
	op(0,args[2]),args[3])
    fi
end: # seriestoseries

listtoseries:=proc ()
    if args[1]<>'stamped' then listtoseries(typecheck(1,args))
    else map(normal,gfun[cat(`listtoseries/`,args[4])](args[2],args[3])) fi
end: # listtoseries

gfun[`listtoseries/egf`]:=proc(l,x)
local i;
    series(convert([seq(op(i,l)*x^(i-1)/(i-1)!,i=1..nops(l)),
	O(x^(nops(l)))],`+`),x,nops(l))
end:

gfun[`listtoseries/Laplace`]:=proc(l,x)
local i;
    series(convert([seq(op(i,l)*x^(i-1)*(i-1)!,i=1..nops(l)),
	O(x^(nops(l)))],`+`),x,nops(l))
end:

gfun[`listtoseries/ogf`]:=proc(l,x)
local i;
    series(convert([seq(op(i,l)*x^(i-1),i=1..nops(l)),
	O(x^(nops(l)))],`+`),x,nops(l))
end: # `listtoseries/ogf`

gfun[`listtoseries/revogf`]:=proc(L,x)
local l, i, nl;
    l:=L;
    while l<>[] and l[1]=0 do l:=subsop(1=NULL,l) od;
    if l=[] then ERROR(`cannot revert 0 series`) fi;
    nl:=nops(l);
    `gfun/powrevert`(series(convert([seq(l[i]*x^i,i=1..nl),x^(nl+1)],
	`+`),x,nl+1),x,nl)
end: # `listtoseries/revogf`

gfun[`listtoseries/revegf`]:=proc(L,x)
local l, i, nl;
    l:=L;
    while l<>[] and l[1]=0 do l:=subsop(1=NULL,l) od;
    if l=[] then ERROR(`cannot revert 0 series`) fi;
    nl:=nops(l);
    `gfun/powrevert`(series(convert([seq(l[i]*x^i/i!,i=1..nl),x^(nl+1)]
	,`+`),x,nl+1),x,nl)
end: # `listtoseries/revegf`

gfun[`listtoseries/lgdogf`]:=proc(L,x)
local l, i, nl;
    l:=L;
    while l<>[] and l[1]=0 do l:=subsop(1=NULL,l) od;
    if l=[] then ERROR(`cannot revert 0 series`) fi;
    nl:=nops(l);
    series(convert([seq(i*l[i+1]*x^(i-1),i=1..nl-1)],`+`)/
	    convert([seq(l[i]*x^(i-1),i=1..nl)],`+`),x,nl-1)
end: # `listtoseries/lgdogf`

gfun[`listtoseries/lgdegf`]:=proc(L,x)
local l, i, nl;
    l:=L;
    while l<>[] and l[1]=0 do l:=subsop(1=NULL,l) od;
    if l=[] then ERROR(`cannot revert 0 series`) fi;
    nl:=nops(l);
    series(convert([seq(l[i]*x^(i-2)/(i-2)!,i=2..nl)],`+`)/
	convert([seq(l[i]*x^(i-1)/(i-1)!,i=1..nl)],`+`),x,nl-1)
end: # `listtoseries/lgdegf`

listtodiffeq:=proc()
local result, ex, methods, method, y, x, s, unkn, expr;
    if args[1]<>'stamped' then RETURN(listtodiffeq(typecheck(2,args))) fi;
    expr:=args[2];unkn:=args[3];methods:=args[4];
    y:=op(0,unkn);x:=op(unkn);ex:=expr;
    for method in methods do
	s:=listtoseries('stamped',ex,x,method);
	userinfo(3,'gfun',`Trying the `,method,s);
	result:=`s2d/s2d`(s,x,y);
	if result<>FAIL then
	    userinfo(2,'gfun','The',method,'`seems to satisfy`',result);
	    RETURN([inicond(s,result,y,x),method])
	fi
    od;
    FAIL
end: # listtodiffeq

seriestodiffeq:=proc ()
    if args[1]<>'stamped' then seriestodiffeq(typecheck(6,args))
    else listtodiffeq('stamped',
	seriestolist('stamped',args[2],'ogf'),args[3],args[4]) fi
end: # seriestodiffeq

listtoalgeq:=proc()
local result, ex, methods, method, y, x, s, unkn, expr;
    if args[1]<>'stamped' then RETURN(listtoalgeq(typecheck(2,args))) fi;
    expr:=args[2];unkn:=args[3];methods:=args[4];
    y:=op(0,unkn);x:=op(unkn);ex:=expr;
    for method in methods do
	s:=listtoseries('stamped',ex,x,method);
	userinfo(3,'gfun',`Trying the `,method,s);
	result:=`s2a/s2a`(s,x,y);
	if result<>FAIL then
	    userinfo(2,'gfun','The',method,'`seems to satisfy`',result);
	    RETURN([result,method])
	fi
    od;
    FAIL
end: # listtoalgeq

seriestoalgeq:=proc ()
    if args[1]<>'stamped' then seriestoalgeq(typecheck(6,args))
    else listtoalgeq('stamped',
	seriestolist('stamped',args[2],'ogf'),args[3],args[4]) fi
end: # seriestoalgeq

listtoratpoly:=proc()
local result, ex, methods, method, s, x, bigO, ord, nbz;
    if args[1]='stamped' then ex:=args[2];x:=args[3];methods:=args[4]
    else RETURN(listtoratpoly(typecheck(3,args))) fi;
    if not type(pade2,procedure) then ERROR(`please load pade2`) fi;
    bigO:=nops(ex);
    for method in methods do
	s:=listtoseries('stamped',ex,x,method);
	userinfo(3,'gfun',`Trying the `,method,s);
	if type(s,series) then ord:=order(s); nbz:=op(2,s)
	else ord:=bigO-1; nbz:=0 fi;
	result:=pade2([1,s],x,ord,'easy');
	if nops(subs(0=NULL,map(coeffs,result,x)))<ord-nbz then
	    s:=series(s+result[1]/result[2],x,ord+3);
	    if s=0 or op(1,s)=O(1) then 
		userinfo(2,'gfun','The',method,'`seems to be`',result);
		RETURN([-result[1]/result[2],method])
	    fi
	fi
    od;
    FAIL
end: # listoratpoly

seriestoratpoly:=proc () # yes, it's stupid to convert it to a list now.
    if args[1]<>'stamped' then seriestoratpoly(typecheck(7,args))
    else listtoratpoly('stamped',
	    seriestolist('stamped',args[2],'ogf'),op(0,args[2]),args[3]) fi
end: # seriestoratpoly

# s2d/s2d
# Input: a series s, its variable x, a name y for the function y(x) whose
#        series it is.
# Output: a linear differential equation satisfied by y(x), if possible.
#         The order of the differential equation is bounded by the global
#         variable maxordereqn. The degree of the coefficients is bounded by
#         the global variable maxdegcoeff.
`s2d/s2d`:=proc (s, x, y)
local serie, i, bigO, lpol, j, check, maxord, nbz;
    if not type(pade2,procedure) then ERROR(`please load pade2`) fi;
    if op(nops(s)-1,s)<>O(1) then bigO:=op(nops(s),s)
    else bigO:=op(nops(s),s)-1 fi; nbz:=op(2,s);
    serie:=[1,s,seq(diff(s,x$i),i=1..minordereqn)];# diff has option remember
    maxord:=min(bigO-1,maxordereqn);
    for i from minordereqn to maxord do
	userinfo(4,'gfun',`Looking for differential equation of order`,i);
	lpol:=pade2(serie,x,min(bigO-i+1,(maxdegcoeff+1)*(i+2)),'easy');
	if lpol<>FAIL and nops(subs(0=NULL,map(coeffs,lpol,x)))<bigO-i+1-nbz
	then
	    check:=map(normal,series(convert(
		[seq(lpol[j]*serie[j],j=1..i+2)],`+`),x,bigO+2));
	    if check=0 or op(1,check)=O(1) then
		RETURN(convert([lpol[1],lpol[2]*y(x),
		    seq(lpol[j]*diff(y(x),x$(j-2)),j=3..i+2)],`+`)) fi
	fi;
	if i=maxord then RETURN(FAIL) fi;
	serie:=[op(serie),diff(serie[i+2],x)]
    od;
    FAIL
end: # `s2d/s2d`

# s2a/s2a
# Input: a series s, its variable x, a name y for the function y(x) whose
#        series it is.
# Output: a polynomial equation satisfied by y(x), if possible.
#         The degree in y of the equation is bounded by the global
#         variable maxdegeqn. The degree of the coefficients is bounded by
#         the global variable maxdegcoeff.
`s2a/s2a`:=proc (s, x, y)
local serie, i, bigO, lpol, check, j, nbz;
    if not type(pade2,procedure) then ERROR(`please load pade2`) fi;
    if op(nops(s)-1,s)<>O(1) then bigO:=op(nops(s),s)
    else bigO:=op(nops(s),s)-1 fi; nbz:=op(2,s);
    serie:=map(series,[seq(s^i,i=0..mindegeqn-1)],x,bigO+2); 
    for i from mindegeqn to maxdegeqn do
	userinfo(4,'gfun',`Looking for polynomial equation of degree`,i);
	serie:=[op(serie),series(serie[nops(serie)]*s,x,bigO+2)];
	lpol:=pade2(serie,x,min(bigO+1,(maxdegcoeff+1)*(i+1)),'easy');
	if lpol<>FAIL and nops(subs(0=NULL,map(coeffs,lpol,x)))<bigO+1-nbz then
	    lpol:=factors(convert([seq(lpol[j]*y^(j-1),j=1..i+1)],`+`))[2];
	    for j in lpol do
		check:=series(subs(y=serie[2],j[1]),x,bigO+2);
		if check=0 or op(1,check)=O(1) then
		    RETURN(subs(y=y(x),j[1])) fi
	    od
	fi
    od;
    FAIL
end: # `s2a/s2a`

listtorec:=proc()
local result, methods, method, u, n, s, unkn, expr;
    if args[1]<>'stamped' then RETURN(listtorec(typecheck(2,args))) fi;
    expr:=args[2];unkn:=args[3];methods:=args[4];
    u:=op(0,unkn);n:=op(unkn);
    for method in methods do
	s:=listtolist('stamped',expr,method);
	userinfo(3,'gfun',`Trying the `,method,s);
	result:=`l2r/l2r`(s,n,u);
	if result<>FAIL then
	    userinfo(2,'gfun','The',method,'`seems to satisfy`',result);
	    RETURN([result,method])
	fi
    od;
    FAIL
end: # listtorec

seriestorec:=proc ()
    if args[1]<>'stamped' then seriestorec(typecheck(6,args))
    else listtorec('stamped',
	    seriestolist('stamped',args[2],'ogf'),args[3],args[4]) fi
end: # seriestorec

`l2r/l2r`:=proc (l,n,u)
local unkncoef, solver, homog, i, j, k, p, sys,
degcoef, ordereqn, ldeg, zerocoeff, eqn, trueinds, dim, inds, sol;
    if type(l,list(rational))
	then solver:=op(ratsolvelin) else solver:=op(gensolvelin) fi;
    unkncoef:=array(0..maxordereqn+1,0..maxdegcoeff);
    for i from 0 to maxordereqn+1 do
	p[i]:=convert([seq(unkncoef[i,j]*n^j,j=0..maxdegcoeff)],`+`)
    od;
    eqn:=convert([seq(p[i]*'op'(n+i,l),i=1..maxordereqn+1),p[0]],`+`);
    sys:=[seq(subs(n=k-1,eqn),k=1..nops(l))];
    for homog from 0 to 1 do for degcoef from mindegcoeff+1 to maxdegcoeff+1 do
    for ordereqn from minordereqn to maxordereqn do
	if homog=0 then ldeg:=[0,degcoef$(ordereqn+1)]
	else ldeg:=[degcoef$(ordereqn+2)] fi;
	userinfo(2,'gfun','`Trying degree sequence`',map(x->x-1,ldeg));
	zerocoeff:=seq(seq(unkncoef[i,j]=0,j=op(i+1,ldeg)..maxdegcoeff),
	    i=0..ordereqn+1),seq(seq(unkncoef[i,j]=0,j=0..maxdegcoeff),
	    i=ordereqn+2..maxordereqn+1);
	eqn:={op(eval(subs(zerocoeff,[op(1..nops(l)-ordereqn,sys)])))}minus{0};
	if eqn={} then next fi;
	trueinds:=indets(eqn,unkncoef[anything,anything]);
	dim:=nops(trueinds);
	if nops(eqn)<dim+1 or dim=1 then next fi;
	inds:={seq(seq(unkncoef[i,j],j=0..op(i+1,ldeg)-1),i=0..ordereqn+1)};
	sol:=solver(eqn,trueinds);
	if sol = FAIL or type(sol,set(anything=0)) then next fi;
	# If all the constant coefficients are zero, something simpler should
	# have been found before.
	if subs(sol,{seq(unkncoef[i,0],i=0..ordereqn+1)} intersect trueinds)=
	{0} then next fi;
	RETURN(eval(subs({k=n,op(sol),seq(j=0,j=inds minus trueinds),
	    zerocoeff},{convert([seq(p[i]*u(n+i-1),i=1..ordereqn+1),p[0]],`+`),
	    seq(u(i-1)=op(i,l),i=1..ordereqn)})))
    od od od;
    FAIL
end: # `l2r/l2r`

inicond:=proc (s, eqn, y ,x)
local order, deq, i;
    deq:=select(has,indets(eqn,'diff(anything,identical(x))'),y(x));
    if deq={} then eqn
    else 
	for order while deq<>{y(x)} do deq:=subs(diff(y(x),x)=y(x),deq) od;
	{eqn,seq((D@@i)(y)(0)=coeff(s,x,i)*i!,i=0..order-2)}
    fi
end: # inicond

listtohypergeom:=proc()
local result, methods, method, s, unkn, expr;
    if args[1]<>'stamped' then RETURN(listtohypergeom(typecheck(3,args))) fi;
    expr:=args[2];unkn:=args[3];methods:=args[4];
    for method in methods do
	s:=listtolist('stamped',expr,method);
	userinfo(3,'gfun',`Trying the `,method,s);
	result:=`l2h/l2h`(s,unkn);
	if result<>FAIL then
	    userinfo(2,'gfun','The',method,'`seems to satisfy`',result);
	    RETURN([result,method])
	fi
    od;
    FAIL
end: # listtohypergeom

seriestohypergeom:=proc ()
    if args[1]<>'stamped' then seriestohypergeom(typecheck(7,args))
    else listtohypergeom('stamped',
	    seriestolist('stamped',args[2],'ogf'),op(0,args[2]),args[3]) fi
end: # seriestohypergeom

`l2h/l2h`:=proc (l, x)
local a, a0, k, eqn, u, v, w, den, i, z, c;
    a:=l;
    for k while op(1,a)=0 do a:=subsop(1=NULL,a) od;
    a0:=op(1,a);k:=k-1;
    if nops(a)<5 then RETURN(FAIL) fi;
    a:=[seq(op(i,a)/a0,i=2..nops(a))];
    eqn:=normal((6*a[4]*a[1]**2*a[2]+9*a[2]*a[3]**2+6*a[4]*a[1]*a[3]-6*
	a[3]**2*a[1]**2+a[2]**2*a[3]*a[1]-16*a[4]*a[2]**2)*x**2+(-32*a[4]
	*a[2]**2+5*a[2]**2*a[3]*a[1]+6*a[4]*a[1]*a[3]+18*a[4]*a[1]**2*
	a[2]+27*a[2]*a[3]**2-24*a[3]**2*a[1]**2)*x+6*a[2]**2*a[3]*a[1]-
	18*a[3]**2*a[1]**2+12*a[4]*a[1]**2*a[2]);
    if eqn=0 then v:=1
    elif degree(eqn,x)=0 then RETURN(FAIL)
    else v:=op(1,[solve(eqn,x)])
    fi;
    den:=normal(4*a[2]**2*v**2-3*a[3]*v**2*a[1]-a[2]*v**2*a[1]**2-3*a[2]*v*
	a[1]**2+8*a[2]**2*v-3*a[3]*v*a[1]-2*a[2]*a[1]**2);
    if den=0 then RETURN(FAIL) fi;
    w:=normal(-2*(2*a[2]**2*v+4*a[2]**2-3*a[3]*v*a[1]-3*a[3]*a[1])*v)/den;
    if type(w,negint) or w=0 then RETURN(FAIL) fi;
    z:=normal(-3*a[3]*v*a[1]**2+a[1]*a[2]**2*v+3*a[3]*v*a[2]-3*a[3]*
	a[1]**2+2*a[2]**2*a[1]);
    if z=0 then RETURN(FAIL) fi;
    u:=-normal(2*a[2]**2*v+4*a[2]**2-3*a[3]*v*a[1]-3*a[3]*a[1])*a[1]/z;
    z:=2*z/den;
    userinfo(3,'gfun',`candidate: hypergeom(`,[u,v],[w],z*x,`)`);
    c:=u*(u+1)*(u+2)*(u+3)*v*(v+1)*(v+2)*(v+3)/w/(w+1)/(w+2)/(w+3)*z^4/24;
    for i from 5 to nops(a) do
	c:=c*(u+i-1)*(v+i-1)*z/(w+i-1)/i;
	if c<>op(i,a) then RETURN(FAIL) fi;
    od;
    userinfo(2,'gfun',`hypergeom found, parameters:`,[u,v],[w],z*x);
    RETURN(simplify(a0*x^k*hypergeom([u,v],[w],z*x),hypergeom))
end: # `l2h/l2h`

#ratpolytocoeff
# Input: a rational function of x
#        x its variable
#        n a name
# Output: the nth coefficient of the Taylor expansion at the origin of f.
ratpolytocoeff:=proc(f,x,n)
local g;
    if not type(fullparfrac,procedure) then
	g:=traperror(readlib(fullparfrac));
	if not type(g,procedure) then
	ERROR(`This function requires the procedure fullparfrac from the share library`)
	fi
    fi;
    g:=fullparfrac(f,x);
    if type(g,`+`) then g:=[op(g)] else g:=[g] fi;
    convert(map(`ratpolytocoeff/elmt`,g,x,n),`+`)
end: # ratpolytocoeff

`ratpolytocoeff/elmt`:=proc(g,x,n)
local k, a, c, i;
    if type(g,function) then 
	op(0,g)(`ratpolytocoeff/elmt`(op(1,g),x,n),op(2..nops(g),g))
    elif type(g,polynom(anything,x)) then 0
    else
	# g must be c(a)*(x-a)^(-k)
	k:=select(has,indets(g,`^`),x);
	if nops(k)<>1 then ERROR(`report this as a bug`,g,x,n) fi;
	k:=op(k);
	a:=x-op(1,k);
	c:=g/k;
	k:=-op(2,k);
	c/(-a)^k*a^(-n)*convert([seq(n+i,i=1..k-1)],`*`)/(k-1)!
    fi
end: # `ratpolytocoeff/elmt`

guesseqn:=proc ()
local y, result, l, x, methods, ll, i;
    if args[1]<>'stamped' then RETURN(guesseqn(typecheck(2,args))) fi;
    l:=args[2];y:=op(0,args[3]);x:=op(args[3]);methods:=args[4];
    # First try to find a rational function
    userinfo(1,'gfun',`Trying to find a rational generating function`);
    for i in methods do
	ll[i]:=listtolist(l,i);
	result:=listtoratpoly('stamped',ll[i],x,['ogf']);
	if result=FAIL then next fi;
	RETURN([denom(result[1])*y(x)-numer(result[1]),i])
    od;
    # Then an algebraic equation
    userinfo(1,'gfun',`Trying to find an algebraic generating function`);
    for i in methods do
	result:=listtoalgeq('stamped',ll[i],y(x),['ogf']);
	if result<>FAIL then RETURN([result[1],i]) fi
    od;
    # Then a linear differential equation
    userinfo(1,'gfun',`Trying to find a linear differential equation`);
    for i in methods do
	result:=listtodiffeq('stamped',ll[i],y(x),['ogf']);
	if result<>FAIL then RETURN([result[1],i]) fi
    od;
    FAIL
end: # guesseqn

guessgf:=proc ()
local interres, y, result, l, x, methods, inds, s, i, ll, j, ord, sol, mini, tmp;
    if args[1]<>'stamped' then RETURN(guessgf(typecheck(3,args))) fi;
    l:=args[2];x:=args[3];methods:=args[4];
    # First try to find a rational function
    userinfo(1,'gfun',`Trying to find a rational generating function`);
    for i in methods do
	ll[i]:=listtolist(l,i);
	result:=listtoratpoly('stamped',ll[i],x,['ogf']);
	if result<>FAIL then RETURN([result[1],i]) fi
    od;
    # Then trap easy hypergeometrics
    userinfo(1,'gfun',`Trying to find an hypergeometric generating function`);
    for i in methods do
	result:=listtohypergeom('stamped',ll[i],x,['ogf']);
	if result<>FAIL then RETURN([result[1],i]) fi
    od;
    # Then algebraic functions
    userinfo(1,'gfun',`Trying to find an algebraic generating function`);
    for i in methods do
	result:=listtoalgeq('stamped',ll[i],y(x),['ogf']);
	if result=FAIL then next fi;
	userinfo(1,'gfun',`Trying to solve the equation`);
	sol:=[solve(result[1],y(x))];
	if nops(sol)=1 then RETURN([sol,i]) fi;
	ord:=0; mini:=0; s:=listtoseries(ll[i],x);
	for j to nops(sol) do
	    tmp:=series(sol-s,x,nops(ll[i]));
	    if tmp=0 then RETURN([sol[j],i])
	    elif type(tmp,series) and order(tmp)>ord then 
		ord:=order(tmp); mini:=j
	    fi
	od;
	if mini=0 then next fi;
	RETURN([sol[j],i])
    od;
    # Then a linear differential equation
    userinfo(1,'gfun',`Trying to find a linear differential equation`);
    for i in methods do
	interres:=listtodiffeq('stamped',ll[i],y(x),['ogf']);
	if interres=FAIL then next fi;
	userinfo(1,'gfun',`Trying to solve it`);
	result:=dsolve(op(1,interres),y(x));
	if result<>FAIL and result<>NULL then
	    inds:=(indets(op(2,result),name) minus indets(l,name))
		minus {x,constants};
	    if inds={} then	RETURN([op(2,result),op(2,interres)]) fi;
	    s:=series(op(2,result),x,nops(l)+1);
	    s:=solve({seq(coeff(s,x,j-1)-op(j,l),j=1..nops(l))},inds);
	    if s<>NULL and type(s,set) then 
		RETURN(subs(s,[op(2,result),op(2,interres)])) fi
	fi
    od;
    FAIL
end: # guessgf

####################### Power Series Reversion #######################

# p should be a series with no constant term and a non-zero linear term.
`gfun/powrevert` := proc (s::series,x,o)
local p, v, k, pv, ppv;
    v:=x/op(1,s);
    p:=convert(s,polynom);
    k:=1;
    while 2*k+1<=o do
	pv:=`gfun/powcompose`(p,v,x,2*k+1);
	ppv:=`gfun/pprimeknowingp`(pv,v,x,2*k+1);
	v:=v-`gfun/powdivide`(pv-x,ppv,x,2*k+1);
	k:=2*k+1
    od;
    if k<o then
	pv:=`gfun/powcompose`(p,v,x,o);
	ppv:=`gfun/pprimeknowingp`(pv,v,x,o);
	v:=v-`gfun/powdivide`(pv-x,ppv,x,o)
    fi;
    series(v+O(x^(o+1)),x,o+1)
end: # powrevert

# pol, pol -> pol
`gfun/powcompose`:=proc (Q,P,x,n) # this assumes P(0)=0.
local m, pm, pr, pr1, l, i, s, p, q;
    p:=`gfun/powtruncate`(P,x,n);
    q:=`gfun/powtruncate`(Q,x,n);
    if n<9 then
	s[0]:=coeff(Q,x,0);
	for i to degree(Q,x) do
	    s[i]:=collect(coeff(Q,x,i)*`gfun/powtruncate`(P,x,n-i+1)^i,x)
	od;
	`gfun/powtruncate`(convert([seq(s[i],i=0..degree(Q,x))],`+`),x,n)
    else
	m:=isqrt(trunc(3.32192809*n/length(n)));
	pm:=`gfun/powtruncate`(p,x,m);
	pr:=1;pr1:=p-pm;
	l:=`gfun/powcomposesimple`(q,pm,x,n);
	s[0]:=l;
	for i to iquo(n,m)+1 do
	    l:=`gfun/pprimeknowingp`(l,pm,x,n-i);
	    pr:=`gfun/powtruncate`(collect(pr*pr1,x),x,n);
	    s[i]:=collect(l*pr/i!,x)
	od;
	`gfun/powtruncate`(convert([seq(s[i],i=0..iquo(n,m)+1)],`+`),x,n)
    fi
end: # powcompose

# pol, pol -> pol
`gfun/powcomposesimple`:=proc (q,p,x,n)
local s, j, pk, i;
    j:=degree(q,x);
    s:=1; while s<j do s:=s*2 od;
    pk[1]:=`gfun/powtruncate`(p,x,n);
    i:=1; while i<s do
	pk[2*i]:=`gfun/powtruncate`(collect(pk[i]^2,x),x,n); i:=2*i od;
    `gfun/powcomposesimpledoit`(q,s,pk,x,n)
end: # powcomposesimple

# pol, pol -> pol
`gfun/powcomposesimpledoit`:=proc (q, s, pk, x, n)
local q1, q2;
    if s=8 then
	`gfun/powtruncate`(collect(coeff(q,x,0)+pk[1]*(coeff(q,x,1)+
	coeff(q,x,3)*pk[2]+pk[4]*(coeff(q,x,5)+coeff(q,x,7)*pk[2]))+
	pk[2]*(coeff(q,x,2)+coeff(q,x,6)*pk[4])+coeff(q,x,4)*pk[4]+
	coeff(q,x,8)*pk[8],x),x,n)
    elif s>8 then
	q1:=`gfun/powtruncate`(q,x,s/2-1);
	q2:=collect((q-q1)/x^(s/2),x);
	`gfun/powtruncate`(`gfun/powcomposesimpledoit`(q1, s/2, pk, x, n)+
	    collect(pk[s/2]*
		`gfun/powcomposesimpledoit`(q2, s/2, pk, x, n-s/2),x),x,n)
    elif s=4 then
	`gfun/powtruncate`(collect(coeff(q,x,0)+pk[1]*(coeff(q,x,1)+
	    coeff(q,x,3)*pk[2])+coeff(q,x,2)*pk[2]+coeff(q,x,4)*pk[4],x),x,n)
    elif s=2 then
	collect(coeff(q,x,0)+coeff(q,x,1)*pk[1]+coeff(q,x,2)*pk[2],x)
    else # ASSERTION s=1 
	subs(x=pk[1],q)
    fi
end: # powcomposesimpledoit

# pol, pol -> pol
`gfun/pprimeknowingp`:=proc (poff, f, x, n)
    `gfun/powdivide`(diff(poff,x),diff(f,x),x,n)
end: # pprimeknowingp

# pol, pol -> pol
`gfun/powdivide`:=proc (p, q, x, n)
local u, b, i, j;
    # assuming q[0]<>0, otherwise a shift of this version is all we need
    # This is slower that convert(series(p/q,x,n+1),polynom), but 
    # uses less memory.
    if subs(x=0,q)=0 then ERROR(`not implemented`) fi;
    for i from 0 to n do
	b[i]:=coeff(q,x,i);
	u[i]:=(coeff(p,x,i)-convert([seq(u[j]*b[i-j],j=0..i-1)],`+`))/b[0]
    od;
    convert([seq(u[i]*x^i,i=0..n)],`+`)
end: # powdivide

`gfun/powtruncate`:=proc (pol, x, n)
local i;
    if degree(pol,x)<=n then pol
    elif ldegree(pol,x)>n then 0
#   p:=0;
#   for i in pol do if degree(i,x)<=n then p:=p+i fi od;
#   p
    else convert([seq(coeff(pol,x,i)*x^i,i=0..n)],`+`)
    fi
end: # powtruncate

######################## Holonomic Functions #########################

#algeqtodiffeq
# Input:  a polynomial in two variables y and z
#         the unknown function y(z)
#         (optional) a set of initial conditions
# Output: a linear differential equation verified by RootOf(P,y)
#         or FAIL if the initial conditions cannot be satisfied
algeqtodiffeq := proc()
local y, z, P, inits, g, u, d, i, Y, deq, j, r, inity, P0;
    P:=formatpoleq([args],'y','z','inits');
    g:=mygcdex(diff(P,y),P,y,'u');
    if has(g,y) then RETURN(algeqtodiffeq(normal(P/g),y(z),inits)) fi;
    d := degree(P,y); userinfo(3,'gfun',`degree is `,d);
    if d<=1 then deq:=subs(y=y(z),P)
    elif not has(P,z) then deq:=y(z)-RootOf(P,y)
    else
	Y[1]:=rem(-u/g*diff(P,z),P,y);
	for i from 2 to d-1 do # compute Y[i] = diff(y,z$i) mod P
	    Y[i]:=rem(collect(diff(Y[i-1],z)+diff(Y[i-1],y)*Y[1],y),P,y) od;
	deq:=lindep(array(1..d+1,1..d,[[1,0$(d-1)],[0,1,0$(d-2)],
	    seq([seq(coeff(Y[i],y,j),j=0..d-1)],i=1..d-1)]),
	    [1,seq((D@@i)(y)(z),i=0..d-1)],z)
    fi;
    userinfo(1,'gfun',`differential equation is`,deq);
#    if inits={} then RETURN(deq) fi;
    P0:=subs(z=0,y=y(0),inits,P);
    if not has(P0,y) and P0<>0 then # this means that the origin 
				    # is a singular point
	if inits={} then RETURN(deq)
	else ERROR(`invalid initial conditions`) fi
    fi;
    if P0<>0 then 
#     This will be the correct way when Maple does not insist on the argument
#     of RootOf being irreducible.  
#	inits:=inits union {y(0)=RootOf(subs(y(0)=_Z,P0))} fi;
	if traperror(evala(RootOf(subs(y(0)=_Z,P0))))<>lasterror then
	    inits:=inits union {y(0)=RootOf(subs(y(0)=_Z,P0))} fi;
    fi;
    inity:=y=subs(inits,y(0));
    inits:={y(0)=op(2,inity)} union subs(y(0)=op(2,inity),inits);
    for i to d-1 do
	r:=traperror((D@@i)(y)(0)=subs([z=0,inity],Y[i]));
	if r<>lasterror then inits:=inits union {r} else break fi
    od;
    inits:=`goodinitvalues/diffeq`(formatdiffeq([deq,y(z)]),y,z,
	select(type,inits,anything=`gfun/free`(y(0))));
    if inits={} then RETURN(deq) else RETURN({deq,op(inits)}) fi
end: # algeqtodiffeq

#diffeqtorec
# Input:  eqn: differential equation (for example output of algeqtodiffeq)
#         y(z): its unknown function
#         u(n): the name of the sequence of Taylor coefficients at the origin
# Output: the linear recurrence satisfied by u(n)
#
diffeqtorec:=proc (eqn,yofz,uofk)
local iniconds, f, y, z, u, k;
    if nargs<>3 then ERROR(`wrong number of arguments`) fi;
    if not type(uofk,function(name)) then ERROR(`invalid argument`,uofk) fi;
    u:=op(0,uofk);k:=op(uofk);
    f:=formatdiffeq([eqn,yofz],'y','z','iniconds');
    `gfun/diffeqtorec/doit`(f,y,z,u,k,iniconds)
end: # diffeqtorec

`gfun/diffeqtorec/doit` := proc(R,y,z,u,k,iniconds)
local l, ini, i, rec, j, minordrec, maxordrec, m, r, dr1, inhdeg, inhpart, p;
    if has(R,[k,u]) then ERROR(`invalid arguments`) fi;
    # initial conditions
    l:={seq(op(2,op(0,op(0,i))),i=indets([iniconds,R],`gfun/initeq`(y))
	minus {y(0),D(y)(0)})};
    ini:=[y(0)=u(0),D(y)(0)=u(1),seq((D@@i)(y)(0)=u(i)*i!,i=l)];
    r:=subs(ini,R); ini:=subs(ini,iniconds);
    # In very special cases, this loop makes it possible to return
    # an inhomogeneous equation of lower order.
    # Ex: z*(-1+z)^3*(D@@2)(y)(z)+(-1+z)^3*D(y)(z)-(-1+z)^3*y(z)-z*(z-3)
    if r[1]<>0 then for inhdeg from 0 do
	p:=1-z;
	for i from 2 to nops(r) while degree(p,z)=1 do p:=gcd(p,r[i]) od;
	if not has(p,z) then break fi;
	r:=[r[1],seq(quo(r[i],1-z,z),i=2..nops(r))]
    od fi;
    # main loop
    minordrec:=min(seq(i-degree(op(i+2,r),z),i=0..nops(r)-2));
    maxordrec:=max(seq(i-ldegree(op(i+2,r),z),i=0..nops(r)-2));
    rec:=array(sparse,minordrec..maxordrec);
    for i from 2 to nops(r) do
	for j from ldegree(op(i,r),z) to degree(op(i,r),z) do
	    rec[i-2-j]:=rec[i-2-j]+coeff(op(i,r),z,j)*
		expand(convert([seq(k+m,m=1-j..i-2-j)],`*`)) od od;
    # inhomogeneous part of the differential equation
    if r[1]=0 then dr1:=-1; inhpart:=0
    else
	dr1:=degree(r[1],z);
	if inhdeg<>0 then
	    inhpart:=expand(convert([seq(coeff(r[1],z,i)*convert(
		[seq(k+minordrec+j,j=1-i..inhdeg-i-1)],`*`),i=0..dr1)],`+`))
		/(inhdeg-1)!;
	    r:=subsop(1=series(r[1]/(1-z)^inhdeg,z,
		max(dr1,maxordrec-minordrec)+1),r)
	else inhpart:=0
	fi
    fi;
    ini:={op(ini),op(map(convert,[seq(subs(k=i,[coeff(r[1],z,i),seq(
	rec[j]*u(i+j),j=max(minordrec,-i)..maxordrec)]),i=0..dr1)],`+`))};
    for i from dr1+1 while i<-minordrec or
	subs(k=i,{seq(rec[maxordrec-j],j=0..(i-dr1-1))})={0} do 
	ini:={op(ini),convert(subs(k=i,[seq(rec[j]*u(i+j),
	    j=max(minordrec,-i)..maxordrec),inhpart]),`+`)} od;
    rec:=listprimpart(
	subs(k=k-minordrec,[inhpart,seq(rec[i],i=minordrec..maxordrec)]),k);
    while nops(rec)>2 and rec[nops(rec)]=0 do
	rec:=subsop(nops(rec)=NULL,rec) od;
    ini:=select(type,`goodinitvalues/rec`(rec,u,k,ini),
	u(integer)=`gfun/free`(u));
    makerec(map(collect,rec,k),u,k,ini)
end: # `gfun/diffeqtorec/doit`

#rectoproc
# Input:  a recurrence and its unknown function.
#         (optional) 'remember'
# Output: a procedure computing values of the sequence.
#
#   If 'remember' is given, then the procedure uses option remember to
# compute values of the sequence in linear time and linear space.
# Otherwise, if the recurrence has constant coefficients, then binary
# powering is used to make the procedure logarithmic time/logarithmic space.
# If the coefficients are not constants, then the procedure is a simple loop
# that will compute the values in linear time/constant space.
#
# The variables that are reserved for the produced code are lowercase
# all the other variables are uppercase.
rectoproc := proc(expr,yofn)
local T,Y,N,R,INITS,ARGLIST,RECSTAT,A,U,I,REMBR,ORDER,N0,n,a,l,res,i, INIVECT,INIMAT, RES, RECLOOP, THRESHOLD;
    R:=formatrec([args[1..2]],'Y','N','INITS');
    REMBR:=(nargs>2) and (args[3]='remember');
    ORDER:=nops(R)-2;
    INITS:=`goodinitvalues/rec`(R,Y,N,INITS);
    if INITS={} then INITS:={seq(Y(I)=Y(I),I=0..ORDER-1)} fi;
    N0:=nops(INITS);
    R:=subs(N=N-ORDER,R);
if MAPLE5.4 then
    ARGLIST:=[n]
else
    ARGLIST:=[n::nonnegint]
fi;
    RECSTAT:=-convert(convert([R[1],seq(R[I+2]*U[I],I=0..ORDER-1)],`+`),
	horner,N)/convert(R[nops(R)],horner,N);
    if REMBR then 	######## Easy case: linear time/linear space###
	T:=subsop(4=NULL,readlib(procmake)(`&proc`(ARGLIST,[],['remember'],
	    subs([N=n,seq(U[I]=`&args`[-2](n-ORDER+I),I=0..ORDER-1)],
	    RECSTAT))))
    else
	RECLOOP:=`&statseq`(op(subs(INITS,
	    [seq(`&:=`(evaln(u.I),Y(N0-ORDER+I)),I=0..ORDER-1)])),
	    `&for`(i,N0,1,n-1,true,`&statseq`(
	        `&:=`(evaln(u.ORDER),
		    subs([N=i,seq(U[I]=evaln(u.I),I=0..ORDER-1)],RECSTAT)),
		seq(`&:=`(evaln(u.I),evaln(u.(I+1))),I=0..ORDER-1))),
	    subs([N=n,seq(U[I]=evaln(u.I),I=0..ORDER-1)],RECSTAT));
	if has(R,N) or R[1]<>0 then##linear time/constant space###########
	    T:=readlib(procmake)(`&proc`(ARGLIST,
		[i,seq(evaln(u.I),I=0..ORDER)],[],RECLOOP))
	else 		 ######## logarithmic time/constant space#######
	    THRESHOLD:=round(evalf(-2*ORDER^2/ln(2)*W(-1,-1/2*ln(2)/ORDER^2)));
	    INIMAT:=[[seq(-R[nops(R)-I]/R[nops(R)],I=1..ORDER)],
		    seq([0$(I-1),1,0$(ORDER-I)],I=1..ORDER-1)];
	    INIVECT:=subs(INITS,[seq(Y(N0-I),I=1..ORDER)]);
	    A:=array(1..ORDER,1..ORDER);RES:=array(1..ORDER);
	    T:=readlib(procmake)(`&proc`(ARGLIST,
		[a,res,i,l,seq(evaln(u.I),I=0..ORDER)],[],
		`&if`(n<=THRESHOLD,
		    RECLOOP,
		`&statseq`(
		    `&:=`(a,'array'(1..ORDER,1..ORDER,INIMAT)),
		    `&:=`(res,'array'(1..ORDER,INIVECT)),
		    `&:=`(l,'convert'(n-N0+1,base,2)),
		    `&if`(l[1]=1,
			`&:=`(res,'array'(1..ORDER,op(3,linalg['multiply'](
			    array(INIMAT),array(1..ORDER,INIVECT)))))),
		    `&if`(l=[1],
			res[1],
		    `&statseq`(
			`&for`(i,'subsop'(1=NULL,'nops'(l)=NULL,l),true,
			`&statseq`(
			    `&:=`(a,'array'(1..ORDER,1..ORDER,subs(A=a,
				op(3,linalg['multiply'](A,A))))),
			    `&if`(i=1,`&:=`(res,'array'(1..ORDER,op(3,subs(
				A=a,RES=res,linalg['multiply'](A,RES)))))))),
			subs(A=a,RES=res,linalg['multiply'](
			    linalg['multiply'](A,A),RES)[1])))))))
	fi
    fi;
    # put the initial conditions in the remember table
    for I in INITS do T(op(op(1,I))):=op(2,I) od;
    op(T)
end: # rectoproc

#rectodiffeq
# Input:  expr: a linear recurrence (with or without initial conditions)
#         a(n): its unknown function
#         f(t): the function sum(a(n)*t^n,n=0..infinity)
# Output: the linear differential equation satisfied by f(t).
#
rectodiffeq := proc(expr,aofn,foft)
local r, a, n, f, t, iniconds;
    if nargs<>3 then ERROR(`wrong number of arguments`) fi;
    if not type(foft,function(name)) then ERROR(`invalid argument`,foft) fi;
    f:=op(0,foft);t:=op(foft);
    r:=formatrec([expr,aofn],'a','n','iniconds');
    iniconds:=`goodinitvalues/rec`(r,a,n,iniconds);
    `gfun/rectodiffeq/doit`(r,a,n,f,t,iniconds)
end: # rectodiffeq

`gfun/rectodiffeq/doit`:=proc (r,u,n,f,z,iniconds)
local order, diffeq, P, k, p, a, rr, ini, i, k0, inds, res, l, c, aa, j;
    if has(r,[f,z]) then ERROR(`invalid arguments`) fi;
    order:=max(op(map(degree,r,n)));
    diffeq:=array(sparse,-1..order);
    k0:=nops(r)-2;
    D(f):=D(f);
    # To keep polynomial coefficients, multiply by z^(nops(r)-2)
    for k from 0 to k0 do
	P:=op(k+2,r);
	for p from 0 to degree(P,n) do
	    a:=subs(n=p-k,P);
	    P:=quo(P-a,n+k-p,n);
	    diffeq[p]:=diffeq[p]+a*z^(k0+p-k);
	    if p>0 then
		diffeq[-1]:=diffeq[-1]-collect(a*z^(k0+p-k)*diff(
		    convert([seq((D@@i)(f)(0)*z^i/i!,i=p..k-1)],`+`),z$p),z)
	    else
		diffeq[-1]:=diffeq[-1]-collect(a*z^(k0+p-k)*convert(
		[seq((D@@i)(f)(0)*z^i/i!,i=p..k-1)],`+`),z) fi od od;
    P:=op(1,r);
    for p from 0 to degree(P,n) do
	rr[p]:=subs(n=-p-1,P);
	P:=quo(P-rr[p],(n+p+1)/(p+1),n) od;
    # listprimpart must be taken here because otherwise the equation
    # may seem to be singular at the origin
    diffeq:=listprimpart([diffeq[-1]*(1-z)^p+z^k0*
	convert([seq(rr[k]*(1-z)^(p-k-1),k=0..p-1)],`+`),
	seq((1-z)^p*diffeq[k],k=0..order)],z);
    # initial conditions
    inds:=map(op,indets(iniconds,u(anything)));
    ini:=solve(subs([seq(u(i)=(D@@i)(f)(0)/i!,i=inds)],iniconds),
	{seq((D@@i)(f)(0),i=inds)});
    diffeq:=subs(ini,diffeq);
    # some initial conditions may correspond to polynomial inhomogeneities
    # at least when the equation is not singular at the origin.
#    if nops(ini)>order then
    inds:=max(op(inds));
    if subs(z=0,diffeq[nops(diffeq)])<>0 and inds<>-infinity and inds>=order
	then
	diffeq:=subsop(1=collect(diffeq[1],z)-convert(series(eval(subs(f(z)=
	    convert([seq((D@@i)(f)(0)*z^i/i!,i=0..inds),O(1)*z^(inds+1)],`+`)
	    ,ini,makediffeqdiff(diffeq,f,z))),z,infinity),polynom),diffeq)
    fi;
    # Case when the inhomogeneous part contains (D@@k)(f)(0)
    # Then make the equation homogeneous.
    if has(diffeq[1],f) then
	l:=select(has,indets(diffeq[1],function),f);
	if not type(diffeq[1],linear(l)) then
	    ERROR(`invalid inhomogeneous part`) fi;
	diffeq:=subs({seq(l[i]=aa[i],i=1..nops(l))},diffeq);
	l:=[seq(aa[i],i=1..nops(l))];
	diffeq:=subsop(1=collect(diffeq[1],l),diffeq);
	for i in l do
	    if has(diffeq[1],i) then
		c:=coeff(diffeq[1],i,1);
		res:=-diff(c,z);
		diffeq:=[res*diffeq[1]+c*diff(diffeq[1],z),
		         res*diffeq[2]+c*diff(diffeq[2],z),
			 seq(res*diffeq[j]+c*diff(diffeq[j],z)+
			    c*diffeq[j-1],j=3..nops(diffeq)),
			 c*diffeq[nops(diffeq)]];
		diffeq:=subsop(1=collect(diffeq[1],l,normal),diffeq)
	    fi
	od;
	if has(diffeq,aa) then ERROR(`some assertion was wrong`) fi;
	diffeq:=map(collect,diffeq,z)
    fi;
    diffeq:=listprimpart(diffeq,z);
    ini:=`goodinitvalues/diffeq`(diffeq,f,z,ini);
    res:=makediffeq(subs(ini,diffeq),f,z);
    if ini<>{} then {res,op(ini)} else res fi
end: # `gfun/rectodiffeq/doit`

# borel, invborel
# Input:  a linear recurrence or differential equation
#         u(n) or y(x) the variable
#         (optional) a flag 'diffeq' saying that it's a differential equation
#             by default it is a recurrence
# Output: the linear recurrence or differential equation in u(n) (or y(x))
#         satisfied by the sequence u(n)/n! in the borel case, u(n)*n! in the
#         invborel case. For differential equations, the equation is the
#         equation satisfied by the generating function of the borel/invborel
#         transform of the sequence of Taylor coefficients.
gfun[invborel]:=proc() `gfun/borelinvborel`(false,args) end:
gfun[Laplace]:=gfun[invborel]:
gfun[borel]:=   proc() `gfun/borelinvborel`(true, args) end:

`gfun/borelinvborel` := proc(borel,expr,aofn)
local a, n;
    if nargs<3 then ERROR(`Not enough arguments`) # not necessary in V.2
    elif nargs=3 then
	if not type(aofn,function(name)) then ERROR(`Invalid argument`,aofn)fi;
	a:=op(0,aofn); n:=op(aofn);
	if borel then `rec*rec`(expr,{n*a(n)=a(n-1),a(0)=1},a(n))
	else `rec*rec`(expr,{a(n)=n*a(n-1),a(0)=1},a(n)) fi
    elif args[4]<>'diffeq' then ERROR(`invalid argument`,args[4])
    else
	rectodiffeq(procname(borel,
	    diffeqtorec(expr,aofn,a(n)),a(n)),a(n),aofn)
    fi
end: # `gfun/borelinvborel`

# diffeq+diffeq
#Input: two differential equations Eq1 and Eq2 in the variable y(z)
#Output: a differential equation satisfied by the sum of a solution of Eq1 and
#        a solution of Eq2.
`diffeq+diffeq` := proc(Eq1,Eq2,yofz)
local y,z,A,d1,d2,d,i,j,f,g,eq1,eq2,C, ini1, ini2, eq, c, C2, ini, eeq1, eeq2;
    eeq1:=formatdiffeq([Eq1,yofz],'y','z','ini1'); d1:=nops(eeq1)-2;
    ini1:=remove(type,`goodinitvalues/diffeq`(eeq1,y,z,ini1),`gfun/identity`);
    eeq2:=formatdiffeq([Eq2,yofz],'y','z','ini2'); d2:=nops(eeq2)-2;
    ini2:=remove(type,`goodinitvalues/diffeq`(eeq2,y,z,ini2),`gfun/identity`);
    userinfo(2,gfun,`computing the sum of two holonomic functions of order`,
	d1,`and`,d2);
    d:=d1+d2; # maximal order of the sum
    if d=0 then
	eq:=normal(op(1,eeq1)/op(2,eeq1)+op(1,eeq2)/op(2,eeq2));
	eq:=denom(eq)*yofz+numer(eq)
    else
	A := array(1..d+1,1..d,sparse);
	C := array(1..d+1,sparse); # constant (rational) terms
	eq1:=[seq(-eeq1[i]/eeq1[d1+2],i=1..d1+1)];
	# column 1..d1: f .. (D@@(d1-1))(f)
	for i to d1 do A[i,i]:=1 od;
	if d1>0 then
	    for i from d1+1 to d+1 do
		c:=A[i-1,d1];
		C[i]:=diff(C[i-1],z)+c*eq1[1];
		A[i,1]:=diff(A[i-1,1],z)+c*eq1[2];
		for j from 2 to d1 do
		    A[i,j]:=diff(A[i-1,j],z)+A[i-1,j-1]+c*eq1[j+1] od od
	else C[1]:=eq1[1];
	    for i from 2 to d+1 do C[i]:=diff(C[i-1],z) od fi;
	eq2:=[seq(-eeq2[i]/eeq2[d2+2],i=1..d2+1)];
	# column d1+1..d1+d2: g .. (D@@(d2-1))(g)
	for i to d2 do A[i,d1+i]:=1 od;
	if d2>0 then
	    C2:=0;
	    for i from d2+1 to d+1 do 
		c:=A[i-1,d];
		C2:=diff(C2,z)+c*eq2[1];
		C[i]:=C[i]+C2;
		A[i,d1+1]:=diff(A[i-1,d1+1],z)+c*eq2[2];
		for j from 2 to d2 do
		    A[i,d1+j]:=diff(A[i-1,d1+j],z)+A[i-1,d1+j-1]+c*eq2[j+1] od
	    od
	else C2:=op(eq2); C[1]:=C[1]+C2;
	    for i from 2 to d+1 do C2:=diff(C2,z); C[i]:=C[i]+C2 od fi;
	eq:=lindep(A,[seq((D@@i)(y)(z)-C[i+1],i=0..d)],z);
    fi;
    # initial conditions
    ini1:=`goodinitvalues/diffeq`(eeq1,y,z,ini1,d-1);
    ini2:=`goodinitvalues/diffeq`(eeq2,y,z,ini2,d-1);
    ini:=`goodinitvalues/diffeq`(formatdiffeq([eq,y(z)]),y,z,
	remove(has,subs(subs(y=f,ini1) union subs(y=g,ini2),
	{seq(i=subs(y=f,i)+subs(y=g,i),
	    i={seq(op(1,i),i=ini1)} intersect {seq(op(1,i),i=ini2)})}),{f,g}));
    if ini={} then eq else {eq,op(ini)} fi
end: # `diffeq+diffeq`

# diffeq*diffeq
#Input: two differential equations Eq1 and Eq2 in the variable y(z)
#Output: a differential equation satisfied by the product of a solution of Eq1
#	 and a solution of Eq2.
`diffeq*diffeq` := proc(Eq1,Eq2,yofz)
local y, z, A,d1,d2,d,i,j,f,g,eeq1,eeq2,eq1,eq2, ini1, ini2, e1, eq, ini,k,Y;
    eeq1:=formatdiffeq([Eq1,yofz],'y','z','ini1'); d1:=nops(eeq1)-2;
    eeq2:=formatdiffeq([Eq2,yofz],'y','z','ini2'); d2:=nops(eeq2)-2;
    userinfo(2,gfun,`computing the product of two holonomic functions of order`
	,d1,`and`,d2);
    if d1=0 then
	eq:=makediffeq(formatdiffeq([numer(eval(subs(yofz=-Y(z)*op(2,eeq1)/
	    op(1,eeq1),makediffeqdiff(eeq2,y,z)))),Y(z)]),y,z);
	d:=d2
    elif d2=0 then
	eq:=makediffeq(formatdiffeq([numer(eval(subs(yofz=-Y(z)*op(2,eeq2)/
	    op(1,eeq2),makediffeqdiff(eeq1,y,z)))),Y(z)]),y,z);
	d:=d1
    else
	if op(1,eeq1)<>0 then d2 else 0 fi;
	if op(1,eeq2)<>0 then d1 else 0 fi;
	d:=d1*d2+"+""; # maximal order of the product
	A := array(1..d+1,1..d,sparse); 
	# (D@@i)(f)*(D@@j)(g) -> column i*d2+j+1, 0<=i<d1, 0<=j<d2
	eq1:=[seq(-eeq1[i]/eeq1[d1+2],i=1..d1+1)];
	eq2:=[seq(-eeq2[i]/eeq2[d2+2],i=1..d2+1)];
	A[1,1]:=1;
	for k from 2 to d+1 do
	    # fg          fg         f g^{(d2-1)}              f^{(d1-1)} g
	    A[k,1]:=diff(A[k-1,1],z)+A[k-1,d2]*eq2[2]+A[k-1,(d1-1)*d2+1]*eq1[2];
	    for j to d2-1 do 			# f g^{(j)}
		A[k,j+1]:=diff(A[k-1,j+1],z)		# f g^{(j)}
		    +A[k-1,j]				# f g^{(j-1)}
		    +A[k-1,d2]*eq2[j+2]			# f g^{(d2-1)}
		    +A[k-1,(d1-1)*d2+j+1]*eq1[2]   	# f^{(d1-1)}g^{(j)}
	    od;
	    for i to d1-1 do 			# f^{(i)} g
		A[k,i*d2+1]:=diff(A[k-1,i*d2+1],z)	# f^{(i)} g
		    +A[k-1,(i-1)*d2+1]			# f^{(i-1)} g
		    +A[k-1,(d1-1)*d2+1]*eq1[i+2]	# f^{(d1-1)} g
		    +A[k-1,(i+1)*d2]*eq2[2];		# f^{(i)} g^{(d2-1)}
		for j to d2-1 do 			# f^{(i)} g^{(j)}
		    A[k,i*d2+j+1]:=		# f^{(i)} g^{(j)}
			diff(A[k-1,i*d2+j+1],z)		# f^{(i)} g^{(j)}
			+A[k-1,(i-1)*d2+j+1]		# f^{(i-1)} g^{(j)}
			+A[k-1,i*d2+(j-1)+1]		# f^{(i)} g^{(j-1)}
			+A[k-1,(i+1)*d2]*eq2[j+2]	# f^{(i)} g^{(d2-1)}
			+A[k-1,(d1-1)*d2+j+1]*eq1[i+2];	# f^{(d1-1)} g^{(j)}
		od od od;
	if op(1,eq2)<>0 then # (D@@i)(f) -> column d1*d2+i+1, 0<=i<d1
	    e1:=d1;
	    for k from d2 to d+1 do 		# f
		A[k,d1*d2+1]:=
		    diff(A[k-1,d1*d2+1],z)		# f
		    +A[k-1,d2]*op(1,eq2)		# f g^{(d2-1)}
		    +A[k-1,(d1+1)*d2]*eq1[2];		# f^{(d1-1)}
		for i to d1-1 do 			# f^{(i)}
		    A[k,d1*d2+i+1]:=
			diff(A[k-1,d1*d2+i+1],z)	# f^{(i)}
			+A[k-1,d1*d2+i]			# f^{(i-1)}
			+A[k-1,(i+1)*d2]*eq2[1]		# f^{(i)} g^{(d2-1)}
			+A[k-1,(d1+1)*d2]*eq1[i+2] 	# f^{(d1-1)}
			    od od
	else e1:=0 fi;
	if op(1,eq1)<>0 then # (D@@i)(g) -> column d1*d2+e1+i+1, 0<=i<d2
	    for k from d1 to d+1 do
		A[k,d1*d2+e1+1]:=
		    diff(A[k-1,d1*d2+e1+1],z)		# g
		    +A[k-1,(d1-1)*d2+1]*eq1[1]		# f^{(d1-1)} g
		    +A[k-1,(d1+1)*d2+e1]*eq2[2];	# g^{(d2-1)}
		for i to d2-1 do
		    A[k,d1*d2+e1+i+1]:=
			diff(A[k-1,d1*d2+e1+i+1],z)	# g^{(i)}
			+A[k-1,d1*d2+e1+i]		# g^{(i-1)}
			+A[k-1,(d1-1)*d2+i+1]*eq1[1]	# f^{(d1-1)} g^{(i)}
			+A[k-1,(d1+1)*d2+e1]		# g^{(d2-1)}
			od od fi;
	eq:=lindep(A,[seq((D@@i)(y)(z),i=0..d)],z);
    fi;
    ini1:=`goodinitvalues/diffeq`(eeq1,y,z,ini1,d-1);
    ini2:=`goodinitvalues/diffeq`(eeq2,y,z,ini2,d-1);
    ini:=`goodinitvalues/diffeq`(formatdiffeq([eq,y(z)]),y,z,
	remove(has,
	    subs(subs(y=f,ini1) union subs(y=g,ini2),
		{seq((D@@i)(y)(0)=(D@@i)(f*g)(0),i=0..d-1)}),{f,g}));
    if ini={} then eq else {eq,op(ini)} fi
end: # `diffeq*diffeq`

# rec+rec
#Input: two linear recurrences rec1 and rec2 in the variable uofn (u(n))
#Output: a linear recurrence satisfied by the sum of a solution of rec1 and
#        a solution of rec2.
#  It used to be that way, but the equation this produces may not have
#  the minimal possible order.
# `rec+rec` := proc(rec1,rec2,uofn)
# local y, z;
#     diffeqtorec(`diffeq+diffeq`(
# 	rectodiffeq(rec1,uofn,y(z)),rectodiffeq(rec2,uofn,y(z)),y(z)),
# 	y(z),uofn)
# end: # `rec+rec`
`rec+rec`:=proc (rec1,rec2,uofn)
local u, n, eq1, d1, ini1, eq2, d2, ini2, d, eq, i, ini, lsubs, sys, f, g, n0, rec, ord, B, X, Y;
    eq1:=formatrec([rec1,uofn],'u','n','ini1'); d1:=nops(eq1)-2;
    eq2:=formatrec([rec2,uofn],'u','n','ini2'); d2:=nops(eq2)-2;
    userinfo(2,gfun,`computing the sum of two holonomic sequences of order`,
	d1,`and`,d2);
    d:=d1+d2; # maximal order of the sum
    if d=0 then
	eq:=normal(eq1[1]/eq1[2]+eq2[1]/eq2[2]);
	eq:=denom(eq)*uofn+numer(eq)
    else
	lsubs:=[n=n+1,seq(X[i]=X[i+1],i=0..d1-2),seq(Y[i]=Y[i+1],i=0..d2-2),
	    X[d1-1]=convert([-eq1[1]/eq1[d1+2],
		seq(-eq1[i]/eq1[d1+2]*X[i-2],i=2..d1+1)],`+`),
	    Y[d2-1]=convert([-eq2[1]/eq2[d2+2],
		seq(-eq2[i]/eq2[d2+2]*Y[i-2],i=2..d2+1)],`+`),
	    seq(Z[i]=Z[i+1],i=0..d)];
	sys[0]:=X[0]+Y[0]-Z[0];
	for i to d do sys[i]:=subs(lsubs,sys[i-1]) od;
	eq:=subs([seq(Z[i]=u(n+i),i=0..d)],
	    lindep(systomatrix([seq(sys[i],i=0..d)],
	    [seq(X[i],i=0..d1),seq(Y[i],i=0..d2)],'B'),convert(B,list),n))
    fi;
    # initial conditions
    rec:=formatrec([eq,uofn],'u','n');
    ord:=nops(rec)-2;
    n0:=max(ord-1,firstnonzero(subs(n=n-ord+1,rec[nops(rec)]),n));
    ini1:=remove(type,`goodinitvalues/rec`(eq1,u,n,ini1,n0),`gfun/identity`);
    ini2:=remove(type,`goodinitvalues/rec`(eq2,u,n,ini2,n0),`gfun/identity`);
    ini:=select(type,`goodinitvalues/rec`(rec,u,n,
	remove(has,subs(subs(u=f,ini1) union subs(u=g,ini2),
	{seq(i=subs(u=f,i)+subs(u=g,i),
	    i={seq(op(1,i),i=ini1)} intersect {seq(op(1,i),i=ini2)})}),{f,g})),
	    u(integer)=`gfun/free`(u));
    if ini={} then eq else {eq,op(ini)} fi
end: # `rec+rec`

# cauchyproduct
#Input: two linear recurrences rec1 and rec2 in the variable uofn (u(n))
#Output: a linear recurrence satisfied by \sum_{k=0}^n{u_kv_{n-k}}, where
# 		 u is a solution of rec1 and v is a solution of rec2.
#  I do not understand why this does not work:
# cauchyproduct := subs(`diffeq+diffeq`=`diffeq*diffeq`,`rec+rec`):
cauchyproduct := proc(rec1,rec2,uofn)
local y, z;
    diffeqtorec(`diffeq*diffeq`(
	rectodiffeq(rec1,uofn,y(z)),rectodiffeq(rec2,uofn,y(z)),y(z)),
	y(z),uofn)
end: # cauchyproduct

# rec*rec
#Input: two linear recurrences rec1 and rec2 in the variable uofn (u(n))
#Output: a linear recurrence satisfied by the product of a solution of
# 		rec1 by a solution of rec2.
`rec*rec` := proc(Eq1,Eq2,uofn)
local u, n, A,d1,d2,d,i,j,f,g,eq1,eq2,ini1, ini2,e1,eq1hom,eq2hom,maxinds1,maxinds2, eq,ini,k, w, eeq1, eeq2;
    eeq1:=formatrec([Eq1,uofn],'u','n','ini1'); d1:=nops(eeq1)-2;
    ini1:=`goodinitvalues/rec`(eeq1,u,n,ini1);
    eeq2:=formatrec([Eq2,uofn],'u','n','ini2'); d2:=nops(eeq2)-2;
    ini2:=`goodinitvalues/rec`(eeq2,u,n,ini2);
    userinfo(2,gfun,`computing the product of two holonomic sequences of order`
	,d1,`and`,d2);
    eq1hom:=evalb(op(1,eeq1)=0); eq2hom:=evalb(op(1,eeq2)=0);
    if eq1hom then 0 else d2 fi; if eq2hom then 0 else d1 fi;
    d:=d1*d2+"+""; # maximal order of the product
    if d=0 then
	eq:=op(2,eeq1)*op(2,eeq2)*u(n)-op(1,eeq1)*op(1,eeq2);
	eq1:=subs(n=n-1,[seq(-eeq1[i]/eeq1[d1+2],i=1..d1+1)]);
	eq2:=subs(n=n-1,[seq(-eeq2[i]/eeq2[d2+2],i=1..d2+1)]);
    elif d1=0 then
	w:=-op(2,eeq1)/op(1,eeq1);
	eq:=numer(normal(op(1,eeq2)+convert([seq(subs(n=n+i-1,w)*op(i+1,eeq2)*
	    u(n+i-1),i=1..d2+1)],`+`)));
	eq1:=subs(n=n-1,[seq(-eeq1[i]/eeq1[d1+2],i=1..d1+1)]);
	eq2:=subs(n=n-1,[seq(-eeq2[i]/eeq2[d2+2],i=1..d2+1)]);
    elif d2=0 then
	w:=-op(2,eeq2)/op(1,eeq2);
	eq:=numer(normal(op(1,eeq1)+convert([seq(subs(n=n+i-1,w)*op(i+1,eeq1)*
	    u(n+i-1),i=1..d1+1)],`+`)));
	eq1:=subs(n=n-1,[seq(-eeq1[i]/eeq1[d1+2],i=1..d1+1)]);
	eq2:=subs(n=n-1,[seq(-eeq2[i]/eeq2[d2+2],i=1..d2+1)]);
    else
    A := array(1..d+1,1..d,sparse); 
    eq1:=subs(n=n-1,[seq(-eeq1[i]/eeq1[d1+2],i=1..d1+1)]);
    eq2:=subs(n=n-1,[seq(-eeq2[i]/eeq2[d2+2],i=1..d2+1)]);
    # u(n+i)*v(n+j) -> column i*d2+j+1, 0<=i<d1, 0<=j<d2
    if not eq2hom then e1:=d1 else e1:=0 fi;
    A[1,1]:=1;
    for k from 2 to d+1 do
	#u_n v_n  u_{n+d1-1} v_{n+d2-1}
	A[k,1]:=A[k-1,d1*d2]*eq1[2]*eq2[2];
	for j to d2-1 do # u_n v_{n+j}
	    A[k,j+1]:=A[k-1,(d1-1)*d2+j]*eq1[2]	# u_{n+d1-1}v_{n+j-1}
		+A[k-1,d1*d2]*eq1[2]*eq2[j+2]	# u_{n+d1-1}v_{n+d2-1}
	od;
	for i to d1-1 do # u_{n+i} v_n
	    A[k,i*d2+1]:=A[k-1,i*d2]*eq2[2]	# u_{n+i-1} v_{n+d2-1}
		+A[k-1,d1*d2]*eq1[i+2]*eq2[2];	# u_{n+d1-1}v_{n+d2-1}
	    for j to d2-1 do # u_{n+i} v_{n+j}
		A[k,i*d2+j+1]:=A[k-1,(i-1)*d2+j]# u_{n+i-1} v_{n+j-1}
		    +A[k-1,i*d2]*eq2[j+2]	# u_{n+i-1} v_{n+d2-1}
		    +A[k-1,(d1-1)*d2+j]*eq1[i+2]# u_{n+d1-1}v_{n+j-1}
		    +A[k-1,d1*d2]*eq1[i+2]*eq2[j+2]# u_{n+d1-1}v_{n+d2-1}
	    od
	od;
	for i to d1*d2 do A[k,i]:=collect(subs(n=n+1,A[k,i]),n) od;
    od;
    if not eq2hom then # u_{n+i} -> column d1*d2+i+1, 0<=i<d1
	for k from d2 to d+1 do # u_n
	    A[k,d1*d2+1]:=
		A[k-1,d1*d2+1]*eq1[2]*eq2[1]	# u_{n+d1-1} v_{n+d2-1}
		+A[k-1,(d1+1)*d2]*eq1[2];	# u_{n+d1-1}
	    for i to d1-1 do # u_{n+i}
		A[k,d1*d2+i+1]:=A[k-1,d1*d2+i]	# u_{n+i-1}
		    +A[k-1,(d1+1)*d2]*eq1[i+2]	# u_{n+d1-1}
		    +A[k-1,i*d2]*eq2[1]		# u_{n+i-1} v_{n+d2-1}
		    +A[k-1,d1*d2]*eq1[i+2]*eq2[1]# u_{n+d1-1}v_{n+d2-1}
	    od;
	    for i from d1*d2+1 to d1*d2+d1 do
		A[k,i]:=collect(subs(n=n+1,A[k,i]),n) od;
	od;
    fi;
    if not eq1hom then # v_{n+i} -> column d1*d2+e1+i+1, 0<=i<d2
	for k from d1 to d+1 do # v_n
	    A[k,d1*d2+e1+1]:=
		A[k-1,(d1+1)*d2+e1]*eq2[2]	# v_{n+d2-1}
		+A[k-1,d1*d2]*eq1[1]*eq2[2];	# u_{n+d1-1} v_{n+d2-1}
	    for i to d2-1 do # v_{n+i}
		A[k,d1*d2+e1+i+1]:=
		    A[k-1,d1*d2+e1+i]		# v_{n+i-1}
		    +A[k-1,(d1-1)*d2+i]*eq1[1]	# u_{n+d1-1} v_{n+i-1}
		    +A[k-1,(d1+1)*d2+e1]*eq2[i+2]# v_{n+d2-1}
		    +A[k-1,d1*d2]*eq1[1]*eq2[i+2]# u_{n+d1-1} v_{n+d2-1}
	    od;
    	    for i from d1*d2+e1+1 to d1*d2+e1+d2 do
		A[k,i]:=collect(subs(n=n+1,A[k,i]),n) od;
	od
    fi;
    eq:=lindep(A,[seq(u(n+i),i=0..d)],n);
    fi;
    maxinds1:=max(op(map(op,indets(ini1,u(anything)))));
    maxinds2:=max(op(map(op,indets(ini2,u(anything)))));
    if d1=0 or (maxinds1<>-infinity and maxinds1>=d1-1) then
	ini1:=`goodinitvalues/rec`(eeq1,u,n,ini1,max(d-1,maxinds2)) fi;
    if d2=0 or (maxinds2<>-infinity and maxinds2>=d2-1) then
	ini2:=`goodinitvalues/rec`(eeq2,u,n,ini2,max(d-1,maxinds1)) fi;
    ini:=subs(subs(u=f,ini1) union subs(u=g,ini2),
	{seq(i=subs(u=f,i)*subs(u=g,i),
	i={seq(op(1,i),i=ini1)} intersect {seq(op(1,i),i=ini2)})});
    if ini={} then eq else {eq,op(ini)} fi
end: # `rec*rec`

# hadamardproduct
#Input: two linear differential equations eq1 and eq2 in the variable yofz
#Output: a linear differential equation satisfied by the Hadamard product
# of any solution of eq1 with any solution of eq2.
hadamardproduct := proc(eq1,eq2,yofz)
local u, n;
    rectodiffeq(`rec*rec`(diffeqtorec(eq1,yofz,u(n)),
	diffeqtorec(eq2,yofz,u(n)),u(n)),u(n),yofz)
end: # hadamardproduct

# algebraicsubs
#Input: a linear differential equation Deq in the variable yofz (which is y(z))
#       a polynomial eq in y and z
#Output: a linear differential equation satisfied by f(y(z)) for any solution
# f of Deq and y of eq.
algebraicsubs := proc(Deq,eq,yofz)
local y, z, deq, P, u, d, i, d1, k, A, C, Dg, g, j, ord_eqn, c, F, f, inhomog,
invtoto, toto, eqn;
    P:=formatpoleq([eq,yofz],'y','z'); d:=degree(P,y);
    deq:=subs(z=y,formatdiffeq([Deq,yofz])); d1:=nops(deq)-2;
    g:=mygcdex(diff(P,y),P,y,'u');
    if has(g,y) then RETURN(algebraicsubs(Deq,normal(P/g),yofz)) fi;
    Dg:=rem(-u/g*diff(P,z),P,y);
    g:=mygcdex(deq[d1+2],P,y,'u');
    if has(g,y) then RETURN(algebraicsubs(Deq,normal(P/g),yofz)) fi;
    deq:=map(rem,[seq(-deq[i]*u/g,i=1..d1+1)],P,y);
    inhomog:=evalb(op(1,deq)<>0);
    # if inhomog then ord_eqn:=d*d1 else ord_eqn:=d*(d1+1) fi;
    # Previous order was wrong. Fixed BS Dec 94.
    if inhomog then ord_eqn:=d*(d1+1) else ord_eqn:=d*d1 fi;
    deq:=convert([op(1,deq),seq(deq[i]*F^(i-1),i=2..nops(deq))],`+`);
    eqn[0]:=f@y;
    toto:=seq((D@@i)(f)@y=F^(i+1),i=0..d1-1),D(z)=1;
    invtoto:=seq(F^(d1-i)=(D@@(d1-i-1))(f)@y,i=0..d1-1);
    for k to ord_eqn do
	eqn[k]:=subs(invtoto,
	    rem(subs([(D@@d1)(f)@y=deq,D(y)=Dg,toto],D(eqn[k-1])),P,y)) od;
    for k from 0 to ord_eqn do eqn[k]:=collect(subs(toto,eqn[k]),[F,y]) od;
    A:=array(1..ord_eqn+1,1..ord_eqn,sparse);
    C:=array(1..ord_eqn+1,sparse);
    # f^{(i)}(g).g^j (i.e. F^i y^j) -> column j*d1+i+1, 0<=i<d1, 0<=j<d
    A[1,1]:=1;
    for k to ord_eqn+1 do for i from 0 to d1-1 do
	c:=coeff(eqn[k-1],F,i+1);
	for j from 0 to d-1 do A[k,j*d1+i+1]:=coeff(c,y,j) od od
    od;
    if inhomog then # g^j -> column d*d1+j, 0<j<d; g^0 -> C
	for k to ord_eqn+1 do
	    c:=coeff(eqn[k-1],F,0);
	    for j to d-1 do A[k,d*d1+j]:=coeff(c,y,j) od;
	    C[k]:=coeff(c,y,0)
	od
    fi;
    lindep(A,[seq((D@@i)(y)(z)-C[i+1],i=0..ord_eqn)],z)
end: # algebraicsubs


#save `gfun.m`;
#quit
