#
## <SHAREFILE=engineer/syrup/syrup.mpl >
## <DESCRIBE>
##                A package for doing symbolic circuit analysis.
##                Syrup is a symbolic circuit analysis tool, it reads an
##                electric circuit description written in SPICE notation,
##                generates the nodal equations, and returns the solutions.
##                It can perform ac, dc, and transient analyses, can handle
##                nested subcircuits, and permits the creation of non-standard
##                circuit elements.
##                AUTHOR: Joe Riel, jsr@sparc.SanDiegoCA.NCR.COM
## </DESCRIBE>
## <UPDATE=R4 >

# This is the complete source code for Syrup, with commments and appended
# help files (TEXT structures).  Reading this file into Maple will install
# Syrup in the library mylib.  If this is not what you want, modify the last
# lines in the file.
#
# This file was created with a tab length of 2.
#
# Copyright 1994 by Joseph Riel
# The source code and object code for Syrup are not intended for commercial
# distribution.  They may be freely used for any purpose and distributed by
# any means provided that no charge, other than handling costs, are made.
# The author shall not be liable for any loss or damage resulting from the
# use of this software.
#
# If you find any bugs, or have suggestions, criticisms, or comments about
# this software, please contact the author at one of the following e-mail
# addresses, preferably the first,
#
#    jsr@sparc.sandiegoca.ncr.com
#    Joseph.Riel@sandiegoca.ncr.com
#
#
# Revision History
# -------------------------
# Rev 1.0   -Initial Release
# Rev 1.01  -Minor change to syrup()
# Rev 1.02  -Minor change to syrup()
# Rev 1.03  -Added global varialbe `syrup/revision`, updated ?syrup/global.
#
# Rev 1.04  -Changed method for identifying element type.  The previous
#            method didn't work when assumptions were put on the global
#            variable of the same name as a type (the reason isn't clear).
#           -Added traperror statments when using sscanf for parsing element
#            lines, this gives useful syntax error messages.
#           -Changed some mislabeled infolevel() statements to userinfo().
#           -Changed _parse to print lines it is ignoring, depending on
#            value of infolevel[syrup].  Need to document infolevels!
#
# Rev 1.05  _convert: -Modified so m, M, and MEG would parse correctly.
#                     -Added traperror statements for syntax checking.
#
#           _parse:   -Added syntax check for number of fields in elements.
#                     -Added statement to unassign all element names,
#                      giving warning if infolevel[syrup] > 0; this was
#                      needed to permit assigned & assumed values in previous
#                      analyses from causing problems.
#                     -Added check for duplicate element names.
#
#           _flatten: -Added statement to copy Ks table from subcircuit to
#                      the calling circuit.
#
#           _couple:  -Changed simplify property from "symbolic" to
#                      "assume=positive".
#
#           syrup:    -Trap division by zero for resistor.
#
# Rev 2.00  general:  -Changed assigned global variables to "slash format".
#                     -Added semiconductor modeling capability.
#                     -Permitted references to nodes and currents in
#                      subcircuits.
#                     -Changed comments.
#
#           _convert: -Removed eval() when returning a Maple expression.
#
#           _parse:   -Changed `syrup/maxnodeargs` to a global variable.
#                     -Added "Vs" and "semiconductor" sets.
#                     -Added test for "0" as a subcircuit formal parameter.
#
#           _flatten: -Added chngvolt & chngcurr substitutions to expand.
#                      references to node voltages and element currents.
#                     -Added check for empty subcircuit.
#                     -Added check for matching nodes in subckt def. & call.
#
#           _expand:  -New function for expanding semiconductor definitions.
#-----------------------------------------------------------------------------
# CONTENTS
#
# Functions
# ---------
# syrup...........................Symbolically solves SPICE decks
#
# `syrup/unassign`................Unassigns created global variables
# `syrup/parse`...................Converts SPICE deck to table structure
# `syrup/parse/GetLineText`.......Gets next line of a SPICE TEXT structure
# `syrup/parse/GetLineTextInit`...Initializes _GetLineInit
# `syrup/parse/GetLineFile`.......Gets next line of a SPICE file
# `syrup/parse/GetLineFileInit`...Initializes _GetLineFile
# `syrup/parse/convert`...........Converts a SPICE value to a Maple value
# `syrup/flatten`.................Flattens table structure
# `syrup/flatten/rcopy`...........Recursively copies tables
# `syrup/expand`..................Expands semiconductor definitions
# `syrup/couple`..................Couples inductors
#
# Assigned Global Variables:
# --------------------------
# `syrup/SpiceDeck`...............Name of the SPICE deck
# `syrup/GetLine`.................Assigned either _GetLineText or _GetLineFile
# `syrup/LineCount`...............Used by _GetLineText
# `syrup/NextLine`................Used by _GetLineFile
# `syrup/maxnodeargs`.............Maximum node arguments in a subcircuit
# `syrup/revision`................Revision level of this release
#
# Unassigned Global Variables:
# ----------------------------
#  s..............................Complex frequency in an ac analysis
#  t..............................Time in a transient analysis
#  v.<node>.......................Voltage at a node
#  i.<element>....................Current through an element
#  <element>......................Name of an element
#
# Help Files:
# -----------
# ?syrup..........................Describes operation of Syrup
# ?syrup[syntax]..................Describes syntax of a Syrup/SPICE deck
# ?syrup[global]..................Describes global variables used by Syrup
# ?syrup[infolevel]...............Describes settings of syrup[infolevel]
# ?syrup[semiconductor]...........Describes semiconductor modeling with Syrup
#
#----------------------------------------------------------------------------

`syrup/revision` := 2.00:

macro(_parse           = `syrup/parse`,
			_GetLine         = `syrup/GetLine`,
			_GetLineText     = `syrup/parse/GetLineText`,
			_GetLineTextInit = `syrup/parse/GetLineTextInit`,
			_GetLineFile     = `syrup/parse/GetLineFile`,
			_GetLineFileInit = `syrup/parse/GetLineFileInit`,
			_flatten         = `syrup/flatten`,
      _rcopy           = `syrup/flatten/rcopy`,
      _expand          = `syrup/expand`,
      _couple          = `syrup/couple`,
      _unassign        = `syrup/unassign`,
      _convert         = `syrup/parse/convert`,
      _maxnodeargs     = `syrup/maxnodeargs`);

#----------------------------------------------------------------------------
# FUNCTION syrup
#
# syrup reads a SPICE deck which describes an electric circuit and,
# depending upon the type of analysis selected, generates, solves, and
# returns the solutions to the nodal equations that describe the circuit.
#-----------
# Input:
#   SpiceDeck...SPICE deck to be analyzed, may be a filename or TEXT structure.
#   analysis....The type of analysis to be performed.  There are three options:
#                      ac - small signal analysis
#                      dc - dc analysis
#                    tran - transient analysis
#   current.....(optional) an unassigned name.
#   voltage.....(optional) an unassigned name.
#  'symbolic'...(optional) flag indicating that numerical values in
#               the SPICE deck should be ignored.
# Output:
#   current.....A set of equations for the current through each element.
#   voltage.....A set of equations for the node voltages.  This is only used
#               for a transient analysis, since an ac or dc analysis return
#               this set.
#   return......Depends on the type of analysis selected.  An ac or dc analysis
#               returns a set of equations describing each node voltage.  A 
#               transient analysis returns a sequence, the first term is a set of
#               differential equations and initial conditions, the second term is
#               the set of variables for which the differential equations should be
#               solved.
# Side Effects:
#
#   A number of global variables are unassigned,
#
#   s.............Complex frequency in an ac analysis
#   t.............Time in a transient analysis
#   v.<node>......Voltage at <node>
#   i.<element>...Current through <element>
#
#  If infolevel[syrup] is unassigned then it will be set to 1.
#------------------
# Local Variables (the interesting ones):
#
#   Node_eq    Table containing the sum of the currents at each node
#   Curr_eqs   Set of equations giving the current through each element
#   Init_eqs   Set of equations giving the initial state of each L or C
#   Volt_eqs   Set of equations giving the voltages between pairs of nodes
#
#   Curr_vars  Set of variables representing the current through each
#              element.  They are not used to solve any equations, but
#              are for information.
#
#   State_vars Set of state variables in a transient analysis.  State
#              variables are the voltage across a capacitor and the current
#              through an inductor.
#
#   Aux_vars   Set of auxiliary variables for which solutions must be found.
#
#   Circuit    Table describing circuit
#                   
#   Pnode      Positive node of an element
#   Nnode      Negative node of an element
#   PCnode     Positive control node of a voltage controlled element
#   NCnode     Negative control node of a voltage controlled element
#   CCur       Control voltage source of a current controlled element
#------------
# Rev 1.0  - Initial Release
# Rev 1.01 - Deleted extraneous _unassign('v',Xnode) statements
# Rev 1.02 - Added _unassign('i',Lm) for ac analysis in syrup main body
# Rev 1.03 - Added `syrup/revision`, updated ?syrup[global]
# Rev 1.04 - Added seq(assign ...) to initialization.  Changed ``.<type>
#            to <type> in case statment.
# Rev 1.05 - Added traperror for division by zero with resistor element.
# Rev 2.00 - Removed multiplesols local variable.
#----------------------------------------------------------------------------

syrup := proc(SpiceDeck:: {name, TEXT},
							analysis:: {identical(ac), identical(dc), identical(tran)},
							current:: name,
							voltage:: name)

options `Copyright 1994 by Joseph Riel`;
local elem, element, elem_name,
			node, Pnode, Nnode, PCnode, NCnode, 
			Aux_vars, State_vars, Curr_vars, vars,
			Volt_eqs, Init_eqs, Curr_eqs, Node_eq, eqs,
			CCur, Velement, Ielement,
			dIdt, dVdt,
			sol, sols,
			m, Lm,
			i,v,
			C,D,E,F,G,H,I,L,R,V,
			Circuit,Symbolic;
global s,t,infolevel;

if nargs < 2 then ERROR(`must have at least two arguments`) fi;

if not assigned(infolevel[syrup]) then infolevel[syrup] := 1 fi;

Symbolic := member('symbolic', {args});

#------------------------------------#
# Parse, flatten, and couple circuit #
#------------------------------------#

Circuit := _parse(SpiceDeck, true, Symbolic);
_flatten(Circuit,true);
_expand(Circuit);
_couple(Circuit);

#----------------#
# Initialization #
#----------------#

Volt_eqs    := {};                     # initialize sets
Aux_vars    := {};
State_vars  := {};
Init_eqs    := {};
Curr_eqs    := {};
Curr_vars   := {};

readlib('unassign')('s','t');          # unassign global variables
for node in Circuit['nodes'] do
	_unassign('v',node);                 # unassign variable v.node	
	Node_eq[node] := 0                   # clear node equation
od;

v.0 := 0;                              # assign ground voltage

seq(assign(i,sscanf(i,`%c`)),i=[C,D,E,F,G,H,I,L,R,V]);

#--------------------#
# Generate Equations #
#--------------------#

if not assigned(Circuit['elements']) then ERROR(`no elements in circuit`) fi;
for elem in entries(Circuit['elements']) do
	element := op(elem);

  #--------------#
  # C: Capacitor #
  #--------------#

	if element['type'] = C then
		elem_name := element['name'];

		if analysis = 'ac' then
			Velement := v.elem_name;
			Ielement := element['value']*s*Velement;
			Aux_vars := Aux_vars union {Velement}
	
		elif analysis = 'dc' then
			next
	
		elif analysis = 'tran' then
			_unassign('v',elem_name,t);
			Velement := v.elem_name(t);
			dVdt := diff(Velement,t);
			Ielement := element['value']*dVdt;
			Aux_vars := Aux_vars union {dVdt};
			State_vars := State_vars union {Velement};
			Init_eqs := Init_eqs union {subs(t=0, Velement) = element['init']}
		fi;
	
		Pnode := element['nodes'][1];
		Nnode := element['nodes'][2];
		Node_eq[Pnode] := Node_eq[Pnode] + Ielement;
		Node_eq[Nnode] := Node_eq[Nnode] - Ielement;
		Volt_eqs := Volt_eqs union {v.Pnode - v.Nnode = Velement};
		_unassign('i', elem_name);
		Curr_eqs := Curr_eqs union {i.elem_name = Ielement}
		
	#--------------------------------------#
	# E: Voltage Controlled Voltage Source #
	#--------------------------------------#

	elif element['type'] = E then
		elem_name := element['name'];
		Pnode := element['nodes'][1];
		Nnode := element['nodes'][2];
		PCnode := element['nodes'][3];
		NCnode := element['nodes'][4];
		_unassign('i', elem_name);
		Ielement := i.elem_name;
		Velement := element['value']*(v.PCnode - v.NCnode);
		Node_eq[Pnode] := Node_eq[Pnode] + Ielement;
		Node_eq[Nnode] := Node_eq[Nnode] - Ielement;
		Volt_eqs := Volt_eqs union {v.Pnode - v.Nnode = Velement};
		Aux_vars := Aux_vars union {Ielement};
		Curr_vars := Curr_vars union {Ielement}

	#--------------------------------------#
	# F: Current Controlled Current Source #
	#--------------------------------------#

	elif element['type'] = F then
		elem_name := element['name'];
		Pnode := element['nodes'][1];
		Nnode := element['nodes'][2];
		CCur := element['control'];
		_unassign('i', CCur);
		Ielement := element['value']*i.CCur;
		Node_eq[Pnode] := Node_eq[Pnode] + Ielement;
		Node_eq[Nnode] := Node_eq[Nnode] - Ielement;
		Curr_eqs := Curr_eqs union {i.elem_name = Ielement}

	#--------------------------------------#
	# G: Voltage Controlled Current Source #
	#--------------------------------------#

	elif element['type'] = G then
		elem_name := element['name'];
		Pnode := element['nodes'][1];
		Nnode := element['nodes'][2];
		PCnode := element['nodes'][3];
		NCnode := element['nodes'][4];
		Ielement := element['value']*(v.PCnode - v.NCnode);
		Node_eq[Pnode] := Node_eq[Pnode] + Ielement;
		Node_eq[Nnode] := Node_eq[Nnode] - Ielement;
		_unassign('i', elem_name);
		Curr_eqs := Curr_eqs union {i.elem_name = Ielement}

	#--------------------------------------#
	# H: Current Controlled Voltage Source #
	#--------------------------------------#

	elif element['type'] = H then
		elem_name := element['name'];
		Pnode := element['nodes'][1];
		Nnode := element['nodes'][2];
		CCur := element['control'];
		_unassign('i', CCur);
		_unassign('i', elem_name);		
		Ielement := i.elem_name;
		Velement := element['value']*i.CCur;
		Node_eq[Pnode] := Node_eq[Pnode] + Ielement;
		Node_eq[Nnode] := Node_eq[Nnode] - Ielement;
		Volt_eqs := Volt_eqs union {v.Pnode - v.Nnode = Velement};
		Aux_vars := Aux_vars union {Ielement};
		Curr_vars := Curr_vars union {Ielement}

	#-------------------#
	# I: Current Source #
	#-------------------#

	elif element['type'] = I then
		elem_name := element['name'];
		Pnode := element['nodes'][1];
		Nnode := element['nodes'][2];
		Ielement := element['value'];
		Node_eq[Pnode] := Node_eq[Pnode] + Ielement;
		Node_eq[Nnode] := Node_eq[Nnode] - Ielement;
		_unassign('i', elem_name);
		Curr_eqs := Curr_eqs union {i.elem_name = Ielement}

	#-------------#
	# L: Inductor #
	#-------------#

	elif element['type'] = L then
		elem_name := element['name'];
	
		if analysis = 'ac' then
			_unassign('i', elem_name);
			Ielement := i.elem_name;
			Velement := element['value']*s*Ielement;
			if assigned(element['M']) then
				for m in op(2,eval(element['M'])) do
					Lm := lhs(m);
					_unassign('i',Lm);
					Velement := Velement + rhs(m)*s*i.Lm
				od
			fi;
			Aux_vars := Aux_vars union {Ielement}
	
		elif analysis = 'dc' then
			_unassign('i', elem_name);
			Ielement := i.elem_name;
			Velement := 0;
			Aux_vars := Aux_vars union {Ielement}
			
		elif analysis = 'tran' then
			_unassign('i', elem_name, t);
			Ielement := i.elem_name(t);
			dIdt := diff(Ielement, t);
			Velement := element['value'] * dIdt;
			if assigned(element['M']) then
				for m in op(2,eval(element['M'])) do
					Lm := lhs(m);
					_unassign('i',Lm,t);
					Velement := Velement + rhs(m)*diff(i.Lm(t),t)
				od
			fi;
			Aux_vars := Aux_vars union {dIdt};
			State_vars := State_vars union {Ielement};
			Init_eqs := Init_eqs union {subs(t=0,Ielement) = element['init']}
		fi;
		Pnode := element['nodes'][1];
		Nnode := element['nodes'][2];
		Node_eq[Pnode] := Node_eq[Pnode] + Ielement;
		Node_eq[Nnode] := Node_eq[Nnode] - Ielement;
		Volt_eqs := Volt_eqs union {Velement = v.Pnode - v.Nnode};
		Curr_vars := Curr_vars union {Ielement}

	#-------------#
	# R: Resistor #
	#-------------#

	elif element['type'] = R then
		elem_name := element['name'];
		Pnode := element['nodes'][1];
		Nnode := element['nodes'][2];
		Ielement := traperror((v.Pnode - v.Nnode)/element['value']);
		if Ielement = lasterror then
			ERROR(`problem with`,elem_name,lasterror) fi;
		Node_eq[Pnode] := Node_eq[Pnode] + Ielement;
		Node_eq[Nnode] := Node_eq[Nnode] - Ielement;
		_unassign('i', elem_name);
		Curr_eqs := Curr_eqs union {i.elem_name = Ielement}

	#-------------------#
	# V: Voltage Source #
	#-------------------#

	elif element['type'] = V then
		elem_name := element['name'];
		Pnode := element['nodes'][1];
		Nnode := element['nodes'][2];
		_unassign('i', elem_name);
		Ielement := i.elem_name;
		Node_eq[Pnode] := Node_eq[Pnode] + Ielement;
		Node_eq[Nnode] := Node_eq[Nnode] - Ielement;
		Volt_eqs := Volt_eqs union {v.Pnode - v.Nnode = element['value']};
		Aux_vars := Aux_vars union {Ielement};
		Curr_vars := Curr_vars union {Ielement}

	else
		userinfo(1, syrup, `Warning: ignoring element of unknown type`, element)
	fi
od;

#-----------------#
# Solve Equations #
#-----------------#

eqs  := {seq(Node_eq[node], node = Circuit['nodes'])} union Volt_eqs;
vars :=	{seq(v.node, node = Circuit['nodes'] minus {0})} union Aux_vars;
sols := traperror([solve(eqs,vars)]);

if sols = lasterror or sols = [] then
	userinfo(1,syrup,`Could not solve equations`);
	RETURN(eqs,vars) fi;
if evalb(1 < nops(sols)) then
	userinfo(1,syrup,`Multiple solutions to `, eqs, vars);
	RETURN(sols)
else
  sol := op(sols)
fi;

#--------------------#
# Transient Analysis #
#--------------------#

if analysis = 'tran' then
	if 2 < nargs and current <> 'symbolic' then
		current := select(has, sol, Curr_vars) union subs(sol, Curr_eqs);
		if 3 < nargs and voltage <> 'symbolic' then
  	  voltage := select((eq,vn)->has(lhs(eq),vn), sol,
				{seq(v.node, node=Circuit['nodes'])} )
		fi
  fi;        
  select(has, sol, diff) union Init_eqs, State_vars

#-------------------#
# AC or DC Analysis #
#-------------------#

elif analysis = 'dc' or analysis = 'ac' then
	if 2 < nargs and current <> 'symbolic' then
	  current := select(has, sol, Curr_vars) union subs(sol, Curr_eqs);
		if 3 < nargs and voltage <> 'symbolic' then
			voltage := select((eq,vn)->has(lhs(eq),vn), sol,
  		               {seq(v.node, node=Circuit['nodes'])})
		fi
  fi;
	select(has, sol, {seq(v.node, node = Circuit['nodes'] minus {0})})
fi

end:

#----------------------------------------------------------------------------
# FUNCTION `syrup/unassign` (aka _unassign)
#
# _unassign unassigns the global variable formed from its parameters.
# If there are only two parameters, "a" and "b", then the variable
# "ab" is unassigned, unless "ab" is "v0" (this prevents unassigning the
# ground node).  If there are three parameters, "a", "b", and "c",
# then the variable "ab(c)" is unassigned.
#
# If infolevel[syrup] >= 2 then _unassign prints a notification that
# it is unassigning the variable.

_unassign := proc(a::name, b::{name,integer}, c::{name})
options `Copyright 1994 by Joseph Riel`;
if nargs < 3 then
	if assigned(a.b) and a <> 'v' and b <> 0 then
		userinfo(2, syrup, `unassigning`, evaln(a.b));
		readlib('unassign')(evaln(a.b))
	fi
elif assigned(a.b(c)) then
		userinfo(2, syrup, `unassigning`, evaln(a.b(c)));
		readlib('unassign')(evaln(a.b(c)))
fi
end:

##########################################################################

_maxnodeargs := 10:

#---------------------------------------------------------------------------
# FUNCTION `syrup/parse` (aka _parse)
#
# _parse reads a SPICE deck and returns a table which describes the circuit.
# _parse calls itself recursively when it encounters a subcircuit definition
# in the SPICE deck, it returns when it encounters an end statement.
#-------------
# Input:
#   SpiceDeck....TEXT structure or the filename of the SPICE deck.
#   InitCall.....True indicates that this is the first call to _parse.
#   Symbolic.....True causes _parse to substitute an element's name for
#                a numeric value in the SPICE deck; elements with symbolic
#                values in the SPICE deck are not affected.  False causes
#                _parse to use the numeric values.
# Output:
#   return.......A table which is a hierarchical description of the SPICE deck.
#                The table has some or all of the following entries:
#
#                 title        Title of the SPICE deck.
#                 nodes        Set of the nodes in the circuit.
#                 elements     Table with entries for each circuit element.
#                 subcircuits  Table with the subcircuit descriptions.
#                 Ks           Table with the inductor coupling coefficients.
#                 Vs           Set of the names of the independent voltage sources.
#                 Xs           Set of the names of the subcircuit calls.
#
#                The index of each entry in the "elements" table is the name of the
#                element, the entry is a sub-table which describes that element;
#                the entries of the sub-table include:
#
#                 type         Single letter indicating the element type (R,C,...)
#                 name         Name of the element (same as the index of sub-table)
#                 nodes        Ordered list of the element's nodes.
#                 value        Value of the element.
#                 init         Initial value of the element, for L's and C's only.
#
#                The "subcircuits" table has the same format as the main table,
#                but without the "title entry".  The main table is thus 
#                hierarchically structured.
#
#                The "Ks" table has the following entries:
#
#                 windings     List of the names of the coupled inductors.
#                 value        Coupling coefficient.
# Side Effects:
#  `syrup/GetLine`...is assigned to a procedure.
#------------
# Rev 1.0  -Initial Release
#
# Rev 1.04 -Added seq(assign ...) to initialization.
#          -Changed ``.<type> to <type> in case statment.
#          -Added traperror statements for syntax checking.
#
# Rev 1.05 -Added statement to unassign element names, this 
#           prevents weird problems with assumed global variables.
#          -Added check for duplicate element names.
#          -Changed sscanf call for K element, inductors are now read
#           as strings rather than being parsed.
#
# Rev 2.00 -Added "Vs" and "semiconductor" entries in the circuit table.
#          -Added test for "0" as a subcircuit formal parameter.
#----------------------------------------------------------------------------

_parse := proc(SpiceDeck:: {name,TEXT}, InitCall:: boolean, Symbolic:: boolean)
options `Copyright 1994 by Joseph Riel`;
local line,circuit,elem,element,elem_type,ic,EndColon,i,
      C,D,E,F,G,H,I,J,K,L,M,Q,R,V,X;

global `syrup/GetLine`, `0`;

#-------------------------------------------------#
# Assign and initialize the _syrupGetLine function #
#-------------------------------------------------#

if InitCall then
	#EndColon := interface(endcolon);        # Save interface(endcolon)
	EndColon := parse();        # Save interface(endcolon)
	if type(SpiceDeck, name) then
		`syrup/GetLine` := _GetLineFile;
		_GetLineFileInit(SpiceDeck)
	else
		`syrup/GetLine` := _GetLineText;
		_GetLineTextInit(SpiceDeck)
	fi;
	circuit['title'] := _GetLine()          # Never used, but might as well
fi;
 
circuit['nodes'] := {};                   # Initialize sets
circuit['Vs'] := {};
circuit['Xs'] := {};
circuit['semiconductors'] := {};

seq(assign(i,sscanf(i,`%c`)),i=[C,D,E,F,G,H,I,J,K,L,M,Q,R,V,X]);

#-----------------------------------#
# process each line of [sub]circuit #
#-----------------------------------#

line := _GetLine();

while substring(line, 1..4) <> `.end`
  and substring(line, 1..4) <> `.END` do
	if sscanf(line, `%1[CDEFGHIJKLMQRVX.*]`) = [``] then
		userinfo(1,syrup,`ignoring line`, line);
		line := _GetLine();
		next
	fi;
	elem := 'elem';
	elem['name'] := op(sscanf(line, `%s`));
  if elem['name'] <> eval(elem['name']) then

# Cesar Farell, 95-10-10:
# 	Since `*` is assigned as a procedure in R4 (as in `*`(a,b) = a*b), then
#	skip this check for `*`.  Since `*` has a special meaning (a comment
#	line in the source), the check does not seem to be needed anyway.
#       Do the same for `.` (not assigned in R4, but perhaps in the future).

                if elem['name'] <> `*` and elem['name'] <> `.` then
		    userinfo(1, syrup, `unassigning`, elem['name']);
		    assign(elem['name'],elem['name'])
                fi;
	fi;
	if elem['name'] = `.subckt`
	or elem['name'] = `.SUBCKT` then
		element := traperror(sscanf(line, cat(`%s%s`,seq(`%a`,i=1.._maxnodeargs))));
		if element = lasterror then ERROR(cat(`problem parsing `,line)) fi;
		if has(element[3..nops(element)],[0]) then
			ERROR(`0 cannot be used as a subcircuit formal parameter`,line) fi;
		circuit['subcircuits'][element[2]] := eval(_parse(SpiceDeck,false,Symbolic));
		circuit['subcircuits'][element[2]]['nodeargs'] := element[3..nops(element)]
	else
		if assigned(circuit['elements'][elem['name']]) then
			ERROR(`Duplicate element name in`, line) fi;
    elem_type := sscanf(elem['name'],`%c`);

		# case statement

		#----------------#
		# C & L elements #
		#----------------#
	
		if has(elem_type, [C,L]) then
			element := traperror(sscanf(line, `%s %a %a %s %s %s %s`));
			if element = lasterror then ERROR(`Illegal node name`,line) fi;
			if nops(element) < 3 then
				ERROR(`Missing nodes`, line, `see ?syrup[syntax]`) fi;
			elem['type'] := elem_type;
			elem['nodes'] := element[2..3];
			if nops(element) > 3 then
				ic := cat(op(element[4..nops(element)]));
				if substring(ic,1..3)=`IC=` or substring(ic,1..3)=`ic=` then
					elem['value'] := elem['name'];
					elem['init'] := _convert(substring(ic,4..length(ic)));
					if Symbolic and type(elem['init'],numeric) then
						if elem_type = C then
							elem['init'] := cat(V,elem['name'],`0`)
						else
							elem['init'] := cat(I,elem['name'],`0`)
						fi
					fi
				else
					elem['value'] := _convert(element[4]);
					if Symbolic and type(elem['value'], numeric) then
						elem['value'] := elem['name']
					fi;
					if nops(element) > 4 then
						ic := cat(op(element[5..nops(element)]));
						if substring(ic,1..3)=`IC=` or substring(ic,1..3)=`ic=` then
							elem['init'] := _convert(substring(ic,4..length(ic)));
							if Symbolic and type(elem['init'],numeric) then
								if elem_type = C then
									elem['init'] := cat(V,elem['name'],`0`)
								else
									elem['init'] := cat(I,elem['name'],`0`)
								fi
							fi
						else
							elem['init'] := 0;
							userinfo(1, syrup, `invalid initial condition, using 0`, line)
						fi
					else
						elem['init'] := 0
					fi
				fi
			else
				elem['value'] := elem['name'];
				elem['init'] := 0
			fi;
			circuit['elements'][elem['name']] := eval(elem);
			circuit['nodes'] := circuit['nodes'] union {op(elem['nodes'])}
	
		#--------------------#
		# R, V, & I elements #
		#--------------------#
	
		elif has(elem_type, [R,V,I]) then
			element := traperror(sscanf(line, `%s %a %a %s`));
			if element = lasterror then ERROR(`Illegal node name`,line) fi;
			if nops(element) < 3 then
				ERROR(`Missing nodes`, line, `see ?syrup[syntax]`) fi;
			elem['type'] := elem_type;
			elem['nodes'] := element[2..3];
			if nops(element) > 3 then
				elem['value'] := _convert(element[4]);
				if Symbolic and type(elem['value'], numeric) then
					elem['value'] := elem['name']
				fi
			else
				elem['value'] := elem['name']
			fi;
			circuit['elements'][elem['name']] := eval(elem);
			circuit['nodes'] := circuit['nodes'] union {op(elem['nodes'])};
			if elem_type = V then
				circuit['Vs'] := circuit['Vs'] union {elem['name']} fi
	
		#----------------#
		# F & H elements #
		#----------------#
	
		elif has(elem_type, [F,H]) then
			element := traperror(sscanf(line, `%s %a %a %a %s`));
			if element = lasterror then
				ERROR(`Illegal node name or control term`,line) fi;
			if nops(element) < 4 then
				ERROR(`Missing nodes or control term`, line, `see ?syrup[syntax]`) fi;
			elem['type'] := elem_type;
			elem['nodes'] := element[2..3];
			elem['control'] := element[4];
			if nops(element) > 4 then
				elem['value'] := _convert(element[5]);
				if Symbolic and type(elem['value'],numeric) then
					elem['value'] := elem['name']
				fi
			else
				elem['value'] := 1
			fi;
			circuit['elements'][elem['name']] := eval(elem);
			circuit['nodes'] := circuit['nodes'] union {op(elem['nodes'])}
	
		#----------------#
		# E & G elements #
		#----------------#
	
		elif has(elem_type, [E,G]) then
			element := traperror(sscanf(line, `%s %a %a %a %a %s`));
			if element = lasterror then ERROR(`Illegal node name`,line) fi;
			if nops(element) < 5 then
				ERROR(`Missing nodes`, line, `see ?syrup[syntax]`) fi;
			elem['type'] := elem_type;
			elem['nodes'] := element[2..5];
			if nops(element) > 5 then
				elem['value'] := _convert(element[6]);
				if Symbolic and type(elem['value'],numeric) then
					elem['value'] := elem['name']
				fi
			else
				elem['value'] := 1
			fi;
			circuit['elements'][elem['name']] := eval(elem);
			circuit['nodes'] := circuit['nodes'] union {op(elem['nodes'])}
	
		#-----------#
		# K element #
		#-----------#
	
		elif elem_type = K then
			element := traperror(sscanf(line, `%s %s %s %s`));
			if element = lasterror then ERROR(`Syntax error in`,line) fi;
			if nops(element) < 3 then
				ERROR(`Missing inductors`, line, `see ?syrup[syntax]`) fi;
			elem['windings'] := [element[2],element[3]];
			if nops(element) > 3 then
				elem['value'] := _convert(element[4]);
				if Symbolic and type(elem['value'], numeric) then
					elem['value'] := elem['name']
				fi
			else
				elem['value'] := elem['name']
			fi;
			circuit['Ks'][elem['name']] := eval(elem)
	
		#-----------#
		# X element #
		#-----------#
	
		elif elem_type = X then
			element := traperror(sscanf(line, cat(`%s`,
			                  seq(`%a`,i=1.._maxnodeargs),`%a`)));
			if element = lasterror then
				ERROR(`Illegal node name or subckt`,line) fi;
			elem['nodes'] := element[2..nops(element)-1];
			elem['subckt'] := element[nops(element)];
			circuit['Xs'] := circuit['Xs'] union {elem['name']};
			circuit['elements'][elem['name']] := eval(elem);
			circuit['nodes'] := circuit['nodes'] union {op(elem['nodes'])}

		#-----------#
		# D element #
		#-----------#
	
		elif elem_type = D then
			element := traperror(sscanf(line, `%s %a %a %s`));
			if element = lasterror then ERROR(`Illegal node name`,line) fi;
			if nops(element) < 4 then
				ERROR(`Missing nodes or subcircuit name`, line, `see ?syrup[syntax]`) fi;
			elem['type'] := elem_type;
			elem['nodes'] := element[2..3];
			elem['subckt'] := element[4];
			circuit['elements'][elem['name']] := eval(elem);
			circuit['nodes'] := circuit['nodes'] union {op(elem['nodes'])};
			circuit['semiconductors'] := circuit['semiconductors']
			                             union {elem['name']};

		#----------------#
		# J,M,Q elements #
		#----------------#
	
		elif has(elem_type, [J,M,Q]) then
			element := traperror(sscanf(line, `%s %a %a %a %s`));
			if element = lasterror then ERROR(`Illegal node name`,line) fi;
			if nops(element) < 5 then
				ERROR(`Missing nodes or subcircuit name`, line, `see ?syrup[syntax]`) fi;
			elem['type'] := elem_type;
			elem['nodes'] := element[2..4];
			elem['subckt'] := element[5];
			circuit['elements'][elem['name']] := eval(elem);
			circuit['nodes'] := circuit['nodes'] union {op(elem['nodes'])};
			circuit['semiconductors'] := circuit['semiconductors']
			                             union {elem['name']};
		#---------#
		# Comment #
		#---------#
				
		elif elem_type = [`*`] then           # do nothing

		#---------#
		# Control #
		#---------#
				
		elif elem_type = [`.`] then
			userinfo(2,syrup,`ignoring control line`,line)

		#----------#
		# No match #
		#----------#
		
		else
			ERROR(`Should never be here`,line)
		fi
	fi;
	line := _GetLine()
od;

#------------------------------------------------#
# Reset interface(endcolon) to the proper value. #
#------------------------------------------------#

if InitCall then
  if EndColon then
		sscanf(`:`, `%a`)
	else
		sscanf(``, `%a`)
	fi
fi;
eval(circuit)
end:

#----------------------------------------------------------------------------
# FUNCTION `syrup/parse/GetLineText` (aka _GetLineText)
#
# _GetLineText returns the next line from the SPICE deck represented by the
# TEXT structure _syrupSpiceDeck.  Continued lines are returned as a single
# string.
#------------
# Input:
#  `syrup/SpiceDeck`....TEXT structure which corresponds to a SPICE deck.
#  `syrup/LineCount`....Integer pointing to the next string in _syrupSpiceDeck.
# Output:
#   return..............A string which is the next line in the SPICE deck given
#                       by the TEXT structure `syrup/SpiceDeck`.  
# Side Effects:
#  `syrup/LineCount`...is increased to point to the next string.
#------------
# Rev 1.0
#----------------------------------------------------------------------------

_GetLineText := proc()
options `Copyright 1994 by Joseph Riel`;
local line, MaxCount;
global `syrup/LineCount`, `syrup/SpiceDeck`;

  MaxCount := nops(`syrup/SpiceDeck`);
	if MaxCount < `syrup/LineCount` then
	  ERROR(`Out of text, check for missing ``.end`` in [sub]circuit`) fi;
	line := op(`syrup/LineCount`, `syrup/SpiceDeck`);
	`syrup/LineCount` := `syrup/LineCount`+1;
	if `syrup/LineCount` <= MaxCount
	and substring(op(`syrup/LineCount`, `syrup/SpiceDeck`), 1..1) = `+` then
		line := cat(line, substring(_GetLineText(`syrup/SpiceDeck`),2..100))
	fi;
	line
end:

#----------------------------------------------------------------------------
# FUNCTION `syrup/parse/GetLineTextInit` (aka _GetLineTextInit)
#
# _GetLineTextInit initializes the global variables used by _GetLineText.
#---------
# Input:
#   SpiceText...........TEXT structure representing a SPICE deck.
# Output:
#   return..............NULL
# Side Effects:
#  `syrup/LineCount`....is assigned 1.
#  `syrup/SpiceDeck`....is assigned the TEXT structure SpiceText.
#------------
# Rev 1.0
#----------------------------------------------------------------------------

_GetLineTextInit := proc(SpiceText:: TEXT)
options `Copyright 1994 by Joseph Riel`;
global `syrup/LineCount`, `syrup/SpiceDeck`;
	`syrup/SpiceDeck` := SpiceText;
	if nops(`syrup/SpiceDeck`) = 0 then ERROR(`empty TEXT structure`) fi;
	`syrup/LineCount` := 1;
	NULL
end:

#----------------------------------------------------------------------------
# FUNCTION `syrup/parse/GetLineFile` (aka _GetLineFile)
#
# _GetLineFile returns the next line from the file SpiceFile.  Continued
# lines are returned as a single string.
#---------------
# Input:
#    `syrup/SpiceDeck`...Name of file which has the SPICE deck.
#---------------
# Output:
#    return..............A string wich is the next line in the file _Filename.
#---------------
# Side Effects:
#   `syrup/NextLine`.....is assigned the string which is first read by the
#                        next call to _GetLineFile.
#------------
# Rev 1.0
#----------------------------------------------------------------------------

_GetLineFile := proc()
options `Copyright 1994 by Joseph Riel`;
local line;
global `syrup/NextLine`, `syrup/SpiceDeck`;
	if `syrup/NextLine` = 0 then
	  ERROR(`Out of text, check for missing ``.end`` in [sub]circuit`) fi;
	line := `syrup/NextLine`;
	`syrup/NextLine` := readline(`syrup/SpiceDeck`);
	if `syrup/NextLine` <> 0 and substring(`syrup/NextLine`, 1..1) = `+` then
		line := cat(line, substring(_GetLineFile(`syrup/SpiceDeck`),2..100))
	fi;
	line
end:

#----------------------------------------------------------------------------
# FUNCTION `syrup/parse/GetLineFileInit` (aka _GetLineFileInit)
#
# _GetLineFileInit initializes the global variables used by _GetLineFile.
#------------
# Input:
#   SpiceFile...........File which has the SPICE deck
# Output:
#   return..............NULL
# Side Effects:
#   `syrup/SpiceDeck`...is assigned the filename SpiceFile.
#   `syrup/NextLine`....is assigned the first line in `syrup/SpiceDeck`.
#------------
# Rev 1.0
#----------------------------------------------------------------------------

_GetLineFileInit := proc(SpiceFile:: name)
options `Copyright 1994 by Joseph Riel`;
global `syrup/NextLine`, `syrup/SpiceDeck`;
	`syrup/SpiceDeck` := SpiceFile;
	while readline(`syrup/SpiceDeck`) <> 0 do od;
	`syrup/NextLine` := readline(`syrup/SpiceDeck`);
	if `syrup/NextLine` = 0 then ERROR(`empty file`) fi;
	NULL
end:

#----------------------------------------------------------------------------
# FUNCTION `syrup/parse/convert` (aka _convert)
#
# _convert converts a string into a Maple expression.  The string is first
# parsed as a regular Maple expression, if there are no errors than the
# resulting Maple expression is returned.  If there is an error than
# the string is interpreted as a number in engineering notation.  Decimal
# numbers and numbers with an appended engineering suffix are converted to
# rationals.  Engineering notation uses a set of single letter prefixes to
# represent various powers of ten.  The prefixes and corresponding values
# are,
#
#   F,f (femto)  1e-15
#   P,p (pico)   1e-12
#   N,n (nano)   1e-9
#   U,u (micro)  1e-6
#   M,m (milli)  1e-3 [it's a pity that SPICE doesn't distinguish lower
#   K   (kilo    1e3   case; M for milli and MEG for mega are ridiculous]
#   MEG (mega)   1e6
#   G   (giga)   1e9  [ed. note: giga is pronounced jiga, with a soft g]
#   T   (tera)   1e12
#------------
# Input:
#    s.........string to be parsed
# Output:
#    return....A Maple expression
#------------
# Rev 1.0  - Initial Release
#
# Rev 1.04 - Added seq(assign ...).  This seems questionable, but it works.
#          - Changed ``.<type> to <type> in case statment.
#          - Added traperror statement for syntax checking.
#
# Rev 1.05 - Fixed problem with m,M, and MEG suffixes.
#          - Modified seq(assign ...), added op() statement
#          - Changed "else ps" to "else eval(ps)".  This needs to be tested
#            thoroughly.
#
# Rev 2.00 - Changed "else eval(ps)" back to "else ps".  This was necessary
#            to prevent references to node voltages and elements currents
#            which were assigned in the calling Maple session from being
#            evaluated.
#----------------------------------------------------------------------------

_convert := proc(s::string)
options `Copyright 1994 by Joseph Riel`;
local ch,ps,ex,i,
      f,F,p,P,u,U,m,M,K,MEG,G,T;
	ps := traperror(parse(s));
	if ps <> lasterror then
		if type(ps,float) then
			convert(ps,rational)
		else
			ps
		fi
	else
		seq(assign(i,op(sscanf(i,`%s`))),i=[f,F,p,P,u,U,m,M,K,MEG,G,T]);
		ps := traperror(sscanf(s,`%e %s`));
		if ps = lasterror or nops(ps) < 2 then ERROR(`Problem converting`, s) fi;
		ch := op(sscanf(ps[2],`%c`));
		if ch = f or ch = F then
			ex := 10^(-15)
		elif ch = p or ch = P then
			ex := 10^(-12)
		elif ch = u or ch = U then
			ex := 10^(-6)
		elif ch = m or op(sscanf(ps[2],`%3s`)) = M then
			ex := 10^(-3)
		elif ch = K then
			ex := 10^3
		elif op(sscanf(ps[2],`%3s`)) = MEG then
			ex := 10^6
		elif ch = G then
			ex := 10^9
		elif ch = T then
			ex := 10^12
		else
			ex := 1
		fi;
		convert(ps[1]*ex,rational)
	fi
end:

#----------------------------------------------------------------------------
# FUNCTION `syrup/flatten` (aka _flatten)
#
# _flatten "flattens" the hierarchical circuit generated by _parse.
# Subcircuit calls are expanded using the subcircuit definition, the
# names of elements in the subcircuits are prefaced with the name of the
# subcircuit call separated by a dot; for example, an element R1 in a
# subcircuit called by element X1 will expand to X1.R1.  Subcircuit
# nodes are similarly prefaced.  Elements in the subcircuit which reference
# other elements, such as F, K, and H, are updated.  Element values which
# reference local nodes or element currents are updated.
#
#  _flatten is called recursively so that when it completes there is one
# level of hierarchy, all subcircuits definitions and subcircuit calls are
# removed.
#------------
# Input:
#   circuit.....A table describing a SPICE deck.  See _parse for table entries.
#------------
# Output:
#   circuit.....is modified.
#   return......NULL.
#------------
# Ver 1.0
# Rev 1.04 -Added seq(assign ...) to initialization.
#          -Changed ``.<type> to <type> in if statement.
# Rev 1.05 -Added statements to copy K elements.
# Rev 2.00 -Moved userinfo statement into for loop.
#          -Added lines to rename and move Vs & semiconductor sets.
#          -Added chngcurr and chngvolt sets which are used to rename
#           element current and node voltage references in value statements.
#          -Added check for empty subcircuit.
#          -Added check for matching nodes in subcircuit definition and call.
#----------------------------------------------------------------------------

_flatten := proc(circuit:: table, InitCall:: boolean)
options `Copyright 1994 by Joseph Riel`;
local subcirc, x, xname, elem, 
			chng, changnodes, chngcurr, chngvolt, k, oldname, i,
			F,H;
	# check for subcircuits
	if circuit['Xs'] = {}	then RETURN(circuit) fi;

	# Assign "safe" values for table indices
	seq(assign(i,sscanf(i,`%c`)),i=[F,H]);
	for xname in circuit['Xs'] do
		userinfo(3, syrup, `flattening subcircuit`,xname);
		x := circuit['elements'][xname];
		subcirc := _rcopy(circuit['subcircuits'][x['subckt']]);
		if type(subcirc, indexed) then
		  ERROR(`Could not find subcircuit definition`, xname) fi;
		if not assigned(subcirc['elements']) then
			ERROR(`No elements in subcircuit`, x['subckt']) fi;

		# recursively flatten subcircuits
		if assigned(subcirc['Xs']) then
			_flatten(subcirc, false)
		fi;

		#	Prefix subcircuit nodes and names with the name of the subcircuit call
		if nops(subcirc['nodeargs']) <> nops(x['nodes']) then
			ERROR(`Different number of nodes in subcircuit call and definition`,
			      xname, x['subckt']) fi;
		chng := traperror(zip((x,y)->(x=y),subcirc['nodeargs'],x['nodes']));
		if chng = lasterror then ERROR(`Problem zipping`,
		                               eval(subcirc['nodeargs']),
		                               eval(x['nodes'])) fi;
		changnodes := subcirc['nodes'] minus {op(subcirc['nodeargs'])} minus {0};
		chng := [op(chng), seq(i = cat(xname,`.`,i), i=changnodes)];
		chngcurr := {seq(cat('i',i)=cat('i',xname,`.`,i),i=subcirc['Vs'])};
		chngvolt := {seq(cat('v',lhs(i))=cat('v',rhs(i)),i=chng)};
		userinfo(4, syrup, `node substitutions`, chng);
		for elem in convert(subcirc['elements'],list) do
			elem['nodes'] := subs(chng,elem['nodes']);
			elem['value'] := subs(chngcurr,chngvolt,elem['value']);
			oldname := elem['name'];
			elem['name'] := cat(xname,`.`,elem['name']);
			if elem['type'] = F or elem['type'] = H then
				elem['control'] := cat(xname,`.`,elem['control'])
			fi;
			subcirc['elements'][elem['name']] := eval(elem);
			assign(subcirc['elements'][oldname],subcirc['elements'][oldname])
		od;

		# Prefix names of coupled inductors with the name of the subcircuit call
		if assigned(subcirc['Ks']) then
			for k in convert(subcirc['Ks'],list) do
				oldname := k['name'];
				k['name'] := cat(xname,`.`,k['name']);
				circuit['Ks'][k['name']]['windings'] :=
						[seq(cat(xname,`.`,i),i=k['windings'])];
				circuit['Ks'][k['name']]['value'] := k['value'];
				circuit['Ks'][k['name']]['name'] := k['name'];
				assign(subcirc['Ks'][oldname],subcirc['Ks'][oldname])
			od;
		fi;
		
		# Prefix Vs & semi. set & combine with circuit
		circuit['Vs'] := circuit['Vs']
		        union {seq(cat(xname,`.`,i), i=subcirc['Vs'])};
		circuit['semiconductors'] := circuit['semiconductors']
		        union {seq(cat(xname,`.`,i),i=subcirc['semiconductors'])};

		# Prefix nodes
		subcirc['nodes'] := subs(chng,subcirc['nodes']);
		subcirc['nodeargs'] := subs(chng,subcirc['nodeargs']);

		# copy subcircuit to circuit
		if assigned(circuit['elements']) then
			circuit['elements'] := convert(map(x->op(op(2,eval(x))),
			                        [circuit['elements'],subcirc['elements']]),
			                        table)
		else
			circuit['elements'] := _rcopy(subcirc['elements'])
		fi;
		circuit['nodes'] := circuit['nodes'] union subcirc['nodes'];

		# remove subcircuit call from 'elements' table
		assign(x,x);
	od;

	# remove circuit['Xs']
	circuit['Xs'] := evaln(circuit['Xs']);

	# remove subcircuits, except in main circuit
	if not InitCall then
		circuit['subcircuits'] := evaln(circuit['subcircuits']) fi;
	NULL
end:

#----------------------------------------------------------------------------
# FUNCTION `syrup/parse/rcopy` (aka _rcopy)
#
# _rcopy duplicates a table or array.  Unlike copy it acts recursively
# so that a table of tables of tables, etc., is duplicated.
#------------
# Input:
#   A........anything
# Output:
#   return...a duplicate of A.
#------------
# Rev 1.0
#----------------------------------------------------------------------------

_rcopy := proc(A::anything)
options `Copyright 1994 by Joseph Riel`;
	if type(A, {table,array}) then
		if type(A, name) then
			map(proc() _rcopy(args) end, eval(A))
		else
			map(proc() _rcopy(args) end, A)
		fi
	elif type(A, {list,set}) then
		map(_rcopy, A)
	else
		A
	fi
end:

#----------------------------------------------------------------------------
# FUNCTION `syrup/expand` (aka _expand)
#
# _expand "expands" a semiconductor subcircuit definition.  It is essentially
# the same as _flatten, except that it works on semiconductors elements,
# D, J, M, and Q, and model definitions rather than subcircuit calls and
# subcircuit definitons.
# 
# Semiconductors are expanded using a subcircuit definition, the names of
# elements in the subcircuits are prefaced with the name of the semiconductor
# separated by a dot; for example, an element R1 in a subcircuit called by
# element Q1 will expand to Q1.R1; nodes are similarly prefaced.  Elements in
# the subcircuit which reference other elements, such as F, K, and H, are
# updated.  Element values which reference local nodes or element currents
# are updated.
#
# Unlike _flatten, _expand is NOT called recursively.
#------------
# Input:
#   circuit.....A table describing a SPICE deck.  See _parse for table entries.
# Output:
#   circuit.....is modified.
#   return......NULL.
#------------
# Rev 2.00
#----------------------------------------------------------------------------

_expand := proc(circuit:: table)
options `Copyright 1994 by Joseph Riel`;
local subcirc, x, xname, elem,
      chng, changnodes, chngcurr, chngvolt,
      k, oldname, i,
			F,H;

	# check for subcircuits
	if circuit['semiconductors'] = {}	then RETURN(circuit) fi;

	# Assign "safe" values for table indices
	seq(assign(i,sscanf(i,`%c`)),i=[F,H]);

	for xname in circuit['semiconductors'] do
		userinfo(3, syrup, `expanding semiconductor`, xname);
		x := circuit['elements'][xname];
		subcirc := _rcopy(circuit['subcircuits'][x['subckt']]);
		if type(subcirc, indexed) then
		  ERROR(`Could not find subcircuit definition`, xname) fi;
		if not assigned(subcirc['elements']) then
			ERROR(`No elements in subcircuit`, x['subckt']) fi;		  

		#	Prefix subcircuit nodes and names and with the name of the semiconductor
		if nops(subcirc['nodeargs']) <> nops(x['nodes']) then
			ERROR(`Different number of nodes in subcircuit call and definition`,
			      xname, x['subckt']) fi;		
		chng := traperror(zip((x,y)->(x=y),subcirc['nodeargs'],x['nodes']));
		if chng = lasterror then ERROR(`Problem zipping`,
		                               eval(subcirc['nodeargs']),
		                               eval(x['nodes'])) fi;
		changnodes := subcirc['nodes'] minus {op(subcirc['nodeargs'])} minus {0};
		chng := [op(chng), seq(i = cat(xname,`.`,i), i=changnodes)];
		chngcurr := {seq(cat('i',i)=cat('i',xname,`.`,i),i=subcirc['Vs'])};
		chngvolt := {seq(cat('v',lhs(i))=cat('v',rhs(i)),i=chng)};
		for elem in convert(subcirc['elements'],list) do
			elem['nodes'] := subs(chng,elem['nodes']);
			elem['value'] := subs(chngcurr,chngvolt,elem['value']);
			oldname := elem['name'];
			elem['name'] := cat(xname,`.`,elem['name']);
			if elem['type'] = F or elem['type'] = H then
				elem['control'] := cat(xname,`.`,elem['control'])
			fi;
			subcirc['elements'][elem['name']] := eval(elem);
			assign(subcirc['elements'][oldname],subcirc['elements'][oldname])
		od;

		# Prefix names of coupled inductors with the name of the semiconductor
		if assigned(subcirc['Ks']) then
			for k in convert(subcirc['Ks'],list) do
				oldname := k['name'];
				k['name'] := cat(xname,`.`,k['name']);
				circuit['Ks'][k['name']]['windings'] :=
						[seq(cat(xname,`.`,i),i=k['windings'])];
				circuit['Ks'][k['name']]['value'] := k['value'];
				circuit['Ks'][k['name']]['name'] := k['name'];
				assign(subcirc['Ks'][oldname],subcirc['Ks'][oldname])
			od;
		fi;
		
		# Prefix nodes
		subcirc['nodes'] := subs(chng,subcirc['nodes']);
		subcirc['nodeargs'] := subs(chng,subcirc['nodeargs']);

		# copy subcircuit to circuit
		if assigned(circuit['elements']) then
			circuit['elements'] := convert(
						map(x->op(op(2,eval(x))),
			                        [circuit['elements'],subcirc['elements']]),
			                        table)
		else
			circuit['elements'] := _rcopy(subcirc['elements'])
		fi;
		circuit['nodes'] := circuit['nodes'] union subcirc['nodes'];

		# remove semiconductor from 'elements' table
		assign(x,x);
	od;

	# remove circuit['semiconductors']
	circuit['semiconductors'] := evaln(circuit['semiconductors']);

	# remove subcircuits
	circuit['subcircuits'] := evaln(circuit['subcircuits']);
	NULL
end:

#----------------------------------------------------------------------------
# FUNCTION `syrup/couple` (aka _couple)
#
# _couple adds entries to the element tables of coupled inductors.
# If a coupling element couples two inductors L1 and L2 with a coefficient k,
# then the mutual inductance is given by m = k*sqrt(L1*L2).  The element
# tables for L1 and L2 are modified, an "M" entry is added which is a
# sub-table with indices for each coupled inductor.
#------------
# Input:
#    circuit....A flattened and expanded circuit table.
# Output:
#    circuit....is modified.
#    return.....NULL
#------------
# Rev 1.0
# Rev 1.05 -changed:  m := simplify(...,'symbolic')
#                to:  m := simplify(...,assume=positive)
#----------------------------------------------------------------------------

`syrup/couple` := proc(circuit:: table)
options `Copyright 1994 by Joseph Riel`;
local k,m,L1,L2,w1,w2;
	if assigned(circuit['Ks']) and nops(circuit['Ks']) <> 0 then
		for k in convert(circuit['Ks'],list) do
			w1 := k['windings'][1];
			w2 := k['windings'][2];
			L1 := circuit['elements'][w1]['value'];
			L2 := circuit['elements'][w2]['value'];
			if type(L1, indexed) then
				ERROR(`Could not find value of inductor`, w1) fi;
			if type(L2, indexed) then
				ERROR(`Could not find value of inductor`, w2) fi;
			userinfo(3,syrup,`coupling inductors`, w1, w2);
			m  := simplify(sqrt(L1*L2)*k['value'], assume=positive);
			circuit['elements'][w1]['M'][w2] := m;
			circuit['elements'][w2]['M'][w1] := m
		od
	fi;
	circuit['Ks'] := evaln(circuit['Ks']);    # remove circuit entry 'Ks'
	NULL
end:



#save syrup,\
     _maxnodeargs,\
     _parse, _convert, _unassign,\
     _GetLineText, _GetLineTextInit, _GetLineFile, _GetLineFileInit,\
     _flatten, _rcopy,\
     _expand,\
     _couple,\
     `syrup/revision`,\
     `syrup.m`;

#quit
