#
## <SHAREFILE=numerics/daub/daub.mpl >
## <DESCRIBE>
## daub          File: daub/daub.map
##               Coefficients, values, graphs, moments, and (anti)derivatives
##               of Daubechies scaling functions/wavelets in Maple V.
##
## daubut        File: daub/daubut.map
##               Accuracy estimation and refinement of approximations for
##               the coefficients of Daubechies low pass filters.
##
## daubcc        File: daub/daubcc.map
##               Correct coefficients of Daubechies low pass filters.
##
## daubdd        File: daubdd.map
##               Find the 2p coefficients of the Daubechies Minium Phase filter
##               using the Deslauriers-Dubuc Lagrange interpolation method.
##
## daublh        File: daublh.map
##               Find the 2p coefficients of the Daubechies Minium Phase filter
##               using the original spectral method of Daubechies.
##
##               AUTHOR: Jacques Gelinas, gelinas@cmr.ca (until June 95)
## </DESCRIBE>
## <UPDATE=R4 >

# daub.mac  - macro definitions for all the global function names
#            in the packages
#
#    daub.map  daubut.map daubcc.map daubdd.map daublh.map
#
# This file is read by daub.map, daubut.map and daubdemo.map (via daub.map).
#
# Changing the macros here could be the simplest, safest way of renaming the
# functions for inclusion in a library.
#
# (Copyright) Jacques Gelinas, Ph.D., Maths, CMR, aout 1994, gelinas@cmr.ca

macro(Digitshook1=`daub/Digitshook1`);
macro(Digitshook2=`daub/Digitshook2`);
macro(Digitshook3=`daub/Digitshook3`);
macro(Dphid=`daub/Dphid`);
macro(Dphir=`daub/Dphir`);
macro(Dpsid=`daub/Dpsid`);
macro(Dpsir=`daub/Dpsir`);
macro(Ndyad=`daub/Ndyad`);
macro(Phid=`daub/Phid`);
macro(Phir=`daub/Phir`);
macro(Psid=`daub/Psid`);
macro(Psir=`daub/Psir`);
macro(c20=`daub/c20`);
macro(daubDr=`daub/daubDr`);
macro(daubDv=`daub/daubDv`);
macro(daubV=`daub/daubV`);
macro(daubVr=`daub/daubVr`);
macro(daubacc=`daub/daubacc`);
macro(daubc=`daub/daubc`);
macro(daubcc=`daub/daubcc`);
macro(daubcr=`daub/daubcr`);
macro(daubdd=`daub/daubdd`);
macro(daublh=`daub/daublh`);
macro(daubs=`daub/daubs`);
macro(daubsys=`daub/daubsys`);
macro(daubv=`daub/daubv`);
macro(daubvr=`daub/daubvr`);
macro(drefine=`daub/drefine`);
macro(dyad=`daub/dyad`);
macro(dyadlg=`daub/dyadlg`);
macro(mDphid=`daub/mDphid`);
macro(mDpsid=`daub/mDpsid`);
macro(mPhid=`daub/mPhid`);
macro(mPsid=`daub/mPsid`);
macro(mphid=`daub/mphid`);
macro(mphir=`daub/mphir`);
macro(mpsid=`daub/mpsid`);
macro(mpsir=`daub/mpsir`);
macro(newtons=`daub/newtons`);
macro(phi=`daub/phi`);
macro(phid=`daub/phid`);
macro(phid=`daub/phid`);
macro(phir=`daub/phir`);
macro(psid=`daub/psid`);
macro(psir=`daub/psir`);
macro(rich=`daub/rich`);
macro(rmb=`daub/rmb`);
macro(rmbi=`daub/rmbi`);
macro(simp=`daub/simp`);
macro(trap=`daub/trap`);
macro(vknown=`daub/vknown`);


daub:=`daub `:
# daub.map - Coefficients, values, graphs, moments, and (anti)derivatives
#            of Daubechies scaling functions/wavelets in Maple V (1.1).
#
# Usage:   > Digits:=20; read `daub.map`; daubc(2); daubv(2);
#          > daubg(2); daubG(2); puzzle(2);
#          > phid(2,2.1); Ndyad:=53; phid(2,2.1);
#            (1 minute on a 486/33/8Meg, DOS+Maple V 1.1)
#
# Version 0.1:  Mai  1994  phi, psi used in Galerkin method for Dy=-y^2,y(0)=1.
# Version 0.5:  Juin 1994  antiderivatives Phi, Psi
#               Juil 1994  daub.tex en francais
# Version 0.6:  Aout 1994  moments of phi, Phi, psi, Psi
# Version 0.7:  Aout 1994  derivatives Dphi, Dpsi, puzzle test
# Version 0.8:  Aout 1994  posted to sci.math.symbolic on 24 aug 94
#                          Bugs reported: Psi name conflict, globals in puzzle
#                                         accuracy of arithmetic in dyad
# Version 0.9:  Aout 1994  accuracy control added to functions using option
#                          remember: if Digits is increased, forget+recompute.
#                          Name change: phi->Phid, Phi->Phid.
# Version 1.0:  Sept 1994  Error control, moments of Dphid, help text.
#
# (Copyright)  Jacques Ge'linas, Coll`ege Militaire Royal de Saint-Jean,
#              Richelain, Quebec, Canada J0J 1R0, aout 1994, gelinas@cmr.ca.
#
# See also
#
#   daub.hlp   - Help text for this package.
#   daubcc.map - Correct coefficients of Daubechies low pass filters,
#                accuracy estimation and refinement of approximations.
#   daubdd.map - Find the 2p coefficients of the Daubechies Minium Phase filter
#                using the Deslauriers-Dubuc Lagrange interpolation method.
#   daublh.map - Find the 2p coefficients of the Daubechies Minium Phase filter
#                using the original spectral method of Daubechies.
#   daubut.map - Accuracy estimation and refinement of approximations for
#                the coefficients of Daubechies low pass filters.
#
# References:
#
#[1] I. Daubechies,  Orthonormal bases of compactly supported wavelets, Comm.
#                    Pure Appl. Math 41 (1988), p. 909-996. (Table on p. 980
#                    lists the coefficients to 12 decimals for p=2..10)
#
#[2] G. Strang, Wavelets and dilation equations. A brief introduction, SIAM
#               Rev., 31 (1989), pp 614-629. (Shows that filter coefficients
#               satisfy a nonlinear system of equations - with many solutions).
#
#[3] I. Daubechies,  Ten Lectures on Wavelets, SIAM, Philadelphia, PA, 1992.
#           (Table 6.1 p. 195 lists the coefficients to 16 decimals for p<11).
#
#[4] Wei-Chang Shann, Jann-Chang Yan, Quadratures involving polynomials and
#     Daubechies wavelets. Preprint, April 27, 1994, Technical Report 9301,
#     Math Dept, National Central Univ, Taiwan.  Anonymous ftp available at
#     dongpo.math.ncu.edu.tw:/pub/shann/publications/9301.ps
#
#    "Scaling equations are used to derive formulae of quadratures involving
#     polynomials and scaling/wavelet functions with compact supports; in
#     particular, those discovered by Daubechies.  It turns out that with a
#     few parameters, which are theoretically exact, these quadratures can be
#     evaluated with algebraic formulae instead of numerical approximations."
#
#    "For p=10 and m=9, the smallest and the largest number in that equation
#     is different by a scale of 10^8. Observe that the set of equations can
#     be replaced by $\sum_k (-1)^k (k-p)^m c_k = 0$  .. an observation
#     of Prof. I.L. Chern of National Taiwan University .. One can write down
#     the jacobian of $F_p$ and perform Newton's iteration to solve $F_p(x)=0$
#     (we use the roots of $F_{p-1}$ appended with two zeros as initial guess)."
#    "We can calculate the scaling coefficients up to p=14 .. in Table 1. All
#     values quoted hereafter are computed in quartic precision on a VAX 6510."
#

#read `numerics/daub/daubut.mpl`;
#read `daub.mac`;   # read by daubut.map
#read `daub.mac`;

print( `daubc(p) - Coefficients of the Minimum Phase Daubechies filter.` ):

# Use Newton's iteration to solve a nonlinear system for the 2p coefficients,
# starting from the 2p-2 values for the filter of order p-1, appended with
# two zeros (ref [4]). For 20 digits and p=4, the number of iterations is 8.
# The initial guess [0,1,1,0...] will also lead to Daubechies Minimum Phase
# filter; other choices give the Least Asymmetric filter or do not converge.

daubc  := p -> Digitshook1( daubcr, p ):
daubcr := proc( p )	            option remember;
        drefine( [ op(daubc(p-1)), 0, 0 ], Digits )
end:
daubcr(1) := [1.,1.]:

print( `daubv(p) - Values of the Daubechies scaling function at integers.` ):

# Values at integers of the scaling function $\phi$ are solutions of a linear
# system obtained from the scaling equation  $\phi(x) = \sum_k c_k \phi(2x-k)$.

daubv :=  p ->  Digitshook1( daubvr, p ):
daubvr := proc( p )         local i, k, v; option remember;   assign( fsolve(
    { seq( v[i] = 0, i = -2*p..min(0,2*p-3) ),
      seq( v[i] = sum( daubc(p)[k+1] * v[2*i - k], k=0..2*p-1 ), i=1..2*p-3 ),
      seq( v[i] = 0, i =  2*p-1..4*p ),
      sum( v[k], k=0..2*p-1 ) = 1         }                                ));
    [ seq( v[k], k=0..2*p-1 ) ]
end:
daubvr(0):=0:


# The global variable Ndyad limits the recursion levels in function evaluation.
# It limits the arguments accuracy, while Digits controls the values accuracy.
# The value of Ndyad should yield the same accuracy for arguments and values.
# Ndyad := 33:
Ndyad := round( evalf( Digits / log10(2), 10 ) ):

print( `Digits - Number of significant decimal digits, default `.Digits.`.` );
print( `Ndyad  - Number of significant binary digits, default `.Ndyad.`.` );

# This conversion function is fast, but imprecise. For example, the integers
# are not returned as is when Digits=10 and Ndyad=60.
dyadh := x -> trunc(evalhf( 2^Ndyad*x )) / 2^Ndyad :

# Convert real arguments to dyadic rationals k/2^j to get a finite recursion.
# Use a remember table to increase the precision for the modulo operation.

dyadlg := proc(N)  option remember;  round( evalf(N * log10(2), 10 ) ) end:

dyad  := proc(x) 
                                         Digits := Digits + dyadlg( Ndyad );
              round( 2^Ndyad * x );
                                         Digits := Digits - dyadlg( Ndyad );
               ""  / 2^Ndyad
end:

print( `phid(p,x) - Daubechies scaling function at dyadic points k/2^Ndyad.`);
print( `psid(p,x) - Daubechies wavelet at dyadic points k/2^Ndyad.` );

# This can be used as is in plot(phi(2,x),x=0..3) whereas one must write
#     plot('phid'(2,x),x=0..3) to avoid calls to phir with x non numeric.
phi  := proc( p, x )
        if hastype(p,name) or hastype(x,name) then RETURN( 'phi(p,x)' ) fi;
        phid(p,x)
end:
psi  := proc( p, x )
        if hastype(p,name) or hastype(x,name) then RETURN( 'psi(p,x)' ) fi;
        psid(p,x)
end:

# The scaling function phi(x) satisfies the dilation equation
# phi(x) = c0 phi(2x) + c1 phi(2x-1), for p=1,
# phi(x) = c0 phi(2x) + c1 phi(2x-1) + c2 phi(2x-2) + c3 phi(2x-3), for p=2.
# Values of phid at dyadic points are computed via the scaling equation,
# after the values at integers are computed (recomputed if Digits increases)

phid := ( p, x ) -> Digitshook3( phir, p, dyad( x ) ):
phir := proc( p, x )	   local xx, i, ia, ib; option remember;
        ib := 2*p-1;
	if x <= 0 or x >= ib then 0 else 
		xx := 2*x;
		ia := max( 0, trunc(xx) - ib );
		ib := min( trunc(xx), ib );
		sum( daubc(p)[i+1]*'phir'(p,xx-i), i=ia..ib )
	fi
end:
phir(0,0):=NULL:

# Reversing the coefficients and alternating the signs gives a wavelet ortho-
# gonal to the scaling function and with vanishing moments of order 0..p-1
# psi(x) = c1 phi(2x) - c0 phi(2x-1), for p=1,
# psi(x) = c3 phi(2x) - c1 phi(2x-1) + c2 phi(2x-2) - c0 phi(2x-3), for p=2.

psid := ( p, x ) -> Digitshook3( psir, p, dyad( x ) ):
psir := proc( p, x )                             local k; option remember;
        sum( (-1)^k * daubc(p)[2*p-k] * 'phir'(p,2*x-k), k=0..2*p-1 ) end:
psir(0,0):=NULL:


print( `Phid(p,x) - Antiderivative of phi(p,x).` );
print( `Psid(p,x) - Antiderivative of psi(p,x).` );

# Values of $Phi(x)=\int phi(x) dx$ at integers 0..2p-1 are computed
# from its scaling equation, where $d_k=c_k/2$ and for $p=2$,
#   Phi(x) = d0 Phi(2x) + d1 Phi(2x-1) + d2 Phi(2x-2) + d3 Phi(2x-3).

daubV := p -> Digitshook1( daubVr, p ):
daubVr := proc( p )         local i, k, v; option remember; assign( fsolve(
    {  seq( v[i] = 0, i = -2*p..0 ),
       seq( v[i] = sum( daubc(p)[k+1] * v[2*i-k]/2, k=0..2*p-1), i=1..2*p-2),
       seq( v[i] = 1, i =  2*p-1..4*p )  }                             ));
     [ seq( v[i], i=0..2*p-1 ) ]
end:
daubVr(0):=0:

# The values of Phid at dyadic points are computed via the scaling equation,
# and values of Psid are deduced from these via the definition of Psid.

Phid  :=    ( p, x ) -> Digitshook3( Phir, p, dyad( x ) ):
Psid  :=    ( p, x ) -> Digitshook3( Psir, p, dyad( x ) ):
Psir  := proc( p, x )                              local k; option remember;
      sum( (-1)^k * daubc(p)[2*p-k]  * 'Phir'(p,2*x-k) / 2, k=0..2*p-1 ) end:
Psir(0,0):=NULL:

Phir:=proc( p, x )	            local xx, i, j, ia, ib; option remember;
        ib := 2*p-1;
	if x <= 0  then  0  elif  x >= ib  then  1  else
		xx := 2*x;
		ia := max( 0, trunc(xx) - ib );
		ib := min( trunc(xx), ib );
		sum( daubc(p)[i+1] *       1        / 2, i=0..ia-1 ) +
		sum( daubc(p)[j+1] * 'Phir'(p,xx-j) / 2, j=ia..ib  )
	fi
end:
Phir(0,0):=NULL:


# The values of $ d phid(x)/dx$ at integers 0..2p-1 are computed
# from its scaling equation, where $d_k = 2 c_k$ and for $p=2$,
#  Dphid(x) = d0 Dphid(2x) + d1 Dphid(2x-1) + d2 Dphid(2x-2) + d3 Dphid(2x-3).
# Normalization via identity  $\sum_k (x-k)\phi(x-k) \equiv \int x \phi(x) dx$.
print( `Dphid(p,x) - Derivative of phi(p,x) at dyadic points.` );
print( `Dpsid(p,x) - Derivative of psi(p,x) at dyadic points.` );

daubDv := p -> Digitshook1( daubDr, p ):
daubDr := proc( p )         local i, j, k, v; option remember; assign( fsolve(
    {  seq( v[i] = 0, i = -2*p..0 ),
       seq( v[i] = sum( daubc(p)[k+1] * v[2*i-k]*2, k=0..2*p-1), i=1..2*p-3),
       seq( v[i] = 0, i =  2*p-1..4*p ),
       sum( v[j] * j, j = 0..2*p-1 ) = -1  }                               ));
     [ seq( v[i], i=0..2*p-1 ) ]
end:
daubDr(0):=0:

# The values of Dphid at dyadic points are computed via the scaling equation,
# and values of Dpsid are deduced from these via the definition of Dpsid.

Dphid  :=    ( p, x ) -> Digitshook3( Dphir, p, dyad( x ) ):
Dpsid  :=    ( p, x ) -> Digitshook3( Dpsir, p, dyad( x ) ):

Dpsir := proc( p, x )                              local k; option remember;
     sum( (-1)^k * daubc(p)[2*p-k] * 'Dphir'(p,2*x-k) * 2, k=0..2*p-1 ) end:
Dpsir(0,0):=NULL:

Dphir:=proc( p, x )	               local xx, i, ia, ib; option remember; 
        ib := 2*p-1;
	if x <= 0 or x >= ib then 0 else 
		xx := 2*x;
		ia := max( 0, trunc(xx) - ib );
		ib := min( trunc(xx), ib );
		sum( daubc(p)[i+1]*'Dphir'(p,xx-i)*2, i=ia..ib );
	fi
end:
Dphir(0,0):=NULL:


print( `mphid(p,m) - Moment of order m of phid(p,x).` );

# Moments of $\phi$ are computed recursively by transforming the integral
# of x^m times the rhs of the scaling equation. For $m<p$, a shortcut exists:
#  \sum (x-k)^m \phi(x-k) \equiv \int x^m \phi(x) dx, for 0 \le m \le p-1.

mphid := ( p, m ) -> Digitshook2( mphir, p, m ):
mphir:= proc( p, m )                         local k, j;  option remember;
 if m < 0 then ERROR( `Negative order `.m.` for moment of phid.` )
 elif m = 0 then 1 else sum( binomial( m, j ) * 'mphid'( p, m - j ) *
   sum( k^j * daubc(p)[k+1], k=1..2*p-1 ), j = 1..m ) / ( 2 * (2^m-1) ) fi
#elif m<p then sum( j^m * phid(j), j = 0..2*p-1 )
end:
mphir(1,0)=1:

print( `mpsid(p,m) - Moment of order m of psid(p,x).` );

# Moments of $\psi$ are computed from its definition and moments of $\phi$.

mpsid := ( p, m ) -> Digitshook2( mpsir, p, m ):
mpsir := proc( p, m )                         local k, j; option remember;
 if m < 0 then ERROR( `Negative order `.m.` for moment of psid.` )
 elif 0<= m and m < p then 0 else
 sum( binomial( m, j ) * 'mphid'( p, m - j ) * sum(
    (-1)^k * k^j * daubc(p)[2*p-k], k=1..2*p-1 ), j = p..m) / ( 2 * 2^m ) fi
end:
mpsir(0,0):=0:


print( `mPhid(p,m) - Moment of order m of Phid(p,x) over [0,2p-1].` );
print( `mPsid(p,m) - Moment of order m of Psid(p,x).` );

# Moments of the antiderivative are related to those of the function.

mPhid := (p,m) -> ( (2*p - 1)^(m + 1) - mphid(p, m + 1) ) / (m + 1):
mPsid := (p,m) -> (                   - mpsid(p, m + 1) ) / (m + 1):


print( `mDphid(p,m) - Moment of order m of Dphid(p,x).` );
print( `mDpsid(p,m) - Moment of order m of Dpsid(p,x).` );

# Moments of the derivative are related to those of the function.

mDphid := proc(p,m)  if m=0 then 0 else -m * mphid(p, m - 1) fi end:
mDpsid := proc(p,m)  if m=0 then 0 else -m * mpsid(p, m - 1) fi end:


# Auxiliary functions for accuracy control of remember tables.
readlib(forget):
rmb  := f -> op( 4, op(f) ):                                  # remember table
rmbi := f -> { indices( rmb(f) ) }:                           #    and indices
vknown := ( f, x ) -> member( x, rmbi(f) ):                   # is value known


# Accuracy control strategy for daubcr(p) (daubvr,daubVr, daubDvr):
# If unknown, store Digits in daubcr(-p) and solve a nonlinear system.
# If known to the same accuracy as Digits, return it.
# If known to less accuracy than Digits, refine it via Newton's method.
# If known to a greater accuracy than Digits, round it and return it,
#                               but do not change the remember table.
Digitshook1 := proc( fr, p )
    if not op(4,op(fr))[-p]=Digits then
        if vknown( fr, [-p] ) then
            if Digits < fr(-p) then
                RETURN( evalf( fr(p) ) )  
            elif fr=`daubcr` then 
                daubcr( p ) := drefine( daubcr( p ), Digits )
            else
                forget( fr, p )
            fi
        fi;
        fr(-p) := Digits
    fi;
    fr( p )
end:

# Accuracy control strategy for mphir(p,m) (mpsir):
# If unknown, store Digits in mphir(-p,m) and compute it by recurrence.
# If known to the same accuracy as Digits, return it.
# If known to less accuracy than Digits, forget and recompute it.
# If known to a greater accuracy than Digits, round it and return it,
#                               but do not change the remember table.
Digitshook2 := proc( fr, p, m )
    if not op(4,op(fr))[-p,m]=Digits then
        if vknown( fr, [-p,m] ) then
            if Digits < fr(-p,m) then
                RETURN( evalf( fr(p,m) ) )  
            else
                forget( fr, p, m )
            fi
        fi;
        fr(-p,m) := Digits
    fi;
    fr( p, m )
end:

# Accuracy control strategy for phir(p,x) (psir,Phir,Psir,Dphir,Dpsir):
# If unknown, store Digits in phir(-p) and recurse 2*x up to integers.
# If known to the same accuracy as Digits, return it.
# If known to less accuracy than Digits, forget [p,x] values for all x.
# If known to a greater accuracy than Digits, round it and return it,
#                               but do not change the remember table.
Digitshook3 := proc( fr, p, x )	   local j;
        if not op(4,op(fr))[-p,0]=Digits then
           if vknown( fr, [-p,0] ) then
               if Digits > fr(-p,0) then
                  for j in rmbi( fr ) do if j[1]=p then
                           forget( fr, op(j) ) fi od
               elif vknown( fr, [p,x] ) then
                  RETURN( evalf( fr(p,x) ) )
               fi
           fi;
           if   fr=psir  then  phid(p,x)
           elif fr=Psir  then  Phid(p,x)
           elif fr=Dpsir then  Dphid(p,x)
           fi;
           if  not op(4,op(fr))[p,0]=0   then
              if fr=phir then
                  for j to 2*p do fr(p,j-1) := daubv(p)[j] od
              elif fr=Phir then
                  for j to 2*p do fr(p,j-1) := daubV(p)[j] od
              elif fr=Dphir then
                  for j to 2*p do fr(p,j-1) := daubDv(p)[j] od
              fi
           fi;
           fr(-p,0) := Digits
        fi;
        fr( p, x )
end:

print(`Jacques Gelinas, Ph.D., Maths, CMR, aout 1994, gelinas@cmr.ca`);

# daubcc.map - Correct coefficients of Daubechies low pass filters.
#
# Usage:
#          daubcc( -1 );        Exact values of coefficients for p=1
#          daubcc( -2 );        Exact values of coefficients for p=2
#          daubcc( -3 );        Exact values of coefficients for p=3
#          daubcc( -4 );        Coefficients of Least Asymmetric filter (50d)
#          daubcc(-10 );        Coefficients of Least Asymmetric filter (12d)
#
#          daubcc(  2 );        Coefficients of Minimum Phase filter (50d)
#          daubcc(  3 );        Coefficients of Minimum Phase filter (50d)
#          daubcc(  8 );        Coefficients of Minimum Phase filter (30d)
#
# See also:
#
#   daubcc.hlp - Help text for this package.
#   daubdd.map - Find the 2p coefficients of the Daubechies Minium Phase filter
#                using the Deslauriers-Dubuc Lagrange interpolation method.
#   daublh.map - Find the 2p coefficients of the Daubechies Minium Phase filter
#                using the original spectral method of Daubechies.
#   daubc      - Find the 2p coefficients of the Daubechies Minium Phase filter
#                using Newton's method to solve a nonlinear system of equations.
#   daubut.map - Accuracy estimation and refinement of approximations for
#                the coefficients of Daubechies low pass filters.
#   Other references at end below.
#
# (Copyright) Jacques Gelinas, Ph.D., Maths, CMR, aout 1994, gelinas@cmr.ca

print(`daubcc(p) - Correct 2p coefficients of Daubechies Minimum Phase filter.`);
print(`daubcc(-p) - Correct 2p coefficients of Daubechies Least Asym. filter.`);

daubcc := proc( p )                                  local rt3, rt10, rt11;

if p = 1 then
    [ 1., 1. ]
elif p = -1 then
    [ 1, 1 ]
elif p = -2 then
    rt3 := sqrt(3);
    [ (1 + rt3)/4, (3 + rt3)/4, (3 - rt3)/4,  (1 - rt3)/4 ]
elif p = -3 then
    rt10 := sqrt(10);
    rt11 := sqrt(5 + 2*rt10);
    [ (1  +   rt10 +   rt11)/16, (5  +   rt10 + 3*rt11)/16,
      (10 - 2*rt10 + 2*rt11)/16, (10 - 2*rt10 - 2*rt11)/16,
      (5  +   rt10 - 3*rt11)/16, (1  +   rt10 -   rt11)/16  ]

# Minimum Phase filter coefficients, as in Daubechies, Table 6.1 p. 195
elif p = 2 then
    [  0.683012701892219323381861585376468091735701313452,
       1.18301270189221932338186158537646809173570131345,
       0.316987298107780676618138414623531908264298686547,
      -0.183012701892219323381861585376468091735701313452  ];
elif p = 3 then
    [  0.470467207784163680753329107571170458301775444775,
       1.14111691583144362576012562965942155819038194191,
       0.650365000526232528506934829034161283173662104719,
      -0.190934415568327361506658215142340916603550889550,
      -0.120832208310396209260263936605331741475437549494,
       0.049817499736883735746532585482919358413168947640  ];
elif p = 4 then
    [  0.325803428051298348274520173070400343558071390335,
       1.01094571509182886288273334771324813018241504757,
       0.892200138246759623171669505301189807116843723552,
      -0.039575026235644643095362668173963023671446088037,
      -0.264507167369039736051684828906799586738419297827,
       0.043616300474177252657739994114962714965129353620,
       0.046503601070981764605495150535209436063504183939,
      -0.014986989330361472445110673654247821476098313162  ]
elif p = 5 then
    [  0.226418982583558357648394607390239244086485773546,
       0.853943542705028332447019135725230487344880512823,
       1.02432694425919707323445443709285738636720730833,
       0.195766961347809348057273865815492400737752421399,
      -0.342656715382934886447596027304935178789483981493,
      -0.045601131883547297484801795103042741405878867426,
       0.109702658642133647044249690529831102222620721621,
      -0.008826800108358254544295173381823788689803802807,
      -0.017791870101954191479502707707992553886829822011,
       0.004717427939067871524803966944143642013049736011  ]
elif p = 6 then
    [  0.15774243200290141672140764352588227033784459441974,
       0.69950381407523566211214208996736818106872670726817,
      1.0622637598817379690950288147093469227016236446186,
       0.44583132293003551254734485775903942166233280096597,
      -0.31998659889212278988389877566324278763973283575664,
      -0.18351806406029515226640992598417447255933736087501,
       0.13788809297474459761057147697071979099935062743560,
       0.038923209708329328286723207041502407093555085350710,
      -0.044663748330189069256882099776506936758286399917484,
       0.00078325115229715577541148663852333631487106997582317,
       0.0067560623629278757137729402338007403592003692002470,
      -0.0015235338056025064552117154222588735801483026855998   ]
elif p = 7 then
    [ 0.110099430745623697782676491210,   0.560791283625525125423860008703,
      1.03114849163619733624439063206,    0.664372482211079348474060665452,
     -0.203513822462692484974628080313,  -0.316835011280665678753006670335,
      0.100846465009388173805108556921,   0.114003445159743152633131052485,
     -0.0537824525896908775264304071028, -0.0234399415642065771438337860681,
      0.0177497923793615278397266849465,  0.000607514995402138708529044356150,
     -0.0025479047181873731708438777340,  0.000500226853122490657259685396407];
elif p = 8 then
   [ 0.0769556221081524731722331959534,   0.442467247152249839723222248553,
     0.955486150427747391162709997625,    0.827816532422391785816825621332,
    -0.0223857353337604354821977428650,  -0.401658632780978100840308353665,
     0.000668194092440237560947099015462, 0.182076356847315490862529086076,
    -0.0245639010456968185119329172158,  -0.0623502066502788562711763047094,
     0.0197721592967015179562591727231,   0.0123688448196318447348933206132,
    -0.00688771925688361816782311297962, -0.000554004548958778644753250958765,
     0.000955229711299252309804307742844,-0.000166137261373225381232367240260 ];
elif p = 9 then
 [ 0.0538503495893256002938669913346943,   0.344834303813955884022358807355540,
   0.855349064359415132589782756279426,    0.929545714366294680245423892304593,
   0.188369549506367558166635349769915,   -0.414751761801876984688928276314652,
  -0.136953549024766197207530847836802,    0.210068342279012384309087440908946,
   0.0434526754612290872323451760576357,  -0.0956472641201940521884294720418809,
   0.000354892813233037828625936094461873, 0.0316241658525117260250768298649003,
-0.00667962022627715496448604520337183, -0.00605496057509013186633408627502427,
 0.00261296728049449139578039971144640,  0.000325814671352184426090225377206920,
-.000356329759021555335019716207412011,.0000556455140343097156546388203627070];
elif p = 10 then
[ 0.0377171575922413777705637759431255,   0.266122182793841789833652873332770,
  0.745575071486466769777677876091071,    0.973628110733639891366619631290027,
  0.397637741769017379022100736335968,   -0.353336201794112600498754105574950,
 -0.277109878720966305712580587644395,    0.180127448533393332685537070907716,
  0.131602987101070014647323012218178,   -0.100966571196779434139432035849558,
 -0.0416592480876016139620934376147523,   0.0469698140973971216920437651915258,
  0.00510043696781447746534190223219333, -0.0151790023358564987876837378676650,
  0.00197332536496320518826708836473664,  0.00281768659019467621145676572652435,
 -.000969947839856410962969128814103162,-.000164709006090777955103110159239787,
 .000132354366851106766368762887969254,-.0000187584156275004083371169971592603];

# Least asymmetric filter coefficients, as in Daubechies, Table 6.3 p. 198
elif p =-4 then
    [ -0.107148901418205818232578625017854429257964798315,
      -0.041910965125059527572313302172420760842728620570,
       0.703739068656299887608653648284114612940729472260,
       1.13665824340764071880863193482718158436774638267,
       0.421234534203577085841977088644070015237886337725,
      -0.140317624178543448538775452978364932861856016340,
      -0.017824701441671155218052111910330198920651011670,
       0.045570345895962257302456820323604109336838254234  ]
elif p = -5 then
    [  0.038654795954767986204493274425874528047387408616,
       0.041746864421483304995640218360039516744547847376,
      -0.055344186116622070287274803783640200140922130601,
       0.281990696853819211914637020701471356238493276661,
       1.02305296689439157530724554147664773819344408392,
       0.896581648379832149833391556891185704310010868015,
       0.023478923136123783186886535643852146721879866359,
      -0.247951362612942487635728744377218409438141678643,
      -0.029842499868661274411350547762734212821789228300,
       0.027632152957807820892059948424521832145089686594  ]
elif p = -6 then
    [  0.021784700327,  0.004936612372, -0.166863215412, -0.068323121587,
       0.694457972958,  1.113892783926,  0.477904371333, -0.102724969862,
      -0.029783751299,  0.063250562660,  0.002499922093, -0.011031867509  ]
elif p = -7 then
    [ 0.003792658534, -0.001481225915, -0.017870431651,  0.043155452582,
      0.096014767936, -0.070078291222,  0.024665659489,  0.758162601964,
      1.085782709814,  0.408183939725, -0.198056706807, -0.152463871896,
      0.005671342686,  0.014521394762  ]
elif p = -8 then
   [ 0.002672793393, -0.000428394300, -0.021145686528,  0.005386388754,
     0.069490465911, -0.038493521263, -0.073462508761,  0.515398670374,
     1.099106630537,  0.680745347190, -0.086653615406, -0.202648655286,
     0.010758611751,  0.044823623042, -0.000766690896, -0.004783458512  ]
elif p = -9 then
   [ 0.001512487309,  -0.000669141509, -0.014515578553,  0.012528896242,
     0.087791251554,  -0.025786445930, -0.270893783503,  0.049882830959,
     0.873048407349,   1.015259790832,  0.337658923602, -0.077172161097,
     0.000825140929,   0.042744433602, -0.016303351226, -0.018769396836,
     0.000876502539,   0.001981193736 ]
elif p = -10 then
 [ 0.00108917044724, 0.000135245019942,-0.0122206426341, -0.00207236392055,
   0.0649509246030,  0.0164188694163,  -0.225558972320,  -0.1002402150130,
   0.667071338429,   1.08825153074,     0.542813011103,  -0.0502565403406,
  -0.0452407723042,  0.0707035675623,   0.00815281678378,-0.0287862319439,
  -0.00113753531067, 0.00649572837868,  0.0000806612029979, -0.000649589896782]
else
   ERROR( `Correct coefficients are not available for p = `. p );
fi;
end:

print(`Jacques Gelinas, Ph.D., Maths, CMR, juin 1994, gelinas@cmr.ca`);

# References:
#
#[1] I. Daubechies,  Orthonormal bases of compactly supported wavelets, Comm.
#                    Pure Appl. Math 41 (1988), p. 909-996. (Table on p. 980
#                    lists the coefficients to 12 decimals for p=2..10)
#
#[2] G. Strang, Wavelets and dilation equations. A brief introduction, SIAM
#       Rev., 31 (1989), pp 614-629. (Shows that the 2p filter coefficients
#       satisfy a nonlinear system of equations - with 2^(p-1) solutions).
#
#[3] I. Daubechies,  Ten Lectures on Wavelets, SIAM, Philadelphia, PA, 1992.
#           (Table 6.1 p. 195 lists the coefficients to 16 decimals for p<11).
#
#[4] Wei-Chang Shann, Jann-Chang Yan, Quadratures involving polynomials and
#     Daubechies wavelets. Preprint, April 27, 1994, Technical Report 9301,
#     Math Dept, National Central Univ, Taiwan.  Anonymous ftp available at
#     dongpo.math.ncu.edu.tw:/pub/shann/publications/9301.ps
#
#    "Scaling equations are used to derive formulae of quadratures involving
#     polynomials and scaling/wavelet functions with compact supports; in
#     particular, those discovered by Daubechies.  It turns out that with a
#     few parameters, which are theoretically exact, these quadratures can be
#     evaluated with algebraic formulae instead of numerical approximations."
#
#    "For p=10 and m=9, the smallest and the largest number in that equation
#     is different by a scale of 10^8. Observe that the set of equations can
#     be replaced by $\sum_k (-1)^k (k-p)^m c_k = 0$  .. an observation
#     of Prof. I.L. Chern of National Taiwan University .. One can write down
#     the jacobian of $F_p$ and perform Newton's iteration to solve $F_p(x)=0$
#     (we use the roots of $F_{p-1}$ appended with two zeros as initial guess)."
#    "We can calculate the scaling coefficients up to p=14 .. in Table 1. All
#     values quoted hereafter are computed in quartic precision on a VAX 6510."
#

# end of daubcc.map
# end of daub.map
#save `daub.m`;

# daubdd.map - Find the 2p coefficients of the Daubechies Minium Phase filter
#              using the Deslauriers-Dubuc Lagrange interpolation method.
#
# Usage:
#
#   Digits:=20;    # One digit is lost to roundoff error for p=2, 3 for p=10.
#                                  Digits := 20`,
#   daubdd(3);     # Coefficients with sum 2, not sqrt(2) as in Daubechies[1,3]
#
#      [.47046720778416368076, 1.1411169158314436258, .65036500052623252853,
#      -.19093441556832736150, -.12083220831039620926, .049817499736883735747]
#
#  "[6]/sqrt(2):   # Shows that the sixth coefficient .0352262918856 should be
#   evalf(",12);   # printed .035226291886, not .035226291882 (typo in [1]).
#
#                                 .0352262918856
#
# See also:
#
#   daubdd.hlp - Help text for this package.
#   daubcc.map - Correct coefficients of Daubechies low pass filters.
#   daublh.map - Find the 2p coefficients of the Daubechies Minium Phase filter
#                using the original spectral method of Daubechies.
#   daubc      - Find the 2p coefficients of the Daubechies Minium Phase filter
#                using Newton's method to solve a nonlinear system of equations.
#   daubut.map - Accuracy estimation and refinement of approximations for
#                the coefficients of Daubechies low pass filters.
#   Other references below.
#
# (Copyright) Jacques Gelinas, Ph.D., Maths, CMR, aout 1994, gelinas@cmr.ca

print(`daubdd(p)  -  The 2p coefficients of Daubechies Minimum Phase filter,`);
print(`              via Deslauriers-Dubuc Lagrange interpolation method.`   );

daubdd := proc( p )                    local c, i, r, j, k, f, x, d, t, cij;
  if p = 1  then RETURN( [1.,1.] ) fi;
  if p < 1  then  ERROR(`Number of filter coefficients cannot be `.2*p ) fi;
  cij := proc(i,j) if i=j then 1 else (1/2-j)/(i-j) fi end;
  c   := [ seq( product( 'cij(i,j)','j'=1-p..p ), i=1..p ) ];
  f   := x^(2*p-1) + sum( ( c[k+1]*x^(2*p) + c[p-k] )*x^(2*k), k=0..p-1 );
  r   := simplify( f / (x+1)^(2*p) );
  d   := [ op(select( x->abs(evalf(x))<1, {fsolve(r,x,'complex')} )), (-1)$p ];
  f   := subs( I=0, expand( convert( map( unapply(x-t,t), d ), `*` ) ) );
#           Many authors use sqrt(2) to normalize the sum of the coefficients.
#           But Gilbert Strang uses 2 in [3]: his Haar filter is [1,1].
  map( unapply( 2*t/subs(x=1,f), t), subs( x=1, [ op(sort(f)) ] ) )
end:

print( `Adapted from the Matlab program of Carl Taswell to Maple V 1.1 by` );
print( `Jacques Gelinas, Ph.D., Maths, CMR, aout 1994, gelinas@cmr.ca` );

# References:
#
#[1] Deslauriers, G., Dubuc, S., Interpolation dyadique; in Fractals;Dimensions
#         non entieres et Applications, ed. G. Cherbit (Paris), 1987, p 44-45.
#
#[2] I. Daubechies,  Orthonormal bases of compactly supported wavelets, Comm.
#                    Pure Appl. Math 41 (1988), p. 909-996. (Table on p. 980
#                    lists the coefficients to 12 decimals for p=2..10, dis-
#                    cussion of relations to Deslauriers-Dubuc work on p. 933).
#
#[3] G. Strang, Wavelets and dilation equations. A brief introduction, SIAM
#       Rev., 31 (1989), pp 614-629. (Shows that the 2p filter coefficients
#       satisfy a nonlinear system of equations - with 2^(p-1) solutions).
#
#[4] I. Daubechies,  Ten Lectures on Wavelets, SIAM, Philadelphia, PA, 1992.
#           (Table 6.1 p. 195 lists the coefficients to 16 decimals for p<11).
#
#[5] Carl Taswell,  WavBox (Wavelet Toolbox for Matlab), email to
#               taswell@sccm.stanford.edu"; anonymous ftp from the directory
#              "/pub/taswell" at the site "simplicity.stanford.edu". 
#     (Read the restrictions on usage of this software in the documentation)
#     % Notes: coefficients are computed from Lagrange polynomials; see
#     %        p. 2474 of M. J. Shensa 1992 IEEE TSP 40:2464.
#     % Notes: generates Daubechies minimum phase asymmetric wavelets;
#     %        see Daubechies textbook Table 6.1, page 195; values differ
#     %        in approximately last 4 digits of precision. 
#     % Copyright (c) 1992 Carl Taswell (taswell@sccm.stanford.edu).

# daublh.map - Find the 2p coefficients of the Daubechies Minium Phase filter
#              using the original spectral method of Daubechies.
#
# Usage:
#
#   Digits:=20:   # One digit is lost to roundoff error for p=2, 2 for p=10.
#
#   daublh(3);    # Coefficients with sum 2, not sqrt(2) as in Daubechies[1,3]
#
#      [.47046720778416368075, 1.1411169158314436257, .65036500052623252852,
#      -.19093441556832736150, -.12083220831039620926, .049817499736883735746]
#
#  "[6]/sqrt(2):  # Shows that the sixth coefficient .0352262918856  should be
#   evalf(",12);  # printed .035226291886, not .035226291882 (typo in [1]).
#
#                                 .0352262918856
#
# See also:
#
#   daublh.hlp - Help text for this package.
#   daubcc.map - Correct coefficients of Daubechies low pass filters.
#   daubdd.map - Find the 2p coefficients of the Daubechies Minium Phase filter
#                using the Deslauriers-Dubuc Lagrange interpolation method.
#   daubc      - Find the 2p coefficients of the Daubechies Minium Phase filter
#                using Newton's method to solve a nonlinear system of equations.
#   daubut.map - Accuracy estimation and refinement of approximations for
#                the coefficients of Daubechies low pass filters.
#   Other references below.
#
# (Copyright) Jacques Gelinas, Ph.D., Maths, CMR, aout 1994, gelinas@cmr.ca

print(`daublh(p)  -  The 2p coefficients of Daubechies Minimum Phase filter,`);
print(`via the spectral method (from the Mma program of Tong Chen, Meng Xu).`);

#             On a 486/33/DOS: 25 sec for p=10 (fast), but the code needs work!
#             Globals:  bpk, clksi, eqk, bx, llt, zlh;
clst := (p,x)   -> subs( x=1, [ op(sort(p)) ] ):    # Coeffs of p(x), decreasing

bp   := (n,y)   -> sum( 'binomial(n-1+bpk,bpk) * y^bpk', 'bpk'=0..n-1 ):
bl   := (n,ksi) -> expand( bp( n, (1-cos(ksi))/2 ) ):
cl   := n       -> clst( bl(n,clksi), cos(clksi) ):
eq1  := (n,x)   -> sum( cl(n)['eqk']*x^(n-'eqk'), 'eqk'=1..n):
aeq  := (n,x)   -> simplify( x^n * eq1( n, (x+1/x)/2 ) ):
beq  := n       -> select(t->abs(evalf(t))<1,{fsolve(aeq(n,bx),bx,'complex')}):
ll   := (n,z)   -> ((1+z)/2)^n * convert( map(unapply(z-llt,llt),beq(n)), `*`):
lnorm:= n       -> subs( I=0, expand( ll(n,1) ) ):      # ll() is called twice
nll  := (n,z)   -> simplify( sqrt(2)^2 * subs(I=0, expand(ll(n,z)))/lnorm(n) ):
daublh2 := n    -> clst( nll(n,zlh), zlh ):             # twice as long
daublh := proc(n)                            local  t;
     if n<1 then ERROR(`Number of filter coefficients cannot be `.(2*n) ) fi;
     clst( subs( I=0, expand( ll(n,zlh) )), zlh );
     map( unapply(t * sqrt(2)^2 / convert(",`+`), t), ")
end:

# Many authors use sqrt(2) to normalize the sum of the coefficients.
# But Gilbert Strang uses 2 in [2]: his Haar filter is [1,1].

print( `Translated from Mathematica 2.0 to Maple V 1.1 by` );
print( `Jacques Gelinas, Ph.D., Maths, CMR, aout 1994, gelinas@cmr.ca` );

# References:
#
#[1] I. Daubechies,  Orthonormal bases of compactly supported wavelets, Comm.
#                    Pure Appl. Math 41 (1988), p. 909-996. (Table on p. 980
#                    lists the coefficients to 12 decimals for p=2..10)
#
#[2] G. Strang, Wavelets and dilation equations. A brief introduction, SIAM
#       Rev., 31 (1989), pp 614-629. (Shows that the 2p filter coefficients
#       satisfy a nonlinear system of equations - with 2^(p-1) solutions).
#
#[3] I. Daubechies,  Ten Lectures on Wavelets, SIAM, Philadelphia, PA, 1992.
#           (Table 6.1 p. 195 lists the coefficients to 16 decimals for p<11).
#
#[4] (The following Mathematica program follows closely the procedure described
#     in [1,3] to find the filter coefficients.)
#
#          "I've put up some packages and notebooks for constructing the
#           classical wavelets: Daubechies, Meyer, Battle-Lemarie.
#           There is nothing new here, my purpose is save others from
#           having to rediscover this wheel (and maybe have someone tell
#           me how to do it better!)."
#  Jack K. Cohen, Center for Wave Phenomena, Golden CO, 80401 (303)273-3512.
#
#  The anonymous ftp site is: hilbert.mines.colorado.edu  or  138.67.12.63 
#  The directory is pub/wavelets
#  Authors: Jack K. Cohen, Tong Chen, Meng Xu,  jkc@dix.mines.colorado.edu
#                  tchen@dix.mines.colorado.edu mxu@dix.mines.colorado.edu
# 
#      (Mathematica source reformatted to save space)
#
# (* Copyright 1992, Tong Chen & Meng Xu *) (* Version: Mathematica 2.0 *)
# (* Name: Daubechies *)(* Authors: Tong Chen & Meng Xu *)
# (* Keywords: dblh, dbm0 *)(* Requirements: none *)(* Warning: none *)
# 
# BeginPackage["Daubechies`"]
# dbm0::usage = "dbm0[n,ksi] gives the frequency domain low pass filter of 
# 		Daubechies type of order n"
# dblh::usage = "dblh[n] gives the coeffients of the low pass wavelet 
# 		filter of Daubechies type of order n"
# Begin["`private`"];
#                                                     (* The polynomial P_n *)
# bp[n_,y_] := Sum[ Binomial[n-1+k,k] y^k, {k,0,n-1} ]
#                                                    (* The function L[ksi] *)
# bl[n_,ksi_]:= Expand[ bp[n,(1-Cos[ksi])/2] ]
#                     (* Strip the coefficients to form the polynomial in z *)
# cl[n_]    := CoefficientList[ bl[n,ksi], Cos[ksi] ]
#                                               (* Form the polynomial in z *)
# eq1[n_,x_]:=Block[{len},len=Length[cl[n]]-1; cl[n].Table[x^i,{i,0,len}] ]
#                                (* The equation for spectral factorization *) 
# aeq[n_,x_]:=Block[{len},len=Length[cl[n]]-1;Simplify[x^len*eq1[n,(x+1/x)/2]]]
#                                                  (* Factorized polynomial *)
# beq[n_]   :=Block[{sl,x,solu,pole={}}, solu = NSolve[aeq[n,x] == 0,x];
#    sl=x/.solu; Map[(If[Abs[#]<=1.,pole=Append[pole,#]])&,sl]; Return[pole] ]
#                                                  (* Get the function m[z] *)
# ll[n_,z_] := ((1+z)/2)^n*Apply[Times,z - beq[n]]
#                                                 (* m[1] for normalization *)
# lnorm[n_] := Re[ll[n,1]]
#                                           (* The normalized function m[z] *)
# nll[n_,z_]:= ll[n,z]/lnorm[n]
#                                            (* The low pass wavelet filter *)
# dblh[n_]  := Reverse[Re[N[ Sqrt[2] * CoefficientList[nll[n,z],z], 16]]] 
#                                                            (* m0 function *)
# dbm0[n_,ksi_] := Block[{allh}, allh = dblh[n];
#        Return[1/Sqrt[2] allh.Table[Cos[i ksi] - I Sin[i ksi],{i,1,2*n}]]; ]
# End[]
# Protect[bp,bl,cl,eq1,aeq,beq,ll,lnorm,nll,dbm0,dblh] EndPackage[]  Null

# end of daublh.map
# end of daubdd.map

# daubut.map - Accuracy estimation and refinement of approximations for
#              the coefficients of Daubechies low pass filters.
#
# Usage:
#
#          daubs( 4 );          Nonlinear system for coefficients, with some
#                               catastrophic cancellation of leading digits.
#                               All 2^(4-1) = 8 solutions can be found by
#
#          evalc(evalf( [ allvalues( solve({op(daubs(4))}), `d` ) ] ));
#
#          drefine( ", 80 );    Refine the solution to 80d.
#
#          daubacc( " );        Verify the accuracy, working with precision
#                               160d to find a more accurate solution.
#
# See also:
#
#   daubut.hlp - Help text for this package.
#   daubcc.map - Correct coefficients for the Daubechies low pass filters.
#   daubdd.map - Find the 2p coefficients of the Daubechies Minium Phase filter
#                using the Deslauriers-Dubuc Lagrange interpolation method.
#   daublh.map - Find the 2p coefficients of the Daubechies Minium Phase filter
#                using the original spectral method of Daubechies.
#   daubc      - Find the 2p coefficients of the Daubechies Minium Phase filter
#                using Newton's method to sove a nonlinear system of equations.
#
# (Copyright) Jacques Gelinas, Ph.D., Maths, CMR, aout 1994, gelinas@cmr.ca
#
# References:
#
#[1] I. Daubechies,  Orthonormal bases of compactly supported wavelets, Comm.
#                    Pure Appl. Math 41 (1988), p. 909-996. (Table on p. 980
#                    lists the coefficients to 12 decimals for p=2..10)
#
#[2] G. Strang, Wavelets and dilation equations. A brief introduction, SIAM
#       Rev., 31 (1989), pp 614-629. (Shows that the 2p filter coefficients
#       satisfy a nonlinear system of equations - with 2^(p-1) solutions).
#
#[3] I. Daubechies,  Ten Lectures on Wavelets, SIAM, Philadelphia, PA, 1992.
#           (Table 6.1 p. 195 lists the coefficients to 16 decimals for p<11).
#
#[4] Wei-Chang Shann, Jann-Chang Yan, Quadratures involving polynomials and
#     Daubechies wavelets. Preprint, April 27, 1994, Technical Report 9301,
#     Math Dept, National Central Univ, Taiwan.  Anonymous ftp available at
#     dongpo.math.ncu.edu.tw:/pub/shann/publications/9301.ps
#
#    "Scaling equations are used to derive formulae of quadratures involving
#     polynomials and scaling/wavelet functions with compact supports; in
#     particular, those discovered by Daubechies.  It turns out that with a
#     few parameters, which are theoretically exact, these quadratures can be
#     evaluated with algebraic formulae instead of numerical approximations."
#
#    "For p=10 and m=9, the smallest and the largest number in that equation
#     is different by a scale of 10^8. Observe that the set of equations can
#     be replaced by $\sum_k (-1)^k (k-p)^m c_k = 0$  .. an observation
#     of Prof. I.L. Chern of National Taiwan University .. One can write down
#     the jacobian of $F_p$ and perform Newton's iteration to solve $F_p(x)=0$
#     (we use the roots of $F_{p-1}$ appended with two zeros as initial guess)."
#    "We can calculate the scaling coefficients up to p=14 .. in Table 1. All
#     values quoted hereafter are computed in quartic precision on a VAX 6510."
#

#read `numerics/daub/daub.mac`;

# The exact values are found by "solve" for p = 1,2,3 as solutions (see [2])
# of the linear system with 1 solution for p=1, 
#    2  =  c0 + c1                      normalization
#    0  =  c0 - c1                      moment 0
# or of the nonlinear system with 2 solutions for p=2,
#    2  =  c0 + c1 +   c2 +   c3        normalization
#    0  =  c0 c2   +  c1 c3             orthogonality
#    0  =  c0 - c1 +   c2 -   c3        moment 0
#    0  =     - c1 + 2 c2 - 3 c3        moment 1
# or of the nonlinear system with 4 solutions for p=3, 
#    2  =  c0 + c1 +   c2    +   c3    +    c4   +  c5       normalization
#    0  =  c0 c2   +  c1 c3  +  c2 c4  +  c3 c5              orthogonality
#    0  =  c0 c4   +  c1 c5                                  orthogonality
#    0  =  c0 - c1 +   c2    -   c3    +    c4   -  c5       moment 0
#    0  =     - c1 + 2 c2    - 3 c3    +  4 c4   -  5 c5     moment 1
#    0  =     - c1 + 4 c2    - 9 c3    +  16 c4   - 25 c5    moment 2
# It is evident that catastrophic cancellation occurs in the last equations.
# For p=4... there are too many solutions (or none) returned by "solve":
#   op( evalc(evalf( [ allvalues( solve({op(daubs(4))}), `d` ) ] )) )
# produces 2 real sols, 2 complex sols, and 4 mirror images of those.
# The system can be obtained by calling the following routine:

print(`daubs(p) - System of 2p equations for Daubechies filter coefficients`);
daubs   := proc( p )	      local  k, m, c0,c1,c2,c3,c4,c5,c6,c7,c8,c9,c10,
                                     c11,c12,c13,c14,c15,c16,c17,c18,c19;
   if p>10 and assigned(c20) then ERROR( `symbol c20 has a value.` ) fi;
   [  2 - sum( 'c.k',                  k=0..2*p-1     ),
     seq( sum( 'c.k * c.(k+2*m)',      k=0..2*(p-m)-1 ),  m=1..p-1 ),
     seq( sum( (-1)^k * 'k'^m * 'c.k', k=0..2*p-1     ),  m=0..p-1 )  ]
end:

# An equivalent system, with less catastrophic cancellation of leading
# significant digits in the last equations - I.L. Chern, Taiwan Univ.
daubsys := proc( c )                    local  k, m, p;   p:=nops(c)/2;
    [  2 - sum( c[k],                    k=1..2*p     ),
      seq( sum( c[k] * c[k+2*m],         k=1..2*(p-m) ),  m=1..p-1 ),
      seq( sum( (-1)^k * (k-p)^m * c[k], k=1..2*p     ),  m=0..p-1 ) ]
end:

# Newton's method for systems can be used, starting from published values.
# Maple's fsolve does not work very well (if at all) since the nonlinear
# system has 2^(p-1) solutions (including complex ones and mirror images).
# It seems difficult to pinpoint the solution so as to give a range (chicken)
# to fsolve, and without a range fsolve does not find one solution (egg).

print(
`drefine(c0,N) - Refine filter coefficients c0 to N digits of precision.` );
drefine := proc( c0, N )	             local c, c1, k, v, old;
        old := Digits; Digits := N + nops(c0);
        v   := [ seq( c[k] , k=1..nops(c0) ) ];
        c1  := newtons( daubsys(v), v, c0 );
        Digits := old;
        evalf( c1, N )
end:

newtons  := proc( sys, xv, x0 )      local k, it, s, X0, F0, J, J0, DX;
    J  := linalg[jacobian]( sys, xv ); 
    X0 := linalg[vector]( evalf(x0) );
    for it to Digits do                              Digits:=Digits+it;
        s  := { seq(xv[k]=X0[k], k=1..nops(xv)) };
        F0 := map((x,y)->subs(y,x), linalg[vector](sys), s);
        J0 := map((x,y)->subs(y,x), J, s);
        DX := linalg[linsolve]( J0, F0 );
        X0 := linalg[matadd]( X0, DX, 1, -1 );          Digits:=Digits-it;
        if linalg[norm](DX) < linalg[norm](X0)*5/10.^Digits then break
        elif it = Digits then ERROR(`No convergence`) fi
    od;
    evalf( convert( X0, list ) )
end:

# This routine tries to estimate the precision of a given solution c.
# N0 is a guess for the precision (default Digits) and c0 is a starting
# value for Newton's method (default c). The value of Digits is doubled
# until a more accurate solution than c is found, or Digits exceeds 400.

print(`daubacc(c) - Accuracy of coefficients c of Daubechies filter.`);
daubacc := proc( c, c0, N0 )               local  c1, N, k, s, oldp;
      if nargs<2 then c1:=c else c1:=c0 fi;
      if nargs<3 then N:=2*Digits else N:=N0 fi;
      c1     :=  drefine( c1, N );
      oldp   :=  Digits; 
      Digits :=  N;
      s      :=  max( seq( abs(c1[k]-evalf(c[k])), k=1..nops(c) ) );
      Digits :=  oldp;
      if s < 10^(2-N) and N < 400 then
            RETURN( daubacc( c, c1, 2*N ) )
      fi;
      if s=0 then N else round( evalf( -log10(s), 10 ) ) fi
end:

# The trapezoidal formula for numerical integration is usefull for
# piecewise or recursive functions where the Maple int procedure
# cannot be used easily. If the error is divided approximately
# by 4 when the number of trapezoids is multiplied by 2, then
#    I - trap(a,b,f,n) ~= ( I - trap(a,b,2*n,f) ) / 4
# which leads to the Simpsons formula. The rich function is
# available when the error is of order 3.
print( `trap(a,b,n,f) - ~Int(f,x=a..b) using n trapezoids.` );
print( `simp(a,b,n,f) - ~Int(f,x=a..b) using n parabolas.` );
print( `rich(a,b,n,f) - ~Int(f,x=a..b) using Richardson extrapolation.` );

simp := (a,b,n,f) -> (4*trap(a,b,2*n,'f') - trap(a,b,n,'f'))/3 :
rich := (a,b,n,f) -> (8*simp(a,b,2*n,'f') - simp(a,b,n,'f'))/7 :
trap := proc(a,b,n,f)         local k, h, s;
	h := evalf( (b-a) / n );
	s := ( f( evalf(a) ) + f( evalf(b) ) ) / 2;
	for k from 1 to n-1 do s := s + f( evalf( a + k*h ) ) od;
	evalf( h * s );
end:

print(`Jacques Gelinas, Ph.D., Maths, CMR, aout 1994, gelinas@cmr.ca`);

# end of daubut.map


#quit






