#
## <SHAREFILE=system/sgml/sgml.mpl >
## <DESCRIBE>
##            This is a Maple package based on Maple's latex command
##                for converting Maple expressions into the ISO standard
##                SGML (Standard Generalized Markup Language). The produced
##                code is suitable for SGML text processing systems which
##                use the international mathematics standard as published by
##                the Association of American Publishers.
##                AUTHOR: Wolfgang Frings, w.frings@kfa-juelich.de
## </DESCRIBE>
## <UPDATE=R4 >

#
#       procedure sgml(expr): produces sgml output that
#               should typeset "expr" in a reasonable format
#
#       sgml(expr, filename ) writes the output to the file specified
#
#       sgml(expr, inline)
#       sgml(expr, filename, inline) produces a inline formula
#
#       The code of this procedure is based on the code from the latex 
#               function. Only the literals and LaTeX commands are changed 
#               to the equivalent SGML codes. 
#                       W. Frings (Nov. 1993) 
#
#       The format of the output finally depends on the
#               degree of simplification of expr, no
#               attempt is made to modify the input expression
#
#       The mathematical format is taken, in general, from the
#               CRC handbook or the Handbook of Mathematical
#               functions.
#
#       The functions sum, int, diff, limit and log are aliased by Sum,
#               Int, Diff, Limit and Log, so that these can be used to
#               prevent evaluation by maple.  E.g.
#               sgml( Int( 1/(x^2+1), x) = int( 1/(x^2+1), x) )
#
#       Function printing can be interfaced by the user by including
#               a function `sgml/<function-name>`.
#
#       Usage:
#               writeto(SgmlFile);       or sgml( YourEquation, SgmlFile );
#               sgml( YourEquation );
#               writeto(terminal);
#
#               (sgml will not print the "\[" or "\]" nor any other
#               delimiter, which the user has to provide).
#
#       Bugs:   Will sometimes produces useless pairs of "{" "}".
#       ***     Prints out "words used" messages rendering it useless.
#           Tables are not handled
#               (space reserved for members of watmaple)
#
#                       Ricardo A. Baeza-Yates (Sept 1986, Feb 1988)
#
#                       based in the eqn function (and consistent with it).
#  ******************** This may no longer be true!!! dgc (12 May 1991) ***
#
#       Minor bug fixes  JSD April/89
#       Minor bug fixes  RBY June/89
#       Major bug fixes  dgc fall 89.
#               Most useless pairs of `{` and `}` have been eliminated
#               from the output.  This allows Sgml to insert line
#               breaks when given a chance.  Sets are still output as
#               an indivisible unit; perhaps this should change (the
#               disadavantage to such a change is that it would no
#               longer be possible to automatically set the brace
#               height based on the content of the set.)
#
#               `sgml/print` now returns a sequence of Sgml tokens in
#               so far as is practical.  This is the first step
#               towards adding code that will be capable of
#               automatically format multi-line equations.
#
#               strings are now printed correctly.
#
#       Minor bug fixes  dgc 12 May 1991.
#            Added a product function, similar to sum.
#            Removed the test that interchanges the order of two
#              summands to a sum if the second is positve, but
#              the first is not.
#            Added code so that series() are more gracefully
#              handled, but the parenthesisation does not appear
#              to  be correct yet.  Munge this by parenthesizing
#              all series.
#            Parentheses added in some cases of complicated
#              operator function calls, e.g., (f+g)(x).  This needs more
#              thought.
#            Multiple [][] in indexed names no longer cause an
#              error.
#            Some minor internal changes.  Some of the `sgml/sgml/NAMES`
#              have become `type/sgml/...` to better reflect their
#              functionality.
#
#       Known outstanding bugs:
#                series still aren't gracefully integrated, but they function
#                  much better than 5.0.
#                no provisions have been made for plain-tex output.
#                Formatting of operators is still a hack.
#                Interface to user functions needs more thought.  Also need to
#                  solve the indexed function problem.
#
#    Minor bug fixes - GL - 1992
 
 
sgml := proc(e::anything,filename::string,disp::string)
local ``, i, sgmllist, sgmlstr, instr, exstr, dispi, filen;
option `Copyright 1992 by the University of Waterloo`;
   if type(e,table) or type(e,procedure) or type(e,array) then
       sgmllist := [ `sgml/print`(eval(e)) ];
   else sgmllist := [ `sgml/print`(e) ];
   fi;
   
   dispi=`display`;

   if nargs = 1
   then 
     filen:=``;
   fi;

   if nargs = 2
   then 
     if args[2]=`inline` 
     then 
        dispi := `inline`; 
        filen:=``;
     else 
       filen:=filename;
     fi;
   fi;

   if nargs = 3
   then 
     if args[3]=`inline` then dispi := `inline`; fi;
     filen:=filename;
   fi;
   
   if dispi = `inline`
   then    
     instr := `<inline-equation><f>` ;
     exstr := `</f></inline-equation>` ;
   else
     instr := `<display-equation><fd><fl>` ;
     exstr := `</fl></fd></display-equation>` ;
   fi;

   if filen <> `` then
       readlib(write):
       open(filename);
   fi;

   if filen <> `` then
        writeln( instr );
   else 	
	lprint( instr ); 
   fi;

   sgmlstr := `` ;
   for i from 1 to nops(sgmllist) do
       if length(sgmlstr)+length(sgmllist[i]) > 70 then
                if filen <> `` then
                        writeln( sgmlstr );
                else lprint( sgmlstr ) fi;
                sgmlstr := ``.(sgmllist[i]);
       else sgmlstr := ``.(sgmlstr).(sgmllist[i]);
       fi;
    od;
    if filen <> `` then
        writeln( sgmlstr );
        writeln( exstr );
        close();
    else
        lprint( sgmlstr );  
        lprint( exstr );
    fi;
end:

################################################################################## 
#
#  The following functions are assumed to be math functions known to
#    sgml (and presumably to maple).  They are printed with a surrounding 
#    roman font tag (<rf>). Their arguments are otherwise processed normally.
#
`sgml/mathops` :=
      {  'arccos', 'arcsin', 'arctan', 'arg', 'cos', 'cosh', 'cot',
         'coth', 'csc', 'deg', 'det', 'dim', 'exp', 'gcd', 'hom',
         'inf', 'ker', 'lg', 'liminf', 'limsup', 'max', 'min',
         'Pr', 'sec', 'sin', 'sinh', 'sup', 'tan', 'tanh'
      } :
 
#
#  The following names are the names of variables or functions that
#    should be converted to (Greek) symbols by substituting the symbol 
#    from the sgml/sgreek set.
#
`sgml/greek` := ['alpha','beta','gamma','delta','epsilon','varepsilon',
                  'zeta','eta','theta','vartheta','iota','kappa','lambda',
                  'mu','nu','xi','pi','varpi','rho','varrho','sigma',
                  'varsigma','tau','upsilon','phi','varphi','chi','psi',
                  'omega','Delta','Theta','Lambda','Xi','Pi','Sigma',
                  'Upsilon','Phi','Psi','Omega'] :
 
`sgml/sgreek` := '['a','b','g','d','e','3','z','H','q',
                   'j','i','k','l','m','n','x','p','2','r','9',
                   's','v','t','u','f','4','c','y','w',
                   'D','Q','L','X','P','S','U','F','Y','W','G']' :

##################################################################################
#
#     Copy prevents strings from evaluating to global names by
#     creating anonymous local names.
 
macro(Copy=`sgml/sgml/copy`) ;
`sgml/sgml/copy` := proc(Name)
option `Copyright 1992 by the University of Waterloo`;
eval(subs('X' = Name,proc() local X; X end)()) end:
 
macro(reverse=`sgml/sgml/reverse`);
reverse := proc() local a,i,ans;
        option `Copyright 1992 by the University of Waterloo`;
        seq(args[nargs-i], i=0...nargs-1)
end:

################################################################################## 
#
#   Names known to Maple that are commonly denoted by a different name
#   in standard math notation should be defined here.
#
`sgml/special_names` := table() :
 
`sgml/special_names`['Pi'] := Copy('`<g>p</g>`') :
`sgml/special_names`['Beta'] := Copy('`<g>b</g>`') :
`sgml/special_names`['Zeta'] := Copy('`<?Eqn TeX sym="\\boldzeta">`') :
`sgml/special_names`['GAMMA'] := Copy('`<g>G</g>`') :
`sgml/special_names`['I'] := Copy('i') :
`sgml/special_names`['E'] := Copy('e') :
`sgml/special_names`['infinity'] := Copy('`&infin;`') :

##################################################################################
`sgml/print` := proc(e)
local sgmllist,   `,`, `[`, `]`, ` `,
      `<fen lp="par">`, `<rp post="par"></fen>`, ``,  `_`, `)` ;
global _SgmlSmallFractionConstant;
option `Copyright 1992 by the University of Waterloo`;
 
if nargs=0 then sgmllist := NULL ;
 
elif nargs>1 then
        sgmllist := `sgml/sgml/commalist`([args], `,`, ` `, ` `)
 
elif type(e,'numeric')
        then sgmllist := `sgml/sgml/numeric`(e)
 
elif type(e,'string') then
        sgmllist := `sgml/sgml/string`(e) ;
 
elif type(e,'indexed') then
        sgmllist :=  `sgml/sgml/indexed`(e) ;
 
elif type(e,'`+`') then
        sgmllist := `sgml/sgml/+`(e)
 
#       Product/quotients
elif type(e,'`*`') then
        sgmllist := `sgml/sgml/*`(e)
 
elif type(e,'`^`') then
        sgmllist := `sgml/sgml/**`(e)
 
elif type(e,'function') then
        sgmllist := `sgml/sgml/function`(e)
 
elif type(e,'relation') then
        sgmllist := `sgml/print`(op(1,e)),
                   `sgml/sgml/relation`(e),
                   `sgml/print`(op(2,e))
 
elif type(e,'set') then
        sgmllist := `sgml/sgml/commalist`( [op(e)], `,`,
                                      `<fen lp="par">`,
                                      `<rp post="par"></fen>`
                                    ); 
elif type(e,'list') then
        sgmllist := `sgml/sgml/commalist`( [op(e)], `,`, `[`, `]`);
elif type(e,'table') then
        sgmllist := `sgml/sgml/table`(e) ;
 
elif type(e,'procedure') and not type(e,'name') and
        member(operator,[op(3,e)]) and member(angle,[op(3,e)]) then
        sgmllist := `sgml/sgml/angleoperator`(e)
 
elif type(e,'procedure') and not type(e,'name') and
        member(operator,[op(3,e)]) and member(arrow,[op(3,e)]) then
        sgmllist := `sgml/sgml/arrowoperator`(e)
 
elif type(e,'series') then
    _SgmlSmallFractionConstant:=1;
    sgmllist := `<fen lp="par">`, `sgml/sgml/series`( e ) , `<rp post="par"></fen>` ;  # The parens are a
                                                       # hack. dgc 12 May 1991.
    _SgmlSmallFractionConstant:=50;
 
elif type(e,'range') then
    sgmllist := `sgml/sgml/range`( e )
 
else    ERROR('`Cannot handle type: `',whattype(e))
fi;
 
RETURN(sgmllist)
end:

################################################################################## 
`sgml/sgml/numeric` := proc(e)
local sgmllist, `<fr>`,`</fr>`,`<nu>`,`</nu>`, `<de>`,`</de>`,`/`, `-`, `}` ;
 
    option `Copyright 1992 by the University of Waterloo`;
    if e<0 then sgmllist := `-`, `sgml/sgml/numeric`(-e)
 
    elif type(e,'integer') then sgmllist := e
 
    elif type(e,'float') then sgmllist := `sgml/sgml/float`( e )
 
    elif type(e,'fraction') then
            if abs(op(1,e)*op(2,e)) > _SgmlSmallFractionConstant then
                            sgmllist := `<fr><nu>`, op(1,e), `</nu>`,
                                         `<de>`, op(2,e), `</de>`, `</fr>`
            else sgmllist := op(1,e), `/`, op(2,e)
            fi
    fi ;
 
RETURN(sgmllist) ;
end :

################################################################################## 
#
#       `sgml/sgml/relation`: relational keywords in sgml
#
`sgml/sgml/relation` := proc(e)
local `&le;`, `&neq;`, `&gt;`, `&lt;`, `&ge;`;
  option `Copyright 1992 by the University of Waterloo`;
  if whattype( e ) = '`<=`' then RETURN( `&le;` )
elif whattype( e ) = '`<>`' then RETURN( `&neq;` )
elif whattype( e ) = '`>`' then RETURN( `&gt;` )
elif whattype( e ) = '`<`' then RETURN( `&lt;` )
elif whattype( e ) = '`>=`' then RETURN( `&ge;` )

else RETURN( whattype( e ) ) fi;
end:

##################################################################################  
#
#      `sgml/sgml/prinpar`: output with parenthesis
#
`sgml/sgml/prinpar` := proc(e)
local `<fen lp="par">`, `<rp post="par"></fen>` ;
     option `Copyright 1992 by the University of Waterloo`;
     `<fen lp="par"> `,`sgml/print`(e),`<rp post="par"></fen>`;
end:
`sgml/prinpar` := `sgml/sgml/prinpar` :    # For historical compatibility
 
################################################################################## 
#
#       `type/sgml/istall`: determine if an expression is tall
#               (more than one line tall)
#
`type/sgml/istall` := proc(e)
local ee ;
option `Copyright 1992 by the University of Waterloo`;
if type(e,'`*`') then
   for ee in e do
       if type(ee,'`^`') and type(op(2,ee),'rational') and op(2,ee)<0
          or type(ee, '`sgml/istall`')
          then RETURN( true )
       fi
   od;
elif type(e,'fraction') and
     abs(op(1,e)*op(2,e)) > _SgmlSmallFractionConstant then
        RETURN( true )
elif type(e,{'string','constant','indexed','numeric'}) then
        RETURN( false )
else    for ee in e do
                if type(ee, '`sgml/istall`') then RETURN( true ) fi
                od
        fi;
false
end:
`sgml/istall` := `sgml/sgml/istall` : # For historical compatibility
 
################################################################################## 
#
#       `type/sgml/isneg`: expression will print with a negative sign in
#               front of it
`type/sgml/isneg` := proc(e)
   option `Copyright 1992 by the University of Waterloo`;
   type(e,'numeric') and e<0 or
   type(e,'`*`') and type(op(1,e),'`sgml/isneg`') or
   type(e,'`+`') and type(op(1,e),'`sgml/isneg`')
# See comments in `sgml/sgml/+`
#-               and type(op(2,e),'`sgml/isneg`')
                                                   or
   type(e,'series') and
      ( type(op(1,e),'`sgml/isneg`') or
        ( op(1,e)=1 and op(2,e)=1 and type(op(0,e),'`sgml/isneg`') )
      )
end:
`sgml/isneg` := `type/sgml/isneg` : # For historical compatibility

################################################################################## 
macro(chars=`sgml/sgml/chars_in_string`) :
`sgml/sgml/chars_in_string` := proc(String) local i; options remember,system,
`Copyright 1992 by the University of Waterloo`;
     ['substring(String,i...i)'$'i'=1...length(String)] end  :
 
`sgml/sgml/string` := subs(
 
__alphabet = map(x -> ''x'',convert(chars(
'`ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz`'
                                    ),'set')),
 
__digits =  map(x -> ''x'', convert(chars('`0123456789`'),'set')),
 
 
proc(Q)
 
local sgmllist, `<g>`, `</g>`, QQ;
options system, remember ;
        if assigned(`sgml/special_names`[``.Q])
            then sgmllist := `sgml/special_names`[evaln(``.Q)] ;
        elif  member(Q,`sgml/greek`,'qq')
             then
		   QQ:=op(qq,`sgml/sgreek`);
                   sgmllist := Copy(`<g>`.QQ.`</g>`) 
        elif # Process math operators the same way as Greek symbols.
             # Done here, rather than in `sgml/sgml/function` in order that
             # sgml(sin) and sgml(sin(x)) print sin the same way.
             has( `sgml/mathops`,Q)
                        then sgmllist := Copy(`<rf>`.Q.`</rf>`)
        elif length(Q)>0
             and has(__alphabet,substring(Q,1..1))
             and {} = convert(chars(Q),'set')
                      minus (__alphabet union __digits)
             then
                if length(Q)=1 then sgmllist := Copy(Q) ;
                else sgmllist :=  `<it>`,Copy(Q),`</it>` ;
                fi ;
        elif length(Q)>0
             and has(__alphabet union {'`_`'},substring(Q,1..1))
             and  {'`_`'} = convert(chars(Q),'set') minus
                                     (__alphabet union __digits)
            then sgmllist := `<it>`,
                            cat(op(subs('`_`'='`_`',chars(Q)))),
                            `</it>` ;
 
        else sgmllist := `<ty>`,
                cat(````, op(subs('`_`'='`_`',
                                  '````'='``````',
                                  '`^`'='`^`',
                                  '`#`'='`&num;`',
                                  '`$`'='`&dollar;`',
                                  '`%`'='`&percnt;`',
                                  '`&`'='`&amp;`',
                                  chars(Q)))
                    ,````),`</ty>` ;
        fi ;
        RETURN(sgmllist) ;
end ):

##################################################################################  
`sgml/sgml/float` := proc( f )
local sign, fracpart, filler, mantissa, ipart, exponent,
      ` `, `-`, `.0`, `* `, `<sup>`, `</sup>` ;
 
option `Copyright 1992 by the University of Waterloo`;
mantissa := abs(op(1,f)) ;
sign := `if`( op(1,f)<0 , '`-`', '` `') ;
exponent := op(2,f) ;
 
if exponent<10 and length(mantissa)+exponent>-10 then
   # Print in the standard floating point format.
    if exponent < 0 then
        ipart := iquo(mantissa,10^(-exponent),'fracpart') ;
        filler := (-exponent)-length(fracpart) ;
    else
        ipart := mantissa*10^exponent ;
        fracpart := 0 ;
        filler := 0 ;
    fi ;
    cat(sign, ipart, op(`if`(fracpart>0,
                             ['`.`',0$filler,fracpart],
                             ['`.0`'])))
else
     cat(sign, iquo(mantissa,10^(length(mantissa)-1)),'`.`',
                   irem(mantissa,10^(length(mantissa)-1))),
         `*`,10, `<sup>`, exponent+length(mantissa)-1, `</sup>`
fi;
end:
 
##################################################################################  
#
# Print tables, matrices, vectors, etc.
#
`sgml/sgml/table` := proc(e)
local sgmllist, `<fen lp="par">`, `<rp post="par"></fen>`, ``, `<inf>`, `</inf>`, `<ty>`, `</ty>`;
        option `Copyright 1992 by the University of Waterloo`;
        if type(e,'matrix') then sgmllist := `sgml/sgml/matrix`(e) ;
        elif type(e,array) and nops([op(2,e)])=2 then
            sgmllist := `sgml/sgml/matrix`(array(convert(e,'listlist'))) ;
            sgmllist := sgmllist, `<inf>`, `sgml/print`(op(2,e)), `</inf>` ;
        elif type(e,vector) then
            sgmllist := `sgml/print`(convert(e,'list')) ;
        elif type(e,array) and nops([op(2,e)])=1 then
            sgmllist := `sgml/print`(array(convert(e,'list'))) ;
            sgmllist := sgmllist, `<inf>`, `sgml/print`(op(2,e)), `</inf>` ;
        else
            sgmllist := `<ty>`,
                `if`(type(e,'array'), Copy('array'), Copy('table')),
                `</ty>`,`<fen lp="par">`,
                `sgml/print`(
                       map((x,e) -> (op(x)=e[op(x)]), [indices(e)], e)
                             ), `<rp post="par"></fen>` ;
        fi ;
        RETURN(sgmllist) ;
end :

##################################################################################   
# The following function determines what to print for undefined
#  matrix entries
#
`sgml/sgml/undefined_entry` := proc(A,i,j)
local `?` ;
   option `Copyright 1992 by the University of Waterloo`;
   `?`
end :
 
##################################################################################  
`sgml/sgml/matrix` := proc(e)
local  i, j, sgmllist, `<rp post="par">`, `</fen>`, 
       `<fen lp="par">`, `<ar>`, `</ar>`, `<arr>`, `</arr>`, `<arc>`, `</arc>`;
 
        option `Copyright 1992 by the University of Waterloo`;
        sgmllist := `<fen lp="sqb">`,`<ar>`;
        for i to linalg['rowdim'](e) do
		sgmllist := sgmllist,`<arr>`;
                for j from 1 to linalg[coldim](e) do
			if not has(e[i,j],e)
                 	 then sgmllist := sgmllist, `<arc>`,
					  (`sgml/print`(e[i,j])), `</arc>`;
       		          else sgmllist := sgmllist, `<arc>`, 
                                `sgml/sgml/undefined_entry`(e,i,j), `</arc>`;
                	fi ;
                od;
                sgmllist := sgmllist,`</arr>`;
        od;
        sgmllist := sgmllist, `</ar>`,`<rp post="sqb">`,`</fen>`
end : # `sgml/sgml/matrix`
 
##################################################################################  
`sgml/sgml/indexed` := proc (e)
# Revised: 12 May 1991 to handle multiple [][] better.
local ee, sgmllist, `_`, `,`, `<inf>`, `</inf>` ;
option `Copyright 1992 by the University of Waterloo`;
 
   sgmllist := `sgml/sgml/commalist`( [op(e)], `,`, ` `, ` ` ) ;
   ee := op(0,e) ;
 
   while type(ee, 'indexed') do
      sgmllist := `sgml/sgml/commalist`( [op(ee)], `,` ),
                  `;`, sgmllist  ;
      ee := op(0,ee) ;
   od;
   `sgml/print`(ee) , `<inf>`, sgmllist, `</inf>` ;
end: # `sgml/sgml/indexed`
 
##################################################################################   
`sgml/sgml/*` := proc(e)
local subexp,den,ee,subee,i,num,sgmllist,`<hsp sp="0.167">`,`<fr>`,`</fr>`,
      `<nu>`, `</nu>` ,`<de>`, `</de>`;
 
        option `Copyright 1992 by the University of Waterloo`;
        sgmllist := NULL ;
        num := 1;
        den := 1;
 
        if type(op(1,e),'rational') and op(1,e)<0 then
                RETURN(Copy('`-`'),`sgml/print`(-e))
        fi;
 
        ee := e  ;
        for subee in [op(ee)] do
                if type(subee,'fraction') then num := num*op(1,subee);
                                             den := den*op(2,subee)
                elif type(subee,'`^`') and type(op(2,subee),'rational') and
                        op(2,subee)<0 then den := den/subee
                else    num := num*subee
                        fi
                od;
        if den<>1 then
            if type(num,'`sgml/istall`') or type(den,'`sgml/istall`') then
                if num <> 1 then
                    sgmllist := `sgml/print`(num);
                                        if (type(num,`+`)) then
                                         sgmllist := `<fen lp="par">`,
                                            sgmllist, `<rp post="par"></fen>`;
                                        fi;
                else sgmllist := NULL
                fi ;
                if type(den,'`*`') then den := [op(den)] else den := [den] fi ;
                for subexp in den do
                    sgmllist := sgmllist, `sgml/print`(1/subexp)
                    od ;
                sgmllist ;
            else
                sgmllist := sgmllist,
                           '`<fr>`','`<nu>`', `sgml/print`(num),'`</nu>`',
                                   '`<de>`', `sgml/print`(den),'`</de>`',
                           `</fr>`
            fi
        else
            i := 1 ;
            for subee in ee do
                    if type(subee,{'`+`','series'}) then
                        sgmllist := sgmllist, `sgml/sgml/prinpar`(subee);
                    elif i<nops(ee)
                     and type(subee,'function')
                     and member(op(0,subee),{diff,Diff}) then
                       sgmllist := sgmllist, `sgml/sgml/prinpar`(subee)
                else sgmllist := sgmllist, `sgml/print`(subee)
                fi ;
                if i<nops(ee) and
                   (i=1 and type(op(1,ee),'integer')
                    or type(subee,'`!`')
                    or type(subee,'string') and length(subee)>1
                    or has([`sgml/print`(subee)], '`<rad>`')
                   )
                   then sgmllist := sgmllist, `<hsp sp="0.265">` ;
                fi ;
            i := i+1 ;
            od
        fi;
        RETURN(sgmllist) ;
end :

##################################################################################   
`sgml/sgml/**` := proc(e)
local sgmllist, `<fr>`, `<nu>`, `<de>`, `<rad>`, `<rcd>`, `<rdx>`, 
      `<sup>`, `</fr>`, `</nu>`, `</de>`, `</rad>`, `</rcd>`, `</rdx>`, 
      `</sup>`, `<hsp sp="0.167">`, `-1` ;
        option `Copyright 1992 by the University of Waterloo`;
        sgmllist := NULL ;
        if     type(op(2,e),'fraction')
           and abs(op(1,op(2,e)))=1
           and abs(op(2,op(2,e)))<10
             then
                if op(2,e)<0 then
                    sgmllist := sgmllist,`<fr>`,`<nu>`,1,`</nu>`,
                                `<de>`,`sgml/print`(1/e), `</de>`, `</fr>` ;
                else 
	          sgmllist := sgmllist, `<rad>` ;
	          sgmllist := sgmllist, `<rcd>`, 
			              `sgml/print`(op(1,e)), `</rcd>` ;
                  if op(2,e)<>1/2 then
                      sgmllist := sgmllist ,`<rdx>`, 1/op(2,e), `</rdx>`
                  fi ;
                  sgmllist := sgmllist, `</rad>` ;
                fi ;
 
        else    if type(op(1,e),{'negative','fraction','series','`+`','`*`'})
                  then sgmllist := sgmllist, `sgml/sgml/prinpar`( op(1,e) );
                  else sgmllist := sgmllist, `sgml/print`(op(1,e))
                fi;
                sgmllist := sgmllist,`<sup>`, `sgml/print`(op(2,e)), `</sup>`
        fi;
        RETURN(sgmllist) ;
end:

##################################################################################   
`sgml/sgml/range` := proc(e)
local `{`, `&ldots;`, `}` ;
   option `Copyright 1992 by the University of Waterloo`;
   `sgml/print`(op(1,e)), `&ldots;`, `sgml/print`(op(2,e))
end :

##################################################################################   
`sgml/sgml/commalist` := proc(List,commaitems,Begin,End)
local B,C,E,i,sgmllist ;
 
    option `Copyright 1992 by the University of Waterloo`;
    if nargs>1 then C := commaitems else C := NULL fi ;
    if nargs>2 then B := Begin      else B := NULL fi ;
    if nargs>3 then E := End        else E := NULL fi ;
 
    if C<>NULL and type(C,'list') then C := op(C) fi ;
    if B<>NULL and type(B,'list') then B := op(B) fi ;
    if E<>NULL and type(E,'list') then E := op(E) fi ;
 
    if nops(List)=0
      then sgmllist := B, E
      else
        sgmllist := B, `sgml/print`(List[1]) ;
        for i from 2 to nops(List) do
                sgmllist := sgmllist, C, `sgml/print`(List[i]) ;
        od;
        sgmllist := sgmllist, E ;
    fi ;
    RETURN(sgmllist) ;
end :

##################################################################################   
`sgml/sgml/angleoperator` := proc(e)
local sgmllist, x ;
        option `Copyright 1992 by the University of Waterloo`;
        sgmllist := `<fen lp="ang"> `,
                   `sgml/print`( e( op(1,e) ) ), `&rarrtl;` ;
        x :=  op(1,e) ; # parameters
        if nops([x]) > 0 then
                sgmllist := sgmllist , `sgml/print`( x );
                fi;
        x := op(2,e);   # local vars
        if nops([x]) > 0 then
                sgmllist := sgmllist, `<rf>`,`|`, `</rf>`, `sgml/print`( x ) ;
                fi;
        sgmllist := sgmllist, `<rp post="ang"></fen>`;
        RETURN(sgmllist) ;
end :
 
##################################################################################  
`sgml/sgml/arrowoperator` := proc(e)
local sgmllist, x ;
        option `Copyright 1992 by the University of Waterloo`;
        x :=  op(1,e); # parameters
        if nops([x]) = 0 then sgmllist := `( `,` )`;
        elif nops([x]) = 1 then sgmllist := `sgml/print`( x );
        else sgmllist := `( `,`sgml/print`( x ),` )`;
        fi;
        sgmllist := sgmllist,`&rarrtl;`;
        x := op(2,e);   # local vars
        if nops([x]) > 0 then
                sgmllist := sgmllist,` local `,`sgml/print`( x ),`&amp; `;
                fi;
        sgmllist := sgmllist, `sgml/print`( e( op(1,e) ) );
        RETURN(sgmllist) ;
end :

##################################################################################   
`sgml/sgml/series` := subs(
 
'PRINTTERM' = proc (c,e,v)
local `<hsp sp="0.167">`, sgmllist ;
    if e=0 then
        `sgml/print`(c)
       elif c=1 or c=-1 then
        `sgml/print`(c*v^e) # Does series(x^2,x,3)^4 cause problems?
                             # dgc 28 Sep 1989.
       else
        if type(c,'`sgml/+`') then
            sgmllist := `sgml/prinpar`(c)
           else
            sgmllist := `sgml/print`(c)
        fi;
        if type(c,'`sgml/sf`') then
            sgmllist := sgmllist, `<hsp sp="0.167">` ;
        fi;
        if e=1 and type(v,'`sgml/+`') then
        # N.B. We know that c<>1 here
            sgmllist := sgmllist, `sgml/prinpar`(v)
           else
            sgmllist := sgmllist, `sgml/print`(v^e)
        fi;
        sgmllist
    fi;
 end, # PRINTTERM

##################################################################################   
proc(e) # `sgml/sgml/series`
local i, v, `<fen lp="par">`, `<rp post="par"></fen>`, `+`, tx, sgmllist, nterms, ispoly, ntrueterms;
 
    v := op(0,e) ;
    nterms := iquo(nops(e),2) ;
    ispoly := evalb( 'O(1)'<> op(nops(e)-1,e) ) ;
    if ispoly then
        ntrueterms := nterms ;
       else
        ntrueterms := nterms - 1;
    fi;
    if ntrueterms > 0 then
        sgmllist := PRINTTERM( op(1,e), op(2,e), v)
       else
        sgmllist := NULL ;
    fi;
    for i from 2 to ntrueterms do
        tx := PRINTTERM( op(2*i-1,e), op(2*i,e), v) ;
        if ``.(tx[1]) = ``.`-` then
            sgmllist := sgmllist, tx ;
           else
            sgmllist := sgmllist, `+`, tx ;
        fi;
    od;
    if not ispoly then
        tx := Copy('`O`'), `sgml/prinpar`(v^op(nops(e),e)) ;
        if nterms > 1 then
            sgmllist := sgmllist, `+`
        fi;
        sgmllist := sgmllist, tx ;
    fi;
    sgmllist
end): # `sgml/sgml/series`

##################################################################################    
`sgml/sgml/+` := proc(e)
local i, ii, sgmllist , `+`,`-` ;
 
    option `Copyright 1992 by the University of Waterloo`;
 
#  The `if` test commented out 12 May 1991 by dgc because of user
#  complaints about the re-ordering of terms.  If this breaks
#  something, or too many users want it the old way, uncomment the if
#  test, but be sure to modify `type/sgml/isneg` appropriately.
#
#-  if    type(op(1,e),'`sgml/isneg`') and
#-    not type(op(2,e),'`sgml/isneg`') then
#-      sgmllist := `sgml/print`(op(2,e)), `-`, `sgml/print`(-op(1,e)) ;
#-      ii := 3
#-  else
         sgmllist := `sgml/print`(op(1,e)) ;
         ii := 2 ;
#-  fi;
 
    for i from ii to nops(e) do
            if not type( op(i,e), '`sgml/isneg`' ) then
                sgmllist := sgmllist, `+`, `sgml/print`(op(i,e)) ;
            else
                sgmllist := sgmllist, `-`, `sgml/print`(-op(i,e))
            fi;
            od;
   RETURN(sgmllist) ;
end :

##################################################################################   
`sgml/sgml/function` := proc(e)
#  There are some outstanding problems here.  First of all, the
#  calling sequence for `sgml/??` names needs to be thought out in
#  more detail.  Part of the problem here is the bad design of
#  readlib, which makes it incredibly painful for people other than
#  members of SCG to define alternate library functions.
#
#  Another problem is that of what to do with indexed names being used
#  as function calls.  At the moment these are caught as a default
#  case of this routine, and the problem is passed along to
#  `./indexed`.  This needs to be changed.  Again SCG needs to make a
#  design decision here.
 
local `<fen lp="par">`, `<rp post="par"></fen>`, `,`, sgmllist, `{`, `}` ;
option `Copyright 1992 by the University of Waterloo`;
 
    if nops(op(0,e))>1 or     #  This test must come first!
                              #  The problem here is expression sequences.
       type(op(0,e),'function') and op(0,op(0,e))='`@`' or
       type(op(0,e),{'`+`','`*`'})  then
 
       # Operators can be complicated.  The above is a test for when
       # bracketing is needed around the operator.  Undoubtedly, this
       # will need to be improved.
       sgmllist := `sgml/prinpar`(op(0,e)) ;
 
    elif type(op(0,e),'string') and assigned(`sgml/`.(op(0,e))) then
         RETURN(`sgml/`.(op(0,e))(op(e))) ;
 
    else sgmllist := `sgml/print`(op(0,e))
    fi ;
 
   sgmllist, `sgml/sgml/commalist`([op(e)],`,`,`<fen lp="par">`,`<rp post="par"></fen>`) ;
end :

##################################################################################   
#
#       Special function printing
#
`sgml/exp` := proc()
local e, `<sup>`, `</sup>` ;
                option `Copyright 1992 by the University of Waterloo`;
                RETURN( e,`<sup>`, `sgml/print`(args), `</sup>` ) end:
 
`sgml/log2` := proc()
local `<lim align="r">`,`<op>`,`<rf>log</fr>`,`<ll>`,`</ll>`,`</lim><fen lp="par">`,`<rp post="par">`,`</fen>`,`</op>`;
                option `Copyright 1992 by the University of Waterloo`;
		RETURN(`<lim align="r">`,`<op>`,`<rf>log</rf>`,`</op>`,`<ll>`,
		       2, `</ll>`,`</lim><fen lp="par">`, `sgml/print`(args),
		       `<rp post="par">`,`</fen>` ) end:
 
`sgml/log10` := proc()
local `<lim align="r">`,`<op>`,`<rf>log</fr>`,`<ll>`,`</ll>`,`</lim><fen lp="par">`,`<rp post="par">`,`</fen>`,`</op>`;
                option `Copyright 1992 by the University of Waterloo`;
                RETURN(`<lim align="r">`,`<op>`,`<rf>log</fr>`,`</op>`,`<ll>`, 10, `</ll>`,`</lim><fen lp="par">`, `sgml/print`(args), `<rp post="par">`,`</fen>` ) end:
 
`sgml/Log` := proc()
local `<lim align="r">`,`<op>`,`<rf>Log</rf>`,`</op>`,`</lim><fen lp="par">`, 
      `<rp post="par">`,`</fen>`;
                option `Copyright 1992 by the University of Waterloo`;
                RETURN(`<lim align="r">`,`<op>`,`<rf>Log</rf>`,`</op>`,`</lim><fen lp="par">`, `sgml/print`(args), `<rp post="par">`,`</fen>` ) end:
 
`sgml/ln` := proc()
local `<lim align="r">`,`<op>`,`<rf>ln</rf>`,`</op>`,`</lim><fen lp="par">`, 
      `<rp post="par">`,`</fen>`;
                option `Copyright 1992 by the University of Waterloo`;
                RETURN(`<lim align="r">`,`<op>`,`<rf>ln</rf>`,`</op>`,`</lim><fen lp="par">`, `sgml/print`(args), `<rp post="par">`,`</fen>` ) end:
 
`sgml/D` := proc()
local `<lim align="r">`,`<op>`,`<rf>D</rf>`,`</op>`,`</lim><fen lp="par">`, 
      `<rp post="par">`,`</fen>`;
                option `Copyright 1992 by the University of Waterloo`;
                RETURN(`<lim align="r">`,`<op>`,`<rf>D</rf>`,`</op>`,`</lim><fen lp="par">`, `sgml/print`(args), `<rp post="par">`,`</fen>` ) end:
 
`sgml/@` := proc()
local `<lim align="r">`,`<op>`,`<rf>&j0;</rf>`,`</op>`,`</lim><fen lp="par">`, 
      `<rp post="par">`,`</fen>`;
        option `Copyright 1992 by the University of Waterloo`;
	RETURN(`<lim align="r">`,`<op>`,`<rf>&j0;</rf>`,`</op>`,`</lim><fen lp="par">`, `sgml/print`(args), `<rp post="par">`,`</fen>` ) end:

 
`sgml/@@` := proc()
local `^` ;
        option `Copyright 1992 by the University of Waterloo`;
        RETURN(`sgml/print`(args[1]), `^`,  `sgml/sgml/prinpar`(args[2]) );
end:
 
`sgml/EXPRSEQ` := proc()
local `,`, `[`, `<hsp sp="0.212">`, `]` ;
    option `Copyright 1992 by the University of Waterloo`;
    RETURN(  `sgml/sgml/commalist`([args],
                               [`,`,`[`,2,`]`, `<hsp sp="0.212">`])  )
end:

################################################################################## 
`sgml/abs` := proc(e)
local `<fen lp="vb"> `, `<rp post="vb"></fen> `;
option `Copyright 1992 by the University of Waterloo`;
        RETURN( `<fen lp="vb">`, `sgml/print`(e) , `<rp post="vb"></fen>` );
end:

################################################################################## 
`sgml/int` := proc(Expr,Bounds)
local intstr, x, a, b,     
      `<hsp sp="0.167">`, `<in align="c">`, `<ll>`,`</ll>`, `<ul>`,`</ul>`, 
      d, `</in>`;
option `Copyright 1992 by the University of Waterloo`;
if nargs<>2 then ERROR(`invalid arguments`) fi;
if type(Bounds,'`=`') and type(op(2,Bounds),'range') then
        x := op(1,Bounds) ;
        a := op(1,op(2,Bounds)) ;
        b := op(2,op(2,Bounds)) ;
	intstr := `<in align="c">`, `<ll>`, `sgml/print`(a), `</ll>`,
                            `<ul>`, `sgml/print`(b), `</ul>`,
                            `</in>`
else    x := Bounds ;
        intstr := `<in align="c"></in>` ;
        if type(Expr,'function') and member(op(0,Expr),{`int`,`Int`})
            then intstr := intstr, `<hsp sp="0.167">`$3
            else intstr := intstr, `<hsp sp="0.167">`$1
            fi ;
fi;
intstr := intstr, `sgml/print`(Expr) ;
if type(Expr,'function') and member(op(0,Expr),{`int`,`Int`}) then
    intstr := intstr, `<hsp sp="0.167">`  fi ;
intstr := intstr, `<hsp sp="0.167">`,  Copy('d'), `sgml/print`(x);
RETURN( intstr ) ;
end:
 
`sgml/Int` := `sgml/int`:

################################################################################## 
`sgml/diff` := proc(a,b)
local `<g>6</g>`,d, n, A , ans ,`<fr>`,`</fr>`,`<nu>`,`</nu>`,`<de>`,`</de>`, 
      `<sup>`, `</sup>`,blist,x,y,i,c;
option `Copyright 1992 by the University of Waterloo`;
 
if nargs<2 then ERROR(`invalid arguments`) fi;
n := 1 ;
A := a;
blist := NULL;
c := args[2..nargs];
while type(A,function) and member(op(0,A),{'diff','Diff'}) do
    blist := op(2..nops(A),A),blist ;
    A := op(1,A) ;
    od ;
blist := [reverse( c, blist)];
 
if nops(blist) > 1 then
    ans :=   `<fr>`, `<nu>`, d , `<sup>`,
        `sgml/print`( nops(blist) ), `</sup>`, `</nu>`, `<de>`;
else
    ans := `<fr>`, `<nu>`, d , `</nu>`,`<de>`;
fi;
x := blist[1];
n := 0;
for y in blist do
  if y = x  then  n := n+1; next fi;
  ans := ans , d , `sgml/print`(x^n) ;
  x := y; n := 1;
od;
ans := ans , d, `sgml/print`(y^n), `</de>`, `</fr>` ;
if type(A,{'name','function'})
    then ans := ans, `sgml/print`(A)
    else ans := ans, `sgml/sgml/prinpar` (A)
fi ;
if nops(indets(A,name)) > 1 then
        RETURN( op(subs(d=`<g>6</g>`,[ans])) );
else
        RETURN(ans);
fi;
end:
 
`sgml/Diff` := `sgml/diff`:

################################################################################## 
`sgml/sum` := proc(a,b)
local sumstr, `<sum align="c">`, `<ll>`, `</ll>`, `<ul>`, `</ul>`, `</sum>`;
option `Copyright 1992 by the University of Waterloo`;
if nargs<>2 then ERROR(`invalid arguments`) fi;
sumstr := `<sum align="c">`;
if type(b,'`=`') and type(op(2,b),'range') then
        sumstr := sumstr,`<ll>`, (`sgml/print`(op(1,b))),`=`,
                          (`sgml/print`(op(1,op(2,b)))),`</ll>`;
        sumstr := sumstr,`<ul>`,(`sgml/print`(op(2,op(2,b)))),`</ul>`, `</sum>`
else    sumstr := sumstr,`</sum>`,(`sgml/print`(b)) fi;
RETURN( sumstr,(`sgml/print`(a)) )
end:
`sgml/Sum` := `sgml/sum`:

################################################################################## 
`sgml/product` := proc(a,b)
local productstr,  `<sum align="c">`, `<ll>`, `</ll>`, `<ul>`, `</ul>`, `</pr>`;
option `Copyright 1992 by the University of Waterloo`;
if nargs<>2 then ERROR(`invalid arguments`) fi;
productstr := `<pr align="c">`;
if type(b,'`=`') and type(op(2,b),'range') then
        productstr := productstr,`<ll>`,(`sgml/print`(op(1,b))),`=`,
                          (`sgml/print`(op(1,op(2,b)))),`</ll>`;
        productstr := productstr,`<ul>`,
                         (`sgml/print`(op(2,op(2,b)))),`</ul>`,`</pr>`;
else    productstr := productstr,`</pr>`,(`sgml/print`(b)) fi;
RETURN( productstr,(`sgml/print`(a)) )
end:
`sgml/Product` := `sgml/product`:
 
################################################################################## 
`sgml/limit` := proc(a,b)
local limstr, dir, `&rarr;`, `<lim align="c">`, `<op><rf>lim</rf></op>`, `<sup>`,`+`,`</sup>`, `-`, `<ll>`, `</ll>`, `</lim>`;
option `Copyright 1992 by the University of Waterloo`;
if nargs<2 then ERROR(`invalid arguments`) fi;
limstr := `<lim align="c">`, `<op><rf>lim</rf></op>` ;
dir := NULL;
if nargs > 2 then
        if args[3] = 'right' then dir := `<sup>`,`+`,`</sup>`;
        elif args[3] = 'left' then dir := `<sup>`,`-`,`</sup>`; fi;
fi;
if type(b,'`=`') then
        limstr := limstr,`<ll>`, `sgml/print`(op(1,b)), `&rarr;`,
                              `sgml/print`(op(2,b)), dir,`</ll>`;
else    limstr := limstr, `sgml/print`(b) fi;
RETURN( limstr,(`sgml/print`(a)),`</lim>` )
end:
 
`sgml/Limit` := `sgml/limit`:
 
################################################################################## 
`sgml/factorial` := proc(x)
local factstr, `!`;
option `Copyright 1992 by the University of Waterloo`;
if not type(x,'numeric') and not type(x,'name')
        then factstr := `sgml/sgml/prinpar`(x)
        else factstr := `sgml/print`(x) fi;
RETURN( factstr,`!` )
end:
 
################################################################################## 
`sgml/binomial` := proc( n, r )
local `\\choose `, `{`, `}` ;
option `Copyright 1992 by the University of Waterloo`;
RETURN( `<fen lp="par">`,`<stk><lyr>`, `sgml/print`(n), `</lyr><lyr>`,
         `sgml/print`(r), `</lyr></stk>`, `rp post="par"></fen>` )
end:

################################################################################## 
#
# `type/sgml/+` tests for sum like expressions (used to determine the
# need for bracketing.)
#
`type/sgml/+` := proc (e)
    option `Copyright 1992 by the University of Waterloo`;
    type(e,'`+`') or         # A genuine sum, or
    ( type(e,'series') and   # A series
        ( nops(e)>2 or       #     with more than two terms, or
            ( nops(e)=2 and  #       a constant part of type sum
 
               ( ( op(2,e)=0 and type(op(1,e),{'`sgml/+`',
                                               '`sgml/isneg`'}) ) or
 
                             #       a degree one term with coeff=1
                             #       and the series variable of type sum, or
                             #       a leading minus sign
 
                 ( op(2,e)=1 and op(1,e)=1 and type(op(0,e),'`sgml/+`')) or
                 ( op(1,e)=-1 )
               )
             )
         )
    )
end: # `type/sgml/+`
 
 
################################################################################## 
#
# `type/sgml/sf` tests for expressions that should have space
# inserted after them if they occur in the middle of a product
#
`type/sgml/sf` := proc (e)
     option `Copyright 1992 by the University of Waterloo`;
     type(e,{'integer','float', '`sgml/short_fraction`', '`!`'})
     or type(e,'string') and length(e)>1
     or has([`sgml/print`(e)], '`<rad>`')
     or type(e,'`*`') and type( op(nops(e),e), '`sgml/sf`')
     or type(e,'series') and not type(e,'`sgml/+`')
                         and (   ( op(2,e)=0 and type(op(1,e),'`sgml/sf`') )
                              or ( op(1,e)=1 and op(2,e)=1
                                            and type(op(0,e),'`sgml/sf`') )
                             )
end: # `type/sgml/sf`

################################################################################## 
`type/sgml/short_fraction` := proc (e)
   option `Copyright 1992 by the University of Waterloo`;
   type(e,'fraction') and abs(op(1,e)*op(2,e)) < _SgmlSmallFractionConstant
end: # `type/sgml/short_fraction`
 
 
_SgmlSmallFractionConstant:=50:
macro(chars=chars,Copy=Copy,reverse=reverse) :



 
#save sgml, `sgml/sgml/series`,`type/sgml/isneg`, `type/sgml/+`,\
        `type/sgml/istall`, `type/sgml/sf`,`type/sgml/short_fraction`,\
        `sgml/@@`, `sgml/@`, `sgml/D`, `sgml/Diff`, `sgml/EXPRSEQ`,\
        `sgml/Int`, `sgml/Limit`, `sgml/Log`, `sgml/Sum`, `sgml/abs`,\
        `sgml/binomial`, `sgml/diff`, `sgml/exp`, `sgml/factorial`, \
        `sgml/greek`,`sgml/sgreek`, `sgml/int`, `sgml/isneg`,\
        `sgml/istall`, `sgml/sgml/**`, `sgml/sgml/*`, `sgml/sgml/+`,\
        `sgml/sgml/chars_in_string`, `sgml/sgml/commalist`,\
        `sgml/sgml/copy`, `sgml/sgml/float`, `sgml/sgml/function`,\
        `sgml/sgml/reverse`, _SgmlSmallFractionConstant,\
        `sgml/sgml/angleoperator`, `sgml/sgml/arrowoperator`,\
        `sgml/product`, `sgml/Product`,\
        `sgml/sgml/matrix`, `sgml/sgml/undefined_entry`,\
        `sgml/sgml/numeric`, `sgml/sgml/prinpar`,\
        `sgml/sgml/range`, `sgml/sgml/relation`, `sgml/sgml/string`,\
        `sgml/sgml/table`, `sgml/limit`, `sgml/ln`, `sgml/log10`,\
        `sgml/log2`, `sgml/mathops`, `sgml/prinpar`, `sgml/print`,\
        `sgml/special_names`, `sgml/sum`,\
        `sgml/sgml/indexed`,\
        `sgml.m`;
#quit
