#
## <SHAREFILE=algebra/summation/summation.mpl >
## <DESCRIBE>
## SEE ALSO: summation.ps
##           Programs for computing indefinite and definite sums which
##           solve many examples Maple's sum command fails on.  Includes an
##           extended version of Gosper's algorithm, an extended version of
##           Zeilberger's algorithm, routines for expressing sums as
##           hypergeometric notation, and tools for simplifying factorials,
##           binomials, GAMMAs, powers, and pochhammers.
##                AUTHOR: Wolfram Koepf, koepf@zib-berlin.de
##                AUTHOR: Gregor St\"olting
## </DESCRIBE>
## <UPDATE=R4 >

# file "summation"
#
# Indefinite and Definite Summation 
# An Implementation of Gosper's Algorithm on Indefinite and Zeilberger's
# Algorithm on Definite Summation
# Authors: Wolfram Koepf & Gregor Stoelting
# Based on Tom Koornwinder's program zeilb
# Version 1.0
#
# References:
#
# R. W. Gosper, Jr.:
# Decision procedure for indefinite hypergeometric summation,
# Proc. Nat. Acad. Sci. USA 75 (1978), 40-42.
#
# D. Zeilberger,
# A fast algorithm for proving terminating hypergeometric identities,
# Discrete Math. 80 (1990), 207-211.
#
# T. H. Koornwinder:
# On Zeilberger's algorithm and its q-analogue: a rigorous description.
# J. of Comput. and Appl. Math. 48 (1993), 91-111.
#
# W. Koepf: 
# Algorithms for the indefinite and definite summation.
# Konrad-Zuse-Zentrum Berlin (ZIB), Preprint SC 94-33 (December 1994).
#
#
zborder:=5:
readlib(factors):
readlib(ifactors):

comppol:=proc(f1,g1,x)
local f,g,n,a,b,c,d,j:
# Tests for polynomials f1 and g1 in x if f1(x)=g1(x+j) for some nonnegative
# integer j, while f1 and g1 are not constant in x
f:=collect(f1,x):
g:=collect(g1,x):
n:=degree(f,x):
if n=0 or n<>degree(g,x) then
    RETURN(-1):
fi:
a:=coeff(f,x,n):
b:=coeff(f,x,n-1):
c:=coeff(g,x,n):
d:=coeff(g,x,n-1):
j:=normal((b*c-a*d)/n/a/c):
if not type(j,nonnegint) then
    RETURN(-1):
fi:
if collect(c*f-a*subs(x=x+j,g),x)=0 then
    RETURN(j):
else
    RETURN(-1):
fi:
end: #comppol

maxshift:=proc(r1,r2,x)
local fr1,fr2,i,j,m,ma,n:
ma:=-1:
fr1:=factors(r1):
fr2:=factors(r2):
m:=nops(op(2,fr1)):
n:=nops(op(2,fr2)):
for i from 1 to m do
 for j from 1 to n do
  ma:=max(ma,comppol(op(1,op(i,op(2,fr1))),op(1,op(j,op(2,fr2))),x)):
 od:
od:
RETURN(ma):
end: #maxshift
 
gosper:=proc()
local A, B, BOdeg,  D, E, DE,f, ORDER,  q, S, TOdeg,
a, al,b, co, co1, co2, deg, deg1, deg2, eq, eqa, eqb, 
g, hasa, i, j,  lal,m, ma, n, nnargs,n0, opi,p, r, ra1, ra2,
s, sol, sola, solb, t, tester,time0, va, vb, var, z, k, k1, k2, func,
tmp:
global infolevel:
if not assigned(infolevel['gosper']) then
 infolevel['gosper'] := 1 
fi:
if nargs<2 then ERROR(`not enough arguments in function call`) fi:
func:=args[1]:
if nargs>4 then ERROR(`too many arguments in function call`) fi:
k:=args[2]:
if nargs>3 then 
  k1:=args[3];
  k2:=args[4]; 
elif nargs= 3 then 
   ERROR(`wrong number of arguments in function call`)
fi:
if type(k,`=`) and type(op(2,k),`..`) and type(op(1,k),string) then
 k1:=op(1,op(2,k));
 k2:=op(2,op(2,k));
 k:=op(1,k);
 nnargs:=4
elif type(k,string) then
 nnargs:=nargs
else ERROR(`wrong type of arguments in function call`)
fi:
time0:=time():
userinfo(100,'gosper',`time=`,time()-time0):
al:=argumentlist(func,{});
lal :=nops(al);
for i from 1 to lal do
 opi:=op(i,al);
 if (has(opi,k) and not(type(opi,linear(k)))) then
  ERROR(`algorithm not applicable`)  
 fi;
od; 
tester:=func;
tester:=
  subs({GAMMA=localone,pochhammer=localone,
  factorial=localone,binomial=localone},tester);
tester:=eval(tester);
if not(type_ratpoly(tester,k))  then
 tester:=simplify_power(tester/subs(k=k-1,tester));
 if not(type_ratpoly(tester,k))  then
  ERROR(`algorithm not applicable`)    
 fi:
fi;
DE:=func/subs(k=k-1,func):
DE:=simplify_combinatorial(DE);
userinfo(2,'gosper',`a(`,k,`)/a(`,k,`-1):= `);
userinfo(2,'gosper',DE);
if not type_ratpoly(DE,k) then 
   userinfo(2,'gosper',`is not rational`);
   ERROR(`algorithm not applicable`)
 else 
  userinfo(2,'gosper',`algorithm applicable`)
fi;
 # E:=simplify(denom(DE)):
 # D:=simplify(numer(DE)):
 D:=numer(DE):
 E:=denom(DE):
p:=1:
q:= D:
r:= E:
ma:=maxshift(q,r,k):
j:=0:
while j <= ma do
    g:=gcd(q,subs(k=k+j,r)):
    if g <> 1 then
        q:=normal(q/g):
        r:=normal(r/subs(k=k-j,g)):
        p:=p*product('subs(k=k-p,g)' , 'p'=0..j-1):
    fi:
    j:=j+1:
od:
# r:=expand(r,k):
# q:=expand(q,k):
# p:=expand(p,k):
userinfo(2,'gosper',`p:=`,p):
userinfo(2,'gosper',`q:=`,q):
userinfo(2,'gosper',`r:=`,r):
userinfo(100,'gosper',`time=`,time()-time0):
tmp:=subs(k=k+1,q)+r:
tmp:=expand(tmp,k):
deg1:=degree(tmp,k):
co1:=coeff(tmp,k,deg1):
tmp:=subs(k=k+1,q)-r:
tmp:=expand(tmp,k):
if tmp=0 then
    deg2:=-1:
else
    deg2:=degree(tmp,k):
fi:
co2:=coeff(tmp,k,deg1-1):
if deg1 <= deg2 then
    deg:=degree(p,k)-deg2
else
    co:= -2*co2/co1:
    co:=simplify(co):
    if type(co,nonnegint) then
        deg:=max(co,degree(p,k)-deg1+1)
    else
        deg:=degree(p,k)-deg1+1:
    fi:
fi:
userinfo(2,'gosper',`degreebound:=`,deg):
if deg < 0 then
    ERROR(`no closed form antidifference exists`):
fi:
userinfo(2,'gosper',`solving equations to find f`):
 f:=sum('a[i]*k^i','i'=0..deg):
 tmp:=subs(k=k+1,q)*f-r*subs(k=k-1,f)-p:
 tmp:=expand(tmp):
 deg1:=degree(tmp,k):
 eq:=seq(coeff(tmp,k,i),i=0..deg1):
 var:= seq(a[i],i=0..deg):
 sol:=solve({eq},{var}):
 if sol=NULL then
  ERROR(`no closed form antidifference exists`):
 else 
  userinfo(2,'gosper',`algorithm successful`):
 fi:
 f:=subs(sol,f):
 for i from 0 to deg do 
  f:=subs({a[i]=0},f) :
 od:
 f:=factor(f);
 userinfo(2,'gosper',`f:=`,f):
 S:=subs(k=k+1,q)/p*f*func:
 userinfo(100,'gosper',`total time =`,time()-time0):
 if nnargs=2 then
    RETURN(factor(S))
 else 
    tmp:=traperror(subs(k=k2,S)-subs(k=k1-1,S));
    if tmp=lasterror then tmp:=limit(S,k=k2)-limit(S,k=k1-1) fi;
    RETURN(factor(tmp))
 fi:
end: #gosper

togamma:= proc(term)
 if not(has(term,factorial) or 
        has(term,binomial)  or has(term,pochhammer)) then
   RETURN(term) fi:
 if (op(0,term)=factorial) then RETURN(GAMMA(op(1,term)+1)) fi;
 if (op(0,term)=pochhammer) then
  if op(1,term) + op(2,term) =0 then 
   RETURN((-1)^(op(2,term))*GAMMA(op(2,term)+1))
  else
   RETURN(GAMMA(op(1,term)+op(2,term))/GAMMA(op(1,term))) 
  fi;
 fi;
 if(op(0,term)=binomial) then
  RETURN(GAMMA(op(1,term)+1)/
   (GAMMA(op(1,term)-op(2,term) +1) *GAMMA(op(2,term)+1))) fi;
RETURN(map(togamma,term)):
end: #togamma

gammatofactorial:= proc(term)
 if not(has(term,GAMMA)) then
   RETURN(term) fi:
 if (op(0,term)=GAMMA) then RETURN(factorial(op(1,term)-1)) fi;
 RETURN(map(gammatofactorial,term)):
end: #gammatofactorial

pochhammertobinomial:= proc(term)
local n;
 if nargs>2 then ERROR(`too many arguments in function call`) fi:
 if not(has(term,pochhammer)) then RETURN(term) fi:
 if nargs=2 then 
  n:=args[2];
  if (op(0,term)=pochhammer and op(2,term)=n) then
   RETURN((-1)^n*binomial(-op(1,term),n)) 
  fi;
 else
  if (op(0,term)=pochhammer) then 
   RETURN((-1)^(op(2,term))*binomial(-op(1,term),op(2,term)))
  fi;
 fi;  
 if nargs=2 then RETURN(map(pochhammertobinomial,term,n))
  else RETURN(map(pochhammertobinomial,term)):
 fi;
end: #pochhammertobinomial

simplify_combinatorial:= proc(term1)
# converts binomials, pochhammers, and factorials in term1 into gammas, and
# applies simplify_gamma and simplify_power to the modified term1
local term;
term:=factor(term1);
term:=simplify_gamma(togamma(term));
term:=simplify_power(term);
# if not(type_ratpoly(term,n)) then
#   term:= simplify(term)  
# fi: 
term:=normal(term);
RETURN(factor(term));
end:  #simplify_combinatorial

simplify_gamma:= proc(term1)
# converts all subexpressions
# gamma(xi) -> gamma(xi + m)/((xi)*(xi+1)*...* (xi+m-1))
# where m is the largest integer , so that a subexpression
# gamma(xj) of term1 exists with xj = xi + m.
local term,high,highl,highlength,j;
term:=term1:
if (not(has(term,GAMMA))) then RETURN(term) fi;
highl:= gamma_arglist(term,{});
if highl = {} then RETURN(term) fi:
highlength:= nops(highl):
for j from  1 to highlength do
 high:= op(j,highl):
 term:= gammashift(term,high):
od:
RETURN(normal(term)):
end: #simplify_gamma

gammashift:=proc()
local x,nminusx,j,term,n:
term:=args[1]:
n:=args[2]:
if not(has(term,GAMMA)) then RETURN(term) fi: 
if nops(term)>1 then RETURN(map(gammashift,term,n)) fi:
if not(has(op(1,term),GAMMA)) then
 x:=  op(1,term):
 nminusx:= n-x:
 if not type(nminusx, integer) then RETURN(term) fi: 
 if nminusx>0 then 
  RETURN( GAMMA(n) / product(n-j,j=1..nminusx))
 else
  RETURN( GAMMA(n) * product(x-j,j=1..-nminusx))
 fi:
fi:
RETURN(term):
end: #gammashift

gamma_arglist:=proc()
local no,i,op1,lhighl,term,highl:
term:=args[1];
highl:=args[2];
if not(has(term,GAMMA)) then RETURN(highl) fi: 
no:=nops(term):
if no>1 then
 for i from 1 to no do
  highl:=gamma_arglist(op(i,term),highl)
 od:
else 
 op1:=op(1,term);
 if not(has(op1,GAMMA)) then
  if highl={} then RETURN({op1}) fi:
  lhighl:=nops(highl):
  for i from 1 to lhighl do
   if type(op1-op(i,highl),integer) then RETURN(highl) fi:
  od:
  highl:=highl union {op1}:
 fi  #not(has(op1,GAMMA))
fi: #no>1
RETURN(highl):
end:


simplify_power:= proc(term1)
local term,high,highl,highlength,j;
term:=term1:
term:=integersub(term);
if not(hastype(term,`^`) or has(term,exp)) then RETURN(term) fi;
term:=map(simplify_power,term);
highl:= powerlist(term,{});
if highl = {} then RETURN(term) fi:
highlength:= nops(highl):
for j from  1 to highlength do
 high:= op(j,highl):
 term:= powershift(term,high):
od:
RETURN(normal(term)):
end:

powershift:=proc()
local nlist,lnlist,base,nbase,x,nminusx,j,term,n,lx:
term:=args[1]:
nlist:=args[2]:
if not(hastype(term,`^`) or has(term,exp)) then RETURN(term) fi:
if (op(0,term) = exp) then term:= E^(op(1,term)) fi; 
if op(0,term)= `^` then
 x:=  expand(op(2,term)):
 if (x=-1) then RETURN(map(powershift,term,nlist)) fi:
 if type(x,`+`) then
  lx:=nops(x);
  RETURN(product(powershift(op(1,term)^op(j,x),nlist),'j'=1..lx))
 fi:
 base:=op(1,term):
 lnlist:=nops(nlist);
 for j from 1 to lnlist do
  n:=op(j,nlist);
  nminusx:= n-x:
  if type(nminusx, integer) then
     RETURN(base^n/base^nminusx)
  fi
 od
else 
 RETURN(factor(expand(map(powershift,term,nlist))))
fi:
RETURN(term):
end:
 
localpower:=proc(a,b)
RETURN(a^b):
end:

powerlist:=proc()
local no,i,op1,op2,lifa,lhighl,hop1,hop2,j,opifa,term,highl,
notyet,hopi,hopi1,lhopi,hopj,l,lx:
term:=args[1];
highl:=args[2];
if not (hastype(term,`^`) or has(term,exp)) then RETURN(highl) fi; 
if evalb(op(0,term) = exp) then term := E^op(1,term) fi;
if evalb(op(0,term) = `^`) then
 op2:=expand(op(2,term));
 if type(op2,integer) then RETURN( powerlist(op(1,term),highl)) fi;
 if type(op2,`+`) then 
  lx:=nops(op2);
  for j from 1 to lx do
   highl:= powerlist(op(1,term)^op(j,op2),highl)
  od
 else
  lhighl:=nops(highl);
  if (lhighl=0) then RETURN({op2}) fi;
  for j from 1 to lhighl do
   if type(op2-op(j,highl),integer) then RETURN(highl)  fi
  od:
  RETURN(highl union {op2})  
 fi
else
 l:=nops(term);
 for j from 1 to l do
  highl:= powerlist(op(j,term),highl)
 od
fi;
RETURN (highl):
end:

integersub:=proc(term)
local  expo,l,i,pf,d,j,ifa,ifa2,opi,op2,base;
if  not (hastype(term,`^`)) then RETURN(term) fi;  
if evalb(op(0,term) = `^`) then
 base:= op(1,term);
 if not type(base,integer) then RETURN(term) fi; 
 expo:=op(2,term);
 if op(1,expo) = -1 then RETURN(1/integersub(base^(-expo))) fi;  
 ifa:= ifactors(base);
 op2:=op(2,ifa);
 if op2 = [] then RETURN(term) fi; 
 ifa2:= op(1,ifa);
 l:=nops(op2);
 for i from 1 to l do
  opi:=op(i,op2);
   pf:=op(1,opi);
   d:=op(2,opi);
   for j from 1 to d do
    ifa2:= [op(ifa2),pf] od;
 od;
 RETURN(convert(map(localpower,ifa2,expo),`*`))
else
 RETURN (map(integersub,term));
fi;
end:

type_ratpoly:= proc(term,n)
RETURN(type(denom(term),polynom(anything, n)) and
       type(numer(term),polynom(anything, n)));
end:

sumrecursion:=proc()
local result1,Ank, mini,maxi,Ank1,An1k,BC,DE,B,C, D, E, ORDER,  S, 
a, al,b, co, co1, co2, deg, deg1, deg2, eq, eqa, eqb, 
g, hasa, i, j, k, lal,m, ma, n, n0, opi,p, r, ra1, ra2,
s, sol, sola, solb, t, tester,tester2,time0, va, vb, var, z:
global infolevel:
if not assigned(infolevel['sumrecursion']) then
 infolevel['sumrecursion'] := 1
fi:
time0:=time():
if nargs>4 then
    ERROR(`too many arguments in function call`):
fi:
if nargs<3 then
    ERROR(`too few arguments in function call`):
fi:
time0:=time():
Ank:=args[1];
k:=args[2]:
n:=args[3]:
al:=argumentlist(Ank,{});
lal :=nops(al);
for i from 1 to lal do
 opi:=op(i,al);
 if (has(opi,k) and not(type(opi,linear(k)))) or
  (has(opi,n) and not(type(opi,linear(n)))) then
  ERROR(`algorithm not applicable`)
 fi;
od;
tester:=
  subs({GAMMA=localone,pochhammer=localone,
  factorial=localone,binomial=localone},Ank);
tester:=eval(tester);
if tester <> 0 then 
 tester:=simplify_power(tester/subs(k=k-1,tester));
 tester2:=simplify_power(tester/subs(n=n-1,tester));
fi;
if not(type_ratpoly(tester,k)) or  not(type_ratpoly(tester2,n)) then
ERROR(`algorithm not applicable`)
fi;
if nargs=3 then
   mini:=1: maxi:=zborder
else
   mini:=args[4]; maxi:=mini
fi:
BC:=traperror(simplify_combinatorial(Ank/subs(n=n-1,Ank)));
if BC = lasterror then 
 userinfo(2,'sumrecursion',`input function equals 0`);
 RETURN(summ(n) -summ(n-1)) 
fi;
# B:= simplify(numer(BC));
# C:= simplify(denom(BC));
B:= numer(BC);
C:= denom(BC);
userinfo(2,'sumrecursion',`F(`,n,`,`,k,`)/F(`,n,`-1,`,k,`):= `);
userinfo(2,'sumrecursion',BC);
if not(type_ratpoly(BC,n)) then
  userinfo(2,'sumrecursion',`is not rational`):
  if nargs = 3 then
   userinfo(2,'sumrecursion',`Koepf extension of Zeilberger algorithm applied`):
   RETURN(extended_sumrecursion(Ank,k,n))
  else
   ERROR(`Zeilberger algorithm not applicable`)
  fi;
fi:
DE:=traperror(simplify_combinatorial(Ank/subs(k=k-1,Ank)));
if DE = lasterror then
 userinfo(2,'sumrecursion',`input function equals 0`);
 RETURN(summ(n) -summ(n-1))
fi;
# D:= simplify(numer(DE));
# E:= simplify(denom(DE));
D:= numer(DE);
E:= denom(DE);
userinfo(2,'sumrecursion',`F(`,n,`,`,k,`)/F(`,n,`,`,k,`-1):= `);
userinfo(2,'sumrecursion',DE);
if not type_ratpoly(DE,k) then
 userinfo(2,'sumrecursion',`is not rational`):
 userinfo(2,'sumrecursion',`Koepf extension of Zeilberger algorithm applied`):
 RETURN(extended_sumrecursion(Ank,k,n))
fi:
userinfo(2,'sumrecursion',`Zeilberger algorithm applicable`);
result1:=-1;
for j from  mini to maxi  do
 if (result1 = -1 ) then
   result1:=sumrecursion2(Ank,B,C,D,E,k,n,j,time0)
 else 
   RETURN(result1);
 fi
od;
if result1 = -1 then 
  ERROR(`no recursion of order:=`.maxi.` found; enlarge zborder`);
fi:
userinfo(100,'sumrecursion',`total time=`,time()-time0):
RETURN(result1);
end:

sumrecursion2:=proc(Ank,B,C,D,E,k,n,ORDER,time0)
local recursion,f,P, P0, P1, Q, R, S, 
a, b, co, co1, co2, deg, deg1, deg2, eq, eqa, eqb,
g, hasa, i, j, m, ma,  n0, p, r, ra1, ra2,
s, sol, t, va, vb, var, z,result1,
tmp:
P0:=product('subs(n=n-i,B)','i'=0..ORDER-1):
for j from 1 to ORDER do
    P0:=P0+b[j]*product('subs(n=n-i,C)','i'=0..j-1)*
        product('subs(n=n-i,B)','i'=j..ORDER-1):
od:
userinfo(2,'sumrecursion',
   `applying Zeilberger algorithm for order:=`,ORDER);
Q:=D*product('subs(n=n-i,k=k-1,B)','i'=0..ORDER-1)/
    E/product('subs(n=n-i,B)','i'=0..ORDER-1):
Q:=normal(Q):
R:=denom(Q):
Q:=numer(Q):
ma:=maxshift(Q,R,k):
j:=0:
P1:=1:
while j <= ma do
    g:=gcd(Q,subs(k=k+j,R)):
    if g <> 1 then
        Q:=normal(Q/g):
        R:=normal(R/subs(k=k-j,g)):
        P1:=P1*product('subs(k=k-p,g)' , 'p'=0..j-1):
    fi:
    j:=j+1:
od:
P:=expand(P0*P1,k):
userinfo(2,'sumrecursion',`p:=`,P): 
userinfo(2,'sumrecursion',`q:=`,Q):
userinfo(2,'sumrecursion',`r:=`,R):
userinfo(2,'sumrecursion',`computing maximal degree of f`):
tmp:=subs(k=k+1,Q)+R:
tmp:=expand(tmp,k):
deg1:=degree(tmp,k):
co1:=coeff(tmp,k,deg1):
tmp:=subs(k=k+1,Q)-R:
tmp:=expand(tmp,k):
if tmp=0 then
    deg2:=-1:
else
    deg2:=degree(tmp,k):
fi:
co2:=coeff(tmp,k,deg1-1):
if deg1 <= deg2 then
    deg:=degree(P,k)-deg2
else
    co:= -2*co2/co1:
    co:=simplify(co):
    if type(co,nonnegint) then
        deg:=max(co,degree(P,k)-deg1+1)
    else
        deg:=degree(P,k)-deg1+1:
    fi:
fi:
if deg < 0 then
    RETURN(-1):
fi:
userinfo(2,'sumrecursion',`degreebound:=`,deg);
userinfo(100,'sumrecursion',`time=`,time()-time0):
userinfo(2,'sumrecursion',`solving equations to find f and p`):
f:=sum('a[i]*k^i','i'=0..deg):
tmp:=subs(k=k+1,Q)*f-R*subs(k=k-1,f)-P:
tmp:=expand(tmp):
deg1:=degree(tmp,k):
eq:=seq(coeff(tmp,k,i),i=0..deg1):
va := [seq(a[i],i=0..deg)]:
vb := [seq(b[i],i=1..ORDER)]:
# lprt(`va := `,va);
# lprt(`vb := `,vb);
var:={op(vb),op(va)}:
sol:=solve({eq},var):
if (sol=NULL) then
    RETURN(-1):
fi:
va:=subs(sol,va):
vb:=subs(sol,vb):
for i from 0 to deg do
  if has(va,a[i]) then 
   va:=subs({a[i]=0},va):
   vb:=subs({a[i]=0},vb):
  fi
od:
for i from 0 to ORDER do
  if has(vb,b[i]) then 
  va:=subs({b[i]=0},va):
  vb:=subs({b[i]=0},vb):
 fi
od:
sol:={seq(a[i]=op(i+1,va),i=0..deg),seq(b[i]=op(i,vb),i=1..ORDER)};
f:=subs(sol,f):
P:=subs(sol,P):
Q:=subs(sol,Q);
userinfo(2,'sumrecursion',`f:=`,f);
userinfo(100,'sumrecursion',`time=`,time()-time0):
recursion:= summ(n) +
 sum(b['i']*summ(n-'i'),'i'=1..ORDER);
recursion:=subs(sol,recursion):
# for i from 0 to deg do
#   recursion:=subs({a[i]=0},recursion) :
# od:
# for i from 0 to ORDER do
#   recursion:=subs({b[i]=0},recursion):
# od:
n0:=-1;
for j from 1 to nops(va) do
    n0:=max(testnonnegintroots(denom(normal(va[j])),n),n0):
od:
for j from 1 to nops(vb) do
    n0:=max(testnonnegintroots(denom(normal(vb[j])),n),n0):
od:
n0:=max(testnonnegintroots(denom(normal(subs(k=n+1,Q))),n),n0):
n0:=max(testnonnegintroots(numer(normal(subs(k=n+1,P))),n),n0):
if (n0 > -1)  then
 userinfo(0,'sumrecursion',`recursion valid for`,n,`>`,n0)
fi:
recursion:=normal(recursion);
recursion:=numer(recursion);
for i from 0 to ORDER do
 recursion:=collect(recursion,summ(n-i));
od: 
userinfo(100,'sumrecursion',`total time=`,time()-time0):
RETURN(map(factor,recursion)):
end: #sumrecursion2

testnonnegintroots:=proc(p,k) local f,fp,i,k0:
k0:=-1:
fp:=traperror(factors(p));
if fp = lasterror then RETURN(-1) fi;
for i from 1 to nops(op(2,fp)) do
    f:=collect(op(1,op(i,op(2,fp))),k):
    if degree(f,k)=1 and coeff(f,k)=1 and type(-subs(k=0,f),nonnegint)
then
        k0:=max(k0,-subs(k=0,f))
    fi:
od:
RETURN(k0):
end:

hyperrecursion:=proc()
local i,time0,k,LOWER,UPPER,LOWERdeg,UPPERdeg,z,n,mini, maxi,
 r,s,result1,B,C,D,E,BC,DE, upl,lol,func,j,isr,tester;
 global infolevel:
if not assigned(infolevel['sumrecursion']) then
 infolevel['sumrecursion'] := 1
fi:
if not assigned(infolevel['hyperrecursion']) then
 infolevel['hyperrecursion'] := infolevel['sumrecursion']
fi:
isr:=infolevel['sumrecursion'] ;
infolevel['sumrecursion'] := infolevel['hyperrecursion'];
result1:=-1;
if nargs>5 then
    ERROR(`too many arguments in function call`):
fi:
if nargs<4 then
    ERROR(`too few arguments in function call`):
fi:
time0:=time():
UPPER:=args[1];
if has(map(type,UPPER,integer),true) then
    ERROR(`some upper index is integer`):
fi:
LOWER:=args[2]:
if has(LOWER,0) or has(map(type,LOWER,negint),true) then
    ERROR(`some lower index is nonpositive integer`):
fi:
z:=args[3]:
n:=args[4]:
if has(z,n) then
    ERROR(`third argument depends on `.n):
fi:
if nargs=4 then
   mini:=1: maxi:=zborder
else
   mini:=args[5]: maxi:=mini
fi:
r:=nops(UPPER);
s:=nops(LOWER);
if op(0,UPPER) = set then UPPER:= convert(UPPER,list) fi;
for i from 1 to r do
 if has(UPPER[i],n) and not type(UPPER[i],linear(n)) then
  ERROR(`some upper index is not linear in `.n)
 fi
od;
if op(0,LOWER) = set then LOWER:= convert(LOWER,list) fi;
for i from 1 to s do
 if has(LOWER[i],n) and not type(LOWER[i],linear(n)) then
  ERROR(`some lower index is not linear in `.n)
 fi
od;
UPPERdeg:=map(coeff,UPPER,n);
LOWERdeg:=map(coeff,LOWER,n);
BC:=product('pochhammer(UPPER[t]-UPPERdeg[t]+k,UPPERdeg[t])/
        pochhammer(UPPER[t]-UPPERdeg[t],UPPERdeg[t])','t'=1..r)*
        product('pochhammer(LOWER[t]-LOWERdeg[t],LOWERdeg[t])/
        pochhammer(LOWER[t]-LOWERdeg[t]+k,LOWERdeg[t])','t'=1..s):
BC:=simplify_combinatorial(BC):
userinfo(2,'sumrecursion',`F(`,n,`,`,k,`)/F(`,n,`-1,`,k,`):= `);
userinfo(2,'sumrecursion',BC);
if not(type_ratpoly(BC,n)) then
   infolevel['sumrecursion']:=isr;
   userinfo(2,'sumrecursion',`is not rational`):
   userinfo(2,'sumrecursion',`Koepf extension of Zeilberger algorithm applied`):
   RETURN(extended_hyperrecursion(args))
fi:
# B:=simplify(numer(BC)):
# C:=simplify(denom(BC)):
B:=numer(BC):
C:=denom(BC):
DE:=product('UPPER[t]+k-1','t'=1..r)/
    product('LOWER[t]+k-1','t'=1..s)/k*z:
DE:=normal(DE):
userinfo(2,'sumrecursion',`F(`,n,`,`,k,`)/F(`,n,`,`,k,`-1):= `);
userinfo(2,'sumrecursion',DE);
userinfo(2,'sumrecursion',`Zeilberger algorithm applicable`);
# D:=simplify(numer(DE)):
# E:=simplify(denom(DE)):
D:=numer(DE);
E:=denom(DE);
result1:=-1;
func:=hyperterm(UPPER,LOWER,z,k);
for j from  mini to maxi  do
 if (result1 = -1 ) then
   result1:=sumrecursion2(func,B,C,D,E,k,n,j,time0)
 else
   infolevel['sumrecursion']:=isr;
   RETURN(result1);
 fi
od;
if result1 = -1 then
  infolevel['sumrecursion']:=isr;
  ERROR(`no recursion of order:=`.maxi.` found; enlarge zborder`);
fi:
RETURN(result1);
end: #hyperrecursion

hyperterm:=proc(upper,lower,z,k)
# converts the representation of a hypergeomeric term
local lu,ll,tmp;
lu:=nops(upper):
ll:=nops(lower):
tmp:= 
product(pochhammer(op(j,upper),k),'j'=1..lu)*z^k/
(product(pochhammer(op(j,lower),k),'j'=1..ll)*factorial(k));
tmp:=eval(subs(pochhammer=poch,tmp));
tmp:=subs(poch=pochhammer,tmp);
tmp:=combine_basis(tmp);
RETURN(tmp);
end: #hyperterm

extended_gosper:=proc()
 local summand,k, al,k1,k2,l,nnargs,newsummand,tk,tmpterm,tmp;
 if nargs < 2 then 
  ERROR (`too few arguments in function call`)
 fi;
 if nargs > 4 then
  ERROR (`too many arguments in function call`)
 fi;
 summand:=args[1];
 k:=args[2];
 if type(k,`=`) and type(op(2,k),`..`) and type(op(1,k),string) then
  k1:=op(1,op(2,k));
  k2:=op(2,op(2,k));
  k:=op(1,k);
  nnargs:=4
 elif type(k,string) then
  nnargs:=nargs
 else ERROR(`wrong type of arguments in function call`)
 fi:
 if nargs=3 then 
  l:=args[3]
 else
  al:=argumentlist(summand,{});
  l:= find_substitution(al,k);
  userinfo(2,'gosper',`Koepf extension of Gosper algorithm applied`):
  userinfo(2,'gosper',`linearizing integer with respect to`,
  k,` is `,l);
 fi;
 if nargs>3 then
  k1:=args[3];
  k2:=args[4];
 fi:
 newsummand:=subs(k=k*l,summand);
 userinfo(2,'gosper',`applying Gosper algorithm to a(`,k,`):=`):
 userinfo(2,'gosper',newsummand);
 tk:=subs(k=k/l,gosper(newsummand,k));
 if nnargs = 3 then
  RETURN(tk)
 elif nnargs = 2 then 
  RETURN(sum(subs(k=k-j,tk),'j'=0..l-1))
 else
  tk:= sum(subs(k=k-j,tk),'j'=0..l-1);
  tmp:=traperror(subs(k=k2,tk)-subs(k=k1-1,tk));
  if tmp=lasterror then tmp:=limit(tk,k=k2)-limit(tk,k=k1-1) fi;
  RETURN(factor(tmp))
 fi;
end: #extended_gosper

extended_sumrecursion:=proc(summand,k,n)
local tmpterm,rule,m,l,i,tl,newsummand,al,BC,B,C,DE,D,E;
al:=argumentlist(summand,{});
 if nargs > 5 then
  ERROR (`too many arguments in function call`)
 fi;
if nargs>3 then 
 m:=args[4] 
else 
 m:= find_substitution(al,n);
 userinfo(2,'sumrecursion',`linearizing integer with respect to`,
 n,` is `,m);
fi;
if nargs>4 then 
 l:=args[5] 
else 
 l:= find_substitution(al,k);
 userinfo(2,'sumrecursion',`linearizing integer with respect to`,
  k,` is `,l);
fi;
if evalb(l=1) and evalb(m=1) then
 DE:= simplify_combinatorial(summand/subs(k=k-1,summand));
# D:= numer(DE);
# E:= denom(DE);
 BC:= simplify_combinatorial(summand/subs(n=n-1,summand));
# B:= numer(BC);
# C:= denom(BC);
 if not(type_ratpoly(DE,k) and type_ratpoly(BC,n)) then
  userinfo(2,'sumrecursion',`F(`,n,`,`,k,`)/F(`,n,`,`,k,`-1):= `);
  userinfo(2,'sumrecursion',DE);
  userinfo(2,'sumrecursion',`is not rational`):
  ERROR(`Zeilberger algorithm not applicable`)
 fi:
fi:
userinfo(2,'sumrecursion',`applying Zeilberger algorithm to`):
userinfo(2,'sumrecursion',`F(`,n,`,`,k,`):=`):
newsummand:=subs(k=k*l,n=n*m,summand);
userinfo(2,'sumrecursion',newsummand);
tmpterm:=sumrecursion(newsummand,k,n);
tmpterm:=subs(n=n/m,tmpterm);
i :=0 ;
while has(tmpterm,summ) do
 tmpterm:=subs(summ((n-i)/m)=SUMM(n-i),tmpterm);
 i:=i+1;
od;
tmpterm:=subs(SUMM=summ,tmpterm);
RETURN(tmpterm);
end: #extended_sumrecursion
 
extended_hyperrecursion:=proc(upper,lower,x,n)
local summand,k,m, isr,tmp,al;
global infolevel:
 if nargs > 5 then
  ERROR (`too many arguments in function call`)
 fi;
if not assigned(infolevel['sumrecursion']) then
 infolevel['sumrecursion'] := 1
fi:
if not assigned(infolevel['hyperrecursion']) then
 infolevel['hyperrecursion'] := infolevel['sumrecursion']
fi:
isr:=infolevel['sumrecursion'] ;
summand:=hyperterm(upper,lower,x,k);
infolevel['sumrecursion'] := infolevel['hyperrecursion'];
if nargs>4 then 
 m:=args[5] 
else 
 al:=argumentlist(summand,{});
 m:= find_substitution(al,n) ;
 userinfo(2,'sumrecursion',`linearizing integer with respect to`,
   n,` is `,m);
fi;
tmp:=extended_sumrecursion(summand,k,n,m,1);
infolevel['sumrecursion'] := isr;
RETURN(tmp);
end: #extended_hyperrecursion
 
WZ:=proc(summand,k,n)
local tmp,gos,m,l;
   if nargs>3 then m:=args[4] else m:=1 fi;
   if nargs>4 then l:=args[5] else l:=1 fi;
   tmp:=summand-subs(n=n-m,summand);
   gos:=extended_gosper(tmp,k,l);
   RETURN(simplify_combinatorial(gos/summand))
end: #WZ

argumentlist := proc(term2,list2)
local term1,list1,head1,j,l;
term1 := expand(term2);
list1 := list2;
l := nops(term1);
if l < 1 then RETURN(list1 union {term1}) fi;
if not (hastype(term1,`^`) or has(term1,binomial) or
 has(term1,GAMMA) or
 has(term1,pochhammer) or has(term1,factorial)) then
 RETURN(list1 union {denom(term1),numer(term1)})
fi;
head1 := op(0,term1);
if head1 = `^` then
 list1 := 
  (list1 union argumentlist(op(1,term1),{}) union
   argumentlist(op(2,term1),{}))
else
  if head1 = GAMMA or  head1 = factorial then
    list1 := (list1 union {op(1,term1)})
  else
    if head1 = pochhammer or head1 = binomial then
      list1 := (list1 union {op(2,term1),op(1,term1)})
    else 
      for j to l do  list1 := argumentlist(op(j,term1),list1) od
    fi
  fi
fi;
RETURN(list1)
end:#argumentlist

find_substitution:=proc(term,n)
local listn,m;
listn := map(linearfactor,term,n);
if listn = {} then RETURN(1) fi;
m:= lcml(listn);
RETURN( m);
end:#find_substitution


linearfactor:=proc(term1,n)
local p,co,d;
d:=degree(term1,n);
if (d  = 0) then RETURN(1) fi ;
p:= denom(coeff(term1,n,1));
if d >1 or (not type(p,integer)) then
   ERROR(`Extended Gosper algorithm not applicable`) fi;
RETURN(p);
end:

lcml:=proc(list1)
# finds least common multiple of a list of integers
local p1,l,list2,i;
l:= nops(list1);
p1:= op(1,list1);
if l = 1 then RETURN( p1 ) fi;
if l = 2 then RETURN(ilcm( p1, op(2,list1))) fi;
list2:=[p1];
for i from 2 to l-1 do
list2:= [op(list2),op(i,list1)];
od;
 RETURN(ilcm(p1,lcml(list2)));
end:

Hypersum:=proc(upper,lower,z,n)
local tmp,ihr,n1,n2,lra2,ra1,ra2,recursion,term1,x,lc1,lc2,noone,i,j,
rat,num,den,numfactors,denfactors,lc,numlist,denlist,opj;
global infolevel:
if not assigned(infolevel['hyperrecursion']) then
 infolevel['hyperrecursion'] := 1
fi:
if not assigned(infolevel['Hypersum']) then
 infolevel['Hypersum'] := infolevel['hyperrecursion']
fi:
ihr:=infolevel['hyperrecursion'] ;
infolevel['hyperrecursion'] := infolevel['Hypersum'];
recursion:=hyperrecursion(upper,lower,z,n);
userinfo(2,'hyperrecursion',`recursion:=`,recursion);
if nops(recursion)>2 then ERROR(`no hypergeometric recurrence found`) fi;
# NEW
if recursion=summ(n) then RETURN(0) fi;
n1:=op(1,eval(op(1,recursion)/subs(summ=localone,op(1,recursion))));
n2:=op(1,eval(op(2,recursion)/subs(summ=localone,op(2,recursion))));
i:= max(n-n1,n-n2);
if i>1 then
ERROR(`not yet implemented`);
fi;
recursion:=subs(n=n+i,recursion);
recursion:=normal(solve(recursion, summ(n+i))/summ(n));
rat:=recursion*(n+1);
num:=numer(rat);
den:=denom(rat);
numfactors:=factors(num);
denfactors:=factors(den);
lc:=lcoeff(num,n)/lcoeff(den,n);
numfactors:=op(2,numfactors);
denfactors:=op(2,denfactors);
numlist:=[];
denlist:=[];
for j from 1 to nops(numfactors) do
  for i from 1 to op(2,op(j,numfactors)) do
  opj:= op(1,op(j,numfactors));
   if has(opj,n) then
    numlist:=[op(numlist),-solve(opj,n)];
   fi;
  od;
od;
# tmp:=numberofzeros(numlist);
# if tmp>0 then RETURN(Sumtohyper(subs(n=n-tmp,Ank),n)) fi;
for j from 1 to nops(denfactors) do
  for i from 1 to op(2,op(j,denfactors)) do
  opj:= op(1,op(j,denfactors));
   if has(opj,n) then
    denlist:=[op(denlist),-solve(opj,n)];
   fi
  od;
od;
# tmp:=numberofzeros(denlist);
# if tmp>0 then RETURN(Sumtohyper(subs(k=k+tmp,Ank),k)) fi;
tmp:=Hyperterm(numlist,denlist,lc,n);
tmp:=eval(subs(binomial=binom,tmp));
tmp:=subs(binom=binomial,tmp);
# tmp:=Hyperterm(op(hypersum2(recursion,n)));
# infolevel[hyperrecursion]:=ihr;
# RETURN(tmp);
end: #Hypersum

hypersum:=proc(upper,lower,z,n)
local tmp,ihr,n1,n2,lra2,ra1,ra2,recursion,term1,x,lc1,lc2,noone,i,j,
rat,num,den,numfactors,denfactors,lc,numlist,denlist,opj;
global infolevel:
if not assigned(infolevel['hyperrecursion']) then
 infolevel['hyperrecursion'] := 1
fi:
if not assigned(infolevel['hypersum']) then
 infolevel['hypersum'] := infolevel['hyperrecursion']
fi:
ihr:=infolevel['hyperrecursion'] ;
infolevel['hyperrecursion'] := infolevel['hypersum'];
recursion:=hyperrecursion(upper,lower,z,n);
userinfo(2,'hyperrecursion',`recursion:=`,recursion);
if nops(recursion)>2 then ERROR(`no hypergeometric recurrence found`) fi;
# NEW
if recursion=summ(n) then RETURN(0) fi;
n1:=op(1,eval(op(1,recursion)/subs(summ=localone,op(1,recursion))));
n2:=op(1,eval(op(2,recursion)/subs(summ=localone,op(2,recursion))));
i:= max(n-n1,n-n2);
if i>1 then
ERROR(`not yet implemented`);
fi;
recursion:=subs(n=n+i,recursion);
recursion:=normal(solve(recursion, summ(n+i))/summ(n));
rat:=recursion*(n+1);
num:=numer(rat);
den:=denom(rat);
numfactors:=factors(num);
denfactors:=factors(den);
lc:=lcoeff(num,n)/lcoeff(den,n);
numfactors:=op(2,numfactors);
denfactors:=op(2,denfactors);
numlist:=[];
denlist:=[];
for j from 1 to nops(numfactors) do
  for i from 1 to op(2,op(j,numfactors)) do
  opj:= op(1,op(j,numfactors));
   if has(opj,n) then
    numlist:=[op(numlist),-solve(opj,n)];
   fi;
  od;
od;
# tmp:=numberofzeros(numlist);
# if tmp>0 then RETURN(Sumtohyper(subs(n=n-tmp,Ank),n)) fi;
for j from 1 to nops(denfactors) do
  for i from 1 to op(2,op(j,denfactors)) do
  opj:= op(1,op(j,denfactors));
   if has(opj,n) then
    denlist:=[op(denlist),-solve(opj,n)];
   fi
  od;
od;
# tmp:=numberofzeros(denlist);
# if tmp>0 then RETURN(Sumtohyper(subs(k=k+tmp,Ank),k)) fi;
tmp:=hyperterm(numlist,denlist,lc,n);
tmp:=eval(subs(binomial=binom,tmp));
tmp:=subs(binom=binomial,tmp);
# end NEW
# tmp:=hyperterm(op(hypersum2(recursion,n)));
# infolevel[hyperrecursion]:=ihr;
# RETURN(pochhammertobinomial(tmp,n));
RETURN(tmp);
end: #hypersum

hypersum2:=proc()
local n,recursion,ihr,n1,n2,lra2,ra1,ra2,term1,x,lc1,lc2,noone,i,j;
recursion:=args[1];
n:=args[2];
if recursion=summ(n) then RETURN([[0], [], 1, n]) fi;
n1:=op(1,eval(op(1,recursion)/subs(summ=localone,op(1,recursion))));
n2:=op(1,eval(op(2,recursion)/subs(summ=localone,op(2,recursion))));
i:= max(n-n1,n-n2);
if i>1 then
ERROR(`not yet implemented`);
fi;
recursion:=solve(recursion, summ(n));
if has(recursion,summ(n-i)) then 
 recursion:= normal(recursion/summ(n-i));
fi;
if has(recursion,summ) then  ERROR(`no hypergeometric recurrence found`) fi;
ra1:= collect(numer(recursion),n);
ra2:= collect(denom(recursion),n);
lc1:=lcoeff(ra1,n);
lc2:=lcoeff(ra2,n);
x:= lc1/lc2;
ra1:=factors(ra1/lc1):
ra2:=factors(ra2/lc2):
ra1:=refactors(ra1,n);
ra2:=refactors(ra2,n):
lra2:=nops(ra2);
# noone :=true;
# for i from 1 to lra2 do
#  if op(i,ra2) = 1 then ra2:= subsop(i=NULL,ra2); noone:= false fi;
# od;
# if noone then ra1:=[op(ra1),1] fi:
RETURN([ra1,ra2,x,n]);
end: #hypersum2

combine_basis:=proc(term)
local constterm,l,opi,esign,i,j,oldterm,newterm, base, expol,expo;
if nops(term)<1 then RETURN(term) fi;
newterm:=[];
constterm:=1;
expol:=[];
if type(term,`*`) then  
 l:=nops(term);
 for i from 1 to l do
  opi:=op(i,term);
  if type(opi,`^`) then
   expo:=op(2,opi);
   base:= op(1,opi) ;
   if type(expo,integer) and expo <0  then
    if not type(base,`^`) then
     constterm:= constterm  * opi;
     base:=1;
    else
     expo:=expo * op(2,base);
     base:=op(1,base);
    fi # if not type
   fi; # if expo
   esign:= sign(expo);
   expo:=expo*esign;
   base:= base^(esign);
   oldterm:=newterm;
   for j from 1 to nops(expol) do 
    if expo = op(j,expol) then 
     newterm:= subsop(j= 
     (op(1,op(j,newterm))* base)^op(j,expol),newterm);
    fi
   od: # for j
   if (oldterm = newterm) then 
    newterm:=[op(newterm),  base^expo];
    expol:=[op(expol),expo] 
   fi #(oldterm = newterm) 
  else
   constterm:= constterm * opi;
  fi # type(opi,`^`) then
 od; #for i 
 newterm:=convert(newterm,`*`);
 newterm:=constterm*newterm;
 RETURN(newterm)
else 
 RETURN(term)
fi
end:

refactors:=proc(f,n)
local i,c,d,g,pol,deg,j:
g:=[]:
for i from 1 to nops(op(2,f)) do
    pol:=op(1,op(i,op(2,f))):
    deg:=op(2,op(i,op(2,f))):
    if degree(pol,n)=1 then
        d:=coeff(pol,n,1);
        c:=coeff(pol,n,0)/d+1:
       for j from 1 to deg do g:= [op(g),c] od
    else
        RETURN(FAIL):
    fi:
od:
RETURN(g):
end:

localone:=proc();RETURN(1);end:

poch := proc(n,k) local i;
   if n=1/2 then (2*k)!/4^k/k!
   elif type(n,integer) then
      if type(n,posint) then (n+k-1)!/(n-1)!
      elif type(n,negint) then (-1)^k*(-n)! / (-n-k)!
      else delta(n,k) fi
   elif type(n,rational) and denom(n)=2 then
      if n>0 then poch(2*n-1,2*k)/(4^k* poch((2*n-1)/2,k))
     elif n<0 then (-1)^(-n+1/2)*poch(1/2,-n+1/2) *
                   poch(1/2,k+n-1/ 2)
      fi
   elif has(n, I) then 'procname(args)'
   elif type(k, integer) then product(n+i, i=0..k-1)
   elif type(n, numeric) then 'procname(args)'
   else pochhammer(n,k); 
   fi
end:

sumtohyper:=proc()
local Ank,k,tmp,iSt;
global infolevel;
if not assigned(infolevel['sumtohyper']) then
 infolevel['sumtohyper'] := 1
fi:
if not assigned(infolevel['Sumtohyper']) then
 infolevel['Sumtohyper'] := infolevel['sumtohyper']
fi:
iSt:=infolevel['Sumtohyper'] ;
infolevel['Sumtohyper'] := infolevel['Sumtohyper'];
if nargs>2 then
    ERROR(`too many arguments in function call`):
fi:
if nargs<2 then
    ERROR(`too few arguments in function call`):
fi:
Ank:=args[1];
k:=args[2]:
tmp:= Sumtohyper(Ank,k);
infolevel['Sumtohyper']:=iSt;
tmp:= eval(subs(Hypergeom=hypergeom,tmp)); 
tmp:=eval(subs(pochhammer=poch,tmp));
tmp:=subs(poch=pochhammer,tmp);
# tmp:=pochhammertobinomial(tmp,n);
RETURN(tmp):
end: # sumtohyper

Sumtohyper:=proc(Ank,k)
local #Ank,k,
Ank1,DE,rat,num,den,numfactors,denfactors,lc,l,numlist,
numdegree,denlist,opj,tmp,
dendegree,i,j;
global infolevel;
if not assigned(infolevel['sumtohyper']) then
 infolevel['sumtohyper'] := 1
fi:
# if nargs>2 then
#     ERROR(`too many arguments in function call`):
# fi:
# if nargs<2 then
#     ERROR(`too few arguments in function call`):
# fi:
# Ank:=args[1];
# k:=args[2]:
Ank1:= subs(k=k+1,Ank);
DE:= simplify_combinatorial(Ank1/Ank);
userinfo(2,'sumtohyper',`a(`,k,`+1)/a(`,k,`):= `);
userinfo(2,'sumtohyper',DE);
if not type_ratpoly(DE,k) then
   userinfo(2,'sumtohyper',`is not rational`):
   ERROR(`cannot be converted into hypergeometric form`);
fi:
rat:=DE*(k+1);
num:=numer(rat);
den:=denom(rat);
numfactors:=factors(num);
denfactors:=factors(den);
lc:=lcoeff(num,k)/lcoeff(den,k);
numfactors:=op(2,numfactors);
denfactors:=op(2,denfactors);
numlist:=[];
denlist:=[];
for j from 1 to nops(numfactors) do
  for i from 1 to op(2,op(j,numfactors)) do
  opj:= op(1,op(j,numfactors));
   if has(opj,k) then 
    numlist:=[op(numlist),-solve(opj,k)];
   fi;
  od;
od;
tmp:=numberofzeros(numlist);
if tmp>0 then RETURN(Sumtohyper(subs(k=k-tmp,Ank),k)) fi;
for j from 1 to nops(denfactors) do
  for i from 1 to op(2,op(j,denfactors)) do
  opj:= op(1,op(j,denfactors));
   if has(opj,k) then    
    denlist:=[op(denlist),-solve(opj,k)];
   fi
  od;
od;
tmp:=numberofzeros(denlist);
if tmp>0 then RETURN(Sumtohyper(subs(k=k+tmp,Ank),k)) fi;
# this line is difficult
# tmp:=traperror(subs(k=0,Ank));
# if tmp=lasterror then tmp:=limit(Ank,k=0) fi;
tmp:=limit(Ank,k=0);
tmp:=tmp*Hypergeom(numlist,denlist,lc);
tmp:=eval(subs(binomial=binom,tmp));
tmp:=subs(binom=binomial,tmp);
# tmp:=eval(subs(pochhammer=poch,tmp));
# tmp:=subs(poch=pochhammer,tmp);
RETURN(tmp)
end: # Sumtohyper

numberofzeros:=proc(list)
local j,tmp;
  tmp:=0;
  for j from 1 to nops(list) do
    if op(j,list)=0 then tmp:=tmp+1 fi;
  od;
  RETURN(tmp);
end: # numberofzeros

binom := proc(n,k) local i;
   if type(n,integer) then
      if type(n,posint) then n!/(k!*(n-k)!)
      elif type(n,negint) then ERROR(`negative integer argument`)
      else delta(0,k) fi
   elif type(k, integer) then product(n-i, i=0..k-1)/k!
   else binomial(n,k);
   fi
end:

summation:=proc(f,k,n)
local localhypersum,upper,lower,z,term;
  localhypersum:=proc(upper,lower,z)
  RETURN(hypersum(upper,lower,z,n));
  end:
  term:=Sumtohyper(f,k);
  term:=subs(Hypergeom=localhypersum,term);
  term:=eval(term);
  RETURN(term);
end:

Summation:=proc(f,k,n)
local localhypersum,upper,lower,z,term;
  localhypersum:=proc(upper,lower,z)
  RETURN(Hypersum(upper,lower,z,n));
  end:
  term:=Sumtohyper(f,k);
  term:=subs(Hypergeom=localhypersum,term);
  term:=eval(term);
  RETURN(term);
end:

#save `summation.m`;
#quit
